/**************************************************************************
* 
* Copyright 2005-2011 by Andrey Butok. Freescale Semiconductor, Inc.
*
**********************************************************************/ /*!
*
* @file fnet_flash.h
*
* @author Andrey Butok
*
* @date Jun-2-2011
*
* @version 0.1.3.0
*
* @brief On-chip Flash Module driver API.
*
***************************************************************************/
#ifndef _FNET_FLASH_H_

#define _FNET_FLASH_H_

#include "fnet_config.h" 

/*! @cond */
#if FNET_CFG_FLASH
/*! @endcond */

/*! @addtogroup fnet_cfm 
*
* The Flash driver provides the ability to 
* reprogram the internal Flash memory while an application is running 
* in normal operational mode.@n
* @n
* The on-chip Flash module is a non-volatile memory (NVM) module 
* integrated  with a CPU. The Flash memory is ideal for program and 
* data storage for single-chip applications, allowing field 
* reprogramming without requiring external programming voltage sources.@n
* @n
*
* For Falsh driver example, refer to FNET demo application source code.@n
* @n
* Configuration parameters:
* - @ref FNET_CFG_FLASH 
* - @ref FNET_CFG_CPU_FLASH_ADDRESS
* - @ref FNET_CFG_CPU_FLASH_SIZE
* - @ref FNET_CFG_CPU_FLASH_PAGE_SIZE
*/
/*! @{ */

#if !FNET_CFG_CPU_FLASH
#error Flash Module Driver is not implemented/tested for your platform!
#endif


/***************************************************************************/ /*!
 *
 * @brief    Erases the specified range of the Flash memory.
 *
 * @param flash_addr      Address in the Flash to erase from.
 *
 * @param bytes           Number of bytes to erase in the Flash memory.
 *
 * @see fnet_flash_memcpy()
 *
 ******************************************************************************
 *
 * This function attempt to erase the number of @c bytes bytes beginning 
 * at @c flash_addr.@n
 * It should be noted that the Flash is block oriented when erasing. 
 * It is not possible to erase a few bytes within a page, the whole page will 
 * be erased.
 * The @c flash_addr parameter may be anywhere within the first page to be 
 * erased and @c flash_addr+ @c bytes may be anywhere in the last block to 
 * be erased. @n
 * Erase page size is defined by @ref FNET_CFG_CPU_FLASH_PAGE_SIZE.
 *
 ******************************************************************************/
void fnet_flash_erase( void *flash_addr, unsigned bytes);

/***************************************************************************/ /*!
 *
 * @brief    Writes the specified data to the Flash memory.
 *
 * @param flash_addr      Address in the Flash to write to.
 *
 * @param src             Pointer to the buffer containing the 
 *                        data to write to the Flash memory.
 *
 * @param bytes           Number of bytes contained in the data buffer
 *                        pointed by @c src.
 *
 * @see fnet_flash_erase()
 *
 ******************************************************************************
 *
 * This function copies the number of @c bytes bytes from the location
 * pointed by @c src directly to the Flash memory pointed by @c flash_addr.
 *
 ******************************************************************************/
void fnet_flash_memcpy( void *flash_addr, const void *src, unsigned bytes  );

/*! @} */


/*! @cond */
#endif
/*! @endcond */

#endif