/* main.c - FlexCAN example for MPC5744P */
/* Description:  Measures eTimer pulse/period measurement */
/* Rev 1.0 Oct 27 2017 D Chung - production version */
/* Copyright NXP Semiconductor, Inc 2016 All rights reserved. */

/*******************************************************************************
* NXP Semiconductor Inc.
* (c) Copyright 2016 NXP Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without
any warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be
held liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO THE PROJECT
BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. IN NO EVENT
SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold NXP harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result of the advise
or assistance supplied CUSTOMER in connection with product, services or goods
supplied under this Agreement.
********************************************************************************
* File              main.c
* Owner             David Chung
* Version           1.0
* Date              Oct-27-2017
* Classification    General Business Information
* Brief             MPC5744P sends/receives CAN message "Hello".
********************************************************************************
* Detailed Description:
* MPC5744P sends CAN message "Hello".  The macro LOOPBACK in
* can.h determine mode of operation of this demo.  LOOPBACK = 1 if using loopback function. This code
* example ships configured for loopback mode.
*
* CAN message "Hello" is sent from either CAN_2.
* Receiver in loopback mode is same CAN as the transmitting CAN (CAN_2).  If
* not in loopback mode, CAN_0 is the receiver.  In all cases, the buffer RxDATA gets the
* message "Hello World".  Check the UART terminal to see operation.  The UART will print
* the RxDATA before and after CAN transmission.
* ------------------------------------------------------------------------------
* Test HW:         Miriac(TM)-EK5744
* MCU:             MPC5744P
* Terminal:        19200, 8N1, None
* Fsys:            160 MHz PLL with 40 MHz crystal reference
* Debugger:        USB Multilink
* Target:          FLASH
* EVB connection:  Enable CAN power by flipping the switches in the red switch
* 					next to power terminal.
* 					SBC-CANH to CAN2H and SBC-CANL to CAN2L. See QSG for pinmap
* 					Only make sure CAN is powered if using loopback mode; no wire
* 					connections necessary
*
********************************************************************************
Revision History:
Version  Date         Author  			Description of Changes
1.0      Oct-27-2017  David Chung	  	Initial version

*******************************************************************************/

#include "derivative.h" /* include peripheral declarations */
#include "can.h"
#include "uart.h"
#include "mode_entry.h"
#include <string.h>

extern void xcptn_xmpl(void);

void peri_clock_gating(void);

extern uint8_t RxDATA[];

__attribute__ ((section(".text")))
int main(void)
{
	uint8_t transmit_done = 0;
	uint8_t intro[] = {"\n\rWelcome to the Miriac(TM)-EK5744 FlexCAN code example!\n\r"};
	uint8_t BeforeMessage[] = {"\n\rBefore CAN transmission, "};
	uint8_t AfterMessage[] = {"\n\rAfter CAN transmission, "};
	uint8_t RxMessage[] = {"RxData reads "};
	uint8_t EmptyMessage[] = {"RxDATA is empty.\n\r"};
	uint8_t period[] = {"."};
	uint8_t newline[] = {"\n\r"};
	uint8_t EndMessage[] = {"\n\rMPC5744P CAN transmission test done.\n\r"};

#if LOOPBACK
	uint8_t LoopbackMessage[] = {"\n\rLoopback mode has been selected.\n\r"};
#endif
	
	xcptn_xmpl ();              /* Configure and Enable Interrupts */

	uint32_t CAN_msg_count = 0;

	peri_clock_gating();   /* configure gating/enabling peri. clocks for modes*/
	                         /* configuraiton occurs after mode transition */

	system160mhz();        /* sysclk=160MHz, dividers configured, mode trans*/

#if !LOOPBACK
	initCAN_0();			/* Initialize FlexCAN_0 & of its buffers for receive if no loopback */
#endif

	initCAN_2();			/* Initialize FlexCAN 2 & one of its buffers for transmit and receive if loopback enabled  */
	LINFlexD_1_Init();		/* Initialize LINFlexD_1 to transmit messages to terminal */

	TransmitData((const char*)intro,(uint32_t)strlen((const char*)intro));

#if LOOPBACK
	TransmitData((const char*)LoopbackMessage,(uint32_t)strlen((const char*)LoopbackMessage));
#endif

	TransmitData((const char*)BeforeMessage,(uint32_t)strlen((const char*)BeforeMessage));
	if(strlen((const char*)RxDATA) == 0){
		TransmitData((const char*)EmptyMessage,(uint32_t)strlen((const char*)EmptyMessage));
	}else{
		TransmitData((const char*)RxMessage,(uint32_t)strlen((const char*)RxMessage));
		TransmitData((const char*)RxDATA,(uint32_t)8);
		TransmitData((const char*)period,(uint32_t)strlen((const char*)period));
		TransmitData((const char*)newline,(uint32_t)strlen((const char*)newline));
	}

	/* Enable DOUT0 LED */
	SIUL2.MSCR[42].B.OBE = 1; //PWENA
	SIUL2.MSCR[45].B.OBE = 1; //PWENB
	SIUL2.MSCR[58].B.OBE = 1; //DOUT0

	/* PWENA/B effectively function as NAND gates */
	SIUL2.GPDO[42].R = 0;
	SIUL2.GPDO[45].R = 0;
	SIUL2.GPDO[58].R = 0; //Turn off DOUT0 LED


	while (1) {

	TransmitMsg();           /* Transmit one message from a FlexCAN 0 buffer */
	ReceiveMsg();            /* Wait for the message to be received at FlexCAN 1 */
	CAN_msg_count++;         /* Increment CAN message counter */

	/* Only print this message once */
	if(!transmit_done){
		TransmitData((const char*)AfterMessage,(uint32_t)strlen((const char*)AfterMessage));
		if(strlen((const char*)RxDATA) == 0){
			TransmitData((const char*)EmptyMessage,(uint32_t)strlen((const char*)EmptyMessage));
		}else{
			TransmitData((const char*)RxMessage,(uint32_t)strlen((const char*)RxMessage));
			TransmitData((const char*)RxDATA,(uint32_t)8);
			TransmitData((const char*)period,(uint32_t)strlen((const char*)period));
			TransmitData((const char*)newline,(uint32_t)strlen((const char*)newline));
		}

		TransmitData((const char*)EndMessage,(uint32_t)strlen((const char*)EndMessage));

		/* Set the flag so it doesn't print again */
		transmit_done = 1;
	}

	SIUL2.GPDO[58].R = 0 ;		//Turn off DOUT0 LED

	}

	return 0;
}

/*****************************************************************************/
/* peri_clock_gating                                                         */
/* Description: Configures enabling clocks to peri modules or gating them off*/
/*              Default PCTL[RUN_CFG]=0, so by default RUN_PC[0] is selected.*/
/*              RUN_PC[0] is configured here to gate off all clocks.         */
/*****************************************************************************/

void peri_clock_gating (void) {
  MC_ME.RUN_PC[0].R = 0x00000000;  /* gate off clock for all RUN modes */
  MC_ME.RUN_PC[1].R = 0x000000FE;  /* config. peri clock for all RUN modes */

  MC_ME.PCTL79.B.RUN_CFG = 0x1; //FlexCAN 0: select peripheral config RUN_PC[1]
//  MC_ME.PCTL204.B.RUN_CFG = 0x1; //LINFlexD_0: select peripheral config RUN_PC[1]

  MC_ME.PCTL77.B.RUN_CFG = 0x1; //FlexCAN 2: select peripheral config RUN_PC[1]
  MC_ME.PCTL91.B.RUN_CFG = 0x1; //LINFlexD_1: select peripheral config RUN_PC[1]

#if !LOOPBACK
  MC_ME.PCTL78.B.RUN_CFG = 0x1; //Repeat for FlexCAN 1 if no loopback feature
#endif
}
