#define LINSCI_C
/******************************************************************************
*                                                       
*       Copyright (C) 2005 Freescale Semiconductor, Inc.
*       All Rights Reserved								              
*														                            
* Filename:     linsci.c                
*														                            
* Revision:      										                    
*														                            
* Functions:    SCI management module
*												                            
* Description:
*
* Notes:        
*
******************************************************************************/

#include "linbase.h"

#pragma MESSAGE DISABLE C4200 /* WARNING C4200: other segment than in previous declaration */

/****************************************************************************
 * All common-purpose RAM variables shall be declared here.
 ***************************************************************************/
/****************************************************************************
 * It require to perform zero page RAM placement optimisation.
 ***************************************************************************/


LIN_BYTE  LIN_TmpSCIStatus;   /* byte to clear SCI status, and to receive byte   */



#if defined(LINAPI_1_0)

/***************************************************************************
 * Function :   l_ifc_init_sci0
 *
 * Description: Initialize SCI to work with LIN bus. 
 *                  Set specified baud rate;
 *                  Disable all SCI interrupts;
 *                  Enable Tx and RX pins
 *
 * Returns:     none
 *
 * Notes:       LIN API service
 *
 **************************************************************************/

void l_ifc_init_sci0( void )
{
    LIN_BYTE    intMask;

    LIN_DBG_SET_PORT_7;
    intMask = LIN_DisableInt();     /* disable interrupts */

    /* Disconnect SCI and set initial state -- this is abort any activity */
    LIN_StateFlags = LIN_FLAG_IGNORE | LIN_FLAG_DISCONNECT;
    
    /* ??? SCI not disabled while initialization */
                                    /* initialization order correct */
    LIN_SCISetBaudRate(LIN_CfgConst.LIN_BaudRate);

    /* one start bit, eight data bits, one stop bit */
    LIN_SC0CR1 = 0;                            
    /* enable transmitter & receiver and disable all SCI interrupts */
    LIN_SC0CR2 = LIN_SCCR2_TE | LIN_SCCR2_RE;  

    /* SCI flags clearing and interrupts enables in ifc_connect */

    LIN_EnableInt(intMask);         /* enable interrupts */
    LIN_DBG_CLR_PORT_7;
}


/***************************************************************************
 * Function :   l_ifc_connect_sci0
 *
 * Description: Connect SCI -- clear all SCI flags,
 *              enable SCI interrupts, reset idle timeout.
 *
 * Returns:     0   -- success
 *              ~0  -- called before l_ifc_init()
 *
 * Notes:       LIN API.
 *              If already connected then do nothing.
 *
 **************************************************************************/
l_bool l_ifc_connect_sci0( void )
{
    l_irqmask   intMask;
    l_bool      ret;

    ret = 0;    /* suppose that connecting OK */

    LIN_DBG_SET_PORT_7;
    intMask = LIN_DisableInt();     /* disable interrupts */

    /* if l_ifc_connect called before l_ifc_init then return error */
    if ( (LIN_StateFlags & LIN_FLAG_NOINIT) != 0 )
    {
        ret = ~0;
    }

    /* we are after l_ifc_init */

    /* if an interrupt occurs here and call l_sys_init() then we in LIN_FLAG_NOINIT
       so we need interrupt disabling -- OPTIM */

    /* if interface is already connected then do nothing */
    else if ( (LIN_StateFlags & LIN_FLAG_DISCONNECT) != 0 )
    {
        /* we are in disconnected state -> connect */

        /* connect SCI hardware */
                                        /* clear SCI status */
        LIN_TmpSCIByte = LIN_SC0SR1;
        LIN_TmpSCIByte = LIN_SC0DRL;

        LIN_SC0CR2 |= LIN_SCCR2_RIE;    /* enable RX complited interrupt */

        /* start to wait break */
        LIN_StateFlags = LIN_FLAG_IGNORE;

        /* reset idle timeout */
        LIN_SetIdleTimeout();               
    }

    LIN_EnableInt(intMask);         /* enable interrupts */
    LIN_DBG_CLR_PORT_7;

    return ret;
}

/***************************************************************************
 * Function :   l_ifc_disconnect_sci0
 *
 * Description: Disconnect SCI -- disable SCI interrupts.
 *
 * Returns:     0   -- success
 *              ~0  -- called before l_ifc_init()
 *
 * Notes:       LIN API.
 *              If already disconnected then do nothing.
 *
 **************************************************************************/
l_bool l_ifc_disconnect_sci0( void )
{
    l_irqmask   intMask;
    l_bool      ret;

    ret = 0;    /* suppose that disconnecting OK */

    LIN_DBG_SET_PORT_7;
    intMask = LIN_DisableInt();     /* disable interrupts */

    /* if l_ifc_disconnect called before l_ifc_init then return error */
    if ( (LIN_StateFlags & LIN_FLAG_NOINIT) != 0 )
    {
        ret = ~0;
    }

    /* we are after l_ifc_init */

    /* if an interrupt occurs here and call l_sys_init() then we in LIN_FLAG_NOINIT
       so we need interrupt disabling -- OPTIM */

    /* if interface is already disconnected then do nothing */
    else if ( (LIN_StateFlags & LIN_FLAG_DISCONNECT) == 0 )
    {
        /* we are in connected state -> disconnect */

        /* disconnect SCI hardware */
        /* enable transmitter and receiver, disable all SCI interrupts */
        LIN_SC0CR2 = LIN_SCCR2_TE | LIN_SCCR2_RE;  

        /* set disconnected state */
        LIN_StateFlags = LIN_FLAG_IGNORE | LIN_FLAG_DISCONNECT;
    }

    LIN_EnableInt(intMask);         /* enable interrupts */
    LIN_DBG_CLR_PORT_7;

    return ret;
}


#else /* !defined(LINAPI_1_0) */


/***************************************************************************
 * Function :   LIN_SCIInit
 *
 * Description: Initialize SCI to work with LIN bus. 
 *                  Set specified baud rate;
 *                  Disable all SCI interrupts;
 *                  Enable Tx and RX pins
 *                  Enable Rx interrupt
 *
 * Returns:     none
 *
 * Notes:       1. SLAVE, MASTER node
 *              2. Required for cosmic linker. If no functions are used 
 *                 from particular .obj file this file is not linked!
 *                 We can make it as macros ? -- OPTIM 
 *
 **************************************************************************/
void LIN_SCIInit( void )
{
    /* ??? SCI not disabled while initialization */
                                    /* initialization order correct */
    LIN_SCISetBaudRate(LIN_CfgConst.LIN_BaudRate);

    /* one start bit, eight data bits, one stop bit */
    LIN_SC0CR1 = 0;                            
    /* enable transmitter & receiver and disable all SCI interrupts */
    LIN_SC0CR2 = LIN_SCCR2_TE | LIN_SCCR2_RE;  

                                    /* clear SCI status */
    LIN_TmpSCIByte = LIN_SC0SR1;
    LIN_TmpSCIByte = LIN_SC0DRL;

    LIN_SC0CR2 |= LIN_SCCR2_RIE;    /* enable RX complited interrupt */
}

#endif  /* !defined(LINAPI_1_0) */

/***************************************************************************
 * Function :   LIN_SCISetBaudRate
 *
 * Description: Programm SCI to particular baud rate
 *
 * Returns:     none
 *
 * Notes:       -   To disable SCI write 0  to SC0BDH and SC0BDL
 *
 **************************************************************************/
void LIN_SCISetBaudRate( LIN_WORD baudRate )
{
    /* set the baud rate divider  */
    LIN_SC0BDH = (LIN_BYTE)((baudRate >> 8) & 0x001f);               
    LIN_SC0BDL = (LIN_BYTE)(baudRate & 0x00ff);         
}

/****************************************************************************/
/****************************************************************************/
/***                     Interrupt Driven Routines                        ***/
/****************************************************************************/
/****************************************************************************/

/***************************************************************************
 * Function :   LIN_ISR_SCI_Interrupt / l_ifc_rx_sci0 (l_ifc_tx_sci0)
 *
 * Description: SCI interrupt
 *              
 * Returns:     none
 *
 * Notes:       
 *              Freescale API -- direct ISR.
 *              LIN API      -- LIN API function --
 *                  l_ifc_rx_sci0 and l_ifc_tx_sci0 also becouse
 *                  S12 has only one SCI interrupt vector.
 *              SCI interrupts:
 *                 RX completed interrupt          enabled forever
 *                 TX enable interrupt             disabled forever
 *                 TX completed interrupt          enabled only while: 
 *                                                 - break symbol send 
 *                                                 - wakeup symbol send 
 *                                                 disabled after processing
 *                 Overrun interrupt               enabled but not processed
 *                                                 and only cleared
 *
 **************************************************************************/
#if defined(LINAPI_1_0)
void l_ifc_rx_sci0( void )
#else /* defined(LINAPI_1_0) */
void LIN_ISR_SCI_Interrupt ( void )
#endif /* defined(LINAPI_1_0) */
{
    LIN_DBG_SET_PORT_0;

#if defined(LINAPI_1_0)
    if ( (LIN_StateFlags & LIN_FLAG_DISCONNECT) == 0 )
    {   
        /* work only if SCI connected */
#endif /* defined(LINAPI_1_0) */

        LIN_TmpSCIStatus = LIN_SC0SR1;
        LIN_TmpSCIByte   = LIN_SC0DRL;                  /* clear all flags */

        if ( LIN_TmpSCIStatus & LIN_SCSR1_RDRF )        /* overrun not processed */
        {   
            if ( LIN_TmpSCIStatus & LIN_SCSR1_FE )
            {   /********************************************************* error interrupt */
                /* only frame error was processed */

    #if defined(MASTER)
                LIN_SC0CR2 &= ~LIN_SCCR2_RE;    /* NB: disable and enable SCI to prevent */
                                                /*     distinction next zero level bit as start bit */
                LIN_DBG_SET_PORT_3;             /* indicate frame error recognized */

                LIN_FrameError();

                LIN_DBG_CLR_PORT_3;

                LIN_SC0CR2 |= LIN_SCCR2_RE;     /* enable SCI Rx */
    #endif /* defined(MASTER) */

    #if defined(SLAVE)
                if ( LIN_TmpSCIByte == 0 )      /* Check SCI  RX data bit if 0 so count next 
                                                   bittime to receive 11 bittime break else frame error */
                {                        
                    LIN_DBG_SET_PORT_5;         /* indicate break symbol recognized  */

                    LIN_FrameError(LIN_NORMALBREAK);  

                    LIN_DBG_CLR_PORT_5;
                }
                else
                {
                    LIN_SC0CR2 &= ~LIN_SCCR2_RE;    /* NB: disable and enable SCI to prevent */
                                                    /*     distinction next zero level bit as start bit */
                    LIN_DBG_SET_PORT_3;             /* indicate frame error recognized */

                    LIN_FrameError(LIN_FRAMEERROR);

                    LIN_DBG_CLR_PORT_3;

                    LIN_SC0CR2 |= LIN_SCCR2_RE;     /* enable SCI Rx */
                }
    #endif /* defined(SLAVE) */
            }   /****************************************************** end error interrupt */    

            else
            {  /******************************************************** receiver interrupt */
                LIN_DBG_SET_PORT_1;

                LIN_RxCompleted();

                LIN_DBG_CLR_PORT_1;
            }  /**************************************************** end receiver interrupt */
        }

        else 
        {      /***************************************************** transmitter interrupt */ 
            LIN_SC0CR2  &= ~( LIN_SCCR2_TCIE );    /* Disable Tx completed */
                                                   /* interrupt we cannt clear it */
#if defined(C32)
            /* To ommit hardware bug in PC9S12DP256 ? -- OPTIM */
            LIN_SC0CR2  |= LIN_SCCR2_RIE;          /* Enable Rx interrupt */
#endif /* defined(C32) */
 
            if ( LIN_TmpSCIStatus & LIN_SCSR1_TC )
            {  /**************************************************** tx completed interrupt */
                LIN_DBG_SET_PORT_2;

                LIN_TxCompleted();

                /* if Rx interrupt for break occures after Tx interrupt
                   we ignore it in LIN_RxCompleted function */

                LIN_DBG_CLR_PORT_2;
            }

    #if defined(LIN_DBG_CHECK_INTERNAL_ERROR)
            else
            {  /****************************** Overrun, TX enable  interrupt not processed */
                LIN_InternalError = LIN_ERROR_8;
                while (1)
                {}
            }
    #endif /* defined(LIN_DBG_CHECK_INTERNAL_ERROR) */
        }

#if defined(LINAPI_1_0)
    }
#endif /* defined(LINAPI_1_0) */

    LIN_DBG_CLR_PORT_0;
}
