#define LINPROT_C
/******************************************************************************
*                                                       
*       Copyright (C) 2005 Freescale Semiconductor, Inc.
*       All Rights Reserved								              
*														                            
* Filename:     linapi.c                
*														                            
* Revision:      										                    
*														                            
* Functions:    LIN API realization
*												                            
* Description:  Some Freescale and VCT API functions realization
*												                            
* Notes:        Realization of another API functions -- in another modules
*
******************************************************************************/

#include "linbase.h"

#if defined(LINAPI_1_0)

/***************************************************************************
 * Function :   l_ifc_ioctl_sci0
 *
 * Description: Protocol-specific functionality.
 * 
 * Returns:     depends on the operation
 *
 * Notes:       Operations:
 *  l_op_getrxerr   -- Provide the Receive Errors counter of the LIN Driver 
 *  l_op_gettxerr   -- Provide the Transmit Errors counter of the LIN Driver 
 *  l_op_clrrxerr   -- Clear the Receive Errors counter of the LIN Driver 
 *  l_op_clrtxerr   -- Clear the Transmit Errors counter of the LIN Driver 
 *  l_op_wakeup     -- Issues the LIN bus wakeup frame transmission 
 *  l_op_getidle    -- Check No-Bus-Activity condition 
 *  l_op_idleclock  -- Update No-Bus-Activity condition counter by 1 
 *
 **************************************************************************/
void l_ifc_ioctl_sci0(i_ioctl_op op, void *pv)
{
    LIN_BYTE        intMask;

    LIN_DBG_SET_PORT_7;

    intMask = LIN_DisableInt();         /* Disable interrupts */

    /* OPTIM:
        switch  -- for COSMIC08 (5 bytes less);
        if      -- for HICROSS08 */

    if (op == l_op_getrxerr)
    {
        *( (LINErrCounterType*) pv) = LIN_ErrRxCounter;
    }

    else if (op == l_op_gettxerr)
    {
        *( (LINErrCounterType*) pv) = LIN_ErrTxCounter;
    }

    else if (op == l_op_clrrxerr)
    {
        LIN_ErrRxCounter = 0;
    }

    else if (op == l_op_clrtxerr)
    {
        LIN_ErrTxCounter = 0;
    }

    else if (op == l_op_wakeup)
    {
        /* possible only if we are connected and doing nothing */
        if ( ( (LIN_StateFlags & LIN_FLAG_DISCONNECT)  == 0 ) && 
             ( (LIN_StateFlags & LIN_FLAG_IGNORE) != 0 ) )
        {
            LIN_StateFlags = LIN_FLAG_WAKEUP_TIMEOUT;

            /* send Wakeup message */    
            LIN_SCISendWakeup();

            /* wait for Tx completed interrupt. All others interrupt should be ignored */
            /* timeout is not required */
            
            *( (l_bool*) pv) = 0;    /* success */
        }
        else
        {
            *( (l_bool*) pv) = ~0;   /* pending or SCI disconnected*/
        }
    }

    else if (op == l_op_getidle)
    {
        if ( LIN_IdleTimeout == 0 )
        {
            *( (l_bool*) pv) = ~0;   /* IDLE */
        }
        else
        {
            *( (l_bool*) pv) = 0;    /* no IDLE */
        }
    }

    else if (op == l_op_idleclock)
    {
        if ( LIN_IdleTimeout != 0  )
        {
            --LIN_IdleTimeout;
        }
    }

    LIN_EnableInt(intMask);             /* Enable interrupts */    

    LIN_DBG_CLR_PORT_7;
}


#else /* defined(LINAPI_1_0) */


#if defined(MASTER)
/***************************************************************************
 * Function :   LIN_RequestMsg
 *
 * Description: Request message. Send message header.
 * 
 * Returns:     LIN_OK                No error, service call has succeeded.                   
 *              LIN_REQ_PENDING       The message request is already pending.                 
 *              LIN_INVALID_MODE      The driver is in the sleep mode during 
 *                                    message request. 
 *              LIN_INVALID_ID        Requested Id is not valid
 *                                    (not in range 0..0x3D).
 *
 * Notes:       1. API Service call
 *              2. Disable and restore interrupts 
 *              3. MASTER node
 *
 **************************************************************************/
LINStatusType LIN_RequestMsg( LINMsgIdType msgId )
{
    LIN_BYTE        intMask;

    /* check requested Id */
    if ( msgId > (LIN_BYTE) LIN_FRAME_MAX_ID )
    {
        /* Id is not valid */
        return LIN_INVALID_ID;
    }

    /* check Sleep state */
    if (LIN_StateFlags & LIN_FLAG_SLEEP )
    {
        /* we don't need interrupt disabling becouse in SLEEP mode driver doing nothing */
        return LIN_INVALID_MODE;                
    }

    /* check pending */        
    /* NB: LIN_FLAG_IGNORE should be cheched after SLEEP */
    else if ( LIN_StateFlags & LIN_FLAG_IGNORE )
    {
        /* we can send new header */

        LIN_DBG_SET_PORT_7;
        intMask = LIN_DisableInt();                 /* disable MCU interrupts */

        /* Change driver state */
        LIN_StateFlags = LIN_FLAG_SEND_BREAK;

        /* Prepare message Header for transmission */

        /* Calculate Identifier Parity          */
        LIN_ProtBuf[1] =  (msgId & ~0x40 | 0x80) ^ ( ((msgId << 2) ^ ((msgId << 3) & 0x80) ^ (msgId << 4) ^ 
                                                   ((msgId << 5) & 0x40)  ^ (msgId << 6) ) & 0xc0);
        /* LIN_ProtBuf[0] -- don't used for Header transmition */

        LIN_BufCounter = 0;     /* used as counter for Header transmission */

        /* Prepare for message Response processing */

        /* Save message Id --
           we don't need cut off 2 most significant bits becouse we are already check Id */
        LIN_SaveIdentifier = msgId;

        /* Save Response data bytes number (1..8) */
        LIN_MsgLength = LinLengthTable[msgId];

        /* Send Break Symbol

               For S12: SCI bit rate decreases to 50% (9->13) and send 0 (10->14). 
                         Then bit rate changes to nominal. */
        /* Send "0" byte via SCI to emulate break symbol of 13 bit long */
            
        LIN_SCISetBaudRate(LIN_CfgConst.LIN_BreakBaudRate);  
                                          /* change SCI baud rate to increase length of */
                                          /* 9 bit time to 13 bit time */

        /* Protection from SCI transmitter busy is not required due to previous LIN_StateFlags check */

        LIN_SCISendZero();             /* send Break -- like 0, but on decreased baud rate */

        /* Wait for tx comleted interrupt. If Rx completed or Frame error interrupt occure - */
        /* ignore it. Timeout is not used. */

        LIN_EnableInt(intMask);                 /* enable MCU interrupts */    
        LIN_DBG_CLR_PORT_7;

        return LIN_OK;
    }

    else
    {   
        /* we are not in IGNORE or SLEEP state */
        return LIN_REQ_PENDING;
    }
}
#endif /* defined(MASTER) */

#if defined(SLAVE)
/***************************************************************************
 * Function :   LIN_Wakeup
 *
 * Description: Send wakeup frame
 * 
 * Returns:     LIN_OK                No error, service call has succeeded.                   
 *              LIN_REQ_PENDING       Another LIN bus frame is currently 
 *                                    processed. Or wakeup process is running
 *              LIN_INVALID_MODE      The driver is in the sleep mode during 
 *                                    message request. 
 *
 * Notes:       1. API Service call
 *              2. MASTER and SLAVE
 *              3. Set Timeout 
 *              4. don`t wait while SCI will be free
 *
 **************************************************************************/
LINStatusType          LIN_Wakeup( void )
{
    LIN_BYTE        intMask;

    if (LIN_StateFlags & LIN_FLAG_SLEEP )
    {
        /* we don't need interrupt disabling becouse in SLEEP mode driver doing nothing */
        return LIN_INVALID_MODE;                
    }
    else if (LIN_StateFlags & LIN_FLAG_IGNORE)
    {
        LIN_DBG_SET_PORT_7;
    
        intMask = LIN_DisableInt();         /* Disable interrupts */
    
        LIN_StateFlags = LIN_FLAG_WAKEUP_TIMEOUT;

        /* send Wakeup message */    
        LIN_SCISendWakeup();

        /* wait for Tx completed interrupt. All others interrupt should be ignored */
        /* timeout is not required */

        LIN_EnableInt(intMask);             /* Enable interrupts */    

        LIN_DBG_CLR_PORT_7;

        return LIN_OK;
    }
    else
    {
        return LIN_REQ_PENDING;
    }
}
#endif /* defined(SLAVE) */

/***************************************************************************
 * Function :   LIN_GotoRun
 *
 * Description: Change the current driver state from SLEEP to RUN
 * 
 * Returns:     none
 *
 * Notes:       1. API Service call
 *              2. MASTER: Set Idle timeout;
 *                         set WakeupDelimiter timeout and wait it
 *              3. SLAVE:  Set Idle timeout and go to IGNORE state
 *              4. If in RUN already -> no any actions
 *              5. DisableInterrupt don't needed becouse in SLEEP mode
 *                 driver do nothing
 *
 **************************************************************************/
void LIN_GotoRun( void )
{
    if ( LIN_StateFlags & LIN_FLAG_SLEEP )                    
    {
        /* we don't need interrupt disabling becouse in SLEEP mode driver doing nothing */
        
        LIN_SetIdleTimeout();                   /* Set idle timeout again */

        LIN_StateFlags = LIN_FLAG_IGNORE;
    }
}

/***************************************************************************
 * Function :   LIN_DriverStatus
 *
 * Description: Return actual status of the LIN Driver.
 * 
 * Returns:     LIN_STATUS_RUN            Driver state is run 
 *                                        or sleep                
 *              LIN_STATUS_IDLE           No-Bus_activity timeout 
 *                                        is expired  
 *              LIN_STATUS_PENDING        LIN bus frame is pending 
 *                                        only for MASTER
 *
 * Notes:       1. API Service call
 *              2. MASTER and SLAVE
 *              3. NB: !!!  LIN_STATUS_RUN     = ~LIN_FLAG_SLEEP     
 *                          LIN_STATUS_PENDING = ~LIN_FLAG_IGNORE
 *                          LIN_STATUS_IDLE    =  (LIN_IdleTimeout == 0)
 **************************************************************************/
LINDriverStatusType    LIN_DriverStatus( void )
{
    LINDriverStatusType retVal;
#if defined(MASTER)
    retVal = ( LIN_StateFlags ^ ( LIN_FLAG_SLEEP | LIN_FLAG_IGNORE )) & ( LIN_FLAG_SLEEP | LIN_FLAG_IGNORE );
#endif /* defined(MASTER) */
#if defined(SLAVE)
    retVal = ( LIN_StateFlags ^ LIN_FLAG_SLEEP ) & ( LIN_FLAG_SLEEP );
#endif /* defined(SLAVE) */

    if ( LIN_IdleTimeout == 0 )
    {
        retVal |=  LIN_STATUS_IDLE;
    } 

    return retVal;
}

#endif /* defined(LINAPI_1_0) */
