#define LINTMR_C
/******************************************************************************
*                                                       
*       Copyright (C) 2005 Freescale Semiconductor, Inc.
*       All Rights Reserved								              
*														                            
* Filename:     lintmr.c                
*														                            
* Revision:      										                    
*														                            
* Functions:    LIN Timeout management module 
*												                            
* Description:  
*
* Notes:        Protocol timeouts -- only for Master node
*
******************************************************************************/

#include "linbase.h"

#pragma MESSAGE DISABLE C4200 /* WARNING C4200: other segment than in previous declaration */

/****************************************************************************
 * All common-purpose RAM variables shall be declared here.
 ***************************************************************************/


LIN_WORD LIN_IdleTimeout;          /* counter for idle timeout */

#if !defined(LINAPI_1_0)

#if defined(MASTER)

static LIN_BYTE LIN_ProtocolTimeout;      /* counter for protocol timeout */

LIN_WORD LIN_LengthCh0Tick;        /* length of one timer Channel 1 overflow tick;
                                      the LIN_LengthCh0Tick is not declared as static due to 
                                      it used as symbol in assembler macro, but Cosmic compiler
                                      eliminate symbol references to static variables */

/****************************************************************************
 * End of common-purpose RAM variables declaration
 ***************************************************************************/


/***************************************************************************
 * Function :   LIN_SetTimeout
 *
 * Description: Set protocol timeout.
 *                  - disable channel 0 interrupt
 *                  - set timeout
 *                  - enable channel 0 interrupt
 *                  - set LIN_ProtocolTimeout value
 *
 * Returns:     none
 *
 * Notes:       none
 *
 **************************************************************************/
void LIN_SetTimeout(LIN_BYTE timeoutIndex)
{
    /* at first, disable channel 0 interrupt */
    LIN_TMSK1 &= ~LIN_TM_C0I;

    /* at second, set timeout */
    LIN_LengthCh0Tick = LIN_MTO_Ticks[timeoutIndex];
    LIN_TC0 = LIN_TCNT + LIN_LengthCh0Tick;

    /* interrupt flag is cleared due TFFCA feature is enabled */ 
    LIN_TMSK1 |= LIN_TM_C0I; /* enable channel 0 interrupt */
   
    /* set LIN_ProtocolTimeout value for this timeout */
    LIN_ProtocolTimeout = LIN_MTO_Counter[timeoutIndex];
}

/***************************************************************************
 * Function :   LIN_CancelTimeout
 *
 * Description: Cancel protocol timeout
 *
 * Returns:     none
 *
 * Notes:       none
 *
 **************************************************************************/
void LIN_CancelTimeout( void )
{
    /* disable channel 0 interrupt */
    LIN_TMSK1 &= ~LIN_TM_C0I;
}

/***************************************************************************
 * Function :   LIN_ISR_Timer0
 *
 * Description: Timer interrupt routine. 
 *              Used for:   LIN_MaxFrameTime, 
 *                          LIN_WakeupDelimiterTime
 *                          timeouts maintenance.
 *              Also used for:  break generation.
 *                 
 * Returns:     none
 *
 * Notes:       
 *
 **************************************************************************/
LIN_INTERRUPT LIN_ISR_Timer0( void ) 
{
    LIN_DBG_SET_PORT_0;
    LIN_DBG_SET_PORT_4;

    --LIN_ProtocolTimeout;

    if ( LIN_ProtocolTimeout == 0)
    {   
        /* timeout expired */

        /* clear interrupt flag and disable channel 0 interrupt */
        LIN_TMSK1 &= ~LIN_TM_C0I;
        LIN_TFLG1 &= ~LIN_TM_C0F;

        LIN_Timeout();
    }
    else
    {
        /* timeout continued */

        /* arm timer Channel 0 for next compare */
        LIN_TC0 = LIN_TCNT + LIN_LengthCh0Tick;
        /* interrupt flag is cleared due TFFCA feature is enabled */ 
    }

    LIN_DBG_CLR_PORT_0;
    LIN_DBG_CLR_PORT_4;
}


#endif /* defined(MASTER) */


/***************************************************************************
 * Function :   LIN_IdleClock
 *
 * Description: User time clock for Idle timeout.
 *              Update "no-bus-activity" condition counter
 *              and check, if the condition meets.
 *                 
 * Returns:     none
 *
 * Notes:       1. API Service call       
 *
 **************************************************************************/
void LIN_IdleClock( void ) 
{
    LIN_BYTE        intMask;

    LIN_DBG_SET_PORT_7;

    intMask = LIN_DisableInt();         /* Disable interrupts -- do we really need it? */

    if ( LIN_IdleTimeout != 0  )
        {
            --LIN_IdleTimeout;
        }

    LIN_EnableInt(intMask);             /* enable interrupts */    

    LIN_DBG_CLR_PORT_7;
}

#endif /* !defined(LINAPI_1_0) */

/***************************************************************************
 * Function :   LIN_SetIdleTimeout
 *
 * Description: Set/reset LIN_IdleTimeout variable from constant
 *                 
 * Returns:     none
 *
 * Notes:       
 *
 **************************************************************************/
void LIN_SetIdleTimeout( void )
{
    LIN_IdleTimeout = LIN_CfgConst.LIN_IdleTimeoutClocks;
}
