#!/usr/bin/env python3

"""

Copyright 2019 NXP.

This software is owned or controlled by NXP and may only be used strictly in accordance with the
license terms that accompany it. By expressly accepting such terms or by downloading, installing,
activating and/or otherwise using the software, you are agreeing that you have read, and that you
agree to comply with and are bound by, such license terms. If you do not agree to be bound by the
applicable license terms, then you may not retain, install, activate or otherwise use the software.

File
++++
/Scripts/sln_viznas_iot_open_boot/open_prog_full.py

Brief
+++++
Manufacturing tool for programming the SLN-VIZNAS-IOT binaries

Usage
++++++++++
$python open_prog_full.py -c CA_NAME -p PREFIX -m MAIN_APP -d BIN_DIR
    - Default CA_NAME is 'prod'. See ota_signing README.md for clarification on CA_NAME
    - Default PREFIX is 'sln_viznas_iot\_'
    - Default MAIN_APP is 'elock_oobe'
    - Default BIN_DIR is '../../Image_Binaries/'

.. versionadded:: 0.0

"""

import os
import sys
import subprocess
import Ivaldi.blhost as blhost
import Ivaldi.sdphost as sdphost
import Ivaldi.helpers as helpers
import Ivaldi.onboard.aws as ob
import time
import base64
from shutil import copyfile
import argparse

TOP_DIR = "../../"
PY3_DIR = TOP_DIR + 'Scripts/'
CA_CERTS_DIR = PY3_DIR + 'ota_signing/ca/certs/'
SIGN_DIR = PY3_DIR + 'ota_signing/sign/'

IMG_DIR_DEF = TOP_DIR + 'Image_Binaries/'
APP_NAME_PREFIX_DEF = 'sln_viznas_iot_'
MAINAPP_NAME_DEF = 'elock_oobe'

ser_num = ''
ca_name = 'prod'


def main():
    """
        Performs the following operations:
            - Erases: 
                - 0x2000000 bytes of flash starting at address 0x60000000 
            - Programs:
                - sln_viznas_iot_bootstrap.bin @ 0x60000000 
                - sln_viznas_iot_bootloader.bin @ 0x60040000 
                - sln_viznas_iot_elock_oobe.bin @ 0x60300000
                - Image CA Root Certificate @ 0x61CC0000 
                - Bank A Signing Certificate @ 0x61D00000 
                - Bootloader Signing Certificate @ 0x61D80000 
                - fica_table.bin @ 0x61FC0000
            - Signs:
                - sln_viznas_iot_bootloader.bin
                - sln_viznas_iot_elock_oobe.bin

        :returns: None
    """
    global ser_num
    global ca_name

    parser = argparse.ArgumentParser()
    parser.add_argument("-c", "--ca-name", type=str, nargs='?',
                        default='', help="Name of CA used to sign images.")
    parser.add_argument("-p", "--prefix", type=str, nargs='?',
                        default=APP_NAME_PREFIX_DEF, help="Prefix for bootloader and main_app binaries")
    parser.add_argument("-m", "--main_app", type=str, nargs='?',
                        default=MAINAPP_NAME_DEF, help="Name of main app (elock_oobe, etc.)")
    parser.add_argument("-d", "--bin_dir", type=str, nargs='?',
                        default=IMG_DIR_DEF, help="Binary files directory")
    args = parser.parse_args()

    if args.ca_name:
        ca_name = args.ca_name
    print("Signing Entity: %s" % ca_name)

    app_prefix = args.prefix
    mainapp = args.main_app
    IMG_DIR = args.bin_dir
    bootstrap_name = app_prefix + 'bootstrap.bin'
    bootloader_name = app_prefix + 'bootloader.bin'
    mainapp_name = app_prefix + mainapp + '.bin'

    sdp = sdphost.SDPHost('0x1fc9', '0x135')

    # Check communication with device
    print('Establishing connection...')
    if sdp.error_status()['ret']:
        print('ERROR: Could not establish communication with device. Power cycle device.')
        sys.exit(1)
    else:
        print('SUCCESS: Communication established with device.')

    # Load flashloader onto device
    print('Loading flashloader...')
    if sdp.write_file('0x20000000', '../../Flashloader/ivt_flashloader.bin')['ret']:
        print('ERROR: Could not write file!')
        sys.exit(1)
    else:
        print('SUCCESS: Flashloader loaded successfully.')

    # Jump to flash loader entry point
    print('Jumping to flashloader entry point...')
    if sdp.jump_to_address('0x20000400')['ret']:
        print('ERROR: Could not jump to address!')
        sys.exit(1)
    else:
        print('SUCCESS: Device jumped to execute flashloader.')

    bl = blhost.BLHost()

    # Poll device to make sure it is ready
    print('Waiting for device to be ready for blhost...')
    waitCount = 0
    while bl.get_property('0x01')['ret']:
        time.sleep(0.5)
        waitCount += 1
        if waitCount == 10:
            print(
                'ERROR: Timeout waiting for device to be ready. Power cycle device and try again.')
            sys.exit(1)

    print('SUCCESS: Device is ready for blhost!')

    # Read out unique ID
    print('Reading device unique ID...')
    prop = bl.get_property('0x12')
    if prop['ret']:
        print('ERROR: Could not read device unique ID!')
        sys.exit(1)
    else:
        ser_num = helpers.encode_unique_id(prop['response'])
        print('SUCCESS: Device serial number is %s' % (ser_num))

    # Write config option block to RAM
    print('Writing memory config option block...')
    if bl.fill_memory('0x2000', '0x4', '0xc0333006')['ret']:
        print('ERROR: Could not fill memory!')
        sys.exit(1)
    else:
        print('SUCCESS: Config option block loaded into RAM.')

    # Configure FlexSPI
    print('Configuring FlexSPI...')
    if bl.configure_memory('0x9', '0x2000')['ret']:
        print('ERROR: Could not configure memory!')
        sys.exit(1)
    else:
        print('SUCCESS: FlexSPI configured.')

    # Erase flash
    print('Erasing flash...')
    if bl.flash_erase_region('0x60000000', '0x2000000')['ret']:
        print('ERROR: Could not erase memory!')
        sys.exit(1)
    else:
        print('SUCCESS: Flash erased.')

    # Write files to flash
    print('Programming flash ...')
    if bl.write_memory('0x60000000', IMG_DIR + bootstrap_name)['ret']:
        print('ERROR: Could not write file to memory!')
        sys.exit(1)
    else:
        print('SUCCESS: File written to flash.')

    print('Programming flash ...')
    if bl.write_memory('0x60040000', IMG_DIR + bootloader_name)['ret']:
        print('ERROR: Could not write file to memory!')
        sys.exit(1)
    else:
        print('SUCCESS: File written to flash.')

    # Program root cert
    print('Programming flash with root cert...')
    root_ca_path = CA_CERTS_DIR + ca_name + '.root.ca.crt.pem'
    root_ca_bin = IMG_DIR + ca_name + '.root.ca.bin'
    helpers.file_format(root_ca_path, root_ca_bin)
    if bl.write_memory('0x61CC0000', root_ca_bin)['ret']:
        print('ERROR: Could not program flash with certificates for this "thing"!')
        sys.exit(1)
    else:
        print('SUCCESS: Programmed flash with certificates for this "thing".')

    # Program app cert
    print('Programming flash with app cert application A...')
    leaf_ca_path = CA_CERTS_DIR + ca_name + '.app.a.crt.pem'
    leaf_ca_bin = IMG_DIR + ca_name + '.app.a.bin'
    helpers.file_format(leaf_ca_path, leaf_ca_bin)
    if bl.write_memory('0x61D00000', leaf_ca_bin)['ret']:
        print('ERROR: Could not program flash with certificates for this "thing"!')
        sys.exit(1)
    else:
        print('SUCCESS: Programmed flash with certificates for this "thing".')

    # Program bootloader cert
    print('Programming flash with app cert for bootloader...')
    if bl.write_memory('0x61D80000', leaf_ca_bin)['ret']:
        print('ERROR: Could not program flash with certificates for this "thing"!')
        sys.exit(1)
    else:
        print('SUCCESS: Programmed flash with certificates for this "thing".')

    # Sign bootloader and application and generate fica table
    try:
        copyfile(IMG_DIR + mainapp_name, SIGN_DIR + mainapp_name)
    except:
        print('ERROR: Unable to copy ' + mainapp_name)
        sys.exit(1)

    try:
        copyfile(IMG_DIR + bootloader_name, SIGN_DIR + bootloader_name)
    except:
        print('ERROR: Unable to copy bootloader!')
        sys.exit(1)

    # NOTE: Script is written to program the bank A image and signing certificate only
    # python3 sign_package.py -p prefix -m main_app -a ca_name.app.a
    cmd = ['python3', 'sign_package.py', '-p', app_prefix,
           '-m', mainapp, '-a', ca_name + '.app.a']
    out = subprocess.run(
        cmd, cwd=SIGN_DIR, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print('SUCCESS: sign_package succeeded.')
    else:
        print("ERROR: sign_package failed!")
        print(str(out.stdout.strip(), 'utf-8', 'strict'))
        print(str(out.stderr.strip(), 'utf-8', 'strict'))
        sys.exit(1)

    try:
        copyfile(SIGN_DIR + 'fica_table.bin', IMG_DIR + 'fica_table.bin')
    except:
        print('ERROR: Unable to copy fica table!')
        sys.exit(1)

    # Program FICA table
    print('Programming FICA table...')
    fica_path = IMG_DIR + 'fica_table.bin'
    if bl.write_memory('0x61FC0000', fica_path)['ret']:
        print('ERROR: Could not program flash with certificates for this "thing"!')
        sys.exit(1)
    else:
        print('SUCCESS: Programmed flash with certificates for this "thing".')

    print('Programming flash...')
    if bl.write_memory('0x60300000', IMG_DIR + mainapp_name)['ret']:
        print('ERROR: Could not write file to memory!')
        sys.exit(1)
    else:
        print('SUCCESS: File written to flash.')

    # Get app entry point
    read_entry_resp = bl.read_memory('0x60002004', '4')
    entry_pnt = 0
    if read_entry_resp['ret']:
        print('ERROR: Could ready memory!')
        sys.exit(1)
    else:
        entry_pnt = helpers.bytes_to_word(read_entry_resp['response'], 4)
        print('SUCCESS: Application entry point at %s' % (entry_pnt))

    # Get initial stack pointer
    read_sp_resp = bl.read_memory('0x60002000', '4')
    stack_pnt = 0
    if read_sp_resp['ret']:
        print('ERROR: Could ready memory!')
        sys.exit(1)
    else:
        stack_pnt = helpers.bytes_to_word(read_sp_resp['response'], 4)
        print('SUCCESS: Application entry point at %s' % (stack_pnt))

    # Execute application
    print('Attemping to execute application...')
    if bl.execute(entry_pnt, '0', stack_pnt)['ret']:
        print('ERROR: Could not execute application!')
        sys.exit(1)
    else:
        print('SUCCESS: Application running.')


if __name__ == '__main__':
    main()
