/*
 * The Clear BSD License
 * Copyright (c) 2017, NXP Semiconductor, Inc.
 * All rights reserved.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided
 * that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS LICENSE.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * sdcard_test.c
 *
 */

#include "main.h"

extern bool done;

#define MSG_STRING_LEN 80
static char msgString[MSG_STRING_LEN];

/*! @brief Data block count accessed in card */
#define DATA_BLOCK_COUNT (5U)
/*! @brief Start data block number accessed in card */
#define DATA_BLOCK_START (2U)
/*! @brief Data buffer size. */
#define DATA_BUFFER_SIZE (FSL_SDMMC_DEFAULT_BLOCK_SIZE * DATA_BLOCK_COUNT)


/*! @brief Card descriptor. */
sd_card_t g_sd;

/* @brief decription about the read/write buffer
* The size of the read/write buffer should be a multiple of 512, since SDHC/SDXC card uses 512-byte fixed
* block length and this driver example is enabled with a SDHC/SDXC card.If you are using a SDSC card, you
* can define the block length by yourself if the card supports partial access.
* The address of the read/write buffer should align to the specific DMA data buffer address align value if
* DMA transfer is used, otherwise the buffer address is not important.
* At the same time buffer address/size should be aligned to the cache line size if cache is supported.
*/
/*! @brief Data written to the card */
SDK_ALIGN(uint8_t g_dataWrite[SDK_SIZEALIGN(DATA_BUFFER_SIZE, SDMMC_DATA_BUFFER_ALIGN_CACHE)],
          MAX(SDMMC_DATA_BUFFER_ALIGN_CACHE, SDMMCHOST_DMA_BUFFER_ADDR_ALIGN));
/*! @brief Data read from the card */
SDK_ALIGN(uint8_t g_dataRead[SDK_SIZEALIGN(DATA_BUFFER_SIZE, SDMMC_DATA_BUFFER_ALIGN_CACHE)],
          MAX(SDMMC_DATA_BUFFER_ALIGN_CACHE, SDMMCHOST_DMA_BUFFER_ADDR_ALIGN));

/*! @brief SDMMC host detect card configuration */
static const sdmmchost_detect_card_t s_sdCardDetect = {
#ifndef BOARD_SD_DETECT_TYPE
    .cdType = kSDMMCHOST_DetectCardByGpioCD,
#else
    .cdType = BOARD_SD_DETECT_TYPE,
#endif
    .cdTimeOut_ms = (3000),
};

/*! @brief SDMMC card power control configuration */
#if defined DEMO_SDCARD_POWER_CTRL_FUNCTION_EXIST
static const sdmmchost_pwr_card_t s_sdCardPwrCtrl = {
    .powerOn = BOARD_PowerOnSDCARD, .powerOnDelay_ms = 500U, .powerOff = BOARD_PowerOffSDCARD, .powerOffDelay_ms = 0U,
};
#endif

static status_t AccessCard(sd_card_t *card, bool isReadOnly)
{
    if (isReadOnly)
    {
    	snprintf(msgString,MSG_STRING_LEN,"\r\nRead one data block......\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);

        if (kStatus_Success != SD_ReadBlocks(card, g_dataRead, DATA_BLOCK_START, 1U))
        {
        	snprintf(msgString,MSG_STRING_LEN,"Read one data block failed.\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
            return kStatus_Fail;
        }

        snprintf(msgString,MSG_STRING_LEN,"Read multiple data blocks......\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);
        if (kStatus_Success != SD_ReadBlocks(card, g_dataRead, DATA_BLOCK_START, DATA_BLOCK_COUNT))
        {
        	snprintf(msgString,MSG_STRING_LEN,"Read multiple data blocks failed.\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
            return kStatus_Fail;
        }
    }
    else
    {
        memset(g_dataWrite, 0x67U, sizeof(g_dataWrite));

        snprintf(msgString,MSG_STRING_LEN,"\r\nWrite/read one data block......\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);
        if (kStatus_Success != SD_WriteBlocks(card, g_dataWrite, DATA_BLOCK_START, 1U))
        {
        	snprintf(msgString,MSG_STRING_LEN,"Write one data block failed.\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
            return kStatus_Fail;
        }

        memset(g_dataRead, 0U, sizeof(g_dataRead));
        if (kStatus_Success != SD_ReadBlocks(card, g_dataRead, DATA_BLOCK_START, 1U))
        {
        	snprintf(msgString,MSG_STRING_LEN,"Read one data block failed.\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
            return kStatus_Fail;
        }

        snprintf(msgString,MSG_STRING_LEN,"Compare the read/write content......\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);
        if (memcmp(g_dataRead, g_dataWrite, FSL_SDMMC_DEFAULT_BLOCK_SIZE))
        {
        	snprintf(msgString,MSG_STRING_LEN,"The read/write content isn't consistent.\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
            return kStatus_Fail;
        }
        snprintf(msgString,MSG_STRING_LEN,"The read/write content is consistent.\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);

    	snprintf(msgString,MSG_STRING_LEN,"Write/read multiple data blocks......\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);
        if (kStatus_Success != SD_WriteBlocks(card, g_dataWrite, DATA_BLOCK_START, DATA_BLOCK_COUNT))
        {
        	snprintf(msgString,MSG_STRING_LEN,"Write multiple data blocks failed.\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
            return kStatus_Fail;
        }

        memset(g_dataRead, 0U, sizeof(g_dataRead));

        if (kStatus_Success != SD_ReadBlocks(card, g_dataRead, DATA_BLOCK_START, DATA_BLOCK_COUNT))
        {
        	snprintf(msgString,MSG_STRING_LEN,"Read multiple data blocks failed.\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
            return kStatus_Fail;
        }

        snprintf(msgString,MSG_STRING_LEN,"Compare the read/write content......\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);
        if (memcmp(g_dataRead, g_dataWrite, FSL_SDMMC_DEFAULT_BLOCK_SIZE))
        {
        	snprintf(msgString,MSG_STRING_LEN,"The read/write content isn't consistent.\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
            return kStatus_Fail;
        }
        snprintf(msgString,MSG_STRING_LEN,"The read/write content is consistent.\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);

    	snprintf(msgString,MSG_STRING_LEN,"Erase multiple data blocks......\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);
        if (kStatus_Success != SD_EraseBlocks(card, DATA_BLOCK_START, DATA_BLOCK_COUNT))
        {
        	snprintf(msgString,MSG_STRING_LEN,"Erase multiple data blocks failed.\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
            return kStatus_Fail;
        }
    }

    return kStatus_Success;
}

static void CardInformationLog(sd_card_t *card)
{
    assert(card);

    snprintf(msgString,MSG_STRING_LEN,"Card size %d * %d bytes\n", card->blockCount, card->blockSize);
	GUI_DispString(msgString);
	PRINTF(msgString);
	snprintf(msgString,MSG_STRING_LEN,"\r\nWorking condition:\r\n");
	GUI_DispString(msgString);
	PRINTF(msgString);

    if (card->operationVoltage == kCARD_OperationVoltage330V)
    {
    	snprintf(msgString,MSG_STRING_LEN,"\r\n  Voltage : 3.3V\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);
    }
    else if (card->operationVoltage == kCARD_OperationVoltage180V)
    {
    	snprintf(msgString,MSG_STRING_LEN,"\r\n  Voltage : 1.8V\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);
    }

    if (card->currentTiming == kSD_TimingSDR12DefaultMode)
    {
        if (card->operationVoltage == kCARD_OperationVoltage330V)
        {
        	snprintf(msgString,MSG_STRING_LEN,"\r\n  Timing mode: Default mode\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
        }
        else if (card->operationVoltage == kCARD_OperationVoltage180V)
        {
        	snprintf(msgString,MSG_STRING_LEN,"\r\n  Timing mode: SDR12 mode\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
        }
    }
    else if (card->currentTiming == kSD_TimingSDR25HighSpeedMode)
    {
        if (card->operationVoltage == kCARD_OperationVoltage180V)
        {
        	snprintf(msgString,MSG_STRING_LEN,"\r\n  Timing mode: SDR25\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
        }
        else
        {
        	snprintf(msgString,MSG_STRING_LEN,"\r\n  Timing mode: High Speed\r\n");
        	GUI_DispString(msgString);
        	PRINTF(msgString);
        }
    }
    else if (card->currentTiming == kSD_TimingSDR50Mode)
    {
    	snprintf(msgString,MSG_STRING_LEN,"\r\n  Timing mode: SDR50\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);
    }
    else if (card->currentTiming == kSD_TimingSDR104Mode)
    {
    	snprintf(msgString,MSG_STRING_LEN,"\r\n  Timing mode: SDR104\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);
    }
    else if (card->currentTiming == kSD_TimingDDR50Mode)
    {
    	snprintf(msgString,MSG_STRING_LEN,"\r\n  Timing mode: DDR50\r\n");
    	GUI_DispString(msgString);
    	PRINTF(msgString);
    }

    snprintf(msgString,MSG_STRING_LEN,"\r\n  Freq : %d HZ\r\n", card->busClock_Hz);
	GUI_DispString(msgString);
	PRINTF(msgString);
}


void sdcard_test_cmd (uint32_t argc, char *argv[])
{
    sd_card_t *card = &g_sd;
    char ch = '0';
    bool isReadOnly;

    card->host.base = SD_HOST_BASEADDR;
    card->host.sourceClock_Hz = SD_HOST_CLK_FREQ;
    /* card detect type */
    card->usrParam.cd = &s_sdCardDetect;
#if defined DEMO_SDCARD_POWER_CTRL_FUNCTION_EXIST
    card->usrParam.pwr = &s_sdCardPwrCtrl;
#endif

    GUI_Clear();

    snprintf(msgString,MSG_STRING_LEN,"\r\n\rLPC54018 SDCARD test\r\n");
    GUI_DispString(msgString);
    PRINTF(msgString);

    /* SD host init function */
    if (SD_HostInit(card) != kStatus_Success)
    {
        snprintf(msgString,MSG_STRING_LEN,"SD host init fail\r\n");
        GUI_DispString(msgString);
        PRINTF(msgString);
        goto exit;
    }

    snprintf(msgString,MSG_STRING_LEN,"Please insert an SD card into the board.\r\n");
    GUI_DispString(msgString);
    PRINTF(msgString);

    /* power off card */
    SD_PowerOffCard(card->host.base, card->usrParam.pwr);

	if (SD_WaitCardDetectStatus(SD_HOST_BASEADDR, &s_sdCardDetect, true) == kStatus_Success)
	{
	    snprintf(msgString,MSG_STRING_LEN,"Card inserted.\r\n");
	    GUI_DispString(msgString);
	    PRINTF(msgString);

		/* reset host once card re-plug in */
		SD_HostReset(&(card->host));
		/* power on the card */
		SD_PowerOnCard(card->host.base, card->usrParam.pwr);
	}
	else
	{
	    snprintf(msgString,MSG_STRING_LEN,"Card detect fail.\r\n");
	    GUI_DispString(msgString);
	    PRINTF(msgString);
		goto exit;
	}

	/* Init card. */
	if (SD_CardInit(card))
	{
	    snprintf(msgString,MSG_STRING_LEN,"SD card init failed.\r\n");
	    GUI_DispString(msgString);
	    PRINTF(msgString);
		goto exit;
	}

	/* card information log */
    CardInformationLog(card);

    /* Check if card is readonly. */
    isReadOnly = SD_CheckReadOnly(card);

    if (kStatus_Success != AccessCard(card, isReadOnly))
    {
		/* access card fail, due to card removal. */
		if (SD_WaitCardDetectStatus(SD_HOST_BASEADDR, &s_sdCardDetect, false) == kStatus_Success)
		{
			snprintf(msgString,MSG_STRING_LEN,"Card removed.\r\n");
			GUI_DispString(msgString);
			PRINTF(msgString);
		}
		/* access card fail, due to transfer error */
		else
		{
			snprintf(msgString,MSG_STRING_LEN,"TEST FAILED - check card inserted correctly and repeat test.\r\n");
			GUI_DispString(msgString);
			PRINTF(msgString);
		}
	}
	else
	{
	    snprintf(msgString,MSG_STRING_LEN,"TEST PASSED.\r\n");
	    GUI_DispString(msgString);
	    PRINTF(msgString);
	}

    SD_Deinit(card);

exit:

    done = false;
    GUI_Display_Prompt();
    cli_display_prompt();
}
