/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2011-2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file tsi.c
*
* @author a13984
*
* @version 0.0.1
*
* @date Aug 23, 2011
*
* @brief application entry point which performs application specific tasks. 
*
*******************************************************************************
*
* provide a demo for how to initialize the PT60, output messages via SCI, 
* flash operations, etc.
* NOTE:
*	printf call may occupy a lot of memory (around 1924 bytes), so please
*	consider your code size before using printf.
******************************************************************************/
#include "common.h"
#include "tsi.h"

/******************************************************************************
* Constants and macros
******************************************************************************/

/******************************************************************************
* Global variables
******************************************************************************/
ELECTRODE_WORK_INFO  g_ElecWorkInfo[ELECTRODE_NUM];
static unsigned short m_uiScanValue[ELECTRODE_NUM];
unsigned char g_ucSignalChangeFlag;
//static  unsigned int m_uiScanValue[ELEC_NUM];       //result buffer array
//static  unsigned char m_ucScanChannel[ELEC_NUM];   //channel buffer array
static  unsigned char m_ucIndex;

void (* const KeyHandle[ELECTRODE_NUM])(void) =
{
	KEY1_Processing,	/* call back functions defined externally */
	KEY2_Processing,
	KEY3_Processing,
	KEY4_Processing,
};

/******************************************************************************
* Local types
******************************************************************************/

/******************************************************************************
* Local function prototypes
******************************************************************************/
void TSI_VaribleInit( void );
void TSI_Init( void );
void TSI_Scan( unsigned short *pResult );
void TSI_Processing( void );
void TSI_CheckState( void );



/******************************************************************************
* Local variables
******************************************************************************/

/******************************************************************************
* Local functions
*****************************************************************************/

/******************************************************************************
* Global functions
******************************************************************************/

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_VaribleInit
   *
   * @brief initialize TSI variables
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/

void TSI_VaribleInit( void )
{
	unsigned char i;
	m_ucIndex = 0;
	for(i=0;i<ELECTRODE_NUM;i++)
	{
		g_ElecWorkInfo[i].State = KEY_RELEASED;
#if( BOARD_TYPE == EVB_BOARD )
		g_ElecWorkInfo[i].Channel = i;          //enable channel 0,1,2,3
#endif
		g_ElecWorkInfo[i].Count = 0;
	}
#if( BOARD_TYPE == TOWER_BOARD )
		g_ElecWorkInfo[0].Channel = 0x0f;          
		g_ElecWorkInfo[1].Channel = 0x0e;  
		g_ElecWorkInfo[2].Channel = 0x01;  
		g_ElecWorkInfo[3].Channel = 0x00;  
#endif
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_Init
   *
   * @brief TSI module initialization
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/

void TSI_Init( void )
{
	unsigned char i,j;
	//
	TSI_CS1_PS = TSI_CS1_PS_16;
	TSI_CS1_NSCN = TSI_CS1_NSCN_32;
	TSI_CS2 = TSI_CS2_REFCHRG_32uA | TSI_CS2_DVOLT_00 | TSI_CS2_EXTCHRG_32uA;
	
	// enable channel
	TSI_PEN0 = TSI_PEN0_VALUE;
	TSI_PEN1 = TSI_PEN1_VALUE;
	// 
	TSI_CS0_STPE = 1;
	// enable interrupt
	//TS1_CS0_TSIIEN = 1;
	// enable TSI
	TSI_CS0_TSIEN = 1;
	
	TSI_VaribleInit();
#if 0	
#if( BOARD_TYPE == TOWER_BOARD )
	  LED0_Init();
	  LED1_Init();
	  LED2_Init();
	  LED3_Init(); 
#endif
#endif	
	// Get baseline
	TSI_Scan( &m_uiScanValue[0] );
	for(i=0;i<ELECTRODE_NUM;i++)
	{
		g_ElecWorkInfo[i].BaseLine = m_uiScanValue[i];
	}
	for(j=0;j<10;j++)
	{
		TSI_Scan( &m_uiScanValue[0] );
		for(i=0;i<ELECTRODE_NUM;i++)
		{
			g_ElecWorkInfo[i].BaseLine = (m_uiScanValue[i]+g_ElecWorkInfo[i].BaseLine)>>1;
		}
	}
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_Scan
   *
   * @brief polling method to complete scanning all of electrode
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/

void TSI_Scan( unsigned short *pResult )
{
	unsigned char i;
	unsigned int uiScanValue;
	for(i=0;i<ELECTRODE_NUM;i++)
	{
		// specify the scanning channel	
		TSI_CS3_TSICH = g_ElecWorkInfo[i].Channel;
		// start scanning
		TSI_CS0 |= 0x01;
		// wait scan complete
		while(!TSI_CS0_EOSF);
		TSI_CS0_EOSF = 1;
		pResult[i] = TSI_CNT;
	
	}
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_Processing
   *
   * @brief process the scanning data to determine the key status
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/

void TSI_Processing( void )
{
	unsigned char i;
	unsigned short uiDelta;
	//start scan
	TSI_Scan( &m_uiScanValue[0] );
	
	// calculate delta
	for(i=0;i<ELECTRODE_NUM;i++)
	{
		g_ElecWorkInfo[i].Delta = m_uiScanValue[i] - g_ElecWorkInfo[i].BaseLine;
	}
	
	// determine key state
	for(i=0;i<ELECTRODE_NUM;i++)
	{
		if( g_ElecWorkInfo[i].Delta > 0 )
		{
			uiDelta = g_ElecWorkInfo[i].Delta;
		}
		else
		{
			uiDelta = 0;
		}
		if( uiDelta > KEY_ON_SENSITIVITY )
		{
			if( g_ElecWorkInfo[i].State == KEY_RELEASED )
			{
				if(g_ElecWorkInfo[i].Count > KEY_OVERTIME)
				{
					g_ElecWorkInfo[i].State = KEY_TOUCHED;
					g_ucSignalChangeFlag |= (0x01<<i);
					g_ElecWorkInfo[i].Count = 0;
				}
				else
				{
					g_ElecWorkInfo[i].Count ++;
				}
			}
			else if( g_ElecWorkInfo[i].State == KEY_TOUCHED )
			{
				//repeat touch
				//
			}
		}
		else if( uiDelta < KEY_OFF_SENSITIVITY )
		{
			if( g_ElecWorkInfo[i].State == KEY_TOUCHED )
			{
				if(g_ElecWorkInfo[i].Count > KEY_OVERTIME)
				{
					g_ElecWorkInfo[i].State = KEY_RELEASED;
					g_ucSignalChangeFlag |= (0x01<<i);
					g_ElecWorkInfo[i].Count = 0;
				}
				else
				{
					g_ElecWorkInfo[i].Count ++;
				}
			}
			else if( g_ElecWorkInfo[i].State == KEY_RELEASED )
			{
				// always released
				//
			}
		}
		else
		{
			// is noise or other
			//
		}
	}
	// check key state,processing
	TSI_CheckState();
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_CheckState
   *
   * @brief check the change of the key status and implement the corresponding function
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/

void TSI_CheckState( void )
{
	unsigned char i;
	if( g_ucSignalChangeFlag != 0 )
	{
		//key state occur change,handle corresponding key
		for(i=0;i<ELECTRODE_NUM;i++)
		{
			if( g_ucSignalChangeFlag&(0x01<<i) )
			{
				//key 0 release or touched
				if( g_ElecWorkInfo[i].State == KEY_TOUCHED)
				{
					//
					KeyHandle[i]();
				}
				//clear flag
				g_ucSignalChangeFlag &= (~(0x01<<i)); 
			}
		}
	}
}



