/*
 * ringbuf.c
 *
 *  Created on: Jul 5, 2018
 *      Author: nxf45772
 */


#include <stdio.h>
#include <string.h>
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"
#include "queue.h"
#include "ringbuf.h"



struct ringbuf {
    char *p_o;                   /**< Original pointer */
    char *volatile p_r;          /**< Read pointer */
    char *volatile p_w;          /**< Write pointer */
    volatile uint32_t fill_cnt;  /**< Number of filled slots */
    uint32_t size;               /**< Buffer size */
    uint32_t block_size;         /**< Block size */
    SemaphoreHandle_t can_read;
    SemaphoreHandle_t can_write;
    SemaphoreHandle_t task_block;
    bool is_done_write;         /**< to prevent infinite blocking for buffer read */
};

static status_t rb_abort_read(ringbuf_handle_t rb);
static status_t rb_abort_write(ringbuf_handle_t rb);
static void rb_release(SemaphoreHandle_t handle);

ringbuf_handle_t rb_create(int size, int block_size)
{
    if (size < 2) {
        return NULL;
    }

    if (size % block_size != 0) {
        return NULL;
    }
    ringbuf_handle_t rb;
    char *buf = NULL;
    bool _success =
        (
            (rb                 = pvPortMalloc(sizeof(struct ringbuf))) &&
            (buf                = pvPortMalloc(size))                &&
            (rb->can_read       = xSemaphoreCreateBinary())             &&
            (rb->task_block     = xSemaphoreCreateMutex())              &&
            (rb->can_write      = xSemaphoreCreateBinary())
        );

    if(!_success){
    	 goto _rb_init_failed;
    }
    memset(buf,0,size);
    rb->p_o = rb->p_r = rb->p_w = buf;
    rb->fill_cnt = 0;
    rb->size = size;
    rb->block_size = block_size;
    rb->is_done_write = false;
    return rb;
_rb_init_failed:
    rb_destroy(rb);
    return NULL;
}

status_t rb_destroy(ringbuf_handle_t rb)
{
    if (rb == NULL) {
        return STATUS_ERROR;
    }
    vPortFree(rb->p_o);
    rb->p_o = NULL;

    vSemaphoreDelete(rb->can_read);
    vSemaphoreDelete(rb->can_write);
    vSemaphoreDelete(rb->task_block);

    rb->can_read = NULL;
    rb->can_write = NULL;
    rb->task_block = NULL;
    vPortFree(rb);
    rb = NULL;
    return STATUS_SUCCESS;
}

status_t rb_reset(ringbuf_handle_t rb)
{
//    if (rb == NULL) {
//        return STATUS_ERROR;
//    }
//    rb->p_r = rb->p_w = rb->p_o;
//    rb->fill_cnt = 0;
//    rb->is_done_write = false;
//
//    int dummy = 0;
//    QueueSetMemberHandle_t active_handle;
//    while ((active_handle = xQueueSelectFromSet(rb->read_set, 0))) {
//        if (active_handle == rb->abort_read) {
//            xQueueReceive(active_handle, &dummy, 0);
//        } else {
//            xSemaphoreTake(active_handle, 0);
//        }
//    }
//    while ((active_handle = xQueueSelectFromSet(rb->write_set, 0))) {
//        if (active_handle == rb->abort_write) {
//            xQueueReceive(active_handle, &dummy, 0);
//        } else {
//            xSemaphoreTake(active_handle, 0);
//        }
//    }
//    rb_release(rb->can_write);
    return STATUS_SUCCESS;
}

int rb_bytes_available(ringbuf_handle_t rb)
{
    return (rb->size - rb->fill_cnt);
}

int rb_bytes_filled(ringbuf_handle_t rb)
{
    return rb->fill_cnt;
}

static void rb_release(SemaphoreHandle_t handle)
{
    xSemaphoreGive(handle);
}

#define rb_block(handle, time) xSemaphoreTake(handle, time)

static int rb_claim_read(ringbuf_handle_t rb, TickType_t ticks_to_wait)
{
    for (;;) {
        if(!xQueueReceive(rb->can_read,NULL,ticks_to_wait)){
            return RB_TIMEOUT;
        }  else {
            xSemaphoreTake(rb->can_read, 0);
            return RB_OK;
        }
    }
}

static int rb_claim_write(ringbuf_handle_t rb, TickType_t ticks_to_wait)
{
    for (;;) {
        if(!xQueueReceive(rb->can_write,NULL,ticks_to_wait)){
            return RB_TIMEOUT;
        } else {
            xSemaphoreTake(rb->can_write, 0);
            return RB_OK;
        }
    }
}


int rb_read(ringbuf_handle_t rb, char *buf, int buf_len, TickType_t ticks_to_wait)
{
    int read_size = 0, remainder = 0;
    int total_read_size = 0;
    int ret_val = 0;
    if (buf_len == 0) {
        return STATUS_ERROR;
    }

    while (buf_len) {
        //block access ringbuf to this thread
        if (rb_block(rb->task_block, ticks_to_wait) != pdTRUE) {
            ret_val =  RB_TIMEOUT;
            goto read_err;
        }

        if (rb->fill_cnt < buf_len) {
            remainder = rb->fill_cnt % 4;
            read_size = rb->fill_cnt - remainder;
            if ((read_size == 0) && rb->is_done_write) {
                read_size = rb->fill_cnt;
            }
        } else {
            read_size = buf_len;
        }

        if (read_size == 0) {
            //there are no data to read, release thread block to allow other threads to write data
            rb_release(rb->task_block);

            //wait for new data
            if (rb->is_done_write) {
                ret_val = RB_DONE;
                goto read_err;
            }
            int ret = rb_claim_read(rb, ticks_to_wait);
            if (RB_OK == ret) {
                continue;
            } else {
                ret_val = ret;
                goto read_err;
            }
        }

        if ((rb->p_r + read_size) > (rb->p_o + rb->size)) {
            int rlen1 = rb->p_o + rb->size - rb->p_r;
            int rlen2 = read_size - rlen1;
            memcpy(buf, rb->p_r, rlen1);
            memcpy(buf + rlen1, rb->p_o, rlen2);
            rb->p_r = rb->p_o + rlen2;
        } else {
            memcpy(buf, rb->p_r, read_size);
            rb->p_r = rb->p_r + read_size;
        }

        buf_len -= read_size;
        rb->fill_cnt -= read_size;
        total_read_size += read_size;
        buf += read_size;
        rb_release(rb->task_block);
        if (buf_len == 0) {
            break;
        }
    }
read_err:
    if (total_read_size > 0) {
        rb_release(rb->can_write);
    }
    return total_read_size > 0 ? total_read_size : ret_val;
}

int rb_write(ringbuf_handle_t rb, char *buf, int buf_len, TickType_t ticks_to_wait)
{
    int write_size;
    int total_write_size = 0;
    int ret_val = 0;
    if (buf_len == 0) {
        return -1;
    }

    while (buf_len) {
        if (rb_block(rb->task_block, ticks_to_wait) != pdTRUE) {
            ret_val =  RB_TIMEOUT;
            goto write_err;
        }
        write_size = rb_bytes_available(rb);

        if (buf_len < write_size) {
            write_size = buf_len;
        }


        if (write_size == 0) {
            //there are no data to read, release thread block to allow other threads to write data
            if (total_write_size > 0) {
                rb_release(rb->can_read);
            }

            rb_release(rb->task_block);
            if (rb->is_done_write) {
                ret_val = RB_DONE;
                goto write_err;
            }
            //wait for new data
            int ret = rb_claim_write(rb, ticks_to_wait);
            if (RB_OK == ret) {
                continue;
            } else {
                ret_val = ret;
                goto write_err;
            }
        }

        if ((rb->p_w + write_size) > (rb->p_o + rb->size)) {
            int wlen1 = rb->p_o + rb->size - rb->p_w;
            int wlen2 = write_size - wlen1;
            memcpy(rb->p_w, buf, wlen1);
            memcpy(rb->p_o, buf + wlen1, wlen2);
            rb->p_w = rb->p_o + wlen2;
        } else {
            memcpy(rb->p_w, buf, write_size);
            rb->p_w = rb->p_w + write_size;
        }

        buf_len -= write_size;
        rb->fill_cnt += write_size;
        total_write_size += write_size;
        buf += write_size;
        rb_release(rb->can_read);
        rb_release(rb->task_block);
        if (buf_len == 0) {
            break;
        }

    }
write_err:
    if (total_write_size > 0) {
        rb_release(rb->can_read);
    }
    return total_write_size > 0 ? total_write_size : ret_val;
}

static status_t rb_abort_read(ringbuf_handle_t rb)
{

    return STATUS_SUCCESS;
}

static status_t rb_abort_write(ringbuf_handle_t rb)
{
    return STATUS_SUCCESS;
}

status_t rb_abort(ringbuf_handle_t rb)
{
	status_t err = rb_abort_read(rb);
    err |= rb_abort_write(rb);
    return err;
}

bool rb_is_full(ringbuf_handle_t rb)
{
    if (rb == NULL) {
        return false;
    }
    return (rb->size == rb->fill_cnt);
}

int rb_size_get(ringbuf_handle_t rb)
{
    if (rb == NULL) {
        return 0;
    }
    return (rb->size);
}

status_t rb_done_write(ringbuf_handle_t rb)
{
    if (rb == NULL) {
        return STATUS_ERROR;
    }
    rb->is_done_write = true;
    rb_release(rb->can_read);
    return STATUS_SUCCESS;
}

bool rb_is_done_write(ringbuf_handle_t rb)
{
    if (rb == NULL) {
        return false;
    }
    return (rb->is_done_write);
}

int rb_get_size(ringbuf_handle_t rb)
{
    if (rb == NULL) {
        return STATUS_ERROR;
    }
    return rb->size;
}
