/****************************************************************************************************/
/**
Copyright (c) 2009 Freescale Semiconductor

\file       Thermostat_TempControl.c
\brief      Controls the HVAC outputs based on the Heating and Cooling SetPoints
\author     Freescale Semiconductor
\author     Technical Information Center
\version    1.0
\date       February/2009
*/
/********************************************************************************************
* Freescale reserves the right to make changes without further notice to any product
* herein to improve reliability, function, or design. Freescale does not assume any
* liability arising out of the application or use of any product, circuit, or software
* described herein; neither does it convey any license under its patent rights nor the
* rights of others. Freescale products are not designed, intended, or authorized for
* use as components in systems intended for surgical implant into the body, or other
* applications intended to support life, or for any other application in which the
* failure of the Freescale product could create a situation where personal injury or
* death may occur. Should Buyer purchase or use Freescale products for any such
* intended or unauthorized application, Buyer shall indemnify and hold Freescale and
* its officers, employees, subsidiaries, affiliates, and distributors harmless against
* all claims, costs, damages, and expenses, and reasonable attorney fees arising out
* of, directly or indirectly, any claim of personal injury or death associated with
* such unintended or unauthorized use, even if such claim alleges that Freescale was
* negligent regarding the design or manufacture of the part.
*
* Freescale is a registered trademark of Freescale, Inc.
********************************************************************************************/	

/*****************************************************************************************************
* Include files
*****************************************************************************************************/
#include "Thermostat_TempControl.h"

/*****************************************************************************************************
* Declaration of module wide FUNCTIONs - NOT for use in other modules
*****************************************************************************************************/
void vfnThermostat_InitGPIO(void);

/*****************************************************************************************************
* Definition of module wide MACROs / #DEFINE-CONSTANTs - NOT for use in other modules
*****************************************************************************************************/

/*****************************************************************************************************
* Declaration of module wide TYPEs - NOT for use in other modules
*****************************************************************************************************/

/*****************************************************************************************************
* Definition of module wide VARIABLEs - NOT for use in other modules
*****************************************************************************************************/

/*****************************************************************************************************
* Definition of module wide (CONST-) CONSTANTs - NOT for use in other modules
*****************************************************************************************************/
#pragma CONST_SEG NON_VOLATILE_DATA
const UINT8 CoolingSetPoints[DAYS_OF_WEEK] = 
{
  23,    //Sunday
  23,    //Monday
  23,    //Tuesday
  23,    //Wednesday
  23,    //Thursday
  23,    //Friday
  23     //Saturday
};

#pragma CONST_SEG NON_VOLATILE_DATA
const UINT8 HeatingSetPoints[DAYS_OF_WEEK] = 
{
  20,    //Sunday
  20,    //Monday
  20,    //Tuesday
  20,    //Wednesday
  20,    //Thursday
  20,    //Friday
  20     //Saturday
};
#pragma CONST_SEG DEFAULT

/*****************************************************************************************************
* Code of project wide FUNCTIONS
*****************************************************************************************************/
UINT8 gu8HoldTemperature;
UINT8 gu8HVACStatus;

/****************************************************************************************************/
/**
* \brief    Initializes the GPIO used for HVAC outputs as well as the Flash to make non-volatile backup of the parameters
* \author   
* \param    void
* \return   void
* \todo     
*/
void vfnTempControlInit(void)
{
  vfnThermostat_InitGPIO();
  FlashInit();
  CopyInRAM();
}

/****************************************************************************************************/
/**
* \brief    Initializes HVAC GPIO
* \author   
* \param    void
* \return   void
* \todo     
*/
void vfnThermostat_InitGPIO(void)
{
  HEAT_RESET_DDR = HVAC_OUT;
  HEAT_SET_DDR = HVAC_OUT;
  
  COOL_RESET_DDR = HVAC_OUT;
  COOL_SET_DDR = HVAC_OUT;
  
  VENT_RESET_DDR = HVAC_OUT;
  VENT_SET_DDR = HVAC_OUT;
  
  HEAT_OFF();
  COOL_OFF();
  VENT_OFF();
  
  #ifdef __LOWEND_TSTAT__
    PTAPE_PTAPE4 = 1;
    PTAPE_PTAPE5 = 1;
    PTADD_PTADD4 = 0;
    PTADD_PTADD5 = 0;
  #endif

}

/****************************************************************************************************/
/**
* \brief    Sends the HVAC Output to their Off State
* \author   
* \param    void
* \return   void
* \todo     
*/
void vfnThermostatResetOutputs(void)
{
  HEAT_SET_PIN = HVAC_PIN_OFF;
  HEAT_RESET_PIN = HVAC_PIN_OFF;
  
  COOL_SET_PIN = HVAC_PIN_OFF;
  COOL_RESET_PIN = HVAC_PIN_OFF;
  
  VENT_SET_PIN = HVAC_PIN_OFF;
  VENT_RESET_PIN = HVAC_PIN_OFF;
}

/*****************************************************************************************************
* \brief    Function to turn on and off the HVAC system 
* \author   Rafael Peralez
* \param    Temperature DayOfWeek
* \return   HVAC_CHANGED_STATUS   HVAC_SAME_STATUS
* \todo     
*/
UINT8 u8ThermostatTempControl(UINT8 Temperature, UINT8 DayOfWeek)
{
  UINT8 CoolSetPoint = CoolingSetPoints[DayOfWeek];
  UINT8 HeatSetPoint = HeatingSetPoints[DayOfWeek];
  UINT8 TempStatus = gu8HVACStatus;
  
  if (FLAG_CHK(HVAC_HOLD_MODE, gu8HVACStatus))
  {
    CoolSetPoint = gu8HoldTemperature;
    HeatSetPoint = gu8HoldTemperature;
  }
  /* When in hold mode the temperature is alerady in either C or F */
  else if (gu8TempMetricSystem == _FAHRENHEIT_)
  {
    CoolSetPoint = (UINT8)((CoolSetPoint * 9) / 5);
    CoolSetPoint += 32;
    HeatSetPoint = (UINT8)((HeatSetPoint * 9) / 5);
    HeatSetPoint += 32;
  }
  
  

  //if (Temperature > CoolingSetPoints[DayOfWeek])
  if (Temperature >= CoolSetPoint)
  {
    if (!FLAG_CHK(HVAC_COOLING, gu8HVACStatus))
    {
      COOL_ON();
      FLAG_SET(HVAC_COOLING, gu8HVACStatus);
    }
  }
  else 
  {
    if (FLAG_CHK(HVAC_COOLING, gu8HVACStatus))
    {
      FLAG_CLR(HVAC_COOLING, gu8HVACStatus);
      COOL_OFF();
    }
  }
  
  //if (Temperature < HeatingSetPoints[DayOfWeek])
  if (Temperature < HeatSetPoint)
  {
    if (!FLAG_CHK(HVAC_HEATING, gu8HVACStatus))
    {
      HEAT_ON();
      FLAG_SET(HVAC_HEATING, gu8HVACStatus);
    }
  }
  else 
  {
    if (FLAG_CHK(HVAC_HEATING, gu8HVACStatus))
    {
      FLAG_CLR(HVAC_HEATING, gu8HVACStatus);
      HEAT_OFF();
    }
  }
  
  if ((FAN_SWITCH_ON) || FLAG_CHK(HVAC_COOLING, gu8HVACStatus) || FLAG_CHK(HVAC_HEATING, gu8HVACStatus))
  {
    if (!FLAG_CHK(HVAC_VENTING, gu8HVACStatus))
    {
      FLAG_SET(HVAC_VENTING, gu8HVACStatus);
      VENT_ON();
    }
  }
  else if ((!FAN_SWITCH_ON) && !(FLAG_CHK((HVAC_HEATING | HVAC_COOLING), gu8HVACStatus)))
  {
    if (FLAG_CHK(HVAC_VENTING, gu8HVACStatus))
    {
      FLAG_CLR(HVAC_VENTING, gu8HVACStatus);
      VENT_OFF();
    }
  }
  
    
  if (TempStatus != gu8HVACStatus)
  {
    return (HVAC_CHANGED_STATUS);
  }
  return (HVAC_SAME_STATUS);
}

/****************************************************************************************************/
/**
* \brief    Sends the Thermostat into Hold Mode to maintain a specific temperature
* \author   
* \param    void
* \return   void
* \todo     
*/
void vfnThermostatRunHold(UINT8 HoldTemperature)
{
  /* Toggle Run/Hold Flag */
  gu8HVACStatus ^= (1<<HVAC_HOLD_MODE);

  gu8HoldTemperature = HoldTemperature; 
}

