/****************************************************************************************************/
/****************************************************************************************************/
/**
Copyright (c) 2009 Freescale Semiconductor

\file       S08_Flash.c
\brief      Source code to write and erase flash in an S08 device
\author     Freescale Semiconductor
\author     Technical Information Center
\version    1.0
\date       February/2009
*/
/********************************************************************************************
* Freescale reserves the right to make changes without further notice to any product
* herein to improve reliability, function, or design. Freescale does not assume any
* liability arising out of the application or use of any product, circuit, or software
* described herein; neither does it convey any license under its patent rights nor the
* rights of others. Freescale products are not designed, intended, or authorized for
* use as components in systems intended for surgical implant into the body, or other
* applications intended to support life, or for any other application in which the
* failure of the Freescale product could create a situation where personal injury or
* death may occur. Should Buyer purchase or use Freescale products for any such
* intended or unauthorized application, Buyer shall indemnify and hold Freescale and
* its officers, employees, subsidiaries, affiliates, and distributors harmless against
* all claims, costs, damages, and expenses, and reasonable attorney fees arising out
* of, directly or indirectly, any claim of personal injury or death associated with
* such unintended or unauthorized use, even if such claim alleges that Freescale was
* negligent regarding the design or manufacture of the part.
*
* Freescale is a registered trademark of Freescale, Inc.
********************************************************************************************/	

/*****************************************************************************************************
* Include files
*****************************************************************************************************/
#include "S08_Flash.h"

/*****************************************************************************************************
* Declaration of module wide FUNCTIONs - NOT for use in other modules
*****************************************************************************************************/

/*****************************************************************************************************
* Definition of module wide MACROs / #DEFINE-CONSTANTs - NOT for use in other modules
*****************************************************************************************************/

/*****************************************************************************************************
* Declaration of module wide TYPEs - NOT for use in other modules
*****************************************************************************************************/

/*****************************************************************************************************
* Definition of module wide VARIABLEs - NOT for use in other modules
*****************************************************************************************************/
extern char __SEG_START_FLASH_ROUTINES[];
extern char __SEG_SIZE_FLASH_ROUTINES[];

/*****************************************************************************************************
* Definition of module wide (CONST-) CONSTANTs - NOT for use in other modules
*****************************************************************************************************/

/*****************************************************************************************************
* Code of project wide FUNCTIONS
*****************************************************************************************************/
#pragma CODE_SEG DEFAULT
/****************************************************************************************************/
/**
* \brief    Initializes the Flash Clock Divider and clear error flags
* \author   
* \param    void
* \return   void
* \todo     
*/
void FlashInit(void)
{
  if (!(FCDIV & 0x80))
  {
    if (FSTAT&0x30)         //Check to see if FACCERR is set
    {
        FSTAT |= 0x30;  //write a 1 to FACCERR to clear
    }
    FCDIV = FLASH_CLOCK;
  }
}

/****************************************************************************************************/
/**
* \brief    Function to either write or erase the flash. Needs to run from RAM
* \author   
* \param    FlashAddress Address to be written or erased
* \param    FlashDataCounter Amount of bytes to be written (use 1 to erase)
* \param    pFlashDataPtr Pointer to where the data to be stored in flash is (useless for Erase)
* \param    FlashCommand Command to be executed
* \return   FLASH_OK or FLASH_ERROR
* \todo     
*/
#pragma CODE_SEG FLASH_ROUTINES
UINT8 Flash_Cmd(UINT16 FlashAddress, UINT8 FlashDataCounter, UINT8* pFlashDataPtr, UINT8 FlashCommand)
{
    /* Check to see if FACCERR or PVIOL is set */
    if (FSTAT&(FSTAT_FACCERR_MASK | FSTAT_FPVIOL_MASK))  
    {         
        /* Clear Flags if set*/
        FSTAT |= FSTAT_FACCERR_MASK | FSTAT_FPVIOL_MASK;  
    }

    if (FlashDataCounter)
    {
      do
      {
          /* Wait for the Last Busrt Command to complete */
          while(!(FSTAT_FCBEF));
          
          /* Write Data into Flash*/
          (*((volatile UINT8 *)(FlashAddress++))) = *pFlashDataPtr;
          pFlashDataPtr++;
          /* Write Command */
          FCMD = FlashCommand;
          /* Put FCBEF at 1 */
          FSTAT = 0x80;
          
          /* Wait at least 4 cycles to read the Error Flags */
          _asm NOP;
          _asm NOP;
          _asm NOP;
          _asm NOP;
          
          /* Check if Flash Access Error or Protection Violation Error are Set */
          if (FSTAT&(FSTAT_FACCERR_MASK | FSTAT_FPVIOL_MASK))
          {     
            /* If so, finish the function returning 1 to indicate error */
            return (FLASH_ERROR);
          }
      }while (--FlashDataCounter);
    }
    /* wait for the last command to complete */
    while (!(FSTAT_FCCF));
    /* Return zero to indicate that the function executed Ok */
    return (FLASH_OK);
}

#pragma CODE_SEG DEFAULT
/*
  This section of the code is the one that copies the routine into RAM. It is following the steps
  documented in Technical Note 228
*/

/*
Start_Copy_In_RAM refers to the begining of the segment ToCopyToRAM. 
This segment contains the functions after they have been copied to RAM.
*/
#define Start_Copy_In_RAM __SEG_START_FLASH_ROUTINES
#define Size_Copy_In_RAM __SEG_SIZE_FLASH_ROUTINES

/**
* \brief    Copies a section of code to RAM
* \author   
* \param    void
* \return   void
* \todo     
* The warning "Non standard conversion used" is because ANSI C doesn't allow to use a pointer to code as data
*/
void CopyInRAM(void) 
{
  char *srcPtr, *dstPtr;
  int count;
  srcPtr = (char *)Start_Copy_In_RAM;
  dstPtr = (char *)&Flash_Cmd;
  for (count = 0; count < (int) Size_Copy_In_RAM;  count++, dstPtr++, srcPtr++) 
  {
    *dstPtr = *srcPtr;
  }
}