/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*  Standard C Included Files */
#include <stdio.h>
#include <string.h>
#include "board.h"
#include "fsl_debug_console.h"
#include "fsl_lcdc.h"
#include "app.h"
#include "fsl_adc.h"

#include "GUI.h"
#include "GUIDRV_Lin.h"
#include "BUTTON.h"
#include "GRAPH.h"
#include "WM.h"
#include "TEXT.h"
#include "WIDGET.h"
#include "FRAMEWIN.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
static void ADC_Configuration(void);

/*******************************************************************************
 * Variables
 ******************************************************************************/
static int DataAdjust;
uint32_t xSize = 0;
uint32_t ySize = 0;
uint32_t graphXSize = 0;
uint32_t graphYSize = 0;
uint32_t graphXPos = 0;
uint32_t graphYPos = 0;
uint32_t DataYSize = 0;
volatile uint32_t g_ticks;
static adc_result_info_t gAdcResultInfoStruct;
adc_result_info_t *volatile gAdcResultInfoPtr = &gAdcResultInfoStruct;
volatile bool gAdcConvSeqAIntFlag;
GRAPH_SCALE_Handle hScaleH, hScaleV;

GRAPH_Handle hGraph;
GRAPH_DATA_Handle hData;
GRAPH_SCALE_Handle hScale;
float ADCVoltage = 0;
float TempValue = 0;
/* Frame end flag. */
static volatile bool s_frameEndFlag;

/* Structure for wave graph */
typedef struct
{
    char *Name;
    int ScaleVOff;
    int DataVOff;
    int GridVOff;
    int NumWaves;
} GRAPH_WAVE;

GRAPH_WAVE TempWave = {
    "Temperature Data", // Name
    159,                // Vertical scale offset in relation to GRAPH_DIV
    156,                // Vertical data  offset in relation to GRAPH_DIV
    21,                 // Vertical grid  offset in relation to GRAPH_DIV
    1                   // Number of waves
};

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief ISR for LCD
 */
void APP_LCD_IRQHandler(void)
{
    uint32_t intStatus = LCDC_GetEnabledInterruptsPendingStatus(APP_LCD);

    LCDC_ClearInterruptsStatus(APP_LCD, intStatus);

    if (intStatus & kLCDC_VerticalCompareInterrupt)
    {
        s_frameEndFlag = true;
    }
    __DSB();
/* Add for ARM errata 838869, affects Cortex-M4, Cortex-M4F Store immediate overlapping
  exception return operation might vector to incorrect interrupt */
#if defined __CORTEX_M && (__CORTEX_M == 4U)
    __DSB();
#endif
}

/*!
 * @brief ISR for Systick Timer
 */
void SysTick_Handler(void)
{
    g_ticks++;
}

/*!
 * @brief ISR for ADC conversion sequence A done.
 */
void DEMO_ADC_IRQ_HANDLER_FUNC(void)
{
    if (kADC_ConvSeqAInterruptFlag == (kADC_ConvSeqAInterruptFlag & ADC_GetStatusFlags(DEMO_ADC_BASE)))
    {
        ADC_GetChannelConversionResult(DEMO_ADC_BASE, DEMO_ADC_SAMPLE_CHANNEL_NUMBER, gAdcResultInfoPtr);
        ADC_ClearStatusFlags(DEMO_ADC_BASE, kADC_ConvSeqAInterruptFlag);
        gAdcConvSeqAIntFlag = true;
    }
}

status_t APP_LCDC_Init(void)
{
    /* Initialize the display. */
    lcdc_config_t lcdConfig;

    LCDC_GetDefaultConfig(&lcdConfig);

    lcdConfig.panelClock_Hz = LCD_PANEL_CLK;
    lcdConfig.ppl = LCD_PPL;
    lcdConfig.hsw = LCD_HSW;
    lcdConfig.hfp = LCD_HFP;
    lcdConfig.hbp = LCD_HBP;
    lcdConfig.lpp = LCD_LPP;
    lcdConfig.vsw = LCD_VSW;
    lcdConfig.vfp = LCD_VFP;
    lcdConfig.vbp = LCD_VBP;
    lcdConfig.polarityFlags = LCD_POL_FLAGS;
    lcdConfig.upperPanelAddr = VRAM_ADDR;
    lcdConfig.bpp = kLCDC_16BPP565;
    lcdConfig.display = kLCDC_DisplayTFT;
    lcdConfig.swapRedBlue = false;

    LCDC_Init(APP_LCD, &lcdConfig, LCD_INPUT_CLK_FREQ);

    /* Trigger interrupt at start of every vertical back porch. */
    LCDC_SetVerticalInterruptMode(APP_LCD, kLCDC_StartOfBackPorch);
    LCDC_EnableInterrupts(APP_LCD, kLCDC_VerticalCompareInterrupt);
    NVIC_EnableIRQ(APP_LCD_IRQn);

    LCDC_Start(APP_LCD);
    LCDC_PowerUp(APP_LCD);

    return kStatus_Success;
}

/*******************************************************************************
 * Application implemented functions required by emWin library
 ******************************************************************************/
void LCD_X_Config(void)
{
    GUI_DEVICE_CreateAndLink(GUIDRV_LIN_16, GUICC_565, 0, 0);

    LCD_SetSizeEx(0, LCD_WIDTH, LCD_HEIGHT);
    LCD_SetVSizeEx(0, LCD_WIDTH, LCD_HEIGHT);

    LCD_SetVRAMAddrEx(0, (void *)VRAM_ADDR);
}

int LCD_X_DisplayDriver(unsigned LayerIndex, unsigned Cmd, void *p)
{
    return 0;
}

void GUI_X_Config(void)
{
    /* Assign work memory area to emWin */
    GUI_ALLOC_AssignMemory((void *)GUI_MEMORY_ADDR, GUI_NUMBYTES);

    GUI_ALLOC_SetAvBlockSize(1024);

    /* Select default font */
    GUI_SetDefaultFont(GUI_FONT_6X8);
}

void GUI_X_Init(void)
{
}

/* Dummy RTOS stub required by emWin */
void GUI_X_InitOS(void)
{
}

/* Dummy RTOS stub required by emWin */
void GUI_X_Lock(void)
{
}

/* Dummy RTOS stub required by emWin */
void GUI_X_Unlock(void)
{
}

/* Dummy RTOS stub required by emWin */
U32 GUI_X_GetTaskId(void)
{
    return 0;
}

void GUI_X_ExecIdle(void)
{
}

GUI_TIMER_TIME GUI_X_GetTime(void)
{
    return g_ticks;
}

void GUI_X_Delay(int Period)
{
}

void *emWin_memcpy(void *pDst, const void *pSrc, long size)
{
    return memcpy(pDst, pSrc, size);
}

static void cbBackgroundWin(WM_MESSAGE *pMsg)
{
    switch (pMsg->MsgId)
    {
        default:
            WM_DefaultProc(pMsg);
    }
}

/*!
 * @brief Display the data points on graph
 *
 * @param hGraph  Handle of GRAPH widget
 * @param hData 	Handle of data object
 */
static void ShowGraph(GRAPH_Handle hGraph, GRAPH_DATA_Handle hData)
{
    int m = 0;
    int Count, Data_xSize, xSize;
    int TimeStart, TimeDiff, TimeStep;
    uint32_t i, Flag;

    /* Get physical X-size of LCD in pixels */
    xSize = LCD_GetXSize();

    /* calculate size of data to be displayed on horizontal scale on graph */
    Data_xSize = xSize - (DISTANCE_TO_BORDER << 1) - (BORDER_LEFT + BORDER_RIGHT);

    /* Add values depending on time */
    TimeStart = GUI_GetTime();
    Flag = 1;
    m = SCALE_H_START_OFFSET;
    Count = 0;

    do
    {
        ADCVoltage = 0;
        TempValue = 0;
        /* Calculate Average of 50 ADC samples */
        for (i = 0; i < 50; i++)
        {
            gAdcConvSeqAIntFlag = false;
            ADC_DoSoftwareTriggerConvSeqA(DEMO_ADC_BASE);

            while (!gAdcConvSeqAIntFlag)
            {
            }

            /* Convert ADC Reading to Voltage. Assuming 3.3V is supply voltage
             * For 12-bit ADC formula is Voltage(in volts) = [ (Decimal Value/4096) * Vsupply ]   */
            ADCVoltage = (((float)gAdcResultInfoStruct.result / 4096) * 3.3);

            /* Convert Voltage to temperature value. Temp (C) = [ (V (in mV) - LLS intercept at 0C)/ (LLS slope) ]
             * values from LPC546xx Datasheet: LLS slope = -2.04 mV/C ; LLS intercept at 0C = 584 mV   */
            TempValue += (((ADCVoltage * 1000) - 584) / (-2.04));
        }
        /* Average of 50 samples*/
        TempValue = TempValue / 50;
				
        printf("TempValue        = %f\r\n", TempValue);

        if (Count >= Data_xSize)
        {
            GRAPH_SCALE_SetOff(hScaleH, m--);
        }
        else
        {
            Count++;
        }

        TimeDiff = GUI_GetTime() - TimeStart;

        /* Add data item to the GRAPH_DATA_YT object */
        GRAPH_DATA_YT_AddValue(hData, TempValue);

        if (Flag)
        {
            Flag = 0;
            GUI_Exec();
            WM_ValidateWindow(hGraph);
        }

        TimeStep = GUI_GetTime() - TimeStart;

        if ((TimeStep - TimeDiff) < TIME_STEP)
        {
            GUI_Delay(TIME_STEP - (TimeStep - TimeDiff));
        }

    } while (1);
}

/*!
 * @brief Display string on the display
 *
 * @param Info  String to display
 */
void GUI_ShowInfo(const char *Info)
{
    uint32_t displayPos = 0;

    /* Get physical X-size of LCD in pixels */
    uint32_t xSize = LCD_GetXSize();
    /* Calculate position to display text*/
    displayPos = xSize / 2;

    /* Set the current foreground color of text */
    GUI_SetColor(GUI_WHITE);
    /* Set the font to be used for text output */
    GUI_SetFont(GUI_FONT_COMIC18B_ASCII);
    /* Display string*/
    GUI_DispStringHCenterAt("LPC54608 emWin Temperature Sensor Demo ", displayPos,
                            (FACTOR_EMWIN * ySize) >> SCREEN_DIV);
}

/*!
 * @bief This routine is called by the GRAPH object before anything is drawn
 *   		 and after the last drawing operation.
 */
static void UserDraw(WM_HWIN hWin, int Stage)
{
    if (Stage == GRAPH_DRAW_LAST)
    {
        char TempText[] = "Temperature";
        char TimeText[] = "Time";
        GUI_RECT Rect = {10, 10, 40, 80};

        GUI_SetFont(&GUI_Font13_ASCII);
        WM_GetInsideRect(&Rect);
        GUI_SetColor(GUI_YELLOW);

        GUI_DispStringInRectEx(TempText, &Rect, GUI_TA_HCENTER, strlen(TempText), GUI_ROTATE_CCW);

        GUI_DispStringInRectEx(TimeText, &Rect, GUI_TA_LEFT | GUI_TA_BOTTOM, strlen(TimeText), GUI_ROTATE_0);
    }
}

/*!
 * @brief Initialize the graph widget to be displayed on screen
 */
void GUI_GraphWidgetInit(void)
{
    /* Get the size of LCD display*/
    xSize = LCD_GetXSize(); /* get physical X-size of LCD in pixels */
    ySize = LCD_GetYSize(); /* get physical Y-size of LCD in pixels */

    /* Determine Size of Graph*/
    /* graphXSize = Screen size subtracted by twice the distance to the border of the screen */
    graphXSize = xSize - (DISTANCE_TO_BORDER * 2);
    /* graphYSize = Screen size subtracted by the Info window size and twice the distance to the windows */
    graphYSize = ySize - INFO_SIZE_Y - (DISTANCE_TO_WINDOW * 2);

    if (graphYSize > MAX_GRAPH_Y_SIZE)
    {
        graphYSize = MAX_GRAPH_Y_SIZE;
    }
    if (graphXSize > MAX_GRAPH_X_SIZE)
    {
        graphXSize = MAX_GRAPH_X_SIZE;
    }

    /* Determine starting position of graph*/
    graphXPos = (xSize - graphXSize) / 2;
    graphYPos = (ySize - graphYSize) / 2;

    /* Adjust starting Y position of graph based on Information window size*/
    if (graphYPos < INFO_SIZE_Y + DISTANCE_TO_WINDOW)
    {
        graphYPos = INFO_SIZE_Y + DISTANCE_TO_WINDOW;
    }

    /*Create a GRAPH widget of a specified size at a specified location*/
    hGraph =
        GRAPH_CreateEx(graphXPos, graphYPos, graphXSize, graphYSize, WM_HBKWIN, WM_CF_SHOW | WM_CF_CONST_OUTLINE, 0, 0);

    /*Set the left, top, right and bottom border of the given GRAPH widget*/
    GRAPH_SetBorder(hGraph, BORDER_LEFT, BORDER_TOP, BORDER_RIGHT, BORDER_BOTTOM);

    /* Set the desired color of the given GRAPH widget */
    GRAPH_SetColor(hGraph, COLOR_BG, GRAPH_CI_BK);         /* Sets Background Color of Graph*/
    GRAPH_SetColor(hGraph, COLOR_BORDER, GRAPH_CI_BORDER); /* Sets the color of the border area*/
    GRAPH_SetColor(hGraph, COLOR_FRAME, GRAPH_CI_FRAME);   /* Sets the color of the thin frame line*/
    GRAPH_SetColor(hGraph, COLOR_GRID, GRAPH_CI_GRID);     /* Sets the color of the grid*/

    /*Set the Visbility of the Grid Lines in Graph*/
    GRAPH_SetGridVis(hGraph, true);

    /* Set the interval between one grid line to next*/
    GRAPH_SetGridDistX(hGraph, GRID_X_INTERVAL);
    GRAPH_SetGridDistY(hGraph, GRID_Y_INTERVAL);

    /* Create a GRAPH_DATA_YT object*/
    hData = GRAPH_DATA_YT_Create(GUI_YELLOW, (xSize - (DISTANCE_TO_BORDER * 2) - BORDER_LEFT), 0, 0);

    /* Sets the alignment of the data */
    GRAPH_DATA_YT_SetAlign(hData, GRAPH_ALIGN_LEFT);

    /* Create Horizontal and Vertical Scale GRAPH_SCALE Objects*/
    hScaleH = GRAPH_SCALE_Create(BORDER_BOTTOM >> 1, GUI_TA_VCENTER, GRAPH_SCALE_CF_HORIZONTAL,
                                 SCALE_TICK_DIST_H); /* Horizontal Scale Object*/
    hScaleV = GRAPH_SCALE_Create(BORDER_LEFT >> 1, GUI_TA_HCENTER, GRAPH_SCALE_CF_VERTICAL,
                                 SCALE_TICK_DIST_V); /* Vertical Scale Object*/

    /* Set a factor used to calculate the numbers to be drawn on scale*/
    GRAPH_SCALE_SetFactor(hScaleH, SCALE_H_FACTOR);

    /* Set the position for showing the scale object within the GRAPH widget */
    GRAPH_SCALE_SetPos(hScaleH, graphYSize - SCALE_H_HEIGHT);
    GRAPH_SCALE_SetPos(hScaleV, SCALE_V_HEIGHT);

    /*Set an offset used to shift the scale object in positive or negative direction*/
    GRAPH_SCALE_SetOff(hScaleH, SCALE_H_START_OFFSET);

    /* Display string on display */
    GUI_ShowInfo(TempWave.Name);

    /* Attache scale object to GRAPH widget*/
    GRAPH_AttachScale(hGraph, hScaleH);
    GRAPH_AttachScale(hGraph, hScaleV);

    DataYSize = graphYSize - BORDER_BOTTOM;

    /* Calculate adjustment value to sets vertical offset used to draw the object data */
    DataAdjust = (DataYSize * TempWave.DataVOff) >> GRAPH_DIV;

    /* set vertical offset used to draw the object data */
    GRAPH_DATA_YT_SetOffY(hData, DataAdjust);

    /* Add an offset used to show the horizontal grid lines */
    GRAPH_SetGridOffY(hGraph, (DataYSize * TempWave.GridVOff) >> GRAPH_DIV);

    /* Set an offset used to shift the scale object in positive or negative direction */
    GRAPH_SCALE_SetOff(hScaleV, (((DataYSize - BORDER_BOTTOM) * TempWave.ScaleVOff) >> GRAPH_DIV));

    /* Attach data object to GRAPH widget */
    GRAPH_AttachData(hGraph, hData);

    /* Set the Text for Scale */
    GRAPH_SetUserDraw(hGraph, UserDraw);

    /* Routine to show graph */
    ShowGraph(hGraph, hData);
}

/*
 * Configure the ADC as normal converter in polling mode.
 */
void ADC_Configuration(void)
{
    adc_config_t adcConfigStruct;
    adc_conv_seq_config_t adcConvSeqConfigStruct;

    /* Configure the converter. */
    adcConfigStruct.clockMode = kADC_ClockSynchronousMode; /* Using sync clock source. */
    adcConfigStruct.clockDividerNumber = 1;                /* The divider for sync clock is 2. */
    adcConfigStruct.resolution = kADC_Resolution12bit;
    adcConfigStruct.enableBypassCalibration = false;
    adcConfigStruct.sampleTimeNumber = 0U;
    ADC_Init(DEMO_ADC_BASE, &adcConfigStruct);

    /* Use the temperature sensor input to channel 0. */
    ADC_EnableTemperatureSensor(DEMO_ADC_BASE, true);

    /* Enable channel DEMO_ADC_SAMPLE_CHANNEL_NUMBER's conversion in Sequence A. */
    adcConvSeqConfigStruct.channelMask =
        (1U << DEMO_ADC_SAMPLE_CHANNEL_NUMBER); /* Includes channel DEMO_ADC_SAMPLE_CHANNEL_NUMBER. */
    adcConvSeqConfigStruct.triggerMask = 0U;
    adcConvSeqConfigStruct.triggerPolarity = kADC_TriggerPolarityPositiveEdge;
    adcConvSeqConfigStruct.enableSingleStep = false;
    adcConvSeqConfigStruct.enableSyncBypass = false;
    adcConvSeqConfigStruct.interruptMode = kADC_InterruptForEachSequence;
    ADC_SetConvSeqAConfig(DEMO_ADC_BASE, &adcConvSeqConfigStruct);
    ADC_EnableConvSeqA(DEMO_ADC_BASE, true); /* Enable the conversion sequence A. */
}

int main(void)
{
    status_t status;

    /* Board pin, clock, debug console init */
    BOARD_InitHardware();

    /* Initialize LCD controller */
    status = APP_LCDC_Init();
    if (status != kStatus_Success)
    {
        PRINTF("LCD init failed\n");
    }
    assert(status == kStatus_Success);

    /* ADC Calibration. */
    if (ADC_DoSelfCalibration(DEMO_ADC_BASE))
    {
        PRINTF("ADC_DoSelfCalibration() Done.\r\n");
    }
    else
    {
        PRINTF("ADC_DoSelfCalibration() Failed.\r\n");
    }

    /* Configure the ADC and Temperature Sensor */
    ADC_Configuration();

    /* Enable the interrupt. */
    /* Enable the interrupt the for sequence A done. */
    ADC_EnableInterrupts(DEMO_ADC_BASE, kADC_ConvSeqAInterruptEnable);

    NVIC_EnableIRQ(DEMO_ADC_IRQ_ID);

    PRINTF("ADC Configuration Done.\r\n");

    /* emWin start */
    GUI_Init();

    /* Configure the systick timer*/
    SysTick_Config(SystemCoreClock / TICKRATE_HZ);

    /*Enable Memory Devices on all windows*/
    WM_SetCreateFlags(WM_CF_MEMDEV);

    /* Set size and default color for the background window */
    WM_SetSize(WM_HBKWIN, LCD_WIDTH, LCD_HEIGHT);

    WM_SetDesktopColor(GUI_WHITE);

    /* Set callback for the background window */
    WM_SetCallback(WM_HBKWIN, cbBackgroundWin);

    /* Select Active window*/
    WM_SelectWindow(WM_HBKWIN);

    /* Set the default background color*/
    GUI_SetBkColor(GUI_BLUE);

    /*Clear the current window */
    GUI_Clear();

    WM_Exec();

    /*Enable the use of Memory Devices*/
    WM_EnableMemdev(WM_HBKWIN);

    /* Set the text mode to be displayed transparent.*/
    GUI_SetTextMode(GUI_TM_TRANS);

    /* Intialize the graph widget */
    GUI_GraphWidgetInit();

    /* Never reaches here*/
    while (1)
        ;
}
