/****************************************************************************
 *
 * MODULE:             TIS/TRP Client/Server
 *
 * COMPONENT:
 *
 * DESCRIPTION:
 * Code to run on JN51xx while TIS/TRP tests are performed.
 */
/****************************************************************************
 *
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164,
 * JN5161, JN5148, JN5142, JN5139].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2016. All rights reserved
 *
 ***************************************************************************/
/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
#include <jendefs.h>
#include <MMAC.h>
#include "app.h"
#include "MicroSpecific.h"
#include "fsl_iocon.h"
#include "fsl_usart.h"
#include "fsl_gpio.h"
#include "CRC.h"
#include "board.h"
#include "stdlib.h"


/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/
#define DEMO_USART_CLK_SRC kCLOCK_Fro32M

#define DEFAULT_CHANNEL             16
#define DEFAULT_TX_POWER_dBm        10
#define FRAMES                      (16)

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/
typedef struct
{
    tsPhyFrame sFrame;
    uint32     u32timeStamp;
    uint8      u8RSSI;
} tsDataFrame;

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/
PRIVATE void vCommand_Task(uint32 u32IntMask);

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/
PUBLIC tsDataFrame sFrameData[FRAMES];

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE uint8 u8ByteCounter = 0;
PRIVATE uint8 u8Packetlength = 0;
PRIVATE uint8 u8Command = 0;
PRIVATE uint8 u8Data = 0;
PRIVATE uint8 u8Chksum = 0;
PRIVATE volatile uint8 u8CurFrame = 0;
PRIVATE uint8 u8ThisFrame = 0;
PRIVATE int g_iLed1FlashRate = 100000;
PRIVATE bool_t bRunning = FALSE;
PRIVATE uint8 u8Chksum;
PRIVATE uint32 u32TimeStamp = 0;
PRIVATE uint8 u8TimeBaseOffset = 0;
//PRIVATE uint8 u8Count=0;

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

PUBLIC void vUartWriteAddtoChkSum(uint8 u8Data){
    USART_WriteBlocking(USART0, &u8Data, 1);

    u8Chksum += u8Data;
}

PUBLIC void vUartWriteBinaryAddtoChkSum( uint8 *pu8Ptr, uint32 u32Len)
{
    uint32 n;
    for(n = 0; n < u32Len; n++){
        USART_WriteBlocking(USART0, pu8Ptr, 1);
        u8Chksum += *pu8Ptr++;
    }
}

PRIVATE void CheckForDataToSend(void)
{
    if (!bRunning)
    {
        return;
    }

    uint8 u8Value = 0;
    if (u8ThisFrame != u8CurFrame)
    {
        uint8 TotalLength = sFrameData[u8ThisFrame].sFrame.u8PayloadLength + 12;

        uint16 u16CRC = crc16ccitt_kermit(&sFrameData[u8ThisFrame].sFrame.uPayload.au8Byte[0], sFrameData[u8ThisFrame].sFrame.u8PayloadLength-2);

        uint16 rxCRC = sFrameData[u8ThisFrame].sFrame.uPayload.au8Byte[sFrameData[u8ThisFrame].sFrame.u8PayloadLength-2];
        rxCRC = (rxCRC<<8) +sFrameData[u8ThisFrame].sFrame.uPayload.au8Byte[sFrameData[u8ThisFrame].sFrame.u8PayloadLength-1];
        //vPrintf("\n\nCRC :%x rxCRC : %x ", u16CRC, rxCRC);

        // Send Header
        USART_WriteBlocking(USART0, &u8Value, 1 );   // NUL
        u8Value = 1;
        USART_WriteBlocking(USART0, &u8Value, 1 );   // SOH

        u8Chksum = 0;
        vUartWriteAddtoChkSum( TotalLength);
        vUartWriteAddtoChkSum( 0x80);          // Data Event

        // Send TimeStamp
        if (sFrameData[u8ThisFrame].u32timeStamp < u32TimeStamp)
        {
            /* Time base must have wrapped */
            u8TimeBaseOffset += 16;
        }
        u32TimeStamp = sFrameData[u8ThisFrame].u32timeStamp;
        vUartWriteAddtoChkSum(((u32TimeStamp >> 28) & 0xff) + u8TimeBaseOffset);
        vUartWriteAddtoChkSum((u32TimeStamp >> 20) & 0xff);
        vUartWriteAddtoChkSum((u32TimeStamp >> 12) & 0xff);
        vUartWriteAddtoChkSum((u32TimeStamp >> 4) & 0xff);
        vUartWriteAddtoChkSum((u32TimeStamp << 4) & 0xff);
        vUartWriteAddtoChkSum(sFrameData[u8ThisFrame].sFrame.u8PayloadLength);

        // Send Data
        vUartWriteBinaryAddtoChkSum(sFrameData[u8ThisFrame].sFrame.uPayload.au8Byte,
        		                    sFrameData[u8ThisFrame].sFrame.u8PayloadLength);

        vUartWriteAddtoChkSum(0);
        vUartWriteAddtoChkSum(sFrameData[u8ThisFrame].u8RSSI);
        //vUartWriteAddtoChkSum(u8Count--);
        if (u16CRC == rxCRC)
        {
            vUartWriteAddtoChkSum( 0x01);
        }
        else
        {
            vUartWriteAddtoChkSum( 0x00);
        }

        u8Chksum = 256 - u8Chksum;
        USART_WriteBlocking(USART0, &u8Chksum, 1);    // ChkSum
        u8Value = 4;
        USART_WriteBlocking(USART0, &u8Value, 1);         // EOT marker

        // Update the frame to check the next one.
        u8ThisFrame++;
        if ( u8ThisFrame >= FRAMES)
        {
            u8ThisFrame = 0;
        }
    }
}

PRIVATE void SendResponse(uint8 u8Data){
    uint8 u8Length = 4;
    uint8 u8PacketID = 0x81;
    uint8 u8CalChksum = 0;

    uint8 u8Value = 0;
    USART_WriteBlocking(USART0, &u8Value, 1);
    u8Value = 1;
    USART_WriteBlocking(USART0, &u8Value, 1);
    USART_WriteBlocking(USART0, &u8Length, 1);
    USART_WriteBlocking(USART0, &u8PacketID, 1);
    USART_WriteBlocking(USART0, &u8Data, 1);

    u8CalChksum = u8Length;
    u8CalChksum += u8PacketID;
    u8CalChksum += u8Data;
    u8CalChksum = 256 - u8CalChksum;

    USART_WriteBlocking(USART0, &u8CalChksum, 1);
    u8Value = 0;
    USART_WriteBlocking(USART0, &u8Value, 1);
}

PRIVATE void ProcessCmd(void){
    switch(u8Command){
        case 0: //Channel
            vMMAC_SetChannelAndPower(u8Data, DEFAULT_TX_POWER_dBm);
            SendResponse(0);
            break;

        case 1: // Mode
            if (u8Data == 0){
                bRunning = FALSE;
                g_iLed1FlashRate = 100000;
            }
            else if (u8Data == 1){
                bRunning = TRUE;
                g_iLed1FlashRate = 50000;
            }
            // ignore case 2 as this is a ping (just send response
            else if (u8Data == 3){
                bRunning = !bRunning;
                if(bRunning){
                	g_iLed1FlashRate = 50000;
                }
                else{
                	g_iLed1FlashRate = 100000;
                }
            }
            SendResponse(0);
            break;
    }
}

bool_t bUartRxDataAvailable(USART_Type *base)
{
    if (base->FIFOSTAT & USART_FIFOSTAT_RXNOTEMPTY_MASK) return TRUE;
    return FALSE;
}


PRIVATE void CheckForDataRXed(void){
    uint8 u8value;
    while (bUartRxDataAvailable(USART0)){
        USART_ReadBlocking(USART0, &u8value, 1);
        switch (u8ByteCounter){
        case 0:
            if (u8value == 0) u8ByteCounter++;
            break;
        case 1:
            if (u8value == 1){
                u8ByteCounter++;
            }
            else{
                u8ByteCounter=0;
            }
            break;
        case 2:
            u8Packetlength = u8value;
            u8ByteCounter++;
            break;
        case 3:
            u8Command = u8value;
            u8ByteCounter++;
            break;
        case 4:
            u8Data = u8value;
            u8ByteCounter++;
            break;
        case 5:
            u8Chksum = u8value;
            u8ByteCounter++;
            break;
        case 6:
            if (u8value == 4) ProcessCmd();
            u8ByteCounter = 0;
            break;
        }
    }
}

/****************************************************************************
 *
 * NAME:
 *
 * DESCRIPTION:
 *
 * RETURNS:
 *
 ****************************************************************************/
int __main(void)
{
    BOARD_InitHardware();

    CLOCK_AttachClk(kFRO48M_to_MAIN_CLK);

    usart_config_t config;

    /* Setup LEDS */
    // PIO4 to set to a digital pin FUNC0
    IOCON_PinMuxSet(IOCON, 0, 4,  IOCON_FUNC0 | IOCON_DIGITAL_EN | IOCON_GPIO_MODE);
    // PIO3 for DK6 LED
    IOCON_PinMuxSet(IOCON, 0, 3,  IOCON_FUNC0 | IOCON_DIGITAL_EN | IOCON_GPIO_MODE);

    // PIO10 to set to a digital pin FUNC0. Pin is Open collector by default.
    // Set bit 3 to 'enable GPIO mode'. The only definition for (1<<3) in IOCON_MODE_REPEATER
    // which doesn't seem an appropriate name here
    IOCON_PinMuxSet(IOCON, 0, 10, IOCON_FUNC0 | IOCON_DIGITAL_EN | IOCON_GPIO_MODE | (1<<3));


    // PIO5 to set to a digital pin FUNC7. Pin is RFRX.
    IOCON_PinMuxSet(IOCON, 0, 5, IOCON_MODE_INACT | IOCON_FUNC7 | IOCON_DIGITAL_EN);

    GPIO_PinWrite(GPIO, 0, 4, 0);     // Function in fsl_gpio.h. Set PIO4 low
    GPIO->DIR[0] |= 1U << 4;          // No function defined in fsl_gpio.h. Set PIO4 to output
    GPIO_PinWrite(GPIO, 0, 3, 0);     // Function in fsl_gpio.h. Set PIO3 low
    GPIO->DIR[0] |= 1U << 3;          // No function defined in fsl_gpio.h. Set PIO3 to output
    GPIO_PinWrite(GPIO, 0, 10, 0);    // Function in fsl_gpio.h. Set PIO10 low
    GPIO->DIR[0] |= 1U << 10;         // No function defined in fsl_gpio.h. Set PIO10 to output

    GPIO_PinWrite(GPIO, 0, 6, 0);     // Function in fsl_gpio.h. Set PIO6 low
    GPIO->DIR[0] |= 1U << 6;          // No function defined in fsl_gpio.h. Set PIO6 to output

    /*
     * config.baudRate_Bps = 115200U;
     * config.parityMode = kUSART_ParityDisabled;
     * config.stopBitCount = kUSART_OneStopBit;
     * config.loopback = false;
     * config.enableTx = false;
     * config.enableRx = false;
     */
    USART_GetDefaultConfig(&config);
    config.baudRate_Bps = 1000000;
    config.enableTx = true;
    config.enableRx = true;

    USART_Init(USART0, &config, CLOCK_GetFreq(DEMO_USART_CLK_SRC));


    /* Initialize radio */
    vMMAC_Enable();
    vMMAC_ConfigureRadio();
    vMMAC_SetChannelAndPower(DEFAULT_CHANNEL, DEFAULT_TX_POWER_dBm);
    vMMAC_SetTxParameters(1, 1, 1, 1);
    vMMAC_SetCcaMode(E_MMAC_CCAMODE_CARRIER);

    /* Enable RX interrupt and handler */
    vMMAC_EnableInterrupts(vCommand_Task);
    vMMAC_ConfigureInterruptSources(E_MMAC_INT_RX_COMPLETE);

    /* Set both PHY Rx and PHY off tune times to 0, and minimise turnaround
       times (TAT and LIFS) */
    vREG_BbcWrite(REG_BBC_PRTT, 0);
    vREG_BbcWrite(REG_BBC_TAT,  0);
    vREG_BbcWrite(REG_BBC_LIFS_TURNAROUND, 0);

    vMMAC_StartPhyReceive(&sFrameData[0].sFrame, E_MMAC_RX_START_NOW | E_MMAC_RX_ALLOW_MALFORMED); // | E_MMAC_RX_ALLOW_FCS_ERROR);

    int iLed1Count = 0;
    int iLed1State = 0;

    u8ThisFrame = u8CurFrame;

    while(1)
    {
        /* LED 0 toggles on and off - faster if receiver is on */
        if (++iLed1Count >= g_iLed1FlashRate)
        {
            iLed1Count = 0;

            iLed1State ^= 1;
            if(iLed1State)
            {
            	GPIO_PinWrite(GPIO, 0, 4, 0);
            	GPIO_PinWrite(GPIO, 0, 3, 0);
            	GPIO_PinWrite(GPIO, 0, 10, 1);
            }
            else
            {
            	GPIO_PinWrite(GPIO, 0, 4, 1);
            	GPIO_PinWrite(GPIO, 0, 3, 1);
            	GPIO_PinWrite(GPIO, 0, 10, 0);
            }
            iLed1Count = 0;
        }

        // Do we need to send it over the UART?
        CheckForDataToSend();

        // Have we received data on the serial port?
        CheckForDataRXed();
    }
}

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/


/****************************************************************************
 *
 * NAME:
 *
 * DESCRIPTION:
 *
 * PARAMETERS:  Name    RW  Usage
 *
 * RETURNS:
 *
 ****************************************************************************/
PRIVATE void vCommand_Task(uint32 u32IntMask)
{
    uint8 u8NextFrame;

    /* Wait for a command to be received */
    if (u32IntMask & E_MMAC_INT_RX_COMPLETE)
    {
    	// Toggle PIO6
    	GPIO_PortToggle(GPIO, 0, 1<<6);
        #if 0
        {
            uint8 i;
            vPrintf("\n\rRXD");
            vPrintf("\n\rLen: %d", sFrame.u8PayloadLength);

            for (i = 0; i < sFrame.u8PayloadLength; i++)
            {
                vPrintf("\n\r[%d]: %x", i, sFrame.uPayload.au8Byte[i]);
            }
        }
        #endif

        // Restart the receiver
        u8NextFrame = u8CurFrame + 1;
        if (u8NextFrame >= FRAMES)
        {
            u8NextFrame = 0;
        }
        vMMAC_StartPhyReceive(&sFrameData[u8NextFrame].sFrame, E_MMAC_RX_START_NOW | E_MMAC_RX_ALLOW_MALFORMED); // | E_MMAC_RX_ALLOW_FCS_ERROR);

        sFrameData[u8CurFrame].u8RSSI = u8MMAC_GetRxLqi(NULL);
        sFrameData[u8CurFrame].u32timeStamp = READ_REG32(REG_BBC_RXTSTP + JN518X_ZBMAC);

        // Update the frame to use the next one.
        u8CurFrame = u8NextFrame;
    }
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
