/*
 * Copyright (c) 2019, NXP B.V.
 * Copyright 2016-2019 NXP
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
#include <jendefs.h>
#include <stdlib.h>

#include <MMAC.h>
#include "fsl_debug_console.h"
#include "app.h"
#include "radio.h"
#include "radio_test_jn518x.h"
#include "MicroSpecific.h"
#include "CRC.h"
#include "Command.h"
#include "fsl_reset.h"
#include "fsl_ctimer.h"
#ifdef DBG_ENABLE
	#include "Printf.h"
	#include "UartBuffered.h"
#endif

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/


#define COMMAND_DELIMITER_LEN			2
#define COMMAND_ID_LEN					2
#define COMMAND_HEADER_LEN				(COMMAND_DELIMITER_LEN + COMMAND_ID_LEN)
#define COMMAND_CRC_LEN					2

#define BUS_CLK_FREQ	CLOCK_GetFreq(kCLOCK_BusClk)
typedef struct
{
	uint32 u32TotalPackets;	/* packets issued by ARB generator and CMD module */
	uint32 u32ProtocolPackets; /* packets issued by ARB generator only */
	uint32 u32TotalLqi;	/* not used in the test */
	uint32 u32TotalRssiMagnitude; /*accumulate RSSI values for ARB generator packets only */
}tsRxTestReport;

/* Timer clock rate = 16MHz / 2^PRESCALE = 16MHz / 16384 = 976Hz i.e. approx 1ms */
#define TIMER_PRESCALE		    14

#define APP_REBOOT_BYTE          3
#define APP_HW_VERSION           5
#define APP_SW_VERSION           3
/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/


/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/
PUBLIC void vCommand_Transmit(uint8 u8Id, uint8 u8Len, const uint8 *pu8Payload);
PRIVATE void vCommand_Process(uint16 u16Id, uint8 u8Len, const uint8 *pu8Payload);
PRIVATE void vCommand_HandleIncoming(const uint8 *pu8Data, uint8 u8Len, uint8 u8Lqi, uint8 u8Rssi);
void ctimer_match0_callback(uint32_t flags);
#ifdef MODE_CMD
PRIVATE void vReorderBytes32(uint8 *Array, uint8 ArraySize);
#endif

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/* Wait for a command to be received */
PUBLIC tsPhyFrame sFrame;
PUBLIC tsPhyFrame sTFrame;

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE bool_t bStreamTestRunning = FALSE;
#ifdef MODE_CMD
PUBLIC volatile bool_t bAckReceived = FALSE;
extern PUBLIC bool_t bFind;
#else
PRIVATE uint8 u8SwVersion = APP_SW_VERSION;
PRIVATE uint8 u8HwVersion = APP_HW_VERSION;
static uint8 u8Channel = DEFAULT_CHANNEL;
#endif

PRIVATE tsRxTestReport sRxTestReport;
static int8 i8Power = DEFAULT_TX_POWER_dBm;

#ifndef MODE_CMD
static ctimer_match_config_t matchConfig0;
#endif




/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/
/****************************************************************************
 *
 * NAME:
 *
 * DESCRIPTION:
 *
 * PARAMETERS:  Name    RW  Usage
 *
 * RETURNS:
 *
 ****************************************************************************/
PUBLIC void vCommand_Task(bool_t bRxPoll)
{
	/* Poll timer and if expired execute callback. */
	if (bStreamTestRunning == TRUE)
	{
	}
	else if (TRUE == bRxPoll)
	{


	    if (u32MMAC_PollInterruptSource(E_MMAC_INT_RX_COMPLETE) == E_MMAC_INT_RX_COMPLETE)
    	{
			#if 0
    		{
    			uint8 i;
    			PRINTF("\n\rRXD");
    			PRINTF("\n\rLen: %d", sFrame.u8PayloadLength);

    			for (i = 0; i < sFrame.u8PayloadLength; i++)
    			{
    				PRINTF("\n\r[%d]: %x", i, sFrame.uPayload.au8Byte[i]);
    			}
    		}
			#endif

    		uint32 u32Rssi = u32REG_RFPModemRead(MF_RSSI_DBM_PACKET);
    		int32 i32Rssi = ((int32)(u32Rssi << 22)) >> 22;
    		i32Rssi = abs(i32Rssi);

    		uint8 u8Lqi= u8MMAC_GetRxLqi(NULL);
   			vCommand_HandleIncoming(&sFrame.uPayload.au8Byte[0], sFrame.u8PayloadLength, u8Lqi, i32Rssi);

   			if(bStreamTestRunning == FALSE){
   				/* Restart Reception */
   				vMMAC_StartPhyReceive(&sFrame, E_MMAC_RX_START_NOW | E_MMAC_RX_ALLOW_MALFORMED | E_MMAC_RX_ALLOW_FCS_ERROR);
   			}
    	}
	}
}


/****************************************************************************
 *
 * NAME:
 *
 * DESCRIPTION:
 *
 * PARAMETERS:  Name    RW  Usage
 *
 * RETURNS:
 *
 ****************************************************************************/
PUBLIC void vCommand_SendChannel(uint32 u32ChannelMask)
{
	uint8 values[4];
	uint8* ptr = (uint8*) &u32ChannelMask;
	values[3] = *ptr++;
	values[2] = *ptr++;
	values[1] = *ptr++;
	values[0] = *ptr++;
	vCommand_Transmit(E_CMD_ID_SET_CHANNEL, 4,  values);
}

PUBLIC void vCommand_SetPower(uint8 u8Channel, int iPower)
{
	#if 1
	if (i8Power<0){
		PRINTF("\n\rPower: -%i", iPower);

	}
	else{
		PRINTF("\n\rPower: %i", iPower);
	}
	#endif

	vMMAC_SetChannelAndPower( u8Channel, iPower);
}

#ifndef MODE_CMD
ctimer_callback_t ctimer_callback_table[] = {
    ctimer_match0_callback, NULL, NULL, NULL, NULL, NULL, NULL, NULL};


/****************************************************************************
 *
 * NAME:
 *
 * DESCRIPTION:
 *
 * PARAMETERS:  Name    RW  Usage
 *
 * RETURNS:
 *
 ****************************************************************************/
PUBLIC void vCommand_StartModulatedWaveTx(uint16 u16Periodms)
{
	#if 1
		PRINTF("\n\rPeriod: %dms\n", u16Periodms);
	#endif

	ctimer_config_t config;

	CTIMER_GetDefaultConfig(&config);

	CTIMER_Init(CTIMER0, &config);

	/* Configuration 0 */
	matchConfig0.enableCounterReset = true;
	matchConfig0.enableCounterStop = false;
	matchConfig0.matchValue = (BUS_CLK_FREQ / 1000) * u16Periodms;
	matchConfig0.outControl = kCTIMER_Output_NoAction;
	matchConfig0.outPinInitState = false;
	matchConfig0.enableInterrupt = true;

    CTIMER_RegisterCallBack(CTIMER0, &ctimer_callback_table[0], kCTIMER_SingleCallback);
    CTIMER_SetupMatch(CTIMER0, kCTIMER_Match_0, &matchConfig0);
    CTIMER_StartTimer(CTIMER0);

	/* Start modulated wave transmission */
    if(bStreamTestRunning == FALSE)
    {
    	/* Not running, save previous phy pctrl state */
    	vRadio_Test_Jn518x_TxPowerTest(TXPT_RUN_PRBS);

        bStreamTestRunning = TRUE;
    }
}


PUBLIC void vCommand_ResetCounters(void ){
#if 1
		PRINTF("\n\rReport from DUT module");
		PRINTF("\n\r totalPackets : %d", sRxTestReport.u32TotalPackets);
		PRINTF("\n\r Protocol Packets : %d", sRxTestReport.u32ProtocolPackets);
		PRINTF("\n\r totalLQI : %d", sRxTestReport.u32TotalLqi);
		PRINTF("\n\r totalRSSIMagnitude : %d", sRxTestReport.u32TotalRssiMagnitude);
		PRINTF("\n\r Channel : %d\n\r>", u8Channel);
#endif

	sRxTestReport.u32TotalPackets = 0;
	sRxTestReport.u32ProtocolPackets = 0;
	sRxTestReport.u32TotalLqi = 0;
	sRxTestReport.u32TotalRssiMagnitude = 0;
}
#endif
/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/
/****************************************************************************
 *
 * NAME:
 *
 * DESCRIPTION:
 *
 * PARAMETERS:  Name    RW  Usage
 *
 * RETURNS:
 *
 ****************************************************************************/
PRIVATE void vCommand_HandleIncoming(const uint8 *pu8Data, uint8 u8Len, uint8 u8Lqi, uint8 u8Rssi)
{
	sRxTestReport.u32TotalPackets++;
	//PRINTF("Received Pkt\r\n");

	sRxTestReport.u32TotalLqi += u8Lqi;
	sRxTestReport.u32TotalRssiMagnitude += u8Rssi;

	#if 0
		PRINTF("\n\r%s - Len: %d LQI: %d RSSI: %d", __FUNCTION__, u8Len, u8Lqi, u8Rssi);
	#endif

	/* Extract CRC from last 2 bytes of frame*/
	uint16 u16RxCRC;
	u16RxCRC   = pu8Data[u8Len - 2];
	u16RxCRC <<= 8;
	u16RxCRC  |= pu8Data[u8Len - 1];

	/* Calculate received data CRC (don't include embedded CRC!) */
	uint16 u16CRC = crc16ccitt_kermit(pu8Data, u8Len - 2);

	#if 0
		PRINTF("\n\rCRC - RX: %x Calc: %x", u16RxCRC, u16CRC);
	#endif

	/* Do they match... */
	if (u16CRC == u16RxCRC)
	{
		/* Does this look like a command frame... */
		if ((pu8Data[0] == 0xDE) && (pu8Data[1] == 0xDE))
		{
			/* Extract the command ID */
			uint16 u16CommandID;
			u16CommandID   = pu8Data[COMMAND_DELIMITER_LEN];
			u16CommandID <<= 8;
			u16CommandID  |= pu8Data[COMMAND_DELIMITER_LEN + 1];

			#if 0
				PRINTF("\n\r%s - Command frame identified, command ID : %d",__FUNCTION__,u16CommandID); //JB
			#endif

			/* Handle the command */
			vCommand_Process(u16CommandID, (u8Len - COMMAND_HEADER_LEN - COMMAND_CRC_LEN), &pu8Data[COMMAND_HEADER_LEN]);

			/* Record we have received a command packet */
			sRxTestReport.u32ProtocolPackets++;
		}
	}
}



#ifdef MODE_CMD
PRIVATE void vCommand_Process(uint16 u16Id, uint8 u8Len, const uint8 *pu8Payload)
{
#if 0
	PRINTF("\n\rID: %x Len: %d", u16Id, u8Len);
#endif

	switch (u16Id){

	case E_CMD_ID_ACK:
		{
			/* ACK received */
			bAckReceived = TRUE;
			PRINTF("Received ACK\r\n");
			if(!bFind){
				PRINTF(">");
			}
		}
		break;

	case E_CMD_ID_GET_POWER_RESPONSE:
	{
		int i8RPower = (int8) pu8Payload[0];
		PRINTF("Remote Power :%d\n", i8RPower);
	}
	break;

	case E_CMD_ID_GET_SW_VERSION_RESPONSE:
	{
		PRINTF("Remote Software Version :%d\n>", pu8Payload[0]);
	}
	break;

	case E_CMD_ID_GET_HW_VERSION_RESPONSE:
	{
		/* Respond with hardware version */
		PRINTF("Remote Hardware Version :%d\n>", pu8Payload[0]);
	}
	break;


	case E_CMD_ID_RX_REPORT:
	{
		/* prepare test report */
		vReorderBytes32((uint8 *)&sRxTestReport, sizeof(sRxTestReport));

		 memcpy(&sRxTestReport,&pu8Payload[0],sizeof(tsRxTestReport));

#if 1
		PRINTF("\n\rReport from DUT module");
		PRINTF("\n\r totalPackets : %d", sRxTestReport.u32TotalPackets);
		PRINTF("\n\r Protocol Packets : %d", sRxTestReport.u32ProtocolPackets);
		PRINTF("\n\r totalLQI : %d", sRxTestReport.u32TotalLqi);
		PRINTF("\n\r totalRSSIMagnitude : %d\n\r>", sRxTestReport.u32TotalRssiMagnitude);
#endif
	}
	break;

	default:
	{
		/* Unknown command, do nothing */
#if 1
		PRINTF("\n\rERROR: Unknown Command");
#endif
	}
	break;
}
}

#else
/****************************************************************************
 *
 * NAME:
 *
 * DESCRIPTION:
 *
 * PARAMETERS:  Name    RW  Usage
 *
 * RETURNS:
 *
 ****************************************************************************/
PRIVATE void vCommand_Process(uint16 u16Id, uint8 u8Len, const uint8 *pu8Payload)
{

	#if 1
		PRINTF("\n\rID: %x Len: %d", u16Id, u8Len);
	#endif

	switch (u16Id)
	{
		case E_CMD_ID_PING:
		{
			/* Ping received, reply with an ACK */
			vCommand_Transmit(E_CMD_ID_ACK, 0, NULL);
		}
		break;

		case E_CMD_ID_SET_CHANNEL:
		{
			uint32 u32ChannelMask;
			u32ChannelMask = pu8Payload[0];
			u32ChannelMask <<= 8;
			u32ChannelMask |= pu8Payload[1];
			u32ChannelMask <<= 8;
			u32ChannelMask |= pu8Payload[2];
			u32ChannelMask <<= 8;
			u32ChannelMask |= pu8Payload[3];

			u8Channel = FF1(u32ChannelMask);

			if((u8Channel >10) && (u8Channel <27)){
				u8Channel -=1;   // FF1 always returns 1 bit higher
				PRINTF("Chan: %d ChannelMask:%8x\r\n", u8Channel,u32ChannelMask);

				/* Now set the channel */
				vMMAC_SetChannel(u8Channel);
			}
		}
		break;

		case E_CMD_ID_SET_POWER:
		{
			i8Power = (int8) pu8Payload[0];

			/* Now set the channel */
			vCommand_SetPower(u8Channel, (int)i8Power);
		}
		break;

		case E_CMD_ID_GET_POWER:
		{
			vCommand_Transmit(E_CMD_ID_GET_POWER_RESPONSE, 1, &i8Power);
		}
		break;


		case E_CMD_ID_REBOOT:
		{
			vMMAC_RadioOff();
			{
				/* Delay to ensure all debug has been transmitted before reset */
				volatile uint32 i;
				for (i = 0; i < 50000; i++);
			}

			/* Simply reset without sending an ACK */
#if (defined JENNIC_CHIP_FAMILY_JN516x) || (defined JENNIC_CHIP_FAMILY_JN517x)
			vAHI_SwReset();
#else
			RESET_SystemReset();
#endif

		}
		break;

		case E_CMD_ID_STREAM:
		{
			/* Extract period (ms) from payload */
			uint16 u16Periodms;
			u16Periodms   = pu8Payload[0];
			u16Periodms <<= 8;
			u16Periodms  |= pu8Payload[1];

			/* Now start transmission of modulated waveform for required time */
			vCommand_StartModulatedWaveTx(u16Periodms);
		}
		break;

		case E_CMD_ID_GET_SW_VERSION:
		{
			/* Respond with software version */
			vCommand_Transmit(E_CMD_ID_GET_SW_VERSION_RESPONSE, sizeof(u8SwVersion), &u8SwVersion);
		}
		break;

		case E_CMD_ID_GET_HW_VERSION:
		{
			/* Respond with hardware version */
			vCommand_Transmit(E_CMD_ID_GET_HW_VERSION_RESPONSE, sizeof(u8HwVersion), &u8HwVersion);
		}
		break;

		case E_CMD_ID_START_RX_TEST:
		{
#if 0
			vPrintf("\n\rE_CMD_ID_START_RX_TEST");
#endif

			/* Reset counters */
			sRxTestReport.u32TotalPackets = 0;
			sRxTestReport.u32ProtocolPackets = 0;
			sRxTestReport.u32TotalLqi = 0;
			sRxTestReport.u32TotalRssiMagnitude = 0;
		}
		break;

		case E_CMD_ID_END_RX_TEST:
		{
#if 0
			vPrintf("\n\rE_CMD_ID_END_RX_TEST");
#endif

			/* Transmit report */
			vCommand_Transmit(E_CMD_ID_RX_REPORT, sizeof(tsRxTestReport), (const uint8 *)&sRxTestReport);
#if 0
			vPrintf("\n\report to CMD module");
			vPrintf("\n\r totalPackets : %d", sRxTestReport.u32TotalPackets);
			vPrintf("\n\r Protocol Packets : %d", sRxTestReport.u32ProtocolPackets);
			vPrintf("\n\r totalLQI : %d", sRxTestReport.u32TotalLqi);
			vPrintf("\n\r totalRSSIMagnitude : %d", sRxTestReport.u32TotalRssiMagnitude);
#endif
		}
		break;

		default:
		{
			/* Unknown command, do nothing */
#if 0
			vPrintf("\n\rERROR: Unknown Command");
#endif
		}
		break;
	}
}
#endif

/****************************************************************************
 *
 * NAME:
 *
 * DESCRIPTION:
 *
 * PARAMETERS:  Name    RW  Usage
 *
 * RETURNS:
 *
 ****************************************************************************/
PUBLIC void vCommand_Transmit(uint8 u8Id, uint8 u8Len, const uint8 *pu8Payload)
{
	#if 0
		vPrintf("\n\r%s - ID: %x Len: %d", __FUNCTION__, u8Id, u8Len);
	#endif

	/* Build PHY layer frame header */
	sTFrame.uPayload.au8Byte[0] = 0xDE;
	sTFrame.uPayload.au8Byte[1] = 0xDE;
	sTFrame.uPayload.au8Byte[2] = 0;
	sTFrame.uPayload.au8Byte[3] = u8Id;

	/* Add payload (if any) */
	if (pu8Payload != NULL)
	{
		uint8 i;
		for (i = 0; i < u8Len; i++)
		{
			sTFrame.uPayload.au8Byte[4 + i] = pu8Payload[i];
		}
	}
	else
	{
#if 0
		vPrintf("\n\rNo Payload", __FUNCTION__);
#endif
	}

	/* Calculate CRC over header and payload */
	uint16 u16CRC = crc16ccitt_kermit(&sTFrame.uPayload.au8Byte[0], (4 + u8Len));

	/* Add CRC as frame footer */
	sTFrame.uPayload.au8Byte[4 + u8Len] = (uint8)(u16CRC >> 8);
	sTFrame.uPayload.au8Byte[5 + u8Len] = (uint8)u16CRC;

	/* Length = header + payload + CRC */
	sTFrame.u8PayloadLength = 4 + u8Len + 2;

	/* Start transmission */
	vMMAC_StartPhyTransmit(&sTFrame, E_MMAC_TX_START_NOW);
#if 0
		vPrintf("\n\rTransmition started. Waiting for completion...", __FUNCTION__);
#endif

	/* Wait for TX to complete before going back to receive mode */
	while (u32MMAC_PollInterruptSource(E_MMAC_INT_TX_COMPLETE) == 0);

#if 0
	vPrintf("\n\rTransmition completed ", __FUNCTION__);
	vPrintf(" Errors: %x", u32MMAC_GetTxErrors());
#endif

	/* Restart Reception */
	vMMAC_StartPhyReceive(&sFrame, E_MMAC_RX_START_NOW | E_MMAC_RX_ALLOW_MALFORMED | E_MMAC_RX_ALLOW_FCS_ERROR);
}

#ifndef MODE_CMD
void ctimer_match0_callback(uint32_t flags)
{
	/* Stop modulated wave transmission */
    if(bStreamTestRunning == TRUE)
    {

    	vRadio_Test_Jn518x_TxPowerTest(TXPT_STOP);
    	CTIMER_StopTimer(CTIMER0);
    	PRINTF("Stream stopped\n");
    	bStreamTestRunning = FALSE;
    }

    vMMAC_Disable();
    vMMAC_Enable();
      vMMAC_ConfigureRadio();
      vMMAC_SetChannelAndPower(DEFAULT_CHANNEL, DEFAULT_TX_POWER_dBm);
      vMMAC_SetTxParameters(1, 1, 1, 1);
      vMMAC_SetCcaMode(E_MMAC_CCAMODE_CARRIER);
      vMMAC_StartPhyReceive(&sFrame, E_MMAC_RX_START_NOW | E_MMAC_RX_ALLOW_MALFORMED | E_MMAC_RX_ALLOW_FCS_ERROR);



      // Disable the interrupts. We will poll the MAC
      NVIC_DisableIRQ(ZIGBEE_MAC_IRQn);
}
#endif

#ifdef MODE_CMD
PRIVATE void vReorderBytes32(uint8 *Array, uint8 ArraySize)
{
	uint8 i;
	uint8 savedbyte0, savedbyte1;

	for (i=0; i<ArraySize; i+=4)
	{
		savedbyte0= Array[i+4-1];
		savedbyte1= Array[i+4-2];
		Array[i+4-1] = Array[i];
		Array[i+4-2] = Array[i+1];
		Array[i+1] = savedbyte1;
		Array[i] = savedbyte0;
	}
}
#endif

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
