/*
 * Copyright 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2019 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef _MCDRV_HVP_56F83738_H_
#define _MCDRV_HVP_56F83738_H_

#include "qs.h"

#include "mcdrv_pwm3ph_pwma.h"
#include "mcdrv_adc_adc12.h"
#include "char_pwrstg.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/* Version info */
#define MCRSP_VER "1.2.0"        /* motor control package version */

/* Application info */
typedef struct _app_ver
{
    char cBoardID[15];
    char cMotorType[4];
    char cAppVer[5];
}app_ver_t;
   

/* Define motor 1 ADC periphery (ADC12) */
#define M1_MCDRV_ADC (MCDRV_ADC12)

/* Define motor 1 3-ph PWM periphery */
#define M1_MCDRV_PWM3PH (MCDRV_PWMA)

/******************************************************************************
 * Clock & PWM definition
 ******************************************************************************/
#define M1_PWM_FREQ (10000)         /* PWM frequency - 10kHz */
#define M1_FOC_FREQ_VS_PWM_FREQ (1) /* FOC calculation is called every n-th PWM reload */
#define M1_SLOW_LOOP_FREQ (1000)    /* Slow loop frequency */

#define M1_FAST_LOOP_TS (1.0 / (M1_PWM_FREQ / M1_FOC_FREQ_VS_PWM_FREQ))
#define M1_SLOW_LOOP_TS (1.0 / (M1_SLOW_LOOP_FREQ))
                                                                                
/* Assignment of PWM channels to motor phases 
 * 0 - PWM channels 0&1
 * 1 - PWM channels 2&3
 * 2 - PWM channels 4&5 
 */
#define M1_PWM_PAIR_PHA (0)
#define M1_PWM_PAIR_PHB (1)
#define M1_PWM_PAIR_PHC (2)

/* adjustable DC-bus over-current protection threshold
   can be set in range 250~7750 mA with 250 mA resolution. 
   Set zero value to turn off */
#define OVER_CURRENT_THRESHOLD (7750)

/* Braking resistor macros */
#define M1_BRAKE_SET()   ioctl(GPIO_F, GPIO_SET_PIN, BIT_6)
#define M1_BRAKE_CLEAR() ioctl(GPIO_F, GPIO_CLEAR_PIN, BIT_6)

/******************************************************************************
 * ADC measurement definition
 ******************************************************************************/
/* Configuration table of ADC channels according to the input pin signals:
 * Valid for Kinetis KV46 HVP board (HVP-KV46F150M) together with HVP-MC3PH
 * 
 * Proper ADC channel assignment needs to follow these rules:
 *   - at least one phase current must be assigned to both ADC modules
 *   - two other phase current channels must be assigned to different ADC modules
 *   - Udcb and auxiliary channels must be assigned to differrent ADC modules
 
 
  Quantity    | ADC module 0                  | ADC module 1
--------------------------------------------------------------------------
 I_phA        | ADCA_CH1                	  | -
 I_phB        | -                             | ADCB_CH2 
 I_phC        | ADCA_CH6  			          | ADCB_CH7 
 U_dcbus      | ADCA_CH3		              | -
 Aux(IPM tmp) | -	                          | ADCB_CH0  
*/

/* Phase current A assingned to ADC0 only */
#define M1_ADC0_PH_A (1)
#define M1_ADC1_PH_A (MCDRV_CHAN_OFF)
/* Phase current A assingned to ADC1 only */
#define M1_ADC0_PH_B (MCDRV_CHAN_OFF)
#define M1_ADC1_PH_B (2)
/* Phase current A assingned to ADC0 and ADC1 */
#define M1_ADC0_PH_C (6)
#define M1_ADC1_PH_C (7)
/* Udc bus voltage is assigned to ADC0 only */
#define M1_ADC0_UDCB (3)
#define M1_ADC1_UDCB (MCDRV_CHAN_OFF)
/* Aux channel is assigned to ADC1 only */
#define M1_ADC0_AUX  (MCDRV_CHAN_OFF)
#define M1_ADC1_AUX  (0)

/* offset measurement filter window */     
#define ADC_OFFSET_WINDOW (3)
/******************************************************************************
 * MC driver macro definition and check - do not change this part
 ******************************************************************************/
/******************************************************************************
 * Define motor 1 ADC control functions
 ******************************************************************************/
#ifdef M1_MCDRV_ADC
#if (M1_MCDRV_ADC == MCDRV_ADC12)
#define M1_MCDRV_ADC_PERIPH_INIT() (InitADC12())
#define M1_MCDRV_ADC_GET(par) \
    MCDRV_Curr3Ph2ShGet(par); \
    MCDRV_VoltDcBusGet(par);  \
    MCDRV_AuxValGet(par);
#define M1_MCDRV_CURR_3PH_CHAN_ASSIGN(par) (MCDRV_Curr3Ph2ShChanAssign(par))
#define M1_MCDRV_CURR_3PH_CALIB_INIT(par) (MCDRV_Curr3Ph2ShCalibInit(par))
#define M1_MCDRV_CURR_3PH_CALIB(par) (MCDRV_Curr3Ph2ShCalib(par))
#define M1_MCDRV_CURR_3PH_CALIB_SET(par) (MCDRV_Curr3Ph2ShCalibSet(par))
#else
#error MCDRV: Unsupported ADC periphery!
#endif
#else
#error MCDRV: ADC periphery was not specified!
#endif

/******************************************************************************
 * Define motor 1 3-ph PWM control functions
 ******************************************************************************/
#ifdef M1_MCDRV_PWM3PH
#if (M1_MCDRV_PWM3PH == MCDRV_PWMA)
#define M1_MCDRV_PWM_PERIPH_INIT() (InitPWMA())
#define M1_MCDRV_PWM3PH_SET(par) (MCDRV_eFlexPwm3PhSet(par))
#define M1_MCDRV_PWM3PH_EN(par) (MCDRV_eFlexPwm3PhOutEn(par))
#define M1_MCDRV_PWM3PH_DIS(par) (MCDRV_eFlexPwm3PhOutDis(par))
#define M1_MCDRV_PWM3PH_FLT_GET(par) (MCDRV_eFlexPwm3PhFltGet(par))
#else
#error MCDRV: Unsupported PWM periphery!
#endif
#else
#error MCDRV: PWM periphery was not specified!
#endif

/* FNB41560 internal thermistor fourth-order approximation coeffs */
#define IPM_TEMP_APROX_P1            FRAC16(-0.1115)
#define IPM_TEMP_APROX_P2            FRAC16(0.8401)
#define IPM_TEMP_APROX_P3            FRAC16(-0.9768)
#define IPM_TEMP_APROX_P4            FRAC16(0.5946)
#define IPM_TEMP_APROX_P5            FRAC16(-0.0225)


/******************************************************************************
 * Global variable definitions
 ******************************************************************************/
extern mcdrv_pwma_pwm3ph_t g_sM1Pwm3ph;
extern mcdrv_adc12_t g_sM1AdcSensor;

/*******************************************************************************
 * API
 ******************************************************************************/
#ifdef __cplusplus
extern "C" {
#endif

void MCDRV_Init_M1(void);
void InitPWMA(void);
void InitADC12(void);
void InitRelay(void);
frac16_t GetIPMTemperature(frac16_t f16RawData);

#ifdef __cplusplus
}
#endif
#endif /* _MCDRV_HVP_56F83738_H_ */
