/*
 * Copyright 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2019 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "mcdrv_hvp56f83783.h"

#include "qs.h"
#include "gpio.h"

/*******************************************************************************
 * Variables
 ******************************************************************************/
/* Structure for 3-phase PWM mc driver */
mcdrv_pwma_pwm3ph_t g_sM1Pwm3ph;

/* Structure for current and voltage measurement */
mcdrv_adc12_t g_sM1AdcSensor;


/*******************************************************************************
 * Code
 ******************************************************************************/

/*!
* @brief   void MCDRV_Init_M1(void)
*           - Motor control driver main initialization
*           - Calls initialization functions of peripherals required for motor
*             control functionality
*
* @param   void
*
* @return  none
*/
void MCDRV_Init_M1(void)
{
    /* Init ADC */
    M1_MCDRV_ADC_PERIPH_INIT();
    
    /* 6-channel PWM peripheral init */
    M1_MCDRV_PWM_PERIPH_INIT();

}

/*!
* @brief   void InitPWMA(void)
*           - Initialization of the eFlexPWMA peripheral for motor M1
*           - 3-phase center-aligned PWM
*
* @param   void
*
* @return  none
*/
void InitPWMA(void)
{
    /* Initialize MC driver */
    g_sM1Pwm3ph.pui32PwmBaseAddress = (arch_sEFPWM *)(&ArchIO.EfpwmA);
    g_sM1Pwm3ph.ui16PhASubNum = M1_PWM_PAIR_PHA; /* PWMA phase A submodule number */ //0
    g_sM1Pwm3ph.ui16PhBSubNum = M1_PWM_PAIR_PHB; /* PWMA phase B submodule number */ //1
    g_sM1Pwm3ph.ui16PhCSubNum = M1_PWM_PAIR_PHC; /* PWMA phase C submodule number */ //2
    
    /* PWM Fault number for over current fault detection */
    g_sM1Pwm3ph.ui16FaultFixNum = 0; /* fault number for FNB41560 10.5A over-current fault detection */
    g_sM1Pwm3ph.ui16FaultAdjNum = 1; /* fault number for adjustable comparator over-current fault detection */
}

/*!
* @brief   void InitADC12(void)
*           - Init ADC motor control drivers
*
* @param   void
*
* @return  none
*/
void InitADC12(void)
{
	/**************************************/
	/* motor M1 ADC driver initialization */
	/**************************************/
	/* offset filter window */
	g_sM1AdcSensor.ui16OffsetFiltWindow = ADC_OFFSET_WINDOW;
	
    /* Peripheral ADC base pointer */    
	g_sM1AdcSensor.pui32AdcBase = (arch_sADC *)(&ArchIO.Adc);
	
	/* Phase current measurement */
	/* Sector 1,6 - measured currents Ic & Ib */
	/* ADC0, channel Ic = M1_ADC0_PH_C, , SAMPLE & RESULT = 0 */
	g_sM1AdcSensor.sCurrSec16.ui16ChanNumPhaC = M1_ADC0_PH_C;
	g_sM1AdcSensor.sCurrSec16.ui16AdcNumPhaC = ADC0;
	/* ADC1, channel Ib = M1_ADC1_PH_B, , SAMPLE & RESULT = 8 */
	g_sM1AdcSensor.sCurrSec16.ui16ChanNumPhaB = M1_ADC1_PH_B;
	g_sM1AdcSensor.sCurrSec16.ui16AdcNumPhaB = ADC1;

	/* Sector 2,3 - measured currents Ia & Ic*/
	/* ADC0, channel Ia = M1_ADC0_PH_A, SAMPLE & RESULT = 0 */
	g_sM1AdcSensor.sCurrSec23.ui16ChanNumPhaA = M1_ADC0_PH_A;
	g_sM1AdcSensor.sCurrSec23.ui16AdcNumPhaA = ADC0;
	/* ADC1, channel Ic = M1_ADC1_PH_C, SAMPLE & RESULT = 8 */
	g_sM1AdcSensor.sCurrSec23.ui16ChanNumPhaC = M1_ADC1_PH_C;
	g_sM1AdcSensor.sCurrSec23.ui16AdcNumPhaC = ADC1;

	/* Sector 4,5 - measured currents Ia & Ib */
	/* ADC0, channel Ia = M1_ADC0_PH_A, SAMPLE & RESULT = 0 */
	g_sM1AdcSensor.sCurrSec45.ui16ChanNumPhaA = M1_ADC0_PH_A;
	g_sM1AdcSensor.sCurrSec45.ui16AdcNumPhaA = ADC0;
	/* ADC1, channel Ib = M1_ADC1_PH_B, SAMPLE & RESULT = 8  */
	g_sM1AdcSensor.sCurrSec45.ui16ChanNumPhaB = M1_ADC1_PH_B;
	g_sM1AdcSensor.sCurrSec45.ui16AdcNumPhaB = ADC1;
	
	/* UDCbus channel measurement */
	/*  channel Udcb = M1_ADC0_UDCB, SAMPLE & RESULT = 1 */
	g_sM1AdcSensor.ui16ChanNumVDcb = M1_ADC0_UDCB;
	g_sM1AdcSensor.ui16AdcNumVDcb = ADC0; 
	
	/* Aux channel measurement */
	/*  channel Aux = M1_ADC1_AUX, SAMPLE & RESULT = 9 */
	g_sM1AdcSensor.ui16ChanNumAux = M1_ADC1_AUX;
	g_sM1AdcSensor.ui16AdcNumAux = ADC1; 	

    /* Assign channels and init all pointers */
    MCDRV_Curr3Ph2ShChanAssignInit(&g_sM1AdcSensor);

}

/*!
*@brief      Set Inrush relay on HVP
*
*@param      none
*            
*@return     none
*/
void InitRelay(void)
{
	uint16_t i,j;
	
	/* delay */
	for(i=0;i<5000;i++)
	{
		for(j=0;j<1000;j++)
		{
			asm(nop);
		}
	}

	/* turn on relay */
	ioctl(GPIO_F, GPIO_SET_PIN, BIT_7);   
    
}

/*!
*@brief      Transform raw FNB41560 IPM temperature measurement to Celsius degrees
*            -fourth degree polynomial approximation is used 
*
*@param      ui16RawData - raw ADC measurement
*            
*@return     float_t
*/
frac16_t GetIPMTemperature(frac16_t f16RawData)
{
    frac16_t f16Temperature;
    
    /* get temperature in Celsius degrees */
    f16Temperature = MLIB_MacSat_F16(IPM_TEMP_APROX_P5,f16RawData,
    				 MLIB_MacSat_F16(IPM_TEMP_APROX_P4,f16RawData,
    				 MLIB_MacSat_F16(IPM_TEMP_APROX_P3,f16RawData,
    			     MLIB_MacSat_F16(IPM_TEMP_APROX_P2,IPM_TEMP_APROX_P1,f16RawData))));

    return f16Temperature; 
}


