/*
 * The Clear BSD License
 * Copyright (c) 2013 - 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided
 * that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS LICENSE.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "board.h"
#include "fsl_spifi.h"

#include "pin_mux.h"
#include <stdbool.h>
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define APP_BOARD_TEST_LED_PORT		2U
#define APP_BOARD_TEST_LED_PIN 		2U


//#define QUAD_MODE
//#define MX25R6435F
#define FLASH_W25Q

#define COMMAND_NUM (11)
#define READ (0)
#define PROGRAM_PAGE (1)
#define GET_STATUS (2)
#define ERASE_SECTOR (3)
#define WRITE_ENABLE (4)
#define WRITE_REGISTER (5)
#define GET_STATUS2 (6)
#define WRITE_DISABLE (7)
#define READ_ID_SINGLE (8)
#define SOFT_RESET_EN (9)
#define SOFT_RESET (10)

#define PAGE_SIZE 							256
#define FROCTRL_HSPDCLK_ENA     (1UL << 30)

/*******************************************************************************
 * Variables
 ******************************************************************************/
spifi_command_t command[COMMAND_NUM] = 
{
#ifdef QUAD_MODE
    {PAGE_SIZE, false, kSPIFI_DataInput, 1, kSPIFI_CommandDataQuad, kSPIFI_CommandOpcodeAddrThreeBytes, 0x6B},
#else		
    {PAGE_SIZE, false, kSPIFI_DataInput, 1, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeAddrThreeBytes, 0x0B},
#endif		
#ifdef FLASH_W25Q
		{PAGE_SIZE, false, kSPIFI_DataOutput, 0, kSPIFI_CommandDataQuad, kSPIFI_CommandOpcodeAddrThreeBytes, 0x32},
#else		
    {PAGE_SIZE, false, kSPIFI_DataOutput, 0, kSPIFI_CommandOpcodeSerial, kSPIFI_CommandOpcodeAddrThreeBytes, 0x38},
#endif
    {4, false, kSPIFI_DataInput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeOnly, 0x05},
    {0, false, kSPIFI_DataOutput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeAddrThreeBytes, 0x20},
    {0, false, kSPIFI_DataOutput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeOnly, 0x06},
#ifdef MX25R6435F
    {4, false, kSPIFI_DataOutput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeOnly, 0x01},
#elseif FLASH_W25Q
    {1, false, kSPIFI_DataOutput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeOnly, 0x31}};
#else		
    {2, false, kSPIFI_DataOutput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeOnly, 0x01},
#endif		
    {4, false, kSPIFI_DataInput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeOnly, 0x35},
    {0, false, kSPIFI_DataOutput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeOnly, 0x04},
    {1, false, kSPIFI_DataInput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeOnly, 0x9F},
    {0, false, kSPIFI_DataOutput, 0, kSPIFI_CommandDataQuad, kSPIFI_CommandOpcodeOnly, 0x66},
    {0, false, kSPIFI_DataOutput, 0, kSPIFI_CommandDataQuad, kSPIFI_CommandOpcodeOnly, 0x99},
};

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
void spifi_init(void);
void delay(void);
/*******************************************************************************
 * Code
 ******************************************************************************/

void check_if_finish()
{
    uint32_t val = 0;
    /* Check WIP bit */
    do
    {
        SPIFI_SetCommand(SPIFI0, &command[GET_STATUS]);
        while ((SPIFI0->STAT & SPIFI_STAT_INTRQ_MASK) == 0U)
        {}
        val = SPIFI_ReadData(SPIFI0);
    } while (val & 0x1);
}


void enable_quad_mode()
{
    /* Write enable */
    SPIFI_SetCommand(SPIFI0, &command[WRITE_ENABLE]);

#ifdef MX25R6435F
		/* Set write register command */
    SPIFI_SetCommand(SPIFI0, &command[WRITE_REGISTER]);

    SPIFI_WriteData(SPIFI0, 0x40);
#elseif FLASH_W25Q
			/* Set write register command */
    SPIFI_SetCommand(SPIFI0, &command[WRITE_REGISTER]);

    SPIFI_WriteData(SPIFI0, 0x02);
#else
    /* get status */
    uint32_t status;
    SPIFI_SetCommand(SPIFI0, &command[GET_STATUS]);
    while ((SPIFI0->STAT & SPIFI_STAT_INTRQ_MASK) == 0U)
    {}
    status = (SPIFI_ReadData(SPIFI0) & 0xFF);
    SPIFI_SetCommand(SPIFI0, &command[GET_STATUS2]);
    while ((SPIFI0->STAT & SPIFI_STAT_INTRQ_MASK) == 0U)
    {}
    status |= ((SPIFI_ReadData(SPIFI0) & 0xFF) << 8);
    status |= (1 << 9);
    /* Set write register command */
      SPIFI_SetCommand(SPIFI0, &command[WRITE_REGISTER]);
    while ((SPIFI0->STAT & SPIFI_STAT_INTRQ_MASK) == 0U)
    {}
    uint8_t * pSpifiDat8 = (uint8_t *) &SPIFI0->DATA;
    *pSpifiDat8 = (uint8_t) (status & 0xFF);
    *pSpifiDat8 = (uint8_t) ((status >> 8) & 0xFF);
    /* Write disable */
    SPIFI_SetCommand(SPIFI0, &command[WRITE_DISABLE]);
#endif
	
    check_if_finish();
}


 void spifi_init(void)
{
    uint32_t sourceClockFreq;
    spifi_config_t config = {0};
    static volatile uint32_t id;
	
		/* set up SPIFI */
    /* Set SPIFI clock source */
		SYSCON->FROCTRL |= FROCTRL_HSPDCLK_ENA;
    CLOCK_AttachClk(kFRO_HF_to_SPIFI_CLK);
    sourceClockFreq = CLOCK_GetSpifiClkFreq();

    /* Set the clock divider */
    CLOCK_SetClkDiv(kCLOCK_DivSpifiClk, (sourceClockFreq / 25000000) - 1U, false);

    /* fix bug with SPIFI reset */
    SYSCON->PRESETCTRLCLR[0] = SYSCON_PRESETCTRL_SPIFI_RST_MASK;
    
    /* Initialize SPIFI */
    SPIFI_GetDefaultConfig(&config);
    SPIFI_Init(SPIFI0, &config);

    /* reset device */
    SPIFI_SetCommand(SPIFI0, &command[SOFT_RESET_EN]);
    while ((SPIFI0->STAT & SPIFI_STAT_INTRQ_MASK) == 0U)
    {}
    SPIFI_SetCommand(SPIFI0, &command[SOFT_RESET]);
    while ((SPIFI0->STAT & SPIFI_STAT_INTRQ_MASK) == 0U)
    {}
  
    /* read ID */
    SPIFI_SetCommand(SPIFI0, &command[READ_ID_SINGLE]);
    while ((SPIFI0->STAT & SPIFI_STAT_INTRQ_MASK) == 0U)
    {}
    id = (SPIFI_ReadDataByte(SPIFI0) & 0xFFFFFFF);
#ifdef QUAD_MODE
    /* Enable Quad mode */
    enable_quad_mode();
#endif
    /* Setup memory command */
    SPIFI_SetMemoryCommand(SPIFI0, &command[READ]);
	
}
 
 
/*!
 * @brief Main function
 */
int main(void)
{
    uint32_t i = 0;
	
    /* Define the init structure for the output LED pin*/
    gpio_pin_config_t led_config = {
        kGPIO_DigitalOutput, 0,
    };
		
    /* Init board hardware. */
    BOARD_InitPins();
    BOARD_BootClockFRO12M();
		
    /* Configure SPIFI */
    spifi_init();

    /* Init output LED GPIO. */
    CLOCK_EnableClock(kCLOCK_Gpio0); 		
    GPIO_PortInit(GPIO, APP_BOARD_TEST_LED_PORT);
    GPIO_PinInit(GPIO, APP_BOARD_TEST_LED_PORT, APP_BOARD_TEST_LED_PIN, &led_config);
    GPIO_PinWrite(GPIO, APP_BOARD_TEST_LED_PORT, APP_BOARD_TEST_LED_PIN, 1);
		
    while (1)
    {
      GPIO_PortToggle(GPIO, APP_BOARD_TEST_LED_PORT, 1u << APP_BOARD_TEST_LED_PIN);
      for (i = 0; i < 250; i++)
      {
         delay();
      }		
    }
}
