/*
 * @brief LPC5410x Power library (hidden)
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "chip.h"
#include <stdlib.h>
#include <string.h>

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
/** @brief The real 'SYSCON', private
 */
typedef struct {
	volatile unsigned int SYSMEMREMAP;								// /< System Remap                (0x000)
	volatile unsigned int AHBMATPRIO;									// /< AHB Matrix Priority         (0x004)
	volatile unsigned int RESERVED0[3];
	volatile unsigned int SYSTCKCAL;									// /< System Tick Calibration     (0x014)
	volatile unsigned int RESERVED1;
	volatile unsigned int NMISRC;											// /< NMI Source select           (0x01C)
	volatile unsigned int ASYNCAPBCTRL;								// /< Asynch APB chiplet control  (0x020)
	volatile unsigned int RESERVED2[7];
	volatile unsigned int SYSRESSTAT;									// /< System Reset Stat           (0x040)
	volatile unsigned int PRESETCTRL[2];							// /< Peripheral Reset Ctrl       (0x044 - 0x048)
	volatile unsigned int PRESETCTRLSET[2];						// /< Peripheral Reset Ctrl Set   (0x04C - 0x050)
	volatile unsigned int PRESETCTRLCLR[2];						// /< Peripheral Reset Ctrl Clr   (0x054 - 0x058)
	volatile unsigned int PIOPORCAP[2];								// /< PIO Power-On Reset Capture  (0x05C - 0x060)
	volatile unsigned int RESERVED3;
	volatile unsigned int PIORESCAP[2];								// /< PIO Pad Reset Capture       (0x068 - 0x06C)
	volatile unsigned int RESERVED4[4];
	volatile unsigned int MAINCLKSELA;								// /< Main Clk sel Source Sel A   (0x080)
	volatile unsigned int MAINCLKSELB;								// /< Main Clk sel Source Sel B   (0x084)
	volatile unsigned int RESERVED5;
	volatile unsigned int ADCASYNCCLKSEL;							// /< ADC Async Clk Sel           (0x08C)
	volatile unsigned int RESERVED6;
	volatile unsigned int CLKOUTCLKSELA;							// /< Clk Out Sel Source A        (0x094)
	volatile unsigned int CLKOUTCLKSELB;							// /< Clk Out Sel Source B        (0x098)
	volatile unsigned int RESERVED7;
	volatile unsigned int SYSPLLCLKSEL;								// /< System PLL Clk Sel          (0x0A0)
	volatile unsigned int RESERVED8[7];
	volatile unsigned int SYSAHBCLKCTRL[2];						// /< AHB Peripheral Clk Enable    (0x0C0 - 0x0C4)
	volatile unsigned int SYSAHBCLKCTRLSET[2];				// /< AHB Peripheral Clk Enable Set(0x0C8 - 0xCC)
	volatile unsigned int SYSAHBCLKCTRLCLR[2];				// /< AHB Peripheral Clk Enable Clr(0x0D0 - 0xD4)
	volatile unsigned int RESERVED9[2];
	volatile unsigned int SYSTICKCLKDIV;							// /< Systick Clock divider       (0x0E0)
	volatile unsigned int RESERVED10[7];
	volatile unsigned int SYSAHBCLKDIV;								// /< Main Clk Divider            (0x100)
	volatile unsigned int RESERVED11;
	volatile unsigned int ADCASYNCCLKDIV;							// /< ADC Async Clk Divider       (0x108)
	volatile unsigned int CLKOUTDIV;									// /< Clk Out Divider             (0x10C)
	volatile unsigned int RESERVED12[4];
	volatile unsigned int FREQMECTRL;									// /< Frequency Measure Control   (0x120)
	volatile unsigned int FLASHCFG;										// /< Flash Config                (0x124)
	volatile unsigned int RESERVED13[8];
	volatile unsigned int VFIFOCTRL;									// /< VFIFO control               (0x148)
	volatile unsigned int RESERVED14[14];
	volatile unsigned int IRCCTRL;										// /< IRC Oscillator Control      (0x184)
	volatile unsigned int RESERVED15[2];
	volatile unsigned int RTCOSCCTRL;									// /< RTC Oscillator Control      (0x190)
	volatile unsigned int RESERVED16[7];
	volatile unsigned int SYSPLLCTRL;									// /< SYS PLL CTRL                (0x1B0)
	volatile unsigned int SYSPLLSTAT;									// /< SYS PLL STAT                (0x1B4)
	volatile unsigned int SYSPLLNDEC;									// /< SYS PLL NDEC                (0x1B8)
	volatile unsigned int SYSPLLPDEC;									// /< SYS Pll PDEC                (0x1BC)
	volatile unsigned int SYSPLLSSCGCTRL[2];					// /< Spread Spectrum control     (0x1C0 - 0x1C4)
	volatile unsigned int RESERVED17[14];
	volatile unsigned int PDSLEEPCFG;									// /< Power Down Sleep Config     (0x200)
	volatile unsigned int RESERVED18[3];
	volatile unsigned int PDRUNCFG;										// /< Power Down Run Config       (0x210)
	volatile unsigned int PDRUNCFGSET;								// /< Power Down Run Config Set   (0x214)
	volatile unsigned int PDRUNCFGCLR;								// /< Power Down Run Config Clr   (0x218)
	volatile unsigned int RESERVED19[9];
	volatile unsigned int STARTER[2];									// /< Start Signal Enable Register     (0x240 - 0x244)
	volatile unsigned int STARTERSET[2];							// /< Start Signal Enable Set Register (0x248 - 0x24C)
	volatile unsigned int STARTERCLR[2];							// /< Start Signal Enable Clr Register (0x250 - 0x254)
	volatile unsigned int RESERVED20[42];
	volatile unsigned int CPUCTRL;										// /< CPU CTRL                         (0x300)
	volatile unsigned int CPBOOT;											// /< COPROCESSOR BOOT                 (0x304)
	volatile unsigned int CPSTACK;										// /< COPROCESSOR STACK                (0x308)
	volatile unsigned int CPUSTAT;										// /< CPU STAT                         (0x30C)
	volatile unsigned int RESERVED21[57];
	volatile unsigned int JTAG_IDCODE;								// /<                                  (0x3F4)
	volatile unsigned int DEVICE_ID0;									// /<                                  (0x3F8)
	volatile unsigned int DEVICE_ID1;									// /<                                  (0x3FC)
} ROM_SYSCON_T;

#define ROM_SYSCON         ((ROM_SYSCON_T          *) LPC_SYSCON_BASE)

/** @brief The real 'PMU/POWER', private
 */
typedef volatile struct {
	volatile uint32_t VDCTRL[4];	                    // /< VD1, VD2, VD3, VD8 Domain Voltage Control (0x00 - 0x0C)
} ROM_POWER_T;

#define ROM_POWER          ((ROM_POWER_T            *) LPC_PMU_BASE)

#define V4_UID						(0x08C1FECE)
#define MAX_CLOCK_KHZ			(96000)
#define MAX_CLOCK_V4_KHZ	(100000)

typedef struct {
	uint32_t data[9];
} pwr_vd_table_t;

/* Override table for parts with partial table in index sector */
/* The list below is the frequency range from 0 to 108MHz;
   bit 0~6 is VD1, bit 7~11 is VD2, bit 12~17 is VD3, bit 18~23 is
   VD8, bit 24~31 are # of flash wait states. */

/* Power table for Rev-C silicon */
static const pwr_vd_table_t vd_table_c[1] = {
	/* LOW POWER mode */
	{
			/* VD1				VD2						VD3					VD8					WS */
		/* 0~12 Mhz */
		{
			POWER_V0950 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V0950 << 18) | (0 << 24),

			/* 12~24 Mhz */
			POWER_V1000 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1000 << 18) | (1 << 24),

			/* 24~36 Mhz */
			POWER_V1050 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1050 << 18) | (2 << 24),

			/* 36~48 Mhz */
			POWER_V1100 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1100 << 18) | (2 << 24),

			/* 48~60 Mhz */
			POWER_V1150 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1150 << 18) | (3 << 24),

			/* 60~72 Mhz */
			POWER_V1200 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1200 << 18) | (3 << 24),

			/* 72~84 Mhz */
			POWER_V1200 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1200 << 18) | (4 << 24),

			/* 84~96 Mhz */
			POWER_V1250 | (POWER_V1200 << 6) | (POWER_V1250 << 12) | (POWER_V1250 << 18) | (5 << 24),

			/* 96~100 Mhz */
			POWER_V1300 | (POWER_V1200 << 6) | (POWER_V1300 << 12) | (POWER_V1300 << 18) | (5 << 24),

		}

	}
};

static const pwr_vd_table_t vd_table[1] = {
	/* LOW POWER mode */
	{
			/* VD1				VD2						VD3					VD8					WS */
		/* 0~12 Mhz */
		{
			POWER_V1000 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1000 << 18) | (0 << 24),

			/* 12~24 Mhz */
			POWER_V1050 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1050 << 18) | (1 << 24),

			/* 24~36 Mhz */
			POWER_V1100 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1100 << 18) | (2 << 24),

			/* 36~48 Mhz */
			POWER_V1150 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1150 << 18) | (2 << 24),

			/* 48~60 Mhz */
			POWER_V1200 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1200 << 18) | (3 << 24),

			/* 60~72 Mhz */
			POWER_V1250 | (POWER_V1200 << 6) | (POWER_V1250 << 12) | (POWER_V1250 << 18) | (3 << 24),

			/* 72~84 Mhz */
			POWER_V1250 | (POWER_V1200 << 6) | (POWER_V1250 << 12) | (POWER_V1250 << 18) | (4 << 24),

			/* 84~96 Mhz */
			POWER_V1300 | (POWER_V1200 << 6) | (POWER_V1300 << 12) | (POWER_V1300 << 18) | (5 << 24)
		}

	}
};

/* Bit definitions for PDRUNCFG */
#define PDRUNCFG_MAINCLK_SHUTOFF  (1 << 0)
#define PDRUNCFG_DEEP_PD          (1 << 1)
#define PDRUNCFG_LP_VD1           (1 << 2)
#define PDRUNCFG_PD_IRC_OSC_EN    (1 << 3)
#define PDRUNCFG_PD_IRC_EN        (1 << 4)
#define PDRUNCFG_PD_FLASH         (1 << 5)
#define PDRUNCFG_PD_EEPROM        (1 << 6)
#define PDRUNCFG_PD_BOD_RESET     (1 << 7)
#define PDRUNCFG_PD_BOD_INTR      (1 << 8)
#define PDRUNCFG_PD_VD2_ANA       (1 << 9)
#define PDRUNCFG_PD_ADC0          (1 << 10)
#define PDRUNCFG_PD_VDDFLASH_ENA  (1 << 11)
#define PDRUNCFG_VDDFLASH_SEL_VD3 (1 << 12)
#define PDRUNCFG_PD_RAM0          (1 << 13)
#define PDRUNCFG_PD_RAM1          (1 << 14)
#define PDRUNCFG_PD_RAM2          (1 << 15)
#define PDRUNCFG_PD_RAM3          (1 << 16)
#define PDRUNCFG_PD_ROM           (1 << 17)
#define PDRUNCFG_PD_VDDHV_ENA     (1 << 18)
#define PDRUNCFG_PD_VD7_ENA       (1 << 19)
#define PDRUNCFG_PD_WDT_OSC       (1 << 20)
#define PDRUNCFG_PD_SYS_PLL0      (1 << 22)
#define PDRUNCFG_PD_VREFP_SW      (1 << 23)
#define PDRUNCFG_PD_32K_OSC       (1 << 24)
#define PDRUNCFG_PD_FLASH_BG      (1 << 25)
#define PDRUNCFG_PD_VD3           (1 << 26)
#define PDRUNCFG_LP_VD2           (1 << 27)
#define PDRUNCFG_LP_VD3           (1 << 28)
#define PDRUNCFG_LP_VD8           (1UL << 29)
#define PDRUNCFG_REQ_DELAY        (1UL << 30)
#define PDRUNCFG_FORCE_RBB        (1UL << 31)

#define PCFG_DEEP_SLEEP (    PDRUNCFG_MAINCLK_SHUTOFF	\
							 | PDRUNCFG_LP_VD1			  \
							 | PDRUNCFG_PD_IRC_OSC_EN	  \
							 | PDRUNCFG_PD_IRC_EN		  \
							 | PDRUNCFG_PD_EEPROM		  \
							 | PDRUNCFG_PD_BOD_INTR		  \
							 | PDRUNCFG_PD_VD2_ANA		  \
							 | PDRUNCFG_PD_ADC0			  \
							 | PDRUNCFG_PD_ROM			  \
							 | PDRUNCFG_PD_VD7_ENA		  \
							 | PDRUNCFG_PD_WDT_OSC		  \
							 | PDRUNCFG_PD_SYS_PLL0		  \
							 | PDRUNCFG_PD_VREFP_SW		  \
							 | PDRUNCFG_LP_VD2			  \
							 | PDRUNCFG_LP_VD3			  \
							 | PDRUNCFG_LP_VD8			  \
							 | PDRUNCFG_REQ_DELAY		  \
							 | PDRUNCFG_PD_32K_OSC		  \
							 )

#define PCFG_POWERDOWN  (    PDRUNCFG_MAINCLK_SHUTOFF	\
							 | PDRUNCFG_LP_VD1			  \
							 | PDRUNCFG_PD_IRC_OSC_EN	  \
							 | PDRUNCFG_PD_IRC_EN		  \
							 | PDRUNCFG_PD_FLASH		  \
							 | PDRUNCFG_PD_EEPROM		  \
							 | PDRUNCFG_PD_BOD_RESET	  \
							 | PDRUNCFG_PD_BOD_INTR		  \
							 | PDRUNCFG_PD_VD2_ANA		  \
							 | PDRUNCFG_PD_ADC0			  \
							 | PDRUNCFG_PD_VDDFLASH_ENA			  \
							 | PDRUNCFG_VDDFLASH_SEL_VD3  \
							 | PDRUNCFG_PD_RAM1			  \
							 | PDRUNCFG_PD_RAM2			  \
							 | PDRUNCFG_PD_RAM3			  \
							 | PDRUNCFG_PD_ROM			  \
							 | PDRUNCFG_PD_VDDHV_ENA	  \
							 | PDRUNCFG_PD_VD7_ENA		  \
							 | PDRUNCFG_PD_WDT_OSC		  \
							 | PDRUNCFG_PD_SYS_PLL0		  \
							 | PDRUNCFG_PD_VREFP_SW		  \
							 | PDRUNCFG_PD_FLASH_BG		  \
							 | PDRUNCFG_LP_VD2			  \
							 | PDRUNCFG_LP_VD3			  \
							 | PDRUNCFG_LP_VD8			  \
							 | PDRUNCFG_REQ_DELAY		  \
							 | PDRUNCFG_FORCE_RBB		  \
							 | PDRUNCFG_PD_32K_OSC		  \
							 )

#define PERIPH_CTRL_MASK_DEEP_SLEEP (PDRUNCFG_PD_BOD_INTR | PDRUNCFG_PD_WDT_OSC | \
                                     PDRUNCFG_PD_32K_OSC)

#define PERIPH_CTRL_MASK_POWERDOWN  (PDRUNCFG_PD_BOD_RESET | PDRUNCFG_PD_BOD_INTR | PDRUNCFG_PD_RAM1 | \
                                     PDRUNCFG_PD_RAM2 | PDRUNCFG_PD_RAM3 | PDRUNCFG_PD_WDT_OSC | PDRUNCFG_PD_32K_OSC)

#define PCFG_DEEP_POWERDOWN 0xFFFFFFFF

#define POWER_NORMAL_VLEVEL(X)      ((X & 0xf)<<0)
#define POWER_NORMAL_FINE_VLEVEL(X) ((X & 0x3)<<4)

#define EN0_FLASH                   (1 << 7)

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/

/* Return the boot ROM version */
static uint32_t getBROMVersion(void)
{
	uint32_t command[5], result[4];

	command[0] = IAP_READ_BOOT_CODE_CMD;
	iap_entry(command, result);

	return result[1];
}

/* Return the UID[1] value */
static uint32_t getUID1(void)
{
	return LPC_SYSCON->DEVICE_ID1;
}

void set_vd_level(uint32_t domain, uint32_t level, uint32_t flevel) 
{
  ROM_POWER->VDCTRL[domain] = POWER_NORMAL_VLEVEL(level) | POWER_NORMAL_FINE_VLEVEL(flevel);
}

static uint32_t __set_voltage_17_0(uint32_t mode, uint32_t desired_freq)
{
	uint32_t error = LPC_OK;

	ROM_SYSCON->PDRUNCFGCLR = PDRUNCFG_LP_VD1 | PDRUNCFG_LP_VD2 | PDRUNCFG_LP_VD3 | PDRUNCFG_LP_VD8;

	if (desired_freq < 20000000) {
		set_vd_level(POWER_VD1, POWER_V0900, POWER_FINE_V_NONE);
		set_vd_level(POWER_VD2, POWER_V1200, POWER_FINE_V_NONE);
		set_vd_level(POWER_VD3, POWER_V1200, POWER_FINE_V_NONE);
		set_vd_level(POWER_VD8, POWER_V0900, POWER_FINE_V_NONE);
	}
	else if (desired_freq < 80000000) {
		set_vd_level(POWER_VD1, POWER_V1100, POWER_FINE_V_NONE);
		set_vd_level(POWER_VD2, POWER_V1200, POWER_FINE_V_NONE);
		set_vd_level(POWER_VD3, POWER_V1200, POWER_FINE_V_NONE);
		set_vd_level(POWER_VD8, POWER_V1100, POWER_FINE_V_NONE);
	}
	else if (desired_freq <= 100000000) {
		set_vd_level(POWER_VD1, POWER_V1200, POWER_FINE_V_NONE);
		set_vd_level(POWER_VD2, POWER_V1200, POWER_FINE_V_NONE);
		set_vd_level(POWER_VD3, POWER_V1200, POWER_FINE_V_NONE);
		set_vd_level(POWER_VD8, POWER_V1200, POWER_FINE_V_NONE);
	}
	else {
		/* Out of range */
		error = ERR_CLK_INVALID_PARAM;
	}

	return error;
}
static uint32_t __set_voltage( uint32_t mode, uint32_t frequency )
{
	uint32_t flashcfg_def;
	uint32_t volt_configure;
	uint32_t max_clock_khz;
	const pwr_vd_table_t *table;
	uint32_t rev = getUID1();
	int val;

	/* Right now, only LOW_POWER mode is supported. */
	if ( mode > 0 ) {
		return ( PWR_ERROR_INVALID_CFG );
	}
	/* Version 3 of ROM is higher speed capable */
	max_clock_khz = MAX_CLOCK_KHZ;
	if ((getBROMVersion() == 0x1101) && (rev == V4_UID)) {
		max_clock_khz = MAX_CLOCK_V4_KHZ;
	}

	/* Check the requested frequency to be sure it is in range */
	if ( (frequency/1000) > max_clock_khz ) {
		return ( ERR_CLK_INVALID_PARAM );
	}

	LPC_SYSCON->PDRUNCFGCLR = PDRUNCFG_LP_VD1 | PDRUNCFG_LP_VD2 | PDRUNCFG_LP_VD3 | PDRUNCFG_LP_VD8;
	flashcfg_def = LPC_SYSCON->FLASHCFG & ~(0xF<<12);

	/* Rev-C silicon has a different table */
	if (rev == V4_UID) {
		table = &vd_table_c[mode];
	} else {
		table = &vd_table[mode];
	}

	if ( frequency <= 12000000 ) {
		val = 0;
	} else if ( frequency <= 24000000 ) {
		val = 1;
	} else if ( frequency <= 36000000 ) {
		val = 2;
	} else if ( frequency <= 48000000 ) {
		val = 3;
	} else if ( frequency <= 60000000 ) {
		val = 4;
	} else if ( frequency <= 72000000 ) {
		val = 5;
	} else if ( frequency <= 84000000 ) {
		val = 6;
	} else  { /* Set to max value for 84+ Mhz */
		if (rev == V4_UID && frequency > 96000000) {
			val = 8;
		} else {
			val = 7;
		}
	}
	volt_configure = table->data[val];

	/* This check handles the case that the vd_table data is not valid.
	 * shouldn't happen in this library, but leaving for completeness. */
	if ( (volt_configure == 0x0) || ( volt_configure == 0xFFFFFFFF) ) {
		/* If the table (used to be the index sector) is not configured correctly, set the highest voltage and longest wait state. */
		set_vd_level(POWER_VD1, POWER_V1300, POWER_FINE_V_NONE);
		set_vd_level(POWER_VD2, POWER_V1200, POWER_FINE_V_NONE);
		set_vd_level(POWER_VD3, POWER_V1300, POWER_FINE_V_NONE);
		set_vd_level(POWER_VD8, POWER_V1300, POWER_FINE_V_NONE);

		/* Set wait states to 5 */
		LPC_SYSCON->FLASHCFG = flashcfg_def | (0x5<<12);
	}
	else {
		set_vd_level(POWER_VD1, (volt_configure & 0xF), ((volt_configure>>4) & 0x3));
		set_vd_level(POWER_VD2, ((volt_configure>>6) & 0xF), ((volt_configure>>10) & 0x3));
		set_vd_level(POWER_VD3, ((volt_configure>>12) & 0xF), ((volt_configure>>16) & 0x3));
		set_vd_level(POWER_VD8, ((volt_configure>>18) & 0xF), ((volt_configure>>22) & 0x3));
		/* Set wait states */
		if ( ((volt_configure>>24)&0xF) != 0 ) {
			/* If wait state is not zero, frequency is greater than IRC. Also set prefetch bit. */
			LPC_SYSCON->FLASHCFG = flashcfg_def | (((volt_configure>>24)&0xF)<<12) | (0x1<<5);
		}
		else {
			LPC_SYSCON->FLASHCFG = flashcfg_def | (((volt_configure>>24)&0xF)<<12);
		}
	}
	return ( LPC_OK );
}

static void __set_pmu_sleep(void)
{
	SCB->SCR &= ~SCB_SCR_SLEEPDEEP_Msk;
	__WFI();
}

static void __set_pmu_deep_sleep(unsigned int peripheral_ctrl)
{
	uint32_t pdruncfg_val, pmsk;

	pmsk = __get_PRIMASK();
	__disable_irq();
	SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;

	pdruncfg_val = ROM_SYSCON->PDRUNCFG;
	ROM_SYSCON->PDSLEEPCFG = (PCFG_DEEP_SLEEP | pdruncfg_val) & ~(peripheral_ctrl & PERIPH_CTRL_MASK_DEEP_SLEEP);

	__WFI();

	SCB->SCR &= ~SCB_SCR_SLEEPDEEP_Msk;
	__set_PRIMASK(pmsk);
}

static void __set_pmu_powerdown(uint32_t peripheral_ctrl)
{
	uint32_t pdruncfg_val, sysahbclkctrl_val, pmsk;
	uint32_t i;
	volatile uint32_t j;
	volatile uint32_t temp = temp;

	pmsk = __get_PRIMASK();
	__disable_irq();
	SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;

	pdruncfg_val = ROM_SYSCON->PDRUNCFG;
	ROM_SYSCON->PDSLEEPCFG = (PCFG_POWERDOWN | pdruncfg_val) & ~(peripheral_ctrl & PERIPH_CTRL_MASK_POWERDOWN);

	sysahbclkctrl_val = ROM_SYSCON->SYSAHBCLKCTRL[0];
	/* TURN ON clock for flash controller */
	ROM_SYSCON->SYSAHBCLKCTRLSET[0] = EN0_FLASH;
	/* Turn VDDHV off and assert FLASH PD */
	ROM_SYSCON->PDRUNCFGSET         = PDRUNCFG_PD_VDDHV_ENA | PDRUNCFG_PD_FLASH;
	/* Assert flash controller reset */
	ROM_SYSCON->PRESETCTRLSET[0]    = EN0_FLASH;

	/* Enter powerdown mode */
	__WFI();

	if (!(pdruncfg_val & PDRUNCFG_PD_FLASH)) {
		/* Turn VDDHV on and deassert FLASH PD */
		ROM_SYSCON->PDRUNCFGCLR = PDRUNCFG_PD_FLASH | PDRUNCFG_PD_VDDHV_ENA;
		/* 150usec start up time for flash bandgap */
#if   defined ( __CC_ARM ) /*------------------RealView Compiler -----------------*/
		/* Optimization -O3 */
		for (i = 0; i <= 305; i++) {
#elif defined ( __ICCARM__ ) /*------------------ ICC Compiler -------------------*/
		/* Optimization High Balanced */
		for (i = 0; i <= 261; i++) {
#elif defined ( __GNUC__ ) /*------------------ GNU Compiler ---------------------*/
		/* Optimization -Os */
		for (i = 0; i <= 202; i++) {
#endif
			j = i;
		}
	}

	/* Deassert flash controller reset */
	ROM_SYSCON->PRESETCTRLCLR[0] = EN0_FLASH;

	/* Restore original pdruncfg */
	ROM_SYSCON->PDRUNCFG = pdruncfg_val;

	/* Dummy read back flash to know that initialisation is completed. Any location above vector table */
	temp = *(volatile uint32_t *) 0x1000;
	if (!(sysahbclkctrl_val & EN0_FLASH)) {
		/* TURN OFF clock for flash controller */
		ROM_SYSCON->SYSAHBCLKCTRLCLR[0] = EN0_FLASH;
	}

	SCB->SCR &= ~SCB_SCR_SLEEPDEEP_Msk;
	__set_PRIMASK(pmsk);
}

static void __set_pmu_deep_powerdown(uint32_t peripheral_ctrl)
{
	/* System can only exit deep powerdown from wakeup reset */
	__disable_irq();
	SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;

	/* The system will enter powerdown on the following call */
	/* Turn off everything except the blocks excluded */
	ROM_SYSCON->PDRUNCFG = PCFG_DEEP_POWERDOWN  & ~(peripheral_ctrl & PDRUNCFG_PD_32K_OSC);

	/* It shouldn't make it here */
	__WFI();
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/
/* Sets up the System PLL given the PLL input frequency and feedback multiplier */
uint32_t Chip_POWER_SetPLL(uint32_t multiply_by, uint32_t input_freq)
{
	if ( (multiply_by == 0) || (((multiply_by*input_freq)/1000) > MAX_CLOCK_V4_KHZ) ) {
		return ( ERR_CLK_INVALID_PARAM );
	}
	/* First parameter is the multiplier, the second parameter is
	   the input frequency in MHz */
	Chip_Clock_SetupSystemPLL(multiply_by, input_freq);

	/* Turn on the PLL by clearing the power down bit */
	Chip_SYSCON_PowerUp(SYSCON_PDRUNCFG_PD_SYS_PLL);

	/* Wait for PLL to lock */
	while (!Chip_Clock_IsSystemPLLLocked()) {}

	return LPC_OK;
}

/* Set optimal system voltage based on passed system frequency */
uint32_t Chip_POWER_SetVoltage(PERF_MODE_T mode, uint32_t desired_freq)
{
	if (getBROMVersion() == 0x1100) {
		return __set_voltage_17_0(mode, desired_freq);
	}

	return __set_voltage((uint32_t)mode, desired_freq);
}

/* Enter sleep/powerdown mode and prepare TBD */
void Chip_POWER_EnterPowerMode(POWER_MODE_T mode, uint32_t peripheral_ctrl)
{
	static uint32_t relocMem[512];
	void (*fptr)(uint32_t);

	switch (mode) {
	case POWER_SLEEP:
		__set_pmu_sleep();
		break;

	case POWER_DEEP_SLEEP:
		__set_pmu_deep_sleep(peripheral_ctrl);
		break;

	case POWER_POWER_DOWN:
		/* relocate to RAM so that Flash can be put to standy mode */
		memcpy(relocMem, (void *)((uint32_t)__set_pmu_powerdown & ~3), sizeof(relocMem));
		fptr = (void (*)(uint32_t))((uint32_t)relocMem + ((uint32_t)__set_pmu_powerdown & 3U));
		fptr(peripheral_ctrl);
		break;

	case POWER_DEEP_POWER_DOWN:
		__set_pmu_deep_powerdown(peripheral_ctrl);
		break;
	}
}

/* Return ROM version */
uint32_t Chip_POWER_GetROMVersion(void)
{
	return getBROMVersion();
}
