/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*  Standard C Included Files */
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "board.h"
#include "fsl_lcdc.h"
#include "app.h"
#include "fsl_mma.h"
#include "fsl_i2c.h"
#include "fsl_debug_console.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define APP_BIT_PER_PIXEL 16
#define COLOR_INDEX_MAX_VALUE 6

#define MIN_XPOS_TILT_ANGLE 10
#define MIN_XNEG_TILT_ANGLE -10
#define MIN_YPOS_TILT_ANGLE 10
#define MIN_YNEG_TILT_ANGLE -10

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

#if (defined(__CC_ARM) || defined(__GNUC__))
__attribute__((aligned(8)))
#elif defined(__ICCARM__)
#pragma data_alignment = 8
#else
#error Toolchain not support.
#endif
static volatile uint16_t (*s_frameBuf)[IMG_HEIGHT][IMG_WIDTH] = (volatile uint16_t(*)[
    IMG_HEIGHT][IMG_WIDTH])SDRAM_BASE_ADDR; /*SDRAM starting address*/

/* The index of the inactive buffer. */
static volatile uint8_t s_inactiveBufsIdx;

/* The new frame address already loaded to the LCD controller. */
static volatile bool s_frameAddrUpdated = false;

/* Convert RGB to RGB565 format*/
#define RGB(r, g, b) (((r >> 3) & 0x001f) | (((g >> 2) & 0x003f) << 5) | (((b >> 3) & 0x001f) << 11))
/* Array of colors to select from */
const uint16_t colorsSelect[8] = {RGB(0, 0, 0),        /* Black */
                                  RGB(255, 0, 0),      /* Red */
                                  RGB(255, 255, 0),    /* Yellow */
                                  RGB(0, 255, 0),      /* Green */
                                  RGB(0, 255, 255),    /* Cyan */
                                  RGB(0, 0, 255),      /* Blue */
                                  RGB(255, 0, 255),    /* Pink */
                                  RGB(255, 255, 255)}; /* White*/

i2c_master_handle_t g_MasterHandle;
/* MMA8652 device address */
const uint8_t g_accel_address[] = {0x1CU, 0x1DU, 0x1EU, 0x1FU};

uint8_t accelGValue = 0;
uint8_t dataScale = 0;
int16_t accelXData = 0;
int16_t accelYData = 0;
int16_t accelZData = 0;
int16_t xAngle = 0;
int16_t yAngle = 0;
int16_t zAngle = 0;
bool startmov = false;

/*******************************************************************************
 * Code
 ******************************************************************************/
void APP_LCD_IRQHandler(void)
{
    uint32_t intStatus = LCDC_GetEnabledInterruptsPendingStatus(APP_LCD);

    LCDC_ClearInterruptsStatus(APP_LCD, intStatus);

    if (intStatus & kLCDC_BaseAddrUpdateInterrupt)
    {
        s_frameAddrUpdated = true;
    }
    __DSB();
}

/* Draw a line based on Start and End pixel*/
static void APP_Draw16bppRGB565Line(uint32_t *line, uint16_t start, uint16_t end, uint16_t color)
{
    uint32_t colorToSet = 0;
    uint16_t i;

    /*
              word0																					word1
    ----------------------------------------------------------------------------------------
    |		halfword(Pixel0)	|  halfword(Pixel1)			|	halfword(Pixel2)		|	halfword(Pixel3) | ...
    -----------------------------------------------------------------------------------------
    */
    /* Determine which word and half-word does pixel belong to. Each word contains 2 pixels*/
    uint16_t sword = start / 2;
    uint16_t shword = start % 2;
    uint16_t eword = end / 2;
    uint16_t ehword = end % 2;

    colorToSet = color;
    colorToSet |= color << 16;

    /* start and end pixels belong in same word*/
    if (sword == eword)
    {
        /* start and end pixels are the same*/
        if (shword == ehword)
        {
            if (shword)
            {
                line[sword] = (uint32_t)((line[sword] & 0x0000ffff) | (uint32_t)(color << 16));
            }
            else
            {
                line[sword] = (uint32_t)((line[sword] & 0xffff0000) | (uint32_t)color);
            }
        }
        else
        {
            /* start and end pixels are different*/
            line[sword] = colorToSet;
        }
    }
    /* start and end pixels are in different words*/
    /*start pixel begins at first halfword of startword and end pixel is second halfword of the endword*/
    else if ((shword == 0) && (ehword == 1))
    {
        for (i = start / 2; i <= end / 2; i++)
        {
            line[i] = colorToSet;
        }
    }
    /*start pixel begins at first halfword of startword and end pixel is first halfword of the endword*/
    else if ((shword == 0) && (ehword == 0))
    {
        for (i = start / 2; i < end / 2; i++)
        {
            line[i] = colorToSet;
        }
        line[i] = (uint32_t)((line[i] & 0xffff0000) | (uint32_t)color);
    }
    /*start pixel is second halfword of startword and end pixel is second halfword of the endword*/
    else if ((shword == 1) && (ehword == 1))
    {
        line[start / 2] = (uint32_t)((line[start / 2] & 0x0000ffff) | (uint32_t)(color << 16));
        for (i = (start / 2) + 1; i <= end / 2; i++)
        {
            line[i] = colorToSet;
        }
    }
    /*start pixel is second halfword of startword and end pixel is first halfword of the endword*/
    else if ((shword == 1) && (ehword == 0))
    {
        line[start / 2] = (line[start / 2] & 0x0000ffff) | (uint32_t)(color << 16);
        for (i = (start / 2) + 1; i < end / 2; i++)
        {
            line[i] = colorToSet;
        }
        line[i] = (uint32_t)((line[i] & 0xffff0000) | (uint32_t)color);
    }
}

static void APP_FillBuffer(void *buffer)
{
    /* Background color of lcd display panel */
    uint8_t bgColorIdx = 0U; /* background color index. Background color of LCD is Black */
    static uint16_t lcdPanelXOrigin = 0U;
    static uint16_t lcdPanelYOrigin = 0U;

    /* Position of the foreground rectangle. */
    static uint16_t upperLeftX = 0U;
    static uint16_t upperLeftY = 0U;
    static uint16_t lowerRightX = (IMG_WIDTH - 1U) / 2U;
    static uint16_t lowerRightY = (IMG_HEIGHT - 1U) / 2U;
    static uint16_t tempY = 0;
    /* Foreground color. */
    static uint8_t fgColorIdx = 1U; /* foreground color index. Color of object */

    /* Increase X and Y positions*/
    static int16_t incX = 5;
    static int16_t incY = 5;

    /* Change color in next frame or not. */
    static bool changeColor = false;
    static bool xDirChange = false;
    static bool yDirChange = false;
    uint32_t i, j;
    uint32_t(*pSdramFbPtr)[IMG_WIDTH / 2] = (uint32_t(*)[IMG_WIDTH / 2])buffer;

    /*
       +------------------------------------------------------------------------+
       |                                                                        |
       |                                                                        |
       |                                                                        |
       |                          Area 1                                        |
       |                                                                        |
       |                                                                        |
       |                  +---------------------------+                         |
       |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
       |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
       |    Area 2        |XXXXXXXXXXXXXXXXXXXXXXXXXXX|       Area 3            |
       |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
       |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
       |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
       |                  +---------------------------+                         |
       |                                                                        |
       |                                                                        |
       |                                                                        |
       |                                                                        |
       |                         Area 4                                         |
       |                                                                        |
       |                                                                        |
       +------------------------------------------------------------------------+
       */

    /* Fill the frame buffer. */
    /* Fill area 1. */
    i = lcdPanelYOrigin;

    if (i < upperLeftY)
    {
        APP_Draw16bppRGB565Line(pSdramFbPtr[i], lcdPanelXOrigin, IMG_WIDTH, colorsSelect[bgColorIdx]);
    }
    /* Go row (line) wise*/
    for (; i < upperLeftY; i++)
    {
        /*Go column wise(scanning pixel in every line)*/
        for (j = lcdPanelXOrigin; j < (IMG_WIDTH) / 2;
             j++) /* Write to SDRAM is a word write. So 2 pixels written at a time*/
        {
            /* Fill entire selected region with background color */
            pSdramFbPtr[i][j] = pSdramFbPtr[lcdPanelYOrigin][j];
        }
    }

    /* Fill Area 2, Rectangle object and Area 3*/
    APP_Draw16bppRGB565Line(pSdramFbPtr[i], lcdPanelXOrigin, upperLeftX, colorsSelect[bgColorIdx]);
    APP_Draw16bppRGB565Line(pSdramFbPtr[i], upperLeftX + 1, lowerRightX, colorsSelect[fgColorIdx]);
    APP_Draw16bppRGB565Line(pSdramFbPtr[i], lowerRightX + 1, IMG_WIDTH, colorsSelect[bgColorIdx]);

    for (i++; i <= lowerRightY; i++)
    {
        for (j = lcdPanelXOrigin; j < IMG_WIDTH / 2; j++)
        {
            pSdramFbPtr[i][j] = pSdramFbPtr[upperLeftY][j];
        }
    }

    /*Fill area 4*/
    tempY = i;

    if (i < IMG_HEIGHT)
    {
        APP_Draw16bppRGB565Line(pSdramFbPtr[i], lcdPanelXOrigin, IMG_WIDTH, colorsSelect[bgColorIdx]);
    }
    for (; i < IMG_HEIGHT; i++)
    {
        /*Go column wise(scanning pixel in every line)*/
        for (j = lcdPanelXOrigin; j < IMG_WIDTH / 2; j++)
        {
            /* Fill entire selected region with background color */
            pSdramFbPtr[i][j] = pSdramFbPtr[tempY][j];
        }
    }

    /* Update X position of object on LCD based on Accelerometer data.
     * X coordinates should increment when angle is positive.
     * Here if angle is greater than MIN_XPOS_TILT_ANGLE the object position is updated.
     */
    if ((xAngle > MIN_XPOS_TILT_ANGLE) && (lowerRightX < (IMG_WIDTH)-1))
    {
        /* Update rectangle X position. */
        upperLeftX += incX;
        lowerRightX += incX;

        xDirChange = true; /*x axis direction has changed*/
    }

    /* X coordinates should decrement when angle is negative. Here if angle is less than MIN_XNEG_TILT_ANGLE the object
     * position is updated*/
    if ((xAngle < MIN_XNEG_TILT_ANGLE) && (upperLeftX > 0))
    {
        /* Update rectangle X position. */
        upperLeftX -= incX;
        lowerRightX -= incX;

        xDirChange = true; /*x axis direction has changed*/
    }

    /* Update Y position of object on LCD based on Accelerometer data.
     * Y coordinates should increment when angle is positive.
     * Here if angle is greater than MIN_YPOS_TILT_ANGLE the object position is updated
     */
    if ((yAngle > MIN_YPOS_TILT_ANGLE) && (lowerRightY < IMG_HEIGHT - 1))
    {
        /* Update rectangle Y position. */
        upperLeftY += incY;
        lowerRightY += incY;

        yDirChange = true; /*y axis direction has changed*/
    }

    /* Y coordinates should decrement when angle is negative. Here if angle is less than MIN_YNEG_TILT_ANGLE the object
     * position is updated*/
    if ((yAngle < MIN_YNEG_TILT_ANGLE) && (upperLeftY > 0))
    {
        /* Update rectangle Y position. */
        upperLeftY -= incY;
        lowerRightY -= incY;

        yDirChange = true; /*y axis direction has changed*/
    }

    changeColor = false;

    /* If the object touches the boundaries , change the color of the object*/
    if (lowerRightX >= IMG_WIDTH - 1)
    {
        if (xDirChange)
        {
            changeColor = true;
            xDirChange = false;
        }
    }

    if (upperLeftX == 0)
    {
        if (xDirChange)
        {
            changeColor = true;
            xDirChange = false;
        }
    }

    if (lowerRightY >= IMG_HEIGHT - 1)
    {
        if (yDirChange)
        {
            changeColor = true;
            yDirChange = false;
        }
    }

    if (upperLeftY == 0)
    {
        if (yDirChange)
        {
            changeColor = true;
            yDirChange = false;
        }
    }

    if (changeColor)
    {
        if (COLOR_INDEX_MAX_VALUE == fgColorIdx)
        {
            fgColorIdx = 1U;
        }
        else
        {
            fgColorIdx++;
        }
    }
}

int main(void)
{
    lcdc_config_t lcdConfig;
    mma_handle_t mmaHandle = {0};
    mma_data_t accelSensorData = {0};
    i2c_master_config_t masterConfig;
    uint8_t numberOfAccelSlvAddr = 0;
    uint8_t i = 0;
    uint8_t readRegResult = 0;
    bool foundDevice = false;

    /* Initialize Board pins, clock and debug console unit*/
    BOARD_InitHardware();

    /*Initialize SDRAM*/
    BOARD_InitSDRAM();

    /*Initialize and setup LCD configuration*/
    s_frameAddrUpdated = false;
    s_inactiveBufsIdx = 1;

    /* Fill frame buffer before setting UP/LPBASE addr and enabling interrupts*/
    APP_FillBuffer((void *)(s_frameBuf[0]));

    LCDC_GetDefaultConfig(&lcdConfig);

    lcdConfig.panelClock_Hz = LCD_PANEL_CLK;            /* LCD Panel Clock*/
    lcdConfig.ppl = LCD_PPL;                            /* Pixels per line */
    lcdConfig.hsw = LCD_HSW;                            /* HSYNC signal pulse width*/
    lcdConfig.hfp = LCD_HFP;                            /* HSYNC front porch value*/
    lcdConfig.hbp = LCD_HBP;                            /* HSYNC back porch value*/
    lcdConfig.lpp = LCD_LPP;                            /* lines per panel*/
    lcdConfig.vsw = LCD_VSW;                            /* VSYNC signal pulse width*/
    lcdConfig.vfp = LCD_VFP;                            /* VSYNC front porch value*/
    lcdConfig.vbp = LCD_VBP;                            /* VSYNC back porch value*/
    lcdConfig.polarityFlags = LCD_POL_FLAGS;            /* Set clock and signal polarity flags */
    lcdConfig.bpp = kLCDC_16BPP565;                     /* bits per pixel*/
    lcdConfig.display = kLCDC_DisplayTFT;               /* display type is TFT display*/
    lcdConfig.swapRedBlue = false;                      /* Do not swap RGB to BGR*/
    lcdConfig.upperPanelAddr = (uint32_t)s_frameBuf[0]; /* Upper panel base address - 64-bit aligned*/

    /* Initialize LCD clock and registers*/
    LCDC_Init(APP_LCD, &lcdConfig, LCD_INPUT_CLK_FREQ);

    /*Enable LCD Interrupts*/
    LCDC_EnableInterrupts(APP_LCD, kLCDC_BaseAddrUpdateInterrupt);
    NVIC_EnableIRQ(APP_LCD_IRQn);

    /* Start Output LCD timing signals*/
    LCDC_Start(APP_LCD);

    /*Enable Power to LCD Panel*/
    LCDC_PowerUp(APP_LCD);

    /*Initialize and setup Accelerometer*/
    mmaHandle.base = BOARD_ACCEL_I2C_BASEADDR;
    mmaHandle.i2cHandle = &g_MasterHandle;

    /*
       * masterConfig->enableMaster = true;
     * masterConfig->baudRate_Bps = 100000U;
       * masterConfig->enableTimeout = false;
       */
    I2C_MasterGetDefaultConfig(&masterConfig);
    I2C_MasterInit(BOARD_ACCEL_I2C_BASEADDR, &masterConfig, I2C_MASTER_CLOCK_FREQUENCY);
    I2C_MasterTransferCreateHandle(BOARD_ACCEL_I2C_BASEADDR, &g_MasterHandle, NULL, NULL);

    numberOfAccelSlvAddr = sizeof(g_accel_address) / sizeof(g_accel_address[0]);

    /* Find the accelerometer device*/
    for (i = 0; i < numberOfAccelSlvAddr; i++)
    {
        mmaHandle.xfer.slaveAddress = g_accel_address[i];

        if (MMA_ReadReg(&mmaHandle, kMMA8652_WHO_AM_I, &readRegResult) == kStatus_Success)
        {
            foundDevice = true;
            break;
        }

        if ((!foundDevice) && (i == numberOfAccelSlvAddr - 1))
        {
            PRINTF("\n\r Accelerometer Device not found ! ");
            while (1)
                ;
        }
    }

    /*Initialize the accelerometer device:- Dynamic range : -4g to +4g; FIFO : disabled; F_READ : Normal Read mode*/
    if (MMA_Init(&mmaHandle) != kStatus_Success)
    {
        return kStatus_Fail;
    }

    /* Get Accelerometer dynamic range */
    if (MMA_ReadReg(&mmaHandle, kMMA8652_XYZ_DATA_CFG, &accelGValue) != kStatus_Success)
    {
        return kStatus_Fail;
    }
    /*Check FS bits[1:0] for dynamic range value : 2g = 0x00; 4g = 0x01; 8g = 0x02 */
    if ((accelGValue & 0x03) == 0x00)
    {
        dataScale = 2;
    }
    else if ((accelGValue & 0x03) == 0x01)
    {
        dataScale = 4;
    }
    else if ((accelGValue & 0x03) == 0x02)
    {
        dataScale = 8;
    }

    PRINTF("\n\r Accelerometer initialized");

    while (1)
    {
        /* Read Accelerometer Sensor Data*/
        if (MMA_ReadSensorData(&mmaHandle, &accelSensorData) != kStatus_Success)
        {
            return kStatus_Fail;
        }

        /* Save each sensor data as a 16 bit result.
        * The 16 bit result is in left-justified format.
        * Shift 4 bits to the right(12-bit resolution) to get the actual value.
        */
        accelXData =
            (int16_t)(((uint16_t)(accelSensorData.accelXMSB << 8)) | ((uint16_t)(accelSensorData.accelXLSB))) / 16U;
        accelYData =
            (int16_t)(((uint16_t)(accelSensorData.accelYMSB << 8)) | ((uint16_t)(accelSensorData.accelYLSB))) / 16U;
        accelZData =
            (int16_t)(((uint16_t)(accelSensorData.accelZMSB << 8)) | ((uint16_t)(accelSensorData.accelZLSB))) / 16U;

        /* Convert raw accelerometer sensor data to angle (normalize to 0-90 degrees). No negative angles. */
        xAngle = (int16_t)floor((double)accelXData * (double)dataScale * 180 / 4096);
        yAngle = (int16_t)floor((double)accelYData * (double)dataScale * 180 / 4096);

        /* Fill the inactive buffer. */
        APP_FillBuffer((void *)s_frameBuf[s_inactiveBufsIdx]);

        /*
         * The buffer address has been loaded to the LCD controller, now
         * set the inactive buffer to active buffer.
         */
        LCDC_SetPanelAddr(APP_LCD, kLCDC_UpperPanel, (uint32_t)(s_frameBuf[s_inactiveBufsIdx]));

        while (!s_frameAddrUpdated)
        {
        }

        s_frameAddrUpdated = false;
        s_inactiveBufsIdx ^= 1U;
    }
}
