// ------------------------------------------------------------------
// Commissioning with Installation Code (Out of Band)
// ------------------------------------------------------------------
// Author:    nlv10677
// Copyright: NXP B.V. 2015. All rights reserved
// ------------------------------------------------------------------

/** \addtogroup nd
* \file
* \brief NDEF Commissioning 'app'
* Based on the incoming NDEF record, and the selected NfcMode, this 'app'
* completes the outgoing NDEF record to be written back to the device.
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#include "atoi.h"
#include "nfcreader.h"
#include "newDb.h"
#include "systemtable.h"
#include "parsing.h"
#include "json.h"
#include "socket.h"
#include "nibbles.h"
#include "jsonCreate.h"
#include "dump.h"
#include "ndef.h"
#include "newLog.h"
#include "audiofb.h"
#include "commission_oob.h"

#define SJ_SOCKET_PORT      "2000"
#define CI_SOCKET_PORT      "2001"

#define INPUTBUFFERLEN   200

//#define COMM_DEBUG
//#define NDEF_DEBUG

#ifdef COMM_DEBUG
#define DEBUG_PRINTF(...) printf(__VA_ARGS__)
#else
#define DEBUG_PRINTF(...)
#endif /* COMM_DEBUG */

// ------------------------------------------------------------------------
// Globals
// ------------------------------------------------------------------------

static char inputBuffer[INPUTBUFFERLEN + 2];

static int success = 0;
static int ready   = 0;

static void * glob_join     = NULL;

static int nfcMode = NFC_MODE_COMMISSION;

static uint8_t handled_seqnr = -1;
static uint8_t out_seqnr = 0;

// ------------------------------------------------------------------
// Attributes
// ------------------------------------------------------------------
    
#define NUMINTATTRS  4
static char * intAttrs[NUMINTATTRS] = { "keyseq", "chan", "tcshaddr", "devid" };

#define NUMSTRINGATTRS  5
static char * stringAttrs[NUMSTRINGATTRS] = {
    "key", "mic", "tcaddr", "pan", "extpan" };
static int    stringMaxlens[NUMSTRINGATTRS] = {
    32,      8,      16,       4,     16 };

// ------------------------------------------------------------------------
// Dump
// ------------------------------------------------------------------------

#ifdef COMM_DEBUG
static void commissionDump ( void ) {
    printf( "\nCommission dump:\n" );
    int i;
    for ( i=0; i<NUMINTATTRS; i++ ) {
        printf( "- %-12s = %d\n",
                intAttrs[i], parsingGetIntAttr( intAttrs[i] ) );
    }
    for ( i=0; i<NUMSTRINGATTRS; i++ ) {
        printf( "- %-12s = %s\n",
                stringAttrs[i], parsingGetStringAttr0( stringAttrs[i] ) );
    }
}
#endif

// -------------------------------------------------------------
// Parsing
// -------------------------------------------------------------

static void com_onError(int error, char * errtext, char * lastchars) {
    printf("onError( %d, %s ) @ %s\n", error, errtext, lastchars);
}

static void com_onObjectStart(char * name) {
    // printf("onObjectStart( %s )\n", name);
    parsingReset();
}

static void com_onObjectComplete(char * name) {
    // printf("onObjectComplete( %s )\n", name);

    if ( strcmp( name, "oobresponse" ) == 0 ) {
#ifdef COMM_DEBUG
        commissionDump();
#endif

    oob_response_info_t * oobresponse = (oob_response_info_t *)glob_join;

    memset( (char *)oobresponse, 0, sizeof( oob_response_info_t ) );

    oobresponse->channel = parsingGetIntAttr( "chan" );
    oobresponse->keyseq  = parsingGetIntAttr( "keyseq" );
    oobresponse->panid   = AtoiHex( parsingGetStringAttr( "pan" ) );
    nibblestr2hex( oobresponse->extpanid, LEN_EXTPAN,
        parsingGetStringAttr0( "extpan" ), LEN_EXTPAN * 2 );
    nibblestr2hex( oobresponse->key, LEN_ICODE,
        parsingGetStringAttr0( "key" ), LEN_ICODE * 2 );
    nibblestr2hex( oobresponse->mic, LEN_MIC,
        parsingGetStringAttr0( "mic" ), LEN_MIC * 2 );
    nibblestr2hex( oobresponse->hostextaddress, LEN_TCADDR,
        parsingGetStringAttr0( "tcaddr" ), LEN_TCADDR * 2 );
    oobresponse->shortaddress  = parsingGetIntAttr( "tcshaddr" );
    oobresponse->deviceid  = parsingGetIntAttr( "devid" );
    //oobresponse->deviceextaddress = ?

    ready = 1;
    }
}

static void com_onArrayStart(char * name) {
    // printf("onArrayStart( %s )\n", name);
}

static void com_onArrayComplete(char * name) {
    // printf("onArrayComplete( %s )\n", name);
}

static void com_onString(char * name, char * value) {
    // printf("onString( %s, %s )\n", name, value);
    parsingStringAttr( name, value );
}

static void com_onInteger(char * name, int value) {
    // printf("onInteger( %s, %d )\n", name, value);
    parsingIntAttr( name, value );
}

// ------------------------------------------------------------------
// Functions
// ------------------------------------------------------------------

/**
* \brief Contacts SecureJoiner to get the SecJoin struct.
* \param linkinfo struct from the device
* \param join struct with (encrypted) network data for device to join
* \retval 1 when OK
* \retval 0 on error
*/

static int commission_icode( oob_request_info_t * oob_request, oob_response_info_t * oob_response ) {

    DEBUG_PRINTF( "Commissioning (ICode) Started\n" );

    success = 1;

    // Check input parameters
    if ( oob_request == NULL || oob_response == NULL ) {
        return( 0 );
    }

    // Make globals of the two output parameters
    glob_join     = oob_response;

    // Prepare creating of JSON oob_request message
    char mac_str[LEN_MAC_NIBBLE + 2];
    char key_str[(LEN_ICODE * 2 ) + 2];
    hex2nibblestr( mac_str, oob_request->mac, LEN_MAC );
    hex2nibblestr( key_str, oob_request->key, LEN_ICODE );

#ifdef COMM_DEBUG
    printf( "Oob Data Request:\n" );
    dump( (char *)oob_request, sizeof( oob_request_info_t ) );
    // printf( "MAC:\n" ); dump( oob_request->mac, LEN_MAC_NIBBLE );
    printf( "MAC = %s\n", mac_str );
    printf( "KeyStr = %s\n", key_str );
#endif

    int len;
    int handle = socketOpen( "localhost", SJ_SOCKET_PORT, 0 );

    if ( handle >= 0 ) {
        DEBUG_PRINTF( "SecJoin socket opened\n" );

        if ( success ) {
            if ( socketWriteString( handle,
                                    jsonOobCommissioningRequest(oob_request->version,
                                                       oob_request->type,
                                                       oob_request->profile,
                                                       mac_str,
                                                       key_str ) ) > 0 ) {
                DEBUG_PRINTF( "Sent oob_request (Icode)\n" );

                jsonReset();

                ready = 0;

                while ( !ready &&
                        ( len = socketReadWithTimeout( handle,
                                                    inputBuffer,
                                                    200, 2 ) ) > 0 ) {
//#ifdef COMM_DEBUG
//                    printf( "Received from socket %d bytes\n", len );
//                    dump( inputBuffer, len );
//#endif

                    int i;
                    for ( i=0; i<len; i++ ) {
                        // printf( "%c", inputBuffer[i] );
                        jsonEat( inputBuffer[i] );
                    }
                }

                if ( ready ) {
                    DEBUG_PRINTF( "Commissioning (Icode) ready\n" );
                } else {
                    DEBUG_PRINTF( "Commissioning (Icode) NOT ready\n" );
                    success = 0;
                }
            } else {
                newLogAdd( NEWLOG_FROM_NFC_DAEMON, "Error writing oob_request to socket" );
                success = 0;
            }
        }

        socketClose( handle );

    } else {
        sprintf( logbuffer, "Error opening port to %s", SJ_SOCKET_PORT );
        newLogAdd( NEWLOG_FROM_NFC_DAEMON, logbuffer );
    }

    DEBUG_PRINTF( "Commission (Icode) exit\n");
    return( success );
}

// ------------------------------------------------------------------------
// Helpers
// ------------------------------------------------------------------------

static int getDeviceDev( int from ) {
    switch ( from ) {
        default:
//        case FROM_SETTINGSAPP:
//        case FROM_GATEWAY:
            return( DEVICE_DEV_UNKNOWN );
/*        case FROM_ROUTER:
            return( DEVICE_DEV_SWITCH );
        case FROM_MANAGER:
            return( DEVICE_DEV_MANAGER );
        case FROM_UISENSOR:
            return( DEVICE_DEV_UISENSOR );
        case FROM_SENSOR:
            return( DEVICE_DEV_SENSOR );
        case FROM_ACTUATOR:
            return( DEVICE_DEV_PUMP );
        case FROM_UI_ONLY:
            return( DEVICE_DEV_UI );
        case FROM_PLUGMETER:
            return( DEVICE_DEV_PLUG );*/
        case FROM_LIGHT_ONOFF:
        case FROM_LIGHT_DIMM:
        case FROM_LIGHT_COLOR_DIMM:
        case FROM_LIGHT_COLOR_TEMP:
        case FROM_LIGHT_COLOR_EXT:
            return( DEVICE_DEV_LAMP );
    }
}

static char * getDeviceType( int from ) {
    switch ( from ) {
        default:
            return( DEVICE_TYPE_UNKNOWN );
        case FROM_MANAGER:
            return( DEVICE_TYPE_MAN_HEATING );
        case FROM_LIGHT_ONOFF:
            return( DEVICE_TYPE_LAMP_ONOFF );
        case FROM_LIGHT_DIMM:
            return( DEVICE_TYPE_LAMP_DIMM  );
        case FROM_LIGHT_COLOR_TEMP:
            return( DEVICE_TYPE_LAMP_TW    );
        case FROM_LIGHT_COLOR_DIMM:
            return( DEVICE_TYPE_LAMP_COLOR );
        case FROM_ROUTER:
            return( DEVICE_TYPE_SWITCH_ONOFF ); /** \todo Temporary use from=router for switches */
    }
}

// ------------------------------------------------------------------------
// DB edit via Control Interface
// ------------------------------------------------------------------------

/**
* \brief Sends dbEdit command to Control Interface
* \param mode 1=add, 0=remove
*/

static void dbEdit( int mode, char * mac, int dev, char * ty ) {

    char mac_str[LEN_MAC_NIBBLE + 2];
    hex2nibblestr( mac_str, mac, LEN_MAC );

    DEBUG_PRINTF( "Mode = %d, mac = %s, dev = %d, ty = %s\n",
            mode, mac_str, dev, ty );

    char * message;
    if ( mode ) {
        message = jsonDbEditAdd( mac_str, dev, ty );
    } else {
        message = jsonDbEditRem( mac_str );
    }
    DEBUG_PRINTF( "Message = %s\n", message );

    int handle = socketOpen( "localhost", CI_SOCKET_PORT, 0 );

    if ( handle >= 0 ) {
        DEBUG_PRINTF( "Control socket opened\n" );

        if ( socketWriteString( handle, message ) > 0 ) {
            DEBUG_PRINTF( "DB-edit sent\n" );
        }

        socketClose( handle );
    } else { 
        sprintf( logbuffer, "Error opening port to %s", CI_SOCKET_PORT );
        newLogAdd( NEWLOG_FROM_NFC_DAEMON, logbuffer );
    }
}

static uint16_t getInstallCodeCrc(uint8_t* pu8InstallCode, uint16_t u16Length)
{
    uint16_t i;
    uint16_t u16Data;
    uint16_t  u16Crc;

    u16Crc = 0xFFFF;

    #define CRC_POLYNOMIAL      0x8408

    if (u16Length > 0) {
        do {
            for (i = 0, u16Data = (uint16_t)0xff & *pu8InstallCode++;
                 i < 8;
                 i++, u16Data >>= 1) {
                if ((u16Crc & 0x0001) ^ (u16Data & 0x0001)) {
                    u16Crc = (u16Crc >> 1) ^ CRC_POLYNOMIAL;
                } else {
                    u16Crc >>= 1;
                }
            }
        } while (--u16Length);
    }
    u16Crc = ~u16Crc;

    return (u16Crc);
}

// ------------------------------------------------------------------
// Handle NFC commissioning with Installation Code (Out of Band)
// ------------------------------------------------------------------

/**
* \brief Based on the incoming NDEF record, and the selected NfcMode,
* completes the outgoing NDEF record to be written back to the device.
* \param handle  Handle to NDEF tag
* \param payload Pointer to NDEF payload array of bytes
* \param plen    Length of the payload array
* \retval 1 when OK
* \retval 0 on error
*/

int commissionHandleNfcOob( int handle, char * payload, int plen ) {
    static int prevNfcMode = -1;
    int ret = 0;

#ifdef COMM_DEBUG
    printf( "Commission handle NFC with ICODE:\n" );
    // dump( payload, plen );
#endif

    commission_icode_t * c = (commission_icode_t *)payload;
    /* change 'c->ntag.DeviceId' endianness from big endian to little endian */
    c->ntag.DeviceId = (( c->ntag.DeviceId >> 8 ) & 0xFF) | (( c->ntag.DeviceId & 0xFF ) << 8);

    if ( c->version == COMMISSION_SUPPORTED_VERSION_ICODE ) {

        nfcMode = newDbSystemGetInt( "nfcmode" );
        DEBUG_PRINTF( "nfcMode = %d\n", nfcMode );

        if ( ( c->ntag.Sequence == handled_seqnr ) && ( prevNfcMode == nfcMode ) ) {
            printf( "Message already handled\n" );
            ret = 1;
        } else {

#ifdef NDEF_DEBUG
            printf("*** c->ntag ***\n");
            printf("c->ntag.Command = %d\n", c->ntag.Command);
            printf("c->ntag.Sequence = %d\n", c->ntag.Sequence);
            printf("c->ntag.DeviceId = ");
            dump( (char*)&c->ntag.DeviceId, sizeof(c->ntag.DeviceId));
            printf("c->ntag.ExtAddress = ");
            dump( (char*)&c->ntag.ExtAddress, LEN_MAC);
            printf("c->ntag.ShortAddress = ");
            dump( (char*)&c->ntag.ShortAddress, sizeof(c->ntag.ShortAddress));
            printf("c->ntag.Channel = %d\n", c->ntag.Channel);
            printf("c->ntag.PanId = ");
            dump( (char*)&c->ntag.PanId, sizeof(c->ntag.PanId));
            printf("c->ntag.ExtPanId = ");
            dump( (char*)&c->ntag.ExtPanId, sizeof(c->ntag.ExtPanId));
            printf("c->ntag.Key = ");
            dump( (char*)&c->ntag.Key, LEN_ICODE);
            printf("c->ntag.Crc = ");
            dump( (char*)&c->ntag.Crc, sizeof(c->ntag.Crc));
#endif
            prevNfcMode = nfcMode;

            if ( nfcMode == NFC_MODE_DECOMMISSION ) {
                DEBUG_PRINTF("Decommission...\n");
                newLogAdd( NEWLOG_FROM_NFC_DAEMON, "Decommission mode" );

                // use NFC_NCI_CMD_FACTORY_RESET. The device should handle a Leave.
                if ( c->nci.Command != NFC_NCI_CMD_FACTORY_RESET ) {
                    memset( (char *)&c->nci, 0, sizeof( commission_nci_t ) );
                    c->nci.DeviceId = FROM_GATEWAY; // FIXME: change From field to DeviceId (0x0000 for ZCB?)
                    c->nci.Sequence   = out_seqnr++;
                    c->nci.Command = NFC_NCI_CMD_FACTORY_RESET;

                    if ( ndef_write( handle, payload, plen, COMMISSION_URI_ICODE, strlen( COMMISSION_URI_ICODE ) ) ) {
                        // Remove device from GW dB (via Control Interface)
                        dbEdit( 0, (char *)c->ntag.ExtAddress, 0, NULL );
                        ret = 1;
                    }
                } else {
                    printf( "Tag already contains FACTORY_RESET command\n" );
                    ret = 1;
                }

                if ( ret ) afbPlayPattern( AFB_PATTERN_TWOMEDIUM );

            } else if ( nfcMode == NFC_MODE_COMMISSION ) {
                DEBUG_PRINTF("Commission...\n");
                newLogAdd( NEWLOG_FROM_NFC_DAEMON, "Commission mode" );

                if ( c->ntag.Command != NFC_NTAG_CMD_JOIN_WITH_ICODE ) {
                    DEBUG_PRINTF( "Ntag command does not match commissioning = 0x%02x\n", c->ntag.Command );
                    ret = 0;
                } else {
                    // check if ICODE & CRC are matching
                    uint16_t currentCrc = (c->ntag.Crc[0] << 8) | c->ntag.Crc[1];
                    uint16_t checkCrc = getInstallCodeCrc(&c->ntag.Key, LEN_ICODE);
                    DEBUG_PRINTF("CRC given = x%04x\n", currentCrc);
                    DEBUG_PRINTF("CRC calculated = x%04x\n", checkCrc);

                    if (checkCrc == currentCrc) {
                        DEBUG_PRINTF("CRCs are matching\n");

                        oob_request_info_t oob_request_info;
                        oob_response_info_t oob_response_info;

                        oob_request_info.version = 0;
                        oob_request_info.type    = 0;
                        oob_request_info.profile = 0;
                        memcpy( oob_request_info.mac, c->ntag.ExtAddress, LEN_MAC );
                        memcpy( oob_request_info.key, c->ntag.Key, LEN_ICODE );

                        ret = commission_icode( &oob_request_info, &oob_response_info );
                        if ( ret ) {
        #ifdef NDEF_DEBUG
                            printf( "oob_response_info :\n" );
                            dump( (char *)&oob_response_info, sizeof( oob_response_info_t ) );
                            printf(" - deviceextaddress = ");
                            dump( (char*)oob_response_info.deviceextaddress, LEN_MAC );
                            printf(" - nwkey    = ");
                            dump( (char*)oob_response_info.key, LEN_ICODE );
                            printf(" - mic      = ");
                            dump( (char*)oob_response_info.mic, LEN_MIC );
                            printf(" - hostextaddress = ");
                            dump( (char*)oob_response_info.hostextaddress, LEN_MAC );
                            printf(" - keyseq   = 0x%02x\n", oob_response_info.keyseq );
                            printf(" - channel  = 0x%02x\n", oob_response_info.channel );
                            printf(" - panid    = 0x%04x\n", oob_response_info.panid );
                            printf(" - extpanid = ");
                            dump( (char*)oob_response_info.extpanid, LEN_EXTPAN );
                            printf(" - shortadd = 0x%04x\n", oob_response_info.shortaddress );
                            printf(" - deviceid = 0x%04x\n", oob_response_info.deviceid );
                            printf(" - status   = 0x%02x\n", oob_response_info.status );
        #endif

                            memset( (char *)&c->nci, 0, sizeof( commission_nci_t ) );
                            c->nci.DeviceId = oob_response_info.deviceid;
                            c->nci.Sequence = out_seqnr++;
                            c->nci.Command = NFC_NCI_CMD_JOIN_WITH_ICODE;
                            c->nci.Channel = oob_response_info.channel;
                            c->nci.KeySeqNum = oob_response_info.keyseq;
                            c->nci.PanId[0] = ( oob_response_info.panid >> 8 ) & 0xFF; // match endianness
                            c->nci.PanId[1] = oob_response_info.panid & 0xFF;
                            memcpy( (char *)c->nci.Mic, oob_response_info.mic, LEN_MIC );  // bytes[12..15] TODO
                            memcpy( (char *)c->nci.Key, oob_response_info.key, LEN_ICODE );
                            memcpy( (char *)c->nci.ExtPanId, oob_response_info.extpanid, LEN_EXTPAN );
                            memcpy( (char *)c->nci.ExtAddress, oob_response_info.hostextaddress, LEN_MAC );
                            c->nci.ShortAddress[0] = ( oob_response_info.shortaddress >> 8 ) & 0xFF; // match endianness
                            c->nci.ShortAddress[1] = oob_response_info.shortaddress & 0xFF;

                            if ( ndef_write( handle, payload, plen, COMMISSION_URI_ICODE, strlen( COMMISSION_URI_ICODE ) ) ) {
                                // Add device to GW dB (via Control Interface)
                                int    dev = getDeviceDev( c->ntag.DeviceId );
                                char * ty  = getDeviceType( c->ntag.DeviceId );

                                DEBUG_PRINTF( "From = %d, dev = %d, ty = %s\n", c->ntag.DeviceId, dev, ty );
                                dbEdit( 1, oob_request_info.mac, dev, ty );
                                ret = 1;
                            } else {
                                ret = 0;
                            }
                        }
                    } else {
                        DEBUG_PRINTF("CRCs are NOT matching\n");
                        ret = 0;
                    }
                }

                if ( ret ) afbPlayPattern( AFB_PATTERN_ONELONG );
            } else {
                DEBUG_PRINTF("No commission or decommission...\n");
            }

            handled_seqnr = c->ntag.Sequence;
        }
    } else {
        printf( "Commission version not supported (%d != %d)\n",
                c->version, COMMISSION_SUPPORTED_VERSION_ICODE );
    }
    printf( "Commission handle returns with %d\n", ret );
    return( ret );
}


// ------------------------------------------------------------------
// Init / Reset
// ------------------------------------------------------------------

/**
* \brief Needs to be called on tag-departure to avoid unnecessary tag writes
*/
void commissionResetOob( void ) {
    handled_seqnr = -1;
}

/**
* \brief Initialize parser for Secure-Joiner response
*/
void commissionInitOob( void ) {
    int i;

    jsonSetOnError(com_onError);
    jsonSetOnObjectStart(com_onObjectStart);
    jsonSetOnObjectComplete(com_onObjectComplete);
    jsonSetOnArrayStart(com_onArrayStart);
    jsonSetOnArrayComplete(com_onArrayComplete);
    jsonSetOnString(com_onString);
    jsonSetOnInteger(com_onInteger);
    jsonReset();

    for ( i=0; i<NUMINTATTRS; i++ ) {
        parsingAddIntAttr( intAttrs[i] );
    }
    for ( i=0; i<NUMSTRINGATTRS; i++ ) {
        parsingAddStringAttr( stringAttrs[i], stringMaxlens[i] );
    }
}

