/*****************************************************************************
 *
 * MODULE:              PER Test
 *
 ****************************************************************************
 *
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5179].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2015. All rights reserved
 ****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/

#include <jendefs.h>
#include <string.h>

#include <AppHardwareApi.h>
#include <AppQueueApi.h>
#include <mac_sap.h>
#include <mac_pib.h>
#include <JPT.h>
#include <tsv_pub.h>
#include <BbcAndPhyRegs.h>

#ifdef PER_DEBUG
#include <Printf.h>
#else
#define vPrintf(...)
#endif

#include "PerTest.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

/* Addresses used in the RF comms */
#define PER_PAN_ID                             0x1626
#define PER_FFD_SHORT_ADDR                     0x3927
#define PER_RFD_SHORT_ADDR                     0x3926

/* system states */
#define E_PER_STATE_STOPPED                    1
#define E_PER_STATE_RUNNING_ACKS               2
#define E_PER_STATE_RUNNING_NO_ACKS            3

/* Command retries */
#define PER_CMD_RETRIES                        100
#define PER_LOCATE_RETRIES                     50

/* Packet lengths (must be 4 or more) */
#define PER_PAYLOAD_LENGTH                     4

/* Initial channels */
#ifndef DEFAULT_CHANNEL
#define DEFAULT_CHANNEL                        11
#endif

#define PER_MASTER_INIT_CHANNEL                DEFAULT_CHANNEL
#define PER_SLAVE_INIT_CHANNEL                 DEFAULT_CHANNEL

#define TSV_ONE_SECOND                         (62500)
#define TICK_PERIOD                            (TSV_ONE_SECOND / 100)

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/* Commands sent from coordinator to endpoint */
typedef enum {
	E_PER_CMD_MODE_ACK    = 0x01,
	E_PER_CMD_MODE_NO_ACK = 0x02,
	E_PER_CMD_MODE_STOP   = 0x03,
	E_PER_CMD_RESTART     = 0x04,

	E_PER_CMD_CFG         = 0x05,

	E_PER_CMD_PER_PACKET  = 0x55,
} tePER_Command;

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

PRIVATE MAC_DcfmIndHdr_s *psMlmeDcfmIndGetBuf(void *pvParam);
PRIVATE MAC_DcfmIndHdr_s *psMcpsDcfmIndGetBuf(void *pvParam);
PRIVATE void vTick(void *pvParam);
PRIVATE void vProcessIncomingMlme(void *pvParam, MAC_DcfmIndHdr_s *psDfcmInd);
PRIVATE void vProcessIncomingMcps(void *pvParam, MAC_DcfmIndHdr_s *psDfcmInd);
PRIVATE void vHandleMcpsDataInd(MAC_McpsDcfmInd_s *psMcpsInd);
PRIVATE void vHandleMcpsDataDcfm(MAC_McpsDcfmInd_s *psMcpsInd);

PRIVATE bool_t bPER_SendConfig(uint8 u8Channel, 
                               uint8 u8PacketLength, 
#ifdef RXPOWERADJUST_SUPPORT
                               uint8 u8InputLevel,
#endif
                               int8 i8PowerLevel);
PRIVATE bool_t bPER_TxCommand(tePER_Command eCommand, uint8 *pu8Payload, uint8 u8PayloadLength, uint32 u32Retries);
PRIVATE bool_t bPER_TxPacket(tePER_Command eCommand, uint8 *pu8Payload, uint8 u8PayloadLength, uint32 u32SequenceNum, bool_t bAck);
PRIVATE bool_t bPER_RxPacket(tePER_Command eCommand);

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

volatile uint8 u8PER_Mode;         /* What mode we're in */
uint8 u8PER_State;                 /* What state the system is in */
uint8 u8PER_Retries;               /* Number of retries to perform */
volatile uint8 u8PER_Channel;      /* What channel we're on */
int8 i8PER_Power;				   /* TX Power level */
uint8 u8PER_PacketLen;             /* PER Packet Length */
#ifdef RXPOWERADJUST_SUPPORT
uint8 u8PER_InputLevel;
#endif

volatile uint32 u32PER_Total;      /* Total number of packets */
volatile uint32 u32PER_Seen;       /* Number of packets seen */
volatile uint32 u32PER_Errors;     /* Number of errored packets seen */
volatile uint8  u8PER_Lqi;         /* Link quality information */

uint32 u32PER_Seq;                   /* Sequence number */

bool_t bPER_InFlight;              /* Is a PER packet currently being sent */

/* Handles for the MAC */
PRIVATE void *s_pvMac;
PRIVATE MAC_Pib_s *s_psMacPib;

volatile bool_t bResponseReceived;

bool_t bSlave;

uint16 u16ShortAddr;
uint16 u16DestAddr;

TSV_Timer_s sPER_PacketTimer;

MAC_MlmeDcfmInd_s sMlmeBuffer;
MAC_McpsDcfmInd_s sMcpsBuffer;

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME:       vPER_MasterInit
 *
 * DESCRIPTION:
 * Initialises the site survey per test master.
 *
 * PARAMETERS: Name     RW  Usage
 * none
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PUBLIC void vPER_MasterInit(void)
{
    u8PER_Channel = PER_MASTER_INIT_CHANNEL;
    u8PER_Retries = 3;

    u32PER_Total = 0;
    u32PER_Seen = 0;
    u32PER_Errors = 0;
    u8PER_Lqi = 0;

    u8PER_Mode = E_PER_MODE_STOPPED;
    u8PER_State = E_PER_STATE_STOPPED;

    bSlave = FALSE;

    u16ShortAddr = PER_FFD_SHORT_ADDR;
    u16DestAddr  = PER_RFD_SHORT_ADDR;

    u32AppApiInit(psMlmeDcfmIndGetBuf,
    		      vProcessIncomingMlme,
    		      NULL,
    		      psMcpsDcfmIndGetBuf,
    		      vProcessIncomingMcps,
    		      NULL);

    /* Set up the MAC handles. Must be called AFTER u32AppQApiInit() */
    s_pvMac = pvAppApiGetMacHandle();
    s_psMacPib = MAC_psPibGetHandle(s_pvMac);

    /* Set Pan ID and short address in PIB (also sets match registers in hardware) */
    MAC_vPibSetPanId(s_pvMac, PER_PAN_ID);
    MAC_vPibSetShortAddr(s_pvMac, u16ShortAddr);

    /* Enable receiver to be on when idle */
    MAC_vPibSetRxOnWhenIdle(s_pvMac, TRUE, FALSE);

    TSV_eOpen(&sPER_PacketTimer, vTick, NULL);
}


/****************************************************************************
 *
 * NAME:       bPER_MasterSetState
 *
 * DESCRIPTION:
 * Changes the state of the master. This function is used to change mode,
 * channel and retries.
 *
 * PARAMETERS: Name     RW  Usage
 *          psState     RW  pointer to a struct containing state information.
 *
 * RETURNS:
 * bool_t   TRUE if the command was passed to the endpoint successfully
 *          FALSE if the coordinator did not acknowledge the command
 *
 ****************************************************************************/
PUBLIC bool_t bPER_MasterSetState(tsPER_MasterState *psState)
{

    int n, i;
    bool_t bDone;
    bool_t bResetCount = FALSE;
    bool_t bRetVal = TRUE;

    uint8 u8PreviousMode = u8PER_Mode;         /* Save previous operating mode */

    u8PER_Retries = psState->u8Retries;

    s_psMacPib->u8MaxFrameRetries = u8PER_Retries;

    eAppApiPlmeSet(PHY_PIB_ATTR_TX_POWER, (uint32)psState->i8PowerLevel);

    /* If the mode has changed, send command to slave,
     * then change mode on master
     */
    if(psState->eMode != u8PER_Mode){

        u8PER_Mode = psState->eMode;

        switch(u8PER_Mode){

        case E_PER_MODE_LOCATE:
            n = 0;
            bDone = FALSE;

            do {

            	vPrintf("\nLocate:");

                n++;

                for(i = 11; i < 27; i++){


                    vPrintf(" %d", i);


                    /*
                     * change master to a new channel, and try sending a
                     * command to the endpoint
                     */
                    eAppApiPlmeSet(PHY_PIB_ATTR_CURRENT_CHANNEL, i);

                    u8PER_Channel = i;

                    if(bPER_TxCommand(E_PER_CMD_MODE_STOP, NULL, 0, 1)){
                        bDone = TRUE;

                        vPrintf("\nDone!");

                        break;
                    }

                }

            } while(n < PER_LOCATE_RETRIES && !bDone);
            u8PER_State = E_PER_STATE_STOPPED;
            bRetVal &= bDone;
            break;

        case E_PER_MODE_STOPPED:
            bRetVal &= bPER_TxCommand(E_PER_CMD_MODE_STOP, NULL, 0, PER_CMD_RETRIES);
            u8PER_State = E_PER_STATE_STOPPED;
            break;

        case E_PER_MODE_RUNNING_ACKS:
            bRetVal &= bPER_TxCommand(E_PER_CMD_MODE_ACK, NULL, 0, PER_CMD_RETRIES);
            bResetCount = TRUE;
            u32PER_Seq = 0;
            u8PER_State = E_PER_STATE_RUNNING_ACKS;
            bPER_TxPacket(E_PER_CMD_PER_PACKET, NULL, u8PER_PacketLen - 5, u32PER_Seq, TRUE);
            break;

        case E_PER_MODE_RUNNING_NO_ACKS:
            bRetVal &= bPER_TxCommand(E_PER_CMD_MODE_NO_ACK, NULL, 0, PER_CMD_RETRIES);
            bResetCount = TRUE;
            u32PER_Seq = 0;
            u8PER_State = E_PER_STATE_RUNNING_NO_ACKS;
            break;

        case E_PER_MODE_RESTART:
        	psState->eMode = u8PreviousMode;
        	return bPER_MasterSetState(psState);
            break;

        }

    }

    /*
     * if the channel has changed, send command to slave
     * then change channel on master
     */
    if (   (psState->u8Channel      != u8PER_Channel)
        || (psState->u8PacketLength != u8PER_PacketLen)
        || (psState->i8PowerLevel   != i8PER_Power)
#ifdef RXPOWERADJUST_SUPPORT
        || (psState->u8InputLevel   != u8PER_InputLevel)
#endif
       )
    {
        u8PER_Channel    = psState->u8Channel;
        u8PER_PacketLen  = psState->u8PacketLength;
        i8PER_Power      = psState->i8PowerLevel;
#ifdef RXPOWERADJUST_SUPPORT
        u8PER_InputLevel = psState->u8InputLevel;
#endif

        bRetVal &= bPER_SendConfig(u8PER_Channel, 
                                   u8PER_PacketLen, 
#ifdef RXPOWERADJUST_SUPPORT
                                   u8PER_InputLevel,
#endif
                                   i8PER_Power);

		/* Set channel */
		eAppApiPlmeSet(PHY_PIB_ATTR_CURRENT_CHANNEL, u8PER_Channel);

		/* Set TX power */
		eAppApiPlmeSet(PHY_PIB_ATTR_TX_POWER, (uint32)i8PER_Power);

#ifdef RXPOWERADJUST_SUPPORT
        if (0 == u8PER_InputLevel)
        {
            vAHI_RadioSetReducedInputPower(TRUE);
        }
        else
        {
            vAHI_RadioSetReducedInputPower(FALSE);
        }
#endif

        bResetCount = TRUE;
        u32PER_Seq = 0;
    }

    if (bResetCount)
    {
        u32PER_Total = 0;
        u32PER_Seen = 0;
        u32PER_Errors = 0;
    }

    return(bRetVal);
}


/****************************************************************************
 *
 * NAME:       vPER_MasterGetState
 *
 * DESCRIPTION:
 * Returns the state of the master,
 *
 * PARAMETERS: Name     RW  Usage
 *          psState     W   pointer to a struct containing state information.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PUBLIC void vPER_MasterGetState(tsPER_MasterState *psState)
{

    psState->eMode = u8PER_Mode;
    psState->u8Channel = u8PER_Channel;
    psState->u8Retries = u8PER_Retries;
    psState->u32Total = u32PER_Total;
    psState->u32Seen = u32PER_Seen;
    psState->u32Errors = u32PER_Errors;
    psState->u8Lqi = u8PER_Lqi;

}


/****************************************************************************
 *
 * NAME:       vPER_SlaveInit
 *
 * DESCRIPTION:
 * Initialises the Site Survey PER Test slave.
 *
 * PARAMETERS: Name     RW  Usage
 * None
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PUBLIC void vPER_SlaveInit(void)
{
    u8PER_Channel = PER_SLAVE_INIT_CHANNEL;
    u8PER_Mode = E_PER_MODE_STOPPED;
    u8PER_State = E_PER_STATE_STOPPED;
    u8PER_Retries = 3;

    u16ShortAddr = PER_RFD_SHORT_ADDR;
    u16DestAddr  = PER_FFD_SHORT_ADDR;

    bSlave = TRUE;

    u32AppApiInit(psMlmeDcfmIndGetBuf,
    		      vProcessIncomingMlme,
    		      NULL,
    		      psMcpsDcfmIndGetBuf,
    		      vProcessIncomingMcps,
    		      NULL);

    /* Set up the MAC handles. Must be called AFTER u32AppQApiInit() */
    s_pvMac = pvAppApiGetMacHandle();
    s_psMacPib = MAC_psPibGetHandle(s_pvMac);

    /* Set channel */
    eAppApiPlmeSet(PHY_PIB_ATTR_CURRENT_CHANNEL, u8PER_Channel);

    /* Set Pan ID and short address in PIB (also sets match registers in hardware) */
    MAC_vPibSetPanId(s_pvMac, PER_PAN_ID);
    MAC_vPibSetShortAddr(s_pvMac, u16ShortAddr);

    /* Enable receiver to be on when idle */
    MAC_vPibSetRxOnWhenIdle(s_pvMac, TRUE, FALSE);

    TSV_eOpen(&sPER_PacketTimer, vTick, NULL);
}


/****************************************************************************
 *
 * NAME:       vPER_SlaveGetState
 *
 * DESCRIPTION:
 * Returns the state of the endpoint
 *
 * PARAMETERS: Name     RW  Usage
 *          psState     W   pointer to a struct containing state information.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PUBLIC void vPER_SlaveGetState(tsPER_SlaveState *psState)
{
    psState->eMode = u8PER_Mode;
    psState->u8Channel = u8PER_Channel;
    psState->u8PacketLength = u8PER_PacketLen;
    psState->i8PowerLevel = i8PER_Power;
#ifdef RXPOWERADJUST_SUPPORT
    psState->u8InputLevel = u8PER_InputLevel;
#endif
}

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

PRIVATE MAC_DcfmIndHdr_s *psMlmeDcfmIndGetBuf(void *pvParam)
{
	/* Return a handle to a buffer */
	return (MAC_DcfmIndHdr_s *)&sMlmeBuffer;
}

PRIVATE MAC_DcfmIndHdr_s *psMcpsDcfmIndGetBuf(void *pvParam)
{
	/* Return a handle to a buffer */
	return (MAC_DcfmIndHdr_s *)&sMcpsBuffer;
}

PRIVATE void vTick(void *pvParam)
{
	bPER_TxPacket(E_PER_CMD_PER_PACKET, NULL, u8PER_PacketLen - 5, ++u32PER_Seq, FALSE);
}

/****************************************************************************
 *
 * NAME: vProcessIncomingMlme
 *
 * DESCRIPTION:
 * Process any incoming managment events from the stack.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  psMlmeInd
 *
 * RETURNS:
 * None.
 *
 * NOTES:
 * None.
 ****************************************************************************/
PRIVATE void vProcessIncomingMlme(void *pvParam, MAC_DcfmIndHdr_s *psDfcmInd)
{
	MAC_MlmeDcfmInd_s *psMlmeInd = (MAC_MlmeDcfmInd_s *)psDfcmInd;
    switch (psMlmeInd->u8Type)
    {
    default:
        break;
    }
}

/****************************************************************************
 *
 * NAME: vProcessIncomingData
 *
 * DESCRIPTION:
 * Process incoming data events from the stack.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  psMcpsInd
 *
 * RETURNS:
 * None.
 *
 * NOTES:
 * None.
 ****************************************************************************/
PRIVATE void vProcessIncomingMcps(void *pvParam, MAC_DcfmIndHdr_s *psDfcmInd)
{
	MAC_McpsDcfmInd_s *psMcpsInd = (MAC_McpsDcfmInd_s *)psDfcmInd;
    /* Only handle incoming data events one device has been started as a
       coordinator */
	switch(psMcpsInd->u8Type)
	{
	case MAC_MCPS_IND_DATA:  /* Incoming data frame */
		vHandleMcpsDataInd(psMcpsInd);
		break;
	case MAC_MCPS_DCFM_DATA: /* Incoming acknowledgement or ack timeout */
		vHandleMcpsDataDcfm(psMcpsInd);
		break;
	default:
		break;
	}
}

/****************************************************************************
 *
 * NAME: vHandleMcpsDataDcfm
 *
 * DESCRIPTION:
 *
 * PARAMETERS:      Name            RW  Usage
 *
 * RETURNS:
 *
 * NOTES:
 ****************************************************************************/
PRIVATE void vHandleMcpsDataDcfm(MAC_McpsDcfmInd_s *psMcpsInd)
{
	if (bSlave) {
		/* Data frame transmission successful */
		if (psMcpsInd->uParam.sDcfmData.u8Status == MAC_ENUM_SUCCESS)
		{
			/* Command is stored in handle */
			switch(psMcpsInd->uParam.sDcfmData.u8Handle){

			case E_PER_CMD_CFG:
				/* Set channel */
				eAppApiPlmeSet(PHY_PIB_ATTR_CURRENT_CHANNEL, u8PER_Channel);

				/* Set TX power */
				eAppApiPlmeSet(PHY_PIB_ATTR_TX_POWER, (uint32)i8PER_Power);

#ifdef RXPOWERADJUST_SUPPORT
                if (0 == u8PER_InputLevel)
                {
                    vAHI_RadioSetReducedInputPower(TRUE);
                }
                else
                {
                    vAHI_RadioSetReducedInputPower(FALSE);
                }
#endif
				break;

			case E_PER_CMD_MODE_NO_ACK:
				u32PER_Seq = 0;
				bPER_TxPacket(E_PER_CMD_PER_PACKET, NULL, u8PER_PacketLen - 5, ++u32PER_Seq, FALSE);
				break;

			default:
				break;

			}
		}

		if ((psMcpsInd->uParam.sDcfmData.u8Handle == E_PER_CMD_PER_PACKET) && (u8PER_State == E_PER_STATE_RUNNING_NO_ACKS))
		{
			if ((u32PER_Seq & 0xF) == 0){
				TSV_eStart(&sPER_PacketTimer, TICK_PERIOD);
			} else {
				bPER_TxPacket(E_PER_CMD_PER_PACKET, NULL, u8PER_PacketLen - 5, ++u32PER_Seq, FALSE);
			}
		}

	} else {
		if (psMcpsInd->uParam.sDcfmData.u8Handle == E_PER_CMD_PER_PACKET)
		{
			switch(u8PER_Mode){
			case E_PER_MODE_RUNNING_ACKS:
				u32PER_Total++;

				switch (psMcpsInd->uParam.sDcfmData.u8Status) {
				case MAC_ENUM_CHANNEL_ACCESS_FAILURE:
					/* CCA error */
					u32PER_Errors++;
					break;
				case MAC_ENUM_SUCCESS:
					u32PER_Seq++;
					u32PER_Seen++;
					u8PER_Lqi = u32REG_PhyRead(REG_PHY_MSTAT) & 0xff;
					break;
				default:
					break;
				}

				bPER_TxPacket(E_PER_CMD_PER_PACKET, NULL, u8PER_PacketLen - 5, u32PER_Seq, TRUE);

				break;

			default:
				break;
			}
		}

	}
}

/****************************************************************************
 *
 * NAME: vHandleMcpsDataInd
 *
 * DESCRIPTION:
 *
 * PARAMETERS:      Name            RW  Usage
 *
 * RETURNS:
 *
 * NOTES:
 ****************************************************************************/
PRIVATE void vHandleMcpsDataInd(MAC_McpsDcfmInd_s *psMcpsInd)
{
    MAC_RxFrameData_s *psFrame;

    psFrame = &psMcpsInd->uParam.sIndData.sFrame;

    /* Check application layer sequence number of frame and reject if it is
       the same as the last frame, i.e. same frame has been received more
       than once. */

    uint32 u32SeqNum = psFrame->au8Sdu[1] << 24;
    u32SeqNum |= psFrame->au8Sdu[2] << 16;
    u32SeqNum |= psFrame->au8Sdu[3] << 8;
    u32SeqNum |= psFrame->au8Sdu[4] << 0;

    tePER_Command eCommand = psFrame->au8Sdu[0];

	if (bSlave) {
		switch (eCommand) {
		case E_PER_CMD_MODE_STOP:
			u8PER_Mode = E_PER_MODE_STOPPED;
			u8PER_State = E_PER_STATE_STOPPED;
			bPER_TxPacket(eCommand, &psFrame->au8Sdu[5], psFrame->u8SduLength - 5, u32SeqNum, TRUE);
			break;

		case E_PER_CMD_MODE_ACK:
			u8PER_Mode = E_PER_MODE_RUNNING_ACKS;
			u8PER_State = E_PER_STATE_RUNNING_ACKS;
			bPER_TxPacket(eCommand, &psFrame->au8Sdu[5], psFrame->u8SduLength - 5, u32SeqNum, TRUE);
			break;

		case E_PER_CMD_MODE_NO_ACK:
			u8PER_Mode = E_PER_MODE_RUNNING_NO_ACKS;
			u8PER_State = E_PER_STATE_RUNNING_NO_ACKS;
			bPER_TxPacket(eCommand, &psFrame->au8Sdu[5], psFrame->u8SduLength - 5, u32SeqNum, TRUE);
			break;

		case E_PER_CMD_PER_PACKET:
			/* Ignore. */
			break;

		case E_PER_CMD_CFG:
			/* Echo command back to master
			 * Command takes affect when acknowledged by master */

			u8PER_Channel    = psFrame->au8Sdu[5];
			u8PER_PacketLen  = psFrame->au8Sdu[6];
			i8PER_Power      = psFrame->au8Sdu[7];
#ifdef RXPOWERADJUST_SUPPORT
            u8PER_InputLevel = psFrame->au8Sdu[8];
#endif
			bPER_TxPacket(eCommand, &psFrame->au8Sdu[5], psFrame->u8SduLength - 5, u32SeqNum, TRUE);
			break;

		default:
			/* Echo command back to master
			 * Command takes affect when acknowledged by master */
			bPER_TxPacket(eCommand, &psFrame->au8Sdu[5], psFrame->u8SduLength - 5, u32SeqNum, TRUE);
			break;
		}
	} else {
		if (eCommand == E_PER_CMD_PER_PACKET) {
	        switch(u8PER_State){

	        case E_PER_STATE_RUNNING_NO_ACKS:
	            /* Received PER packet from slave */
	        	if (eCommand == E_PER_CMD_PER_PACKET)
	        	{
	        		if (u32SeqNum < u32PER_Total)
	        		{
	        			/* Slave has restarted */
	        			u32PER_Seen = 0;
	        			u32PER_Errors = 0;
	        		}
					u32PER_Total = u32SeqNum;
					u32PER_Seen++;

	        	}
	            break;

	        default:
	        	break;
	        }

	        u8PER_Lqi = psFrame->u8LinkQuality;
		} else {
			vPrintf(" RX: %d", eCommand);
			bResponseReceived = TRUE;
		}
	}
}


PRIVATE bool_t bPER_SendConfig(uint8 u8Channel, 
                               uint8 u8PacketLength, 
#ifdef RXPOWERADJUST_SUPPORT
                               uint8 u8InputLevel,
#endif
                               int8 i8PowerLevel)
{
	uint8 au8Payload[] = {u8Channel, 
	                      u8PacketLength, 
	                      i8PowerLevel, 	                      
#ifdef RXPOWERADJUST_SUPPORT
	                      u8InputLevel
#else
                          0
#endif	                      
	                     };

	return bPER_TxCommand(E_PER_CMD_CFG, au8Payload, sizeof(au8Payload), PER_CMD_RETRIES);
}

/****************************************************************************
 *
 * NAME:       bPER_TxCommand
 *
 * DESCRIPTION:
 * Sends commands to the slave, ie, change channel, change mode.
 *
 * PARAMETERS: Name     RW  Usage
 *          eCommand  R   Command value to send in payload
 *          u32Retries  R   Number of attempts at sending command
 *
 * RETURNS:
 * bool_t   TRUE if message sent ok
 *          FALSE if message was not delivered
 *
 ****************************************************************************/
PRIVATE bool_t bPER_TxCommand(tePER_Command eCommand, uint8 *pu8Payload, uint8 u8PayloadLength, uint32 u32Retries)
{

    int i;
    bool_t bAck = FALSE;

    vPrintf("\nCMD: %d ...", eCommand);

    for(i = 0; i < u32Retries; i++){
    	bResponseReceived = FALSE;
        if(bPER_TxPacket(eCommand, pu8Payload, u8PayloadLength, (uint8)i, TRUE) == TRUE){
            bAck = bPER_RxPacket(eCommand);
        }

        if(bAck){
            break;
        }

    }

    if (bAck) {
    	vPrintf(" OK");
    } else {
    	vPrintf(" FAILED");
    }

    return(bAck);
}


/****************************************************************************
 *
 * NAME:       bPER_TxPacket
 *
 * DESCRIPTION:
 * Transmits a packet with ack request and a 4 byte payload.
 *
 * PARAMETERS: Name         RW  Usage
 *          u32Payload      R   4 Byte payload
 *          u8SequenceNum   R   Sequence number to use
 *
 * RETURNS:
 * bool_t   TRUE if message sent ok and we got an ACK
 *          FALSE if message was not delivered
 *
 ****************************************************************************/
PRIVATE bool_t bPER_TxPacket(tePER_Command eCommand, uint8 *pu8Payload, uint8 u8PayloadLength, uint32 u32SequenceNum, bool_t bAck)
{
    MAC_McpsReqRsp_s  sMcpsReqRsp;
    MAC_McpsSyncCfm_s sMcpsSyncCfm;
    uint8 *pu8Packet;

    /* Create frame transmission request */
    sMcpsReqRsp.u8Type = MAC_MCPS_REQ_DATA;
    sMcpsReqRsp.u8ParamLength = sizeof(MAC_McpsReqData_s);
    /* Store command type in handle */
    sMcpsReqRsp.uParam.sReqData.u8Handle = eCommand;
    /* Use short address for source */
    sMcpsReqRsp.uParam.sReqData.sFrame.sSrcAddr.u8AddrMode = 2;
    sMcpsReqRsp.uParam.sReqData.sFrame.sSrcAddr.u16PanId = PER_PAN_ID;
    sMcpsReqRsp.uParam.sReqData.sFrame.sSrcAddr.uAddr.u16Short = u16ShortAddr;
    /* Use short address for destination */
    sMcpsReqRsp.uParam.sReqData.sFrame.sDstAddr.u8AddrMode = 2;
    sMcpsReqRsp.uParam.sReqData.sFrame.sDstAddr.u16PanId = PER_PAN_ID;
    sMcpsReqRsp.uParam.sReqData.sFrame.sDstAddr.uAddr.u16Short = u16DestAddr;
    /* Frame requires ack but not security, indirect transmit or GTS */
    if (bAck) {
    	sMcpsReqRsp.uParam.sReqData.sFrame.u8TxOptions = MAC_TX_OPTION_ACK;
    } else {
    	sMcpsReqRsp.uParam.sReqData.sFrame.u8TxOptions = 0;
    }

    pu8Packet = sMcpsReqRsp.uParam.sReqData.sFrame.au8Sdu;

    pu8Packet[0] = eCommand;

    pu8Packet[1] = u32SequenceNum >> 24 & 0xFF;
    pu8Packet[2] = u32SequenceNum >> 16 & 0xFF;
    pu8Packet[3] = u32SequenceNum >> 8  & 0xFF;
    pu8Packet[4] = u32SequenceNum >> 0  & 0xFF;

    if (pu8Payload != NULL) {
    	memcpy(&pu8Packet[5], pu8Payload, u8PayloadLength);
    } else {
    	memset(&pu8Packet[5], 0x55, u8PayloadLength);
    }

    /* Set frame length */
    sMcpsReqRsp.uParam.sReqData.sFrame.u8SduLength = 5 + u8PayloadLength;

    /* Request transmit */
    vAppApiMcpsRequest(&sMcpsReqRsp, &sMcpsSyncCfm);

    return(sMcpsSyncCfm.u8Status != MAC_MCPS_CFM_ERROR);
}


/****************************************************************************
 *
 * NAME:       bPER_RxPacket
 *
 * DESCRIPTION:
 * Looks for a received packet containing the given payload.
 *
 * PARAMETERS: Name         RW  Usage
 *          u8Payload       R   1 Byte payload to check for
 *
 * RETURNS:
 * bool_t   TRUE if message received contained the expected payload
 *          FALSE if either no message or incorrect payload
 *
 ****************************************************************************/
PRIVATE bool_t bPER_RxPacket(tePER_Command eCommand)
{
    bool_t bOk = FALSE;
    volatile int i = 0;

    /* wait for a received packet */
    while((i++ < 500000) && (bOk == FALSE)){

    	if (bResponseReceived) {
    		bOk = TRUE;
    	}
    }

    return(bOk);

}


/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
