/*****************************************************************************
 *
 * MODULE:              PER Test Master
 *
 ****************************************************************************
 *
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5179].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2015. All rights reserved
 ****************************************************************************/

/****************************************************************************/
/***        Include Files                                                 ***/
/****************************************************************************/

#include <jendefs.h>
#include <string.h>

#include <AppHardwareApi.h>
#include <PeripheralRegs.h>
#include <AppQueueApi.h>
#include <mac_sap.h>
#include <mac_pib.h>

#include "Button.h"
#include "LcdDriver.h"
#include "LcdDraw.h"
#include "LcdFont.h"

#include "PerTest.h"

#include "UartBuffered.h"
#include "Printf.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

#define UART_TO_PC          E_AHI_UART_0            /* Uart to PC           */
#define BAUD_RATE           E_AHI_UART_RATE_115200  /* Baud rate to use     */

#define KEY_REPEAT          20

#define MAX_POWER_LEVEL     0

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

typedef enum {
    RV_CHANGE_CHANNEL,
    RV_CHANGE_MODE,
    RV_STOPSTART,
} teReturnValue;

typedef enum {
	E_ENTRY_UINT8,
	E_ENTRY_INT8,
	E_ENTRY_FUNCTION,
	E_ENTRY_SELECT
} teEntryType;

typedef struct {
	char        *pcTitle;
	teEntryType eType;
	union {
		struct {
			uint8 *pu8Value;
			uint8 u8Min;
			uint8 u8Max;
			uint8 u8Step;
		} tsValueUint8;
		struct {
			int8  *pi8Value;
			int8  i8Min;
			int8  i8Max;
			int8  i8Step;
		} tsValueInt8;
		struct {
			uint8 *pu8Value;
			uint8 u8Count;
			char  **ppcOptions;
		} tsSelection;
		struct {
			void (*pvFunction)(void);
		} tsFunction;
	};
} tsMenuEntry;

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

PRIVATE void vChooseOptions(void);
PRIVATE void vUpdateDisplays(bool_t bUpdateLcd, bool_t bUpdateSerialConsole);
PRIVATE uint8 u8UpdateUI(void);

PRIVATE bool_t bReadButtons(uint8 *u8Key, const char *pcKeyMapping);
PRIVATE bool_t bReadKey(uint8 *u8Key);
PRIVATE void vTickTimerISR(void);

PRIVATE void vDrawLcdDisplay(uint32 u32Xoffset);
PRIVATE void vPutC_LCD(uint8 u8Data);
PRIVATE void vLcdClearArea(uint8 u8LeftColumn, uint8 u8TopRow, uint8 u8Width, uint8 u8Height);
PRIVATE void vPutC(uint8 u8Data);

PRIVATE void vStart(void);

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

bool_t bEndpointOk = TRUE;
bool_t bUseLcdButtons = TRUE;
tePT_Mode eRunMode;
uint32 u32LastSeen = 0;

tsPER_MasterState sMasterData;             /* holds master state data */

tsUartFifo sFifo;                               /* TX & RX Buffers for UART */

/* scratchpad area and cursor info for building lcd contents */
char acLcdShadow[8][30];
uint8 u8LcdX = 0, u8LcdY = 0;

volatile uint32 u32Debounce = 0;

bool_t bStartPER = FALSE;

uint8  u8Channel      = DEFAULT_CHANNEL;
int8   i8PowerLevel   = MAX_POWER_LEVEL;
uint8  u8Retries      = 3;
uint8  u8PacketLength = 5;

#ifdef RXPOWERADJUST_SUPPORT
uint8  u8InputLevel = 10;
#endif

tsMenuEntry asMenuEntries[] = {
		{ "Channel",     E_ENTRY_UINT8,     .tsValueUint8 = { &u8Channel,      11,  26 ,                            1  } },
		{ "Power Level", E_ENTRY_INT8,      .tsValueInt8  = { &i8PowerLevel,  -32,  MAX_POWER_LEVEL,                1  } },
		{ "Retries",     E_ENTRY_UINT8,     .tsValueUint8 = { &u8Retries,      0,   7,                              1  } },
		{ "Packet Len",  E_ENTRY_UINT8,     .tsValueUint8 = { &u8PacketLength, 5,   (MAC_MAX_DATA_PAYLOAD_LEN - 2), 10 } },
#ifdef RXPOWERADJUST_SUPPORT
		{ "RX Power",    E_ENTRY_UINT8,     .tsValueUint8 = { &u8InputLevel,   0,   10,                             10 } },
#endif
		{ "Start" ,      E_ENTRY_FUNCTION,  .tsFunction   = { vStart } },
};

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

PUBLIC void AppColdStart(void)
{

    /* Disable watchdog if enabled by default */
    #ifdef WATCHDOG_ENABLED
    vAHI_WatchdogStop();
    #endif

    /* Initialise the hardware API */
    u32AHI_Init();

    /* Wait for clock */
    while (bAHI_GetClkSource() == TRUE);
    vAHI_OptimiseWaitStates();

    /* Set up UART 0: enable interrupts for RX and TX */
    vUartInit(UART_TO_PC, BAUD_RATE, &sFifo);   /* uart for console */
    vInitPrintf((void*)vPutC);
    vPrintf(CLR_SCREEN HOME "AN1212 PER Test Master JN" STR(JENNIC_CHIP) " " STR(JENNIC_MAC) "\n");

    /* set up the tick timer to run at 100Hz with interrupt */
    vAHI_TickTimerInit((void*)vTickTimerISR);
    vAHI_TickTimerInterval(32000000 / 100);
    vAHI_TickTimerConfigure(E_AHI_TICK_TIMER_RESTART);
    vAHI_TickTimerIntEnable(TRUE);

    /* Set up buttons */
    vLcdButtonInit();

    /* Disable Lcd if button held down or LCD is not present. */
    if (u8LcdButtonRead() & (LCD_BOARD_BUTTON_S2_VAL | LCD_BOARD_BUTTON_S3_VAL | LCD_BOARD_BUTTON_S4_VAL)) {
    	bUseLcdButtons = FALSE;
    }

    vLcdResetDefault();

    /* Show startup options screen */
    vChooseOptions();

    /* Initialise site survey PER test master */
    vPrintf("\nStarting as master");
    vPER_MasterInit();

    /* Locate the endpoint */
    vPrintf("\nSearching for endpoint");
    sMasterData.eMode = E_PER_MODE_LOCATE;
    eRunMode = sMasterData.eMode;
    vUpdateDisplays(TRUE, TRUE);
    bEndpointOk = bPER_MasterSetState(&sMasterData);
    vUpdateDisplays(TRUE, TRUE);

    if(!bEndpointOk){
        vPrintf("\nFailed to contact the endpoint!");
    }

    /* Start the PER test running in Ack's mode */
    sMasterData.eMode = E_PER_MODE_RUNNING_ACKS;
    eRunMode = sMasterData.eMode;
    bEndpointOk = bPER_MasterSetState(&sMasterData);

    while (1) {

        switch (u8UpdateUI()) {

        case RV_CHANGE_MODE:
            if(eRunMode == E_PER_MODE_RUNNING_NO_ACKS){
                eRunMode = E_PER_MODE_RUNNING_ACKS;
            } else {
                eRunMode = E_PER_MODE_RUNNING_NO_ACKS;
            }
            if(sMasterData.eMode != E_PER_MODE_STOPPED){
                sMasterData.eMode = eRunMode;
            }
            break;

        case RV_CHANGE_CHANNEL:
            sMasterData.u8Channel++;
            if(sMasterData.u8Channel > 26)
            {
                  sMasterData.u8Channel = 11;
            }
            break;

        case RV_STOPSTART:
            if(sMasterData.eMode != E_PER_MODE_STOPPED){
                sMasterData.eMode = E_PER_MODE_STOPPED;
            } else {
                sMasterData.eMode = eRunMode;
            }
            break;

        }

        bEndpointOk = bPER_MasterSetState(&sMasterData);

    }
}


PUBLIC void AppWarmStart(void)
{
    AppColdStart();
}

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

PRIVATE void vChooseOptions(void)
{

    uint8  n;
    uint8  u8Key;
    uint8  u8MenuEntries = sizeof(asMenuEntries) / sizeof(asMenuEntries[0]);

    bool_t bUpdateDisplay = TRUE;
    uint8  u8Selection = 0;
    uint8  u8LastSelection = 0;

    vInitPrintf((void*)vPutC);

    /* Ascii clear screen */
    vPrintf(CURSOR(2, 1) CLR_DOWN);

    for (n = 0; n < u8MenuEntries; n++) {
        if(n == u8Selection){
        	vPrintf(">");
        } else {
        	vPrintf(" ");
        }

        vPrintf("%s", asMenuEntries[n].pcTitle);

    	switch (asMenuEntries[n].eType) {
    	case E_ENTRY_INT8:
    		vPrintf("\t: %i", *asMenuEntries[n].tsValueInt8.pi8Value);
    		break;
    	case E_ENTRY_UINT8:
    		vPrintf("\t: %d", *asMenuEntries[n].tsValueUint8.pu8Value);
    		break;
    	case E_ENTRY_SELECT:
        	vPrintf("\t: %s", asMenuEntries[n].tsSelection.ppcOptions[*asMenuEntries[n].tsSelection.pu8Value]);
    		break;
    	default:
    		break;
    	}

    	vPrintf("\n");
    }

    vPrintf("Select Entry: (Up / Down Arrow)\n"
            "-/+: (Left / Right Arrow)");

    /* Return cursor to current selection */
    vPrintf("\x1B[%d;%dH" , u8Selection + 2, 1);

    vInitPrintf((void*)vPutC_LCD);  /* Printf now writes to lcd shadow area */

    vLcdClear();

    vPrintf("         Options\n");

    for (n = 0; n < u8MenuEntries; n++) {
    	vPrintf(" %s\n", asMenuEntries[n].pcTitle);
    }

    while (n++ < 6) {
    	vPrintf("\n");
    }

    vPrintf("       Down      -       +");

    vDrawLcdDisplay(5);             /* write to lcd module */
    vLcdRefreshAll();

    do {

        if(bUpdateDisplay){
        	bUpdateDisplay = FALSE;

            vInitPrintf((void*)vPutC_LCD);  /* Printf now writes to lcd shadow area */

            for (n = 0; n < u8MenuEntries; n++) {
            	vPrintf("\n");
                if(n == u8Selection){
                	vPrintf(">");
                } else {
                	vPrintf(" ");
                }
            }

            vDrawLcdDisplay(0);            /* write to lcd module */

            vLcdRefreshArea(0, 0, 8, 8);               /* display new data */

            for (n = 0; n < u8MenuEntries; n++) {
            	vPrintf("\n");
            	switch (asMenuEntries[n].eType) {
            	case E_ENTRY_INT8:
            		vPrintf(": %i", *asMenuEntries[n].tsValueInt8.pi8Value);
            		break;
            	case E_ENTRY_UINT8:
            		vPrintf(": %d", *asMenuEntries[n].tsValueUint8.pu8Value);
            		break;
            	case E_ENTRY_SELECT:
            		vPrintf(": %s", asMenuEntries[n].tsSelection.ppcOptions[*asMenuEntries[n].tsSelection.pu8Value]);
            		break;
            	default:
            		break;
            	}
            }

            vLcdClearArea(74, 0, 128 - 74, 7);

            vDrawLcdDisplay(74);            /* write to lcd module */

            vLcdRefreshArea(74, 0, 128 - 74, 7);               /* display new data */

            vInitPrintf((void*)vPutC);

            vPrintf("\r");

            /* Remove marker from current line and move to new line*/
            if (u8Selection != u8LastSelection) {
                vPrintf(" \x1B[%d;%dH>" , u8Selection + 2, 1);
                u8LastSelection = u8Selection;
            }

			switch (asMenuEntries[u8Selection].eType) {
			case E_ENTRY_INT8:
				vPrintf("\t\t: %i" CLR_END, *asMenuEntries[u8Selection].tsValueInt8.pi8Value);
				break;
			case E_ENTRY_UINT8:
				vPrintf("\t\t: %d" CLR_END, *asMenuEntries[u8Selection].tsValueUint8.pu8Value);
				break;
        	case E_ENTRY_SELECT:
        		vPrintf("\t\t: %s" CLR_END, asMenuEntries[u8Selection].tsSelection.ppcOptions[*asMenuEntries[u8Selection].tsSelection.pu8Value]);
        		break;
			default:
				break;
			}

        }

        /* Handle any button presses */
        if (bReadButtons(&u8Key, "BDC") || bReadKey(&u8Key))
        {
        	tsMenuEntry *psEntry = &asMenuEntries[u8Selection];

        	switch (u8Key) {
        	case 'A':
				/* Up */
				if (u8Selection > 0) {
					u8Selection--;
				} else {
					u8Selection = (u8MenuEntries - 1);
				}
				bUpdateDisplay = TRUE;
				break;
        	case 'B':
				/* Down */
				if (u8Selection < (u8MenuEntries - 1)) {
					u8Selection++;
				} else {
					u8Selection = 0;
				}
				bUpdateDisplay = TRUE;
				break;
        	case 'D':
				switch (psEntry->eType) {
				case E_ENTRY_UINT8:
					if ((*psEntry->tsValueUint8.pu8Value) >= (psEntry->tsValueUint8.u8Min + psEntry->tsValueUint8.u8Step))
					{
						(*psEntry->tsValueUint8.pu8Value) -= psEntry->tsValueUint8.u8Step;
						bUpdateDisplay = TRUE;
					}
					break;
				case E_ENTRY_INT8:
					if ((*psEntry->tsValueInt8.pi8Value) >= (psEntry->tsValueInt8.i8Min  + psEntry->tsValueInt8.i8Step))
					{
						(*psEntry->tsValueInt8.pi8Value) -= psEntry->tsValueInt8.i8Step;
						bUpdateDisplay = TRUE;
					}
					break;
				case E_ENTRY_SELECT:
					if ((*psEntry->tsSelection.pu8Value) > 0)
					{
						(*psEntry->tsSelection.pu8Value)--;
						bUpdateDisplay = TRUE;
					}
					break;
				default:
					break;
				}
				break;
        	case 'C':
				switch (psEntry->eType) {
				case E_ENTRY_UINT8:
					if ((*psEntry->tsValueUint8.pu8Value) <= (psEntry->tsValueUint8.u8Max - psEntry->tsValueUint8.u8Step))
					{
						(*psEntry->tsValueUint8.pu8Value) += psEntry->tsValueUint8.u8Step;
						bUpdateDisplay = TRUE;
					}
					break;
				case E_ENTRY_INT8:
					if ((*psEntry->tsValueInt8.pi8Value) <= (psEntry->tsValueInt8.i8Max - psEntry->tsValueInt8.i8Step))
					{
						(*psEntry->tsValueInt8.pi8Value) += psEntry->tsValueInt8.i8Step;
						bUpdateDisplay = TRUE;
					}
					break;
				case E_ENTRY_SELECT:
					if ((*psEntry->tsSelection.pu8Value) < psEntry->tsSelection.u8Count)
					{
						(*psEntry->tsSelection.pu8Value)++;
						bUpdateDisplay = TRUE;
					}
					break;
				case E_ENTRY_FUNCTION:
					psEntry->tsFunction.pvFunction();
					bUpdateDisplay = TRUE;
					break;
				default:
					break;
				}
				break;
        	case '\n':
        	case '\r':
				switch (psEntry->eType) {
				case E_ENTRY_FUNCTION:
					psEntry->tsFunction.pvFunction();
					bUpdateDisplay = TRUE;
					break;
				default:
					break;
				}
				break;
        	}
        }

    } while (bStartPER == FALSE);

    /* Copy all to master data */
    sMasterData.u8PacketLength = u8PacketLength;
    sMasterData.i8PowerLevel   = i8PowerLevel;
    sMasterData.u8Channel      = u8Channel;
    sMasterData.u8Retries      = u8Retries;
#ifdef RXPOWERADJUST_SUPPORT
    sMasterData.u8InputLevel   = u8InputLevel;
#endif
}

/****************************************************************************
 *
 * NAME:       vStart
 *
 * DESCRIPTION:
 * Menu callback to start PER testing.
 *
 * PARAMETERS: Name     RW  Usage
 * none
 *
 * RETURNS:
 * none
 *
 ****************************************************************************/

PRIVATE void vStart(void)
{
	bStartPER = TRUE;
}

/****************************************************************************
 *
 * NAME:       u8UpdateUI
 *
 * DESCRIPTION:
 * Updates the user interface (Updates LCD and console displays and checks
 * the buttons for any presses. Returns when a button has been pressed.
 *
 * PARAMETERS: Name     RW  Usage
 * none
 *
 * RETURNS:
 * uint8    Keypress value
 *
 ****************************************************************************/

PRIVATE uint8 u8UpdateUI(void)
{
    uint8  u8Key;
    bool_t bOuterLoopFinished = FALSE;
    uint8  u8RetVal = 0;

    do
    {

        /* Doze CPU until an interrupt occurs */
        vAHI_CpuDoze();

        /* get updated per test data */
        vPER_MasterGetState(&sMasterData);

        vUpdateDisplays(TRUE, TRUE);

        if(sMasterData.u32Total > 50000){
            sMasterData.eMode = E_PER_MODE_RESTART;
            bEndpointOk = bPER_MasterSetState(&sMasterData);
        }


        /* Handle any button presses */
        if (bReadKey(&u8Key) || bReadButtons(&u8Key, "123")){
            switch(u8Key){

            case '1':
                /* Change channel */
                bOuterLoopFinished = TRUE;
                u8RetVal = RV_CHANGE_CHANNEL;
                break;

            case '2':
                /* Stop/Start */
                bOuterLoopFinished = TRUE;
                u8RetVal = RV_STOPSTART;
                break;

            case '3':
                /* Ack/No ack */
                bOuterLoopFinished = TRUE;
                u8RetVal = RV_CHANGE_MODE;
                break;

            }

        }

    } while (bOuterLoopFinished == FALSE);

    return u8RetVal;
}


/****************************************************************************
 *
 * NAME:       vUpdateDisplays
 *
 * DESCRIPTION:
 * Updates the displays generated on the LCD module and the serial console.
 *
 * PARAMETERS: Name         RW  Usage
 *          bUpdateLcd      R   Set to TRUE to update LCD
 *          bUpdateSerial   R   Set to TRUE to update serial console
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/

PRIVATE void vUpdateDisplays(bool_t bUpdateLcd, bool_t bUpdateSerialConsole)
{
    static tePT_Mode eLastMode;

    char   acStart[] = "Start";
    char   acStop[] = "Stop ";

    char *pcStart;

    uint8 x;
    uint16 u16Per;
    uint16 u16Bad;
    /* If we are receiving packets, turn LED 0 on */
    if(sMasterData.u32Seen == u32LastSeen)
    {
        /*
         * Since the LQI value holds the level for the last received packet,
         * we need to set it to 0 ourselves if we stop seeing any.
         */
        sMasterData.u8Lqi = 0;
    }
    u32LastSeen = sMasterData.u32Seen;

    /* Calculate new PER and CCA Fail values */
    if(sMasterData.u32Total == 0){
        u16Per = 0;
        u16Bad = 0;
    } else {
        u16Per = (uint16)(1000 - ((sMasterData.u32Seen * 1000) / sMasterData.u32Total));
        u16Bad = (uint16)((sMasterData.u32Errors * 1000) / sMasterData.u32Total);
    }

    /*
     * Figure out which mode we're in and point to
     * relevant text for start/stop button
     */
    if(sMasterData.eMode == E_PER_MODE_STOPPED){
        pcStart = acStart;
    } else {
        pcStart = acStop;
    }

    /* Update the LCD display content if required */
    if(bUpdateLcd){

        vInitPrintf((void*)vPutC_LCD);  /* Printf now writes to lcd shadow area */

    	if (sMasterData.eMode != eLastMode)
    	{
    		/* write the static parts of the display */
    		vLcdClear();

            switch(eRunMode){

            case E_PER_MODE_LOCATE:
                vPrintf("Locating slave```");
                vDrawLcdDisplay(0);
                break;

            case E_PER_MODE_RUNNING_ACKS:
                /* write the static parts of the display */

                vPrintf("Channel                  LQI\n"

                        "Mode\n"
                        "Seen\n"
                        "Total\n"
                        "PER %%\n"
                        "CCA Fail %%\n"
                        "Retries\n"
                        "         Chan  %s Mode",
                        pcStart);


                vDrawLcdDisplay(0);             /* write to lcd module */
                break;

            case E_PER_MODE_RUNNING_NO_ACKS:
                /* write the static parts of the display */

                vPrintf("Channel                  LQI\n"
                        "Mode\n"
                        "Seen\n"
                        "Total\n"
                        "PER %%\n"
                        "\n"
                        "\n"
                        "         Chan  %s Mode",
                        pcStart);

                vDrawLcdDisplay(0);             /* write to lcd module */
                break;

            default:
            	break;

            }

            vLcdRefreshAll();
    	}

        switch (eRunMode) {

        case E_PER_MODE_LOCATE:
            break;

        case E_PER_MODE_RUNNING_ACKS:
        	vLcdClearArea(60, 0, 50, 7);
            /* write the dynamic parts of the display */
            vPrintf("%d\nAck\n%d\n%d\n%d`%d\n%d`%d\n%d",sMasterData.u8Channel,
                                                         sMasterData.u32Seen,
                                                         sMasterData.u32Total,
                                                         u16Per / 10, u16Per % 10,
                                                         u16Bad / 10, u16Bad % 10,
                                                         sMasterData.u8Retries);

            vDrawLcdDisplay(60);            /* write to lcd module */
            break;

        case E_PER_MODE_RUNNING_NO_ACKS:
        	vLcdClearArea(60, 0, 50, 7);
            /* write the dynamic parts of the display */
            vPrintf("%d\nNoAck\n%d\n%d\n%d`%d",sMasterData.u8Channel,
                                                sMasterData.u32Seen,
                                                sMasterData.u32Total,
                                                u16Per / 10, u16Per % 10);

            vDrawLcdDisplay(60);            /* write to lcd module */
            break;

        default:
        	break;

        }

        if(bEndpointOk == FALSE){

            vLcdWriteText("Err  ", 0, 100);

        } else {

        	vLcdClearArea(110, 1, 18, 6);

            /* draw the LQI bar */
            for(x=0;x<5;x++){
                vLcdDrawLine(120 - x, 55, 120 - x, 55 - ((((sMasterData.u8Lqi * 100) / 255) * 47) / 100));
            }

        }

        vLcdRefreshArea(60, 0, 128-60, 7);               /* display new data */
    }

    /* Update the serial console content if required */
    if(bUpdateSerialConsole){
    	vInitPrintf((void*)vPutC);      /* Printf now writes to uart console */

    	if (sMasterData.eMode != eLastMode)
    	{
			switch (eRunMode) {

			case E_PER_MODE_LOCATE:
				break;

			case E_PER_MODE_RUNNING_ACKS:
				/* write the static parts of the display */
				vPrintf(CURSOR(2, 1) CLR_DOWN);
				vPrintf("Channel\n"
						"Mode\t\tAck\n"
						"Seen\n"
						"Total\n"
						"PER %%\n"
						"CCA Fail %%\n"
						"Retries\n");
				break;

			case E_PER_MODE_RUNNING_NO_ACKS:
				/* write the static parts of the display */
				vPrintf(CURSOR(2, 1) CLR_DOWN);
				vPrintf("Channel\n"
						"Mode\t\tNoAck\n"
						"Seen\n"
						"Total\n"
						"PER %%\n"
						"LQI\n");
				break;

			default:
				break;

			}
    	}

		switch(eRunMode){

		case E_PER_MODE_LOCATE:
			break;

		case E_PER_MODE_RUNNING_ACKS:
			/* write the dynamic parts of the display */
			vPrintf(CURSOR(2, 1));
			vPrintf("\t\t%d\n"
					"\n"
					"\t\t%d\n"
					"\t\t%d\n"
					"\t\t" CLR_END "%d.%d\n"
					"\t\t" CLR_END "%d.%d\n"
					"\t\t%d\n"
					"Chan (1)  %s (2)  Mode (3)",
					sMasterData.u8Channel,
					sMasterData.u32Seen,
					sMasterData.u32Total,
					u16Per / 10, u16Per % 10,
					u16Bad / 10, u16Bad % 10,
					sMasterData.u8Retries,
					pcStart);
			break;

		case E_PER_MODE_RUNNING_NO_ACKS:
			/* write the dynamic parts of the display */
			vPrintf(CURSOR(2, 1));
			vPrintf("\t\t%d\n"
					"\n"
					"\t\t%d\n"
					"\t\t%d\n"
					"\t\t" CLR_END "%d.%d\n"
					"\t\t%d\n"
					"Chan (1)  %s (2)  Mode (3)",
					sMasterData.u8Channel,
					sMasterData.u32Seen,
					sMasterData.u32Total,
					u16Per / 10, u16Per % 10,
					sMasterData.u8Lqi, pcStart);
			break;

		default:
			break;

		}

    }

    eLastMode = sMasterData.eMode;
}

/****************************************************************************
 *
 * NAME:       bReadButtons
 *
 * DESCRIPTION:
 * Checks for button presses.
 *
 * PARAMETERS:  Name    RW  Usage
 *              u8Key   W   Pointer to a memory location that will receive the
 *                          key press value.
 *
 * RETURNS:
 * bool_t, TRUE if a button was pressed, FALSE if not
 *
 ****************************************************************************/
PRIVATE bool_t bReadButtons(uint8 *u8Key, const char *pcKeyMapping)
{
    bool_t bKeyPress = FALSE;

    if (bUseLcdButtons)
    {
		switch(u8LcdButtonRead() & (LCD_BOARD_BUTTON_S2_VAL | LCD_BOARD_BUTTON_S3_VAL | LCD_BOARD_BUTTON_S4_VAL))
		{
		case LCD_BOARD_BUTTON_S2_VAL: /* Switch 2 */
			if(!u32Debounce)
			{
				u32Debounce = KEY_REPEAT;
				*u8Key = pcKeyMapping[0];
				bKeyPress = TRUE;
			}
			break;

		case LCD_BOARD_BUTTON_S3_VAL: /* Switch 3 */
			if(!u32Debounce)
			{
				u32Debounce = KEY_REPEAT;
				*u8Key = pcKeyMapping[1];
				bKeyPress = TRUE;
			}
			break;

		case LCD_BOARD_BUTTON_S4_VAL: /* Switch 4 */
			if(!u32Debounce)
			{
				u32Debounce = KEY_REPEAT;
				*u8Key = pcKeyMapping[2];
				bKeyPress = TRUE;
			}
			break;

		default:
			break;

		}
	}

    return(bKeyPress);
}

/****************************************************************************
 *
 * NAME:       bReadButtons
 *
 * DESCRIPTION:
 * Checks for button presses.
 *
 * PARAMETERS:  Name    RW  Usage
 *              u8Key   W   Pointer to a memory location that will receive the
 *                          key press value.
 *
 * RETURNS:
 * bool_t, TRUE if a button was pressed, FALSE if not
 *
 ****************************************************************************/
PRIVATE bool_t bReadKey(uint8 *u8Key)
{
	bool_t bDataReady = FALSE;

	/* Consume all available characters to stop characters being buffered
	 * faster than updates to the display.
	 */
    while (bUartRxDataAvailable(UART_TO_PC)) {
        *u8Key = u8UartRead(UART_TO_PC);
        bDataReady = TRUE;
    }

    return bDataReady;
}


/****************************************************************************
 *
 * NAME:       vTickTimerISR
 *
 * DESCRIPTION:
 * Interrupt handler for the tick timer
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vTickTimerISR(void)
{
    if(u32Debounce)
    {
        u32Debounce--;
    }
}

/****************************************************************************
 *
 * NAME:       vDrawLcdDisplay
 *
 * DESCRIPTION:
 * Writes characters from the LCD scratchpad to the LCD driver
 *
 * PARAMETERS: Name     RW  Usage
 *          u32Xoffset  R   Offset in pixels from the left side of the
 *                          display where content is to be written.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/

PRIVATE void vDrawLcdDisplay(uint32 u32Xoffset)
{
    int n;

    for(n = 0; n < 8; n++)
    {
        vLcdWriteText(&acLcdShadow[n][0], n, u32Xoffset);
        acLcdShadow[n][0] = '\0';
    }

    u8LcdX = 0;
    u8LcdY = 0;

}

/****************************************************************************
 *
 * NAME:       vPutC_LCD
 *
 * DESCRIPTION:
 * Writes characters to the scratchpad area for constructing
 * the LCD content
 *
 * PARAMETERS: Name     RW  Usage
 *          u8Data      R   Character to write to LCD
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/

PRIVATE void vPutC_LCD(uint8 u8Data)
{

    switch(u8Data)
    {

    case '\n':
        acLcdShadow[u8LcdY][u8LcdX++] = '\0';
        u8LcdY++;
        break;

    case '\r':
        acLcdShadow[u8LcdY][u8LcdX++] = '\0';
        u8LcdX = 0;
        break;

    case '-':
        acLcdShadow[u8LcdY][u8LcdX++] = ']';
        acLcdShadow[u8LcdY][u8LcdX] = '\0';
        break;

    case '+':
        acLcdShadow[u8LcdY][u8LcdX++] = '\\';
        acLcdShadow[u8LcdY][u8LcdX] = '\0';
        break;

    default:
        acLcdShadow[u8LcdY][u8LcdX++] = u8Data;
        acLcdShadow[u8LcdY][u8LcdX] = '\0';
        break;

    }

}

/****************************************************************************
 *
 * NAME:       vLcdClearArea
 *
 * DESCRIPTION:
 * Clears an area of the LCD shadow memory. Accepts parameters in the same
 * format as vLcdRefreshArea.
 *
 * PARAMETERS: Name     RW  Usage
 *             u8LeftColumn    R   Left column of refresh area (0-127)
 *             u8TopRow        R   Top character row of refresh area (0-7)
 *             u8Width         R   Number of columns (1-128)
 *             u8Height        R   Number of character rows (1-8)
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vLcdClearArea(uint8 u8LeftColumn, uint8 u8TopRow, uint8 u8Width, uint8 u8Height)
{
	uint8 u8Row;

	for (u8Row = u8TopRow; u8Row < (u8TopRow + u8Height); u8Row++) {
		memset(&au8Shadow[LCD_SHADOW_BUFFER_LOCATION(u8Row, u8LeftColumn)], 0, u8Width);
	}
}

/****************************************************************************
 *
 * NAME:       vPutC
 *
 * DESCRIPTION:
 * Writes characters to the UART connected to the PC
 *
 * PARAMETERS: Name     RW  Usage
 *          u8Data      R   Character to write to the UART
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/

PRIVATE void vPutC(uint8 u8Data)
{
    vUartWrite(UART_TO_PC, u8Data);
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
