/****************************************************************************
 *
 * MODULE:              JN-AN-1180 Demo endpoint code
 *
 * DESCRIPTION:
 * Endpoint for demonstrator. Synchronises to coordinator and sends
 * back data, sets up sensor reads.
 */
/****************************************************************************
*
* This software is owned by NXP B.V. and/or its supplier and is protected
* under applicable copyright laws. All rights are reserved. We grant You,
* and any third parties, a license to use this software solely and
* exclusively on NXP products [NXP Microcontrollers such as JN5148, JN5142, JN5139].
* You, and any third parties must reproduce the copyright and warranty notice
* and any other legend of ownership on each copy or partial copy of the
* software.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.

* Copyright NXP B.V. 2012. All rights reserved
*/

/****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
#include "jendefs.h"
#include "AlsDriver.h"
#include "HtsDriver.h"
#include "HomeSensorConfig.h"
#include <AppHardwareApi.h>
#include <AppQueueApi.h>
#include <AppApi.h>
#include <mac_sap.h>
#include <mac_pib.h>
#include "LedControl.h"
#include "Button.h"
#include "LcdDriver.h"
#include "LightingBoard.h"
#include <string.h>

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/
/* Timing values */
#define FIFTY_MS_IN_32KHZ     1600

#ifdef USE_2006_SECURITY
#define FRAME_TYPE_DATA         (1)

#define TOTAL_SEC_ENTRIES       (1)
#define TOTAL_SECURITY_KEYS     (1)
#define TOTAL_KEY_USAGE_ENTRIES (1)
#define TOTAL_LEVEL_DESCRIPTORS (1)
#endif

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/
/* Key (button) press values as used internally */
typedef enum
{
    E_KEY_0 = LCD_BOARD_BUTTON_S1_VAL

} teKeyValues;

/* State machine states */
typedef enum
{
    E_STATE_OFF,
    E_STATE_SCANNING,
    E_STATE_SYNCING,
    E_STATE_ASSOCIATING,
    E_STATE_RUNNING,
    E_STATE_TX_DATA,
    E_STATE_READ_SENSORS,
    E_STATE_ASSOC_TO_SCAN
} teState;

/* State machine is as follows:

      E_STATE_OFF
            |
            | Initialised
            V
      E_STATE_SCANNING
            |
            | Scan complete, expected beacon found
            V
      E_STATE_SYNCING
            |
            | Expected beacon seen
            V
      E_STATE_ASSOCIATING
            |
            | Association complete
            V
 +--> E_STATE_RUNNING
 |          |
 |          | Beacon seen
 |          V
 |    E_STATE_TX_DATA
 |          |
 |          | Transmission complete (success or fail)
 |          V
 |    E_STATE_READ_SENSORS
 |          |
 |          | Sensors read
 |          |
 +----------+

*/

/* All variables with scope throughout module are in one structure */
typedef struct
{
    /* Transceiver (basically anything TX/RX not covered elsewhere) */
    struct
    {
        uint8   u8CurrentTxHandle;
        uint8   u8PrevRxBsn;
    } sTransceiver;

    /* Controls (switch, light level alarm) */
    struct
    {
        uint8   u8Switch;
        uint8   u8LightAlarmLevel;
    } sControls;

    /* Sensor data, stored between read and going out in frame */
    struct
    {
        uint8   u8TempResult;
        uint8   u8HtsResult;
        uint8   u8AlsResult;
    } sSensors;

    /* Settings specific to light sensor */
    struct
    {
        uint16 u16Hi;
        uint16 u16Lo;
    } sLightSensor;

    /* System (state, assigned address, channel) */
    struct
    {
        teState eState;
        uint16  u16ShortAddr;
        uint8   u8ThisNode;
        uint8   u8Channel;
    } sSystem;
} tsDemoData;

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/
PRIVATE void   vInitSystem(void);
PRIVATE void   vInitEndpoint(void);
PRIVATE void   vStartScan(uint32 u32ChannelsToScan);
PRIVATE void   vProcessInterrupts(void);
PRIVATE void   vStartSync(void);
PRIVATE void   vProcessIncomingMcps(MAC_McpsDcfmInd_s *psMcpsInd);
PRIVATE void   vProcessIncomingMlme(MAC_MlmeDcfmInd_s *psMlmeInd);
PRIVATE void   vProcessRead(void);
PRIVATE void   vProcessRxBeacon(MAC_MlmeDcfmInd_s *psMlmeInd);
PRIVATE void   vProcessTxBlock(void);
PRIVATE uint8  u8FindMin(uint8 u8Val1, uint8 u8Val2);
PRIVATE void   vDisplayError(char *pcString, uint32 u32Data);


/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

/* Handles from the MAC */
PRIVATE void *s_pvMac;
PRIVATE MAC_Pib_s *s_psMacPib;

tsDemoData sDemoData;

#ifdef USE_2006_SECURITY
PRIVATE MAC_KeyDescriptor_s           asKeyDescriptor[TOTAL_SECURITY_KEYS];
PRIVATE MAC_KeyIdLookupDescriptor_s   asKeyIdLookupDescriptor[TOTAL_SECURITY_KEYS];
PRIVATE MAC_KeyUsageDescriptor_s      asKeyUsageNwk[TOTAL_KEY_USAGE_ENTRIES];
PRIVATE MAC_DeviceDescriptor_s        asDeviceDescriptor[TOTAL_SEC_ENTRIES];
PRIVATE MAC_KeyDeviceDescriptor_s     asKeyDeviceList[TOTAL_SEC_ENTRIES];
PRIVATE MAC_SecurityLevelDescriptor_s asSecurityLevelDescriptor[TOTAL_LEVEL_DESCRIPTORS];

PRIVATE const uint8 au8DefaultKeySource[8] = {0x12, 0x34, 0x56, 0x78,
                                              0x9a, 0xbc, 0xde, 0xf0};
#endif

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/
/****************************************************************************
 *
 * NAME: AppColdStart
 *
 * DESCRIPTION:
 * Entry point for application. Initialises system, starts scan then
 * processes interrupts.
 *
 * RETURNS:
 * void, never returns
 *
 ****************************************************************************/
PUBLIC void AppColdStart(void)
{


    /* Disable watchdog if enabled by default */
    #ifdef WATCHDOG_ENABLED
    vAHI_WatchdogStop();
    #endif

    /* General initialisation: reset hardware */
    vInitSystem();

    while (1)
    {
        /* Initialise software elements */
        vInitEndpoint();

        /* Perform scan to find demo coordinator */
        vStartScan(DEMO_CHANNEL_BITMAP);

        /* Run main loop. This processes interrupts util the 'reset' key
           combination is pressed */
        vProcessInterrupts();
    }
}

/****************************************************************************
 *
 * NAME: AppWarmStart
 *
 * DESCRIPTION:
 * Entry point for application from boot loader. Simply jumps to AppColdStart
 * as, in this instance, application will never warm start.
 *
 * RETURNS:
 * Never returns.
 *
 ****************************************************************************/
PUBLIC void AppWarmStart(void)
{
    AppColdStart();
}

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/
/****************************************************************************
 *
 * NAME: vInitSystem
 *
 * DESCRIPTION:
 * Initialises stack and hardware. Also sets non-default values in the
 * 802.15.4 PIB and starts the first read of the light sensor. Subsequent
 * reads of this sensor occur automatically.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vInitSystem(void)
{
    /* Initialise stack and hardware interface, and register peripheral
       interrupts with AppQueueApi handler */
    (void)u32AHI_Init();
    (void)u32AppQApiInit(NULL, NULL, NULL);

    /* Set up the MAC handles. Must be called AFTER u32AppQApiInit() */
    s_pvMac = pvAppApiGetMacHandle();
    s_psMacPib = MAC_psPibGetHandle(s_pvMac);

    bWhite_LED_Enable();
    bWhite_LED_SetLevel(100);
    bWhite_LED_Off();

    bRGB_LED_Enable();
    bRGB_LED_SetLevel(255, 0, 255);
    bRGB_LED_Off();

    /* Set up peripheral hardware */
    vALSreset();
    vHTSreset();


    /* Set Pan ID in PIB (also sets match register in hardware) */
    MAC_vPibSetPanId(s_pvMac, DEMO_PAN_ID);

    /* Start ALS now: it automatically keeps re-sampling after this */
    vALSstartReadChannel(0);

#ifdef USE_2006_SECURITY
    /* Enable security. Using Key ID mode 1 (default key source + 1 byte
       index) and security level 5 (encryption plus authentication with 4-byte
       MIC). Only require security for data frames. Node can only communicate
       with coordinator, so only one key and one security entry needed

       So:
       The Key Descriptor table has 1 entry
         - the entry points to a unique Key ID Lookup Descriptor
         - the entry points to a unique Key Device Descriptor
         - the entry points to a single Key Usage Descriptor
       The Device Descriptor table has 1 entry
         - the entry is pointed to by the Key Device Descriptor

       Also:
       The Security Level Descriptor table has 1 entry (1 per secured frame
       type)

       Unlike the coordinator, some data can only be configured once the
       device has associated, as we use the assigned short address to
       determine the key and the key ID
     */

    /* Set up Security Level Descriptor table, which defines the minimum
       security level allowed for specific frame types. In this case we only
       care about data frames which must always use at least security level 5
     */
    asSecurityLevelDescriptor[0].u8FrameType = FRAME_TYPE_DATA;
    asSecurityLevelDescriptor[0].u8MinimumSecurity = 5;
    asSecurityLevelDescriptor[0].bOverideSecurityMinimum = FALSE;

    /* Key Usage Descriptor list */
    asKeyUsageNwk[0].u8FrameType = FRAME_TYPE_DATA;

    /* Set up Key ID Lookup table. All key lookups are configured as the
      default key source plus a byte for a unique index (to be determined
      after association); 9 bytes in total. The lookup data size is set as
      1, which means 9(!) */
    memcpy(asKeyIdLookupDescriptor[0].au8LookupData,
           au8DefaultKeySource,
           8);
    asKeyIdLookupDescriptor[0].au8LookupData[8] = 0; /* Key index */
    asKeyIdLookupDescriptor[0].u8LookupDataSize = MAC_SECURITY_KEYID_LENGTH_MODE_1;

    /* Key Device Descriptor. Note that as u32DeviceDescriptorHandle is
       index into the asDeviceDescriptor array */
    asKeyDeviceList[0].u32DeviceDescriptorHandle = 0;
    asKeyDeviceList[0].bUniqueDevice = FALSE;
    asKeyDeviceList[0].bBlacklisted = FALSE;

    /* Key descriptors: for each key, Device list (Key Device Descriptor)
       is a list of one, unique to the key. Same for the Key ID Lookup
       Descriptor. All keys share the same Key Usage Descriptor
       (list of frame types). Each key is unique, but in this example the
       value is based on the key number for simplicity */
    asKeyDescriptor[0].psKeyDeviceList = &asKeyDeviceList[0];
    asKeyDescriptor[0].u8KeyDeviceListEntries = 1;
    asKeyDescriptor[0].psKeyIdLookupDescriptor = &asKeyIdLookupDescriptor[0];
    asKeyDescriptor[0].u8KeyIdLookupEntries = 1;
    asKeyDescriptor[0].psKeyUsageList = asKeyUsageNwk;
    asKeyDescriptor[0].u8KeyUsageListEntries = TOTAL_KEY_USAGE_ENTRIES;
    asKeyDescriptor[0].au32SymmetricKey[0] = 0x01020304;
    asKeyDescriptor[0].au32SymmetricKey[1] = 0x05060708;
    asKeyDescriptor[0].au32SymmetricKey[2] = 0x090a0b0c;
    asKeyDescriptor[0].au32SymmetricKey[3] = 0;

    /* Device Descriptor table is mostly concerned with addresses. These
       will be filled in properly when each device joins */
    asDeviceDescriptor[0].u16PanId = 0;
    asDeviceDescriptor[0].u16Short = 0;
    asDeviceDescriptor[0].sExt.u32L = 0;
    asDeviceDescriptor[0].sExt.u32H =  0;
    asDeviceDescriptor[0].u32FrameCounter = 0;
    asDeviceDescriptor[0].bExempt = FALSE;

    /* Configure the MAC... enable security... */
    vAppApiSetSecurityMode(MAC_SECURITY_2006);
    s_psMacPib->bMacSecurityEnabled = TRUE;

    /* ... set the default lookup key (same as the value in the Key ID Lookup
       Descriptors) ... */
    memcpy(s_psMacPib->au8MacDefaultKeySource, au8DefaultKeySource, 8);

    /* ... set the Key Descriptor table... */
    s_psMacPib->psMacKeyTable = asKeyDescriptor;
    s_psMacPib->u8MacKeyTableEntries = TOTAL_SECURITY_KEYS;

    /* ... set the Security Level Descriptor table... */
    s_psMacPib->psMacSecurityLevelTable = asSecurityLevelDescriptor;
    s_psMacPib->u8MacSecuirtyLevelTableEntries = TOTAL_LEVEL_DESCRIPTORS;

    /* ... and set the Device Descriptor table */
    s_psMacPib->psMacDeviceTable = asDeviceDescriptor;
    s_psMacPib->u8MacDeviceTableEntries = TOTAL_SEC_ENTRIES;
#endif
}

/****************************************************************************
 *
 * NAME: vInitEndpoint
 *
 * DESCRIPTION:
 * Initialises software structures and variables.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vInitEndpoint(void)
{

    /* Set defaults for software */
    sDemoData.sTransceiver.u8CurrentTxHandle = 0;
    sDemoData.sControls.u8Switch = 0;
    sDemoData.sControls.u8LightAlarmLevel = 0;
    sDemoData.sSensors.u8TempResult = 0;
    sDemoData.sSensors.u8HtsResult = 0;
    sDemoData.sSensors.u8AlsResult = 0;
    sDemoData.sSystem.eState = E_STATE_OFF;
    sDemoData.sSystem.u16ShortAddr = 0xffff;
    sDemoData.sSystem.u8ThisNode = 0;

    /* Set light sensor values to 'wrong' ends of range, so the first time
       a value is read they will get updated */
    sDemoData.sLightSensor.u16Hi = 0;
    sDemoData.sLightSensor.u16Lo = 65535;
}

/****************************************************************************
 *
 * NAME: vStartScan
 *
 * DESCRIPTION:
 * Sends an MLME request to the 802.15.4 to start an actve scan. If the
 * returned confirmation is not 'deferred', a fatal error is assumed.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vStartScan(uint32 u32ChannelsToScan)
{
    MAC_MlmeReqRsp_s  sMlmeReqRsp;
    MAC_MlmeSyncCfm_s sMlmeSyncCfm;

    sDemoData.sSystem.eState = E_STATE_SCANNING;

    /* Request scan */
    sMlmeReqRsp.u8Type = MAC_MLME_REQ_SCAN;
    sMlmeReqRsp.u8ParamLength = sizeof(MAC_MlmeReqScan_s);
    sMlmeReqRsp.uParam.sReqScan.u8ScanType = MAC_MLME_SCAN_TYPE_ACTIVE;
    sMlmeReqRsp.uParam.sReqScan.u32ScanChannels = u32ChannelsToScan;
    sMlmeReqRsp.uParam.sReqScan.u8ScanDuration = 4;
    vAppApiMlmeRequest(&sMlmeReqRsp, &sMlmeSyncCfm);

    /* Check immediate response */
    if (sMlmeSyncCfm.u8Status != MAC_MLME_CFM_DEFERRED)
    {
        /* Unexpected result: scan request should result in a deferred
           confirmation (i.e. we will receive it later) */
        vDisplayError("Scan request returned error",
                      ((uint32)sMlmeSyncCfm.u8Status) << 8
                      | (uint32)sMlmeSyncCfm.uParam.sCfmScan.u8Status);
    }
}

/****************************************************************************
 *
 * NAME: vStartSync
 *
 * DESCRIPTION:
 * Sends an MLME request to the 802.15.4 to start synchronisation on the
 * channel determined by a previous scan. There is no confirmation for this.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vStartSync(void)
{
    MAC_MlmeReqRsp_s  sMlmeReqRsp;
    MAC_MlmeSyncCfm_s sMlmeSyncCfm;

    sDemoData.sSystem.eState = E_STATE_SYNCING;

    /* Set the coordinator source address before syncing to avoid PAN conflict
       indication */
    s_psMacPib->u16CoordShortAddr = DEMO_COORD_ADDR;

    /* Create sync request on channel selected during scan */
    sMlmeReqRsp.u8Type = MAC_MLME_REQ_SYNC;
    sMlmeReqRsp.u8ParamLength = sizeof(MAC_MlmeReqSync_s);
    sMlmeReqRsp.uParam.sReqSync.u8Channel = sDemoData.sSystem.u8Channel;
    sMlmeReqRsp.uParam.sReqSync.u8TrackBeacon = TRUE;

    /* Put in sync request. There is no deferred confirm for this, we just get
       a SYNC-LOSS later if it didn't work */
    vAppApiMlmeRequest(&sMlmeReqRsp, &sMlmeSyncCfm);
}

/****************************************************************************
 *
 * NAME: vStartAssociate
 *
 * DESCRIPTION:
 * Sends an MLME request to the 802.15.4 to request an association with the
 * coordinator found during a previous scan. If the  returned confirmation is
 * not 'deferred', a fatal error is assumed.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vStartAssociate(void)
{
    MAC_MlmeReqRsp_s  sMlmeReqRsp;
    MAC_MlmeSyncCfm_s sMlmeSyncCfm;

    sDemoData.sSystem.eState = E_STATE_ASSOCIATING;

    /* Create associate request. We know short address and PAN ID of
       coordinator as this is preset and we have checked that received
       beacon matched this */
    sMlmeReqRsp.u8Type = MAC_MLME_REQ_ASSOCIATE;
    sMlmeReqRsp.u8ParamLength = sizeof(MAC_MlmeReqAssociate_s);
    sMlmeReqRsp.uParam.sReqAssociate.u8LogicalChan = sDemoData.sSystem.u8Channel;
    sMlmeReqRsp.uParam.sReqAssociate.u8Capability = 0x80; /* We want short address, other features off */
    sMlmeReqRsp.uParam.sReqAssociate.u8SecurityEnable = FALSE;
    sMlmeReqRsp.uParam.sReqAssociate.sSecurityData.u8SecurityLevel = 0;
    sMlmeReqRsp.uParam.sReqAssociate.sCoord.u8AddrMode = 2;
    sMlmeReqRsp.uParam.sReqAssociate.sCoord.u16PanId = DEMO_PAN_ID;
    sMlmeReqRsp.uParam.sReqAssociate.sCoord.uAddr.u16Short = DEMO_COORD_ADDR;

    /* Put in associate request and check immediate confirm. Should be
       deferred, in which case response is handled by event handler */
    vAppApiMlmeRequest(&sMlmeReqRsp, &sMlmeSyncCfm);
    if (sMlmeSyncCfm.u8Status != MAC_MLME_CFM_DEFERRED)
    {
        /* Unexpected result */
        vDisplayError("Associate request returned error",
                      ((uint32)sMlmeSyncCfm.u8Status) << 8
                      | (uint32)sMlmeSyncCfm.uParam.sCfmAssociate.u8Status);
    }
}

/****************************************************************************
 *
 * NAME: vProcessInterrupts
 *
 * DESCRIPTION:
 * Loops around checking for upward events from the Application Queue API.
 * To save power, the CPU is set to doze at the start of the loop. If an event
 * is generated, the CPU is woken up and processes it, then the code here is
 * allowed to continue.
 *
 * For each type of event, the queue is checked and, if there is something to
 * process, the appropriate handler is called. The buffer is then returned to
 * the Application Queue API. This continues until all events have been
 * processed.
 *
 * Note that the buttons are configured to generate interrupts when they are
 * pressed.
 *
 * RETURNS:
 * Never returns
 *
 ****************************************************************************/
PRIVATE void vProcessInterrupts(void)
{
    MAC_MlmeDcfmInd_s *psMlmeInd;
    MAC_McpsDcfmInd_s *psMcpsInd;
    AppQApiHwInd_s    *psAHI_Ind;

    /* Wait for events */
    while (1)
    {

        /* Doze CPU: it will wake when interrupt occurs, then process
           interrupts, then this code will be able to process queue */
        vAHI_CpuDoze();

        /* Check for anything on the MCPS upward queue */
        do
        {
            psMcpsInd = psAppQApiReadMcpsInd();
            if (psMcpsInd != NULL)
            {
                vProcessIncomingMcps(psMcpsInd);
                vAppQApiReturnMcpsIndBuffer(psMcpsInd);
            }
        } while (psMcpsInd != NULL);

        /* Check for anything on the MLME upward queue */
        do
        {
            psMlmeInd = psAppQApiReadMlmeInd();
            if (psMlmeInd != NULL)
            {
                vProcessIncomingMlme(psMlmeInd);
                vAppQApiReturnMlmeIndBuffer(psMlmeInd);
            }
        } while (psMlmeInd != NULL);

        /* Check for anything on the AHI upward queue */
        do
        {
            psAHI_Ind = psAppQApiReadHwInd();
            if (psAHI_Ind != NULL)
            {
                /* We don't bother to pass the queue data as we are only
                   expecting key press interrupts and, if we get anything else,
                   reading the keys won't have any unfortunate side effects */
                vAppQApiReturnHwIndBuffer(psAHI_Ind);
            }
        } while (psAHI_Ind != NULL);

    }
}

/****************************************************************************
 *
 * NAME: vProcessIncomingMcps
 *
 * DESCRIPTION:
 * Deals with any incoming MCPS data events. Only deferred confirmations of
 * frame transmission requests are specifically handled, to move the state
 * machine forward.
 *
 * PARAMETERS: Name        RW  Usage
 *             psMcpsInd   R   Pointer to structure containing MCPS event
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vProcessIncomingMcps(MAC_McpsDcfmInd_s *psMcpsInd)
{
    /* Process MCPS indication by checking if it is a confirmation of our
       outgoing frame */
    if ((psMcpsInd->u8Type == MAC_MCPS_DCFM_DATA)
        && (sDemoData.sSystem.eState == E_STATE_TX_DATA))
    {
        if (psMcpsInd->uParam.sDcfmData.u8Handle == sDemoData.sTransceiver.u8CurrentTxHandle)
        {
            /* Increment handle for next time. Increment failures */
            sDemoData.sTransceiver.u8CurrentTxHandle++;

            /* Start to read sensors. This takes a while but rather than
               wait for an interrupt we just poll and, once finished, move back
               to the running state to wait for the next beacon. Not a power
               saving solution! */
            sDemoData.sSystem.eState = E_STATE_READ_SENSORS;
            vProcessRead();
            sDemoData.sSystem.eState = E_STATE_RUNNING;
        }
    }
}

/****************************************************************************
 *
 * NAME: vProcessIncomingMlme
 *
 * DESCRIPTION:
 * Deals with any incoming MCPS events. There are 4 events that are handled:
 *
 * 1. Scan deferred confirmation
 *    If the scan saw a beacon from the expected coordinator, the stack is
 *    asked to start synchronisation. Otherwise, it tries to scan again.
 *
 * 2. Associate deferred confirmation
 *    If the association was successful, the short address is stored and the
 *    device enters the 'running' state, in which it will respond to beacons
 *    by sending back a frame of sensor information. If the association was
 *    not successful, it is attempted again.
 *
 * 3. Becon notify indication
 *    If synchronising, this is used to indicate that synchronisation is
 *    complete and association is atempted. If in 'running' state, it is
 *    used to trigger the sending of a frame containing sensor information
 *    to the coordinator.
 *
 * 4. Sync loss indication
 *    Device starts another scan.
 *
 * PARAMETERS: Name        RW  Usage
 *             psMlmeInd   R   Pointer to structure containing MLME event
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vProcessIncomingMlme(MAC_MlmeDcfmInd_s *psMlmeInd)
{
    MAC_PanDescr_s *psPanDesc;
    int i;

    /* We respond to several MLME indications and confirmations, depending
       on mode */
    switch (psMlmeInd->u8Type)
    {
    case MAC_MLME_DCFM_SCAN:

        /* Only respond to this if scanning */
        if (sDemoData.sSystem.eState == E_STATE_SCANNING)
        {

            /* Make sure it is what we're after */
            switch (psMlmeInd->uParam.sDcfmScan.u8Status)
            {
            case MAC_ENUM_SUCCESS:
                if (psMlmeInd->uParam.sDcfmScan.u8ScanType == MAC_MLME_SCAN_TYPE_ACTIVE)
                {
                    /* Determine which, if any, network contains demo coordinator.
                       Algorithm for determining which network to connect to is
                       beyond the scope of 802.15.4, and we use a simple approach
                       of matching the required PAN ID and short address, both of
                       which we already know */
                    i = 0;
                    while (i < psMlmeInd->uParam.sDcfmScan.u8ResultListSize)
                    {
                        psPanDesc = &psMlmeInd->uParam.sDcfmScan.uList.asPanDescr[i];

                        if ((psPanDesc->sCoord.u16PanId == DEMO_PAN_ID)
                            && (psPanDesc->sCoord.u8AddrMode == 2)
                            && (psPanDesc->sCoord.uAddr.u16Short == DEMO_COORD_ADDR))
                        {
                            /* Matched so start to synchronise and associate */
                            sDemoData.sSystem.u8Channel = psPanDesc->u8LogicalChan;
                            vStartSync();
                            return;
                        }
                        i++;
                    }

                    if (psMlmeInd->uParam.sDcfmScan.u32UnscannedChannels != 0)
                    {
                        /* More channels to try */
                        vStartScan(psMlmeInd->uParam.sDcfmScan.u32UnscannedChannels);
                    }
                    else
                    {
                        /* Failed to find coordinator: keep trying */
                         vStartScan(DEMO_CHANNEL_BITMAP);
                     }
                }
                break;

            case MAC_ENUM_NO_BEACON:
                /* Failed to find coordinator: keep trying */
                vStartScan(DEMO_CHANNEL_BITMAP);
                break;

            default:
                break;
            }
        }
        break;


    case MAC_MLME_DCFM_ASSOCIATE:
        /* Only respond if in associating mode */
        switch (sDemoData.sSystem.eState)
        {
        case E_STATE_ASSOCIATING:
            if (psMlmeInd->uParam.sDcfmAssociate.u8Status == MAC_ENUM_SUCCESS)
            {
                /* Store short address */
                sDemoData.sSystem.u16ShortAddr = psMlmeInd->uParam.sDcfmAssociate.u16AssocShortAddr;

                /* Node is a value between 0 and the number of endpoints - 1.
                   It is determined by the short address */
                sDemoData.sSystem.u8ThisNode = sDemoData.sSystem.u16ShortAddr - DEMO_ENDPOINT_ADDR_BASE;

#ifdef USE_2006_SECURITY
                /* Can now update security data based on recieved information */
                asKeyIdLookupDescriptor[0].au8LookupData[8]
                    = sDemoData.sSystem.u8ThisNode + 1; /* Key index */

                /* Key is partly derived from this as well */
                asKeyDescriptor[0].au32SymmetricKey[3] = 0x0d0e0f00 + sDemoData.sSystem.u8ThisNode;

                /* Can fill in coordinator's details: extended address will
                   have been set by MAC from received Association Response
                   frame */
                asDeviceDescriptor[0].u16PanId = DEMO_PAN_ID;
                asDeviceDescriptor[0].u16Short = DEMO_COORD_ADDR;
                asDeviceDescriptor[0].sExt.u32L = s_psMacPib->sCoordExtAddr.u32L;
                asDeviceDescriptor[0].sExt.u32H =  s_psMacPib->sCoordExtAddr.u32H;
                asDeviceDescriptor[0].u32FrameCounter = 0;
#endif

                /* We are now in the running state */
                sDemoData.sSystem.eState = E_STATE_RUNNING;
            }
            else
            {
                /* Try, try again */
                vStartAssociate();
            }
            break;

        case E_STATE_ASSOC_TO_SCAN:
            vStartScan(DEMO_CHANNEL_BITMAP);
            break;

        default:
            break;
        }
        break;

    case MAC_MLME_IND_BEACON_NOTIFY:
        /* Beacon has arrived. If in sync'ing mode, should only see this
           once synchronisation has been achieved. If running, use this
           to time reading of sensors. In other states, ignore it */
        switch (sDemoData.sSystem.eState)
        {
        case E_STATE_SYNCING:
            /* Can now associate */
            vStartAssociate();
            break;

        case E_STATE_RUNNING:
            /* Process received data in beacon payload */
            vProcessRxBeacon(psMlmeInd);
            break;

        default:
            break;
        }
        break;

    case MAC_MLME_IND_SYNC_LOSS:
        /* Lost sync with coordinator so try to find coordinator again */
        if (sDemoData.sSystem.eState != E_STATE_ASSOCIATING)
        {
            vStartScan(DEMO_CHANNEL_BITMAP);
        }
        else
        {
            sDemoData.sSystem.eState = E_STATE_ASSOC_TO_SCAN;
        }
        break;

    default:
        break;
    }
}


/****************************************************************************
 *
 * NAME: vProcessRead
 *
 * DESCRIPTION:
 * Gets the current readings from each sensor. If the light level causes the
 * low light alarm to be triggered, an LED is illuminated.
 *
 * RETURNS:
 * void
 *
 * NOTES:
 * This is not an efficient way to read the sensors as much time is wasted
 * waiting for the humidity and temperature sensor to complete. The sensor
 * pulls a DIO line low when it is ready, and this could be used to generate
 * an interrupt to indicate when data is ready to be read.
 *
 ****************************************************************************/
PRIVATE void vProcessRead(void)
{
    uint16 u16LightSensor;
    uint16 u16Diff;

    /* Read light level, adjust to range 0-6. This sensor automatically starts
       a new conversion afterwards so there is no need for a 'start read' */
    u16LightSensor = u16ALSreadChannelResult();

    /* Adjust the high and low values if necessary, and obtain the
       difference between them */
    if (sDemoData.sLightSensor.u16Hi < u16LightSensor)
    {
        sDemoData.sLightSensor.u16Hi = u16LightSensor;
    }

    if (sDemoData.sLightSensor.u16Lo > u16LightSensor)
    {
        sDemoData.sLightSensor.u16Lo = u16LightSensor;
    }

    u16Diff = sDemoData.sLightSensor.u16Hi - sDemoData.sLightSensor.u16Lo;

    /* Work out the current value as a value between 0 and 6 within the
       range of values that have been seen previously */
    if (u16Diff)
    {
        sDemoData.sSensors.u8AlsResult = (uint8)(((uint32)(u16LightSensor - sDemoData.sLightSensor.u16Lo) * 6) / (uint32)u16Diff);
    }
    else
    {
        sDemoData.sSensors.u8AlsResult = 0;
    }

    /* Set LED 1 based on light level */
    if ((sDemoData.sSensors.u8AlsResult <= sDemoData.sControls.u8LightAlarmLevel)
        && (sDemoData.sControls.u8LightAlarmLevel < 7))
    {

        (void)bRGB_LED_On();
    }
    else
    {
        (void)bRGB_LED_Off();
    }

    /* Read temperature, 0-52 are acceptable. Polls until result received */
    vHTSstartReadTemp();
    sDemoData.sSensors.u8TempResult = u8FindMin((uint8)u16HTSreadTempResult(), 52);

    /* Read humidity, 0-104 are acceptable. Polls until result received */
    vHTSstartReadHumidity();
    sDemoData.sSensors.u8HtsResult = u8FindMin((uint8)u16HTSreadHumidityResult(), 104);
}

/****************************************************************************
 *
 * NAME: vProcessTxBlock
 *
 * DESCRIPTION:
 * Creates an MCPS request to transmit a frame and passes it to the 802.15.4
 * stack. The payload contains sensor information.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vProcessTxBlock(void)
{
    MAC_McpsReqRsp_s  sMcpsReqRsp;
    MAC_McpsSyncCfm_s sMcpsSyncCfm;
    uint8 *pu8Payload;

    /* Create frame transmission request */
    sMcpsReqRsp.u8Type = MAC_MCPS_REQ_DATA;
    sMcpsReqRsp.u8ParamLength = sizeof(MAC_McpsReqData_s);
    /* Set handle so we can match confirmation to request */
    sMcpsReqRsp.uParam.sReqData.u8Handle = sDemoData.sTransceiver.u8CurrentTxHandle;
    /* Use short address for source */
    sMcpsReqRsp.uParam.sReqData.sFrame.sSrcAddr.u8AddrMode = 2;
    sMcpsReqRsp.uParam.sReqData.sFrame.sSrcAddr.u16PanId = DEMO_PAN_ID;
    sMcpsReqRsp.uParam.sReqData.sFrame.sSrcAddr.uAddr.u16Short = sDemoData.sSystem.u16ShortAddr;
    /* Use short address for destination */
    sMcpsReqRsp.uParam.sReqData.sFrame.sDstAddr.u8AddrMode = 2;
    sMcpsReqRsp.uParam.sReqData.sFrame.sDstAddr.u16PanId = DEMO_PAN_ID;
    sMcpsReqRsp.uParam.sReqData.sFrame.sDstAddr.uAddr.u16Short = DEMO_COORD_ADDR;
    /* Frame requires ack but not security, indirect transmit or GTS */
    sMcpsReqRsp.uParam.sReqData.sFrame.u8TxOptions = MAC_TX_OPTION_ACK;
    /* Payload is 8 bytes (room for expansion) and contains:
         ID byte so coordinator knows this is sensor information
         Previous beacon sequence number (for debug, can tell if beacon missed)
         Virtual switch value
         Sensor values: temp, humidity, light
    */
    sMcpsReqRsp.uParam.sReqData.sFrame.u8SduLength = 8;
    pu8Payload = sMcpsReqRsp.uParam.sReqData.sFrame.au8Sdu;
    pu8Payload[0] = DEMO_ENDPOINT_IDENTIFIER;
    pu8Payload[1] = sDemoData.sTransceiver.u8PrevRxBsn;
    pu8Payload[2] = sDemoData.sControls.u8Switch;
    pu8Payload[3] = sDemoData.sSensors.u8TempResult;
    pu8Payload[4] = sDemoData.sSensors.u8HtsResult;
    pu8Payload[5] = sDemoData.sSensors.u8AlsResult;
    pu8Payload[6] = 0;
    pu8Payload[7] = 0;

#ifdef USE_2006_SECURITY
    /* Specify security information. Key ID is mode 1 (Key determined by 1 octet key index and macDefaultKeySource) */
    sMcpsReqRsp.uParam.sReqData.sFrame.sSecurityData.u8SecurityLevel = 5;
    sMcpsReqRsp.uParam.sReqData.sFrame.sSecurityData.u8KeyIdMode = MAC_SECURITY_KEYID_MODE_1;
    sMcpsReqRsp.uParam.sReqData.sFrame.sSecurityData.u8KeyIndex = sDemoData.sSystem.u8ThisNode + 1;
#else
    sMcpsReqRsp.uParam.sReqData.sFrame.sSecurityData.u8SecurityLevel = 0;
#endif

    /* Request transmit */
    vAppApiMcpsRequest(&sMcpsReqRsp, &sMcpsSyncCfm);
}

/****************************************************************************
 *
 * NAME: vProcessRxBeacon
 *
 * DESCRIPTION:
 * Processes the beacon payload received from the coordinator. Checks that
 * it is valid for the demo (8 bytes, with an identifier in the first byte)
 * then stores data passed from coordinator (light level alarm, remote switch
 * value). After processing reception, calls vProcessTxBlock to send a frame
 * back.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  psMlmeInd       R   Pointer to MLME event structure
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vProcessRxBeacon(MAC_MlmeDcfmInd_s *psMlmeInd)
{
    uint8 *pu8Payload;
    uint8 u8RxVal;
    uint8 u8Bsn;

    /* There has been a beacon notify, so get configuration from beacon
       payload and re-synchronise timing */
    if ((psMlmeInd->uParam.sIndBeacon.u8SDUlength != 8)
        || (psMlmeInd->uParam.sIndBeacon.u8SDU[0] != DEMO_BEACON_IDENTIFIER))
    {
        /* Not the payload we were looking for */
        return;
    }

    pu8Payload = psMlmeInd->uParam.sIndBeacon.u8SDU;

    /* Store beacon sequence number */
    u8Bsn = psMlmeInd->uParam.sIndBeacon.u8BSN;

    /* Read local 'relay' control and light low alarm level, as specified by
       the user at the coordinator. Relay control is emulated by illuminating
       an LED on the board */
    u8RxVal = pu8Payload[1 + sDemoData.sSystem.u8ThisNode];
    if(u8RxVal & 0x01)
    {
        bWhite_LED_On();
    }
    else
    {
        bWhite_LED_Off();
    }

    sDemoData.sControls.u8LightAlarmLevel = (u8RxVal >> 1) & 0x7;

    if (((uint8)(u8Bsn - sDemoData.sTransceiver.u8PrevRxBsn)) > 7)
    {
        sDemoData.sTransceiver.u8PrevRxBsn = u8Bsn;

        /* Respond to beacon by sending a frame and changing state to wait for
           it to complete */
        vProcessTxBlock();
        sDemoData.sSystem.eState = E_STATE_TX_DATA;
    }
}

/****************************************************************************
 *
 * NAME: u8FindMin
 *
 * DESCRIPTION:
 * Returns the smallest of two values.
 *
 * PARAMETERS:      Name    RW  Usage
 *                  u8Val1  R   First value to compare
 *                  u8Val2  R   Second value to compare
 *
 * RETURNS:
 * uint8, lowest of two input values
 *
 ****************************************************************************/
PRIVATE uint8 u8FindMin(uint8 u8Val1, uint8 u8Val2)
{
    if (u8Val1 < u8Val2)
    {
        return u8Val1;
    }
    return u8Val2;
}

/****************************************************************************
 *
 * NAME: vDisplayError
 *
 * DESCRIPTION:
 * Used to display fatal errors, by sending them to UART0 if this approach
 * has been enabled at compile time.
 *
 * Sits in an endless loop afterwards.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  pcErrorMessage  R   Message to display
 *                  u32Data         R   Data to display
 *
 * RETURNS:
 * void, never returns
 *
 ****************************************************************************/
PRIVATE void vDisplayError(char *pcErrorMessage, uint32 u32Data)
{
    /* Fatal error has occurred, so loop indefinitely */
    while (1);
}


/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
