/*****************************************************************************
 *
 * MODULE:              Production Test PER Test Master
 *
 * COMPONENT:           $RCSfile: ProdTestPerMaster.c,v $
 *
 * VERSION:             $Name:  $
 *
 * REVISION:            $Revision: 1.5 $
 *
 * DATED:               $Date: 2007/07/18 07:57:18 $
 *
 * STATUS:              $State: Exp $
 *
 * AUTHOR:
 *
 * DESCRIPTION:
 *
 * CHANGE HISTORY:
 *
 * $Log: ProdTestPerMaster.c,v $
 * Revision 1.5  2007/07/18 07:57:18
 * Added GDB Hooks
 *
 * Revision 1.4  2007/02/19 14:07:29
 * Updated to use BoardLib for peripheral drivers
 *
 * Revision 1.3  2006/08/31 11:49:32
 * Modified title block
 *
 * Revision 1.2  2006/08/30 11:00:59
 * Initial channel changed
 *
 * Revision 1.1  2006/08/29 13:06:21
 * First release
 *
 */
/****************************************************************************
*
* This software is owned by NXP B.V. and/or its supplier and is protected
* under applicable copyright laws. All rights are reserved. We grant You,
* and any third parties, a license to use this software solely and
* exclusively on NXP products [NXP Microcontrollers such as JN5148, JN5142, JN5139].
* You, and any third parties must reproduce the copyright and warranty notice
* and any other legend of ownership on each copy or partial copy of the
* software.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.

* Copyright NXP B.V. 2012. All rights reserved
*/

/****************************************************************************/
/****************************************************************************/
/***        Include Files                                                 ***/
/****************************************************************************/

#include <jendefs.h>
#include <AppHardwareAPI.h>
#include <JPT.h>

#include "Button.h"
#include "ALSdriver.h"
#include "HTSdriver.h"
#include "LcdDriver.h"
#include "UartBuffered.h"
#include "Printf.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

#define UART_TO_PC			E_AHI_UART_0			/* Uart to PC 			*/
#define BAUD_RATE			E_AHI_UART_RATE_38400 	/* Baud rate to use 	*/

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

typedef enum {
    RV_CHANGE_CHANNEL,
    RV_CHANGE_MODE,
    RV_CHANGE_RETRIES,
    RV_STOPSTART,
    RV_IDLE
} teReturnValue;

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

PRIVATE void vUpdateDisplays(bool_t bUpdateLcd, bool_t bUpdateSerialConsole);
PRIVATE uint8 u8UpdateUI(void);
PRIVATE void vDrawLcdDisplay(uint32 u32Xoffset);
PRIVATE void vPutC_LCD(uint8 u8Data);
PRIVATE void vPutC(uint8 u8Data);

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

uint8  u8KeysDown = 0;
bool_t bSlaveOk = TRUE;
uint8 u8RunMode;

tsJPT_SSPT_MasterState sMasterData;				/* holds master state data */

tsUartFifo sFifo;								/* TX & RX Buffers for UART */

/* scratchpad area and cursor info for building lcd contents */
uint8 au8LcdShadow[8][30];
uint8 u8LcdX = 0, u8LcdY = 0;

#define PROD_TEST

#ifdef PROD_TEST
bool_t bProdTestDone = FALSE;
#endif

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

PUBLIC void AppColdStart(void)
{

	char acCommand[4] = "123";

	uint8 u8Result;
	uint32 i;

	/* Disable watchdog if enabled by default */
	#ifdef WATCHDOG_ENABLED
	vAHI_WatchdogStop();
	#endif

	/* Initialise the hardware API */
	u32AHI_Init();

	/* Initialise the production test API */
	u32JPT_Init();

	/* Initialise the radio */
	bJPT_RadioInit(E_JPT_MODE_HIPOWER);

    vButtonInitFfd();
    vLcdResetDefault();


    /* Set up UART 0: enable interrupts for RX and TX */
	vUartInit(UART_TO_PC, BAUD_RATE, &sFifo);	/* uart for console */
	vInitPrintf((void*)vPutC);

	/* Initialise site survey PER test master */
	vJPT_SSPT_MasterInit();

#ifdef PROD_TEST
	vPrintf("DCTS");

    /* Set up peripheral hardware */
    vALSreset();
    vHTSreset();

    /* Start ALS now: it automatically keeps re-sampling after this */
    vALSstartReadChannel(0);

    /* Read temperature, 0-52 are acceptable */
    vHTSstartReadTemp();
	for(i = 0; i < 20000; i++);
    u8Result = (uint8)u16HTSreadTempResult();
    if(u8Result > 0 && u8Result < 52){
		vPrintf("DCTP");
	}

    /* Read humidity, 0-104 are acceptable */
    vHTSstartReadHumidity();
	for(i = 0; i < 20000; i++);
    u8Result = (uint8)u16HTSreadHumidityResult();
    if(u8Result > 0 && u8Result < 104){
		vPrintf("DCHP");
	}

    /* Read light level, adjust to range 0-6 in a slightly non-linear way */
    u8Result = (uint8)u16ALSreadChannelResult() >> 6;
	if(u8Result <= 6){
		vPrintf("DCLP");
	}

	vPrintf("DCSC");

#endif


	/* Locate the slave */
	sMasterData.u8Mode = E_JPT_SSPT_MODE_LOCATE;
	u8RunMode = sMasterData.u8Mode;
	sMasterData.u8Channel = 14;
	sMasterData.u8Retries = 3;
	vUpdateDisplays(TRUE, FALSE);
	bSlaveOk = bJPT_SSPT_MasterSetState(&sMasterData);
	vUpdateDisplays(TRUE, FALSE);

#ifdef PROD_TEST
	if(!bSlaveOk){
		vPrintf("DCF0");
	}
#endif


	/* Start the PER test running in Ack's mode */
	sMasterData.u8Mode = E_JPT_SSPT_MODE_RUNNING_ACKS;
	u8RunMode = sMasterData.u8Mode;
	bSlaveOk = bJPT_SSPT_MasterSetState(&sMasterData);

    while(1){

        /* Run PER test */
        switch(u8UpdateUI()){

        case RV_CHANGE_MODE:
			if(u8RunMode == E_JPT_SSPT_MODE_RUNNING_NO_ACKS){
				u8RunMode = E_JPT_SSPT_MODE_RUNNING_ACKS;
			} else {
				u8RunMode = E_JPT_SSPT_MODE_RUNNING_NO_ACKS;
			}
			if(sMasterData.u8Mode != E_JPT_SSPT_MODE_STOPPED){
				sMasterData.u8Mode = u8RunMode;
			}
            break;

        case RV_CHANGE_CHANNEL:
			sMasterData.u8Channel++;
			if(sMasterData.u8Channel > 26){
				sMasterData.u8Channel = 11;
			}
           	break;

		case RV_STOPSTART:
			if(sMasterData.u8Mode != E_JPT_SSPT_MODE_STOPPED){
				sMasterData.u8Mode = E_JPT_SSPT_MODE_STOPPED;
			} else {
				sMasterData.u8Mode = u8RunMode;
			}
			break;

		case RV_CHANGE_RETRIES:
			if(++sMasterData.u8Retries > 7){
				sMasterData.u8Retries = 0;
			}
			break;

        }

#ifdef PROD_TEST

		while(bUartRxDataAvailable(UART_TO_PC)){

			acCommand[0] = acCommand[1];
			acCommand[1] = acCommand[2];
			acCommand[2] = u8UartRead(UART_TO_PC);

			if(acCommand[0] == 'D' && acCommand[1] == 'S'){
				sMasterData.u8Channel = acCommand[2] - 'A' + 1;
				if(sMasterData.u8Channel > 26) sMasterData.u8Channel = 26;
				if(sMasterData.u8Channel < 11) sMasterData.u8Channel = 11;
			}
		}

		bSlaveOk = bJPT_SSPT_MasterSetState(&sMasterData);

#endif

    }
}


PUBLIC void AppWarmStart(void)
{
    AppColdStart();
}

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME:       u8UpdateUI
 *
 * DESCRIPTION:
 * Updates the user interface (Updates LCD and console displays and checks
 * the buttons for any presses. Returns when a button has been pressed.
 *
 * PARAMETERS: Name     RW  Usage
 * none
 *
 * RETURNS:
 * uint8	Keypress value
 *
 ****************************************************************************/

PRIVATE uint8 u8UpdateUI(void)
{

    uint8  u8NewKeysDown;
    bool_t bOuterLoopFinished = FALSE;
    int    i;
    uint8  u8Count = 0;
    uint8  u8RetVal = RV_IDLE;

	do
	{

		/* get updated per test data */
		vJPT_SSPT_MasterGetState(&sMasterData);

		if (u8Count++ < 20) {
			vUpdateDisplays(TRUE, FALSE);
		} else {
			u8Count = 0;
			vUpdateDisplays(TRUE, TRUE);
		}

		if(sMasterData.u32Total > 50000){
			sMasterData.u8Mode = E_JPT_SSPT_MODE_RESTART;
			bSlaveOk = bJPT_SSPT_MasterSetState(&sMasterData);
		}

		/* Short delay for key debounce */
		for (i = 0; i < 100000; i++);

		/* Process key press */
		u8NewKeysDown = u8ButtonReadFfd();

		if (u8NewKeysDown != u8KeysDown){

			u8KeysDown = u8NewKeysDown;

			switch (u8KeysDown) {

			case BUTTON_0_MASK:
				/* Change channel */
				bOuterLoopFinished = TRUE;
				u8RetVal = RV_CHANGE_CHANNEL;
				break;

			case BUTTON_1_MASK:
				/* Stop/Start */
				bOuterLoopFinished = TRUE;
				u8RetVal = RV_STOPSTART;
				break;

			case BUTTON_2_MASK:
				/* Ack/No ack */
				bOuterLoopFinished = TRUE;
				u8RetVal = RV_CHANGE_MODE;
				break;

			case BUTTON_3_MASK:
				bOuterLoopFinished = TRUE;
				u8RetVal = RV_CHANGE_RETRIES;

			}
		}

#ifdef PROD_TEST

		if(bUartRxDataAvailable(UART_TO_PC)) bOuterLoopFinished = TRUE;
#endif
	} while (bOuterLoopFinished == FALSE);

    return u8RetVal;
}


/****************************************************************************
 *
 * NAME:       vUpdateDisplays
 *
 * DESCRIPTION:
 * Updates the displays generated on the LCD module and the serial console.
 *
 * PARAMETERS: Name     	RW  Usage
 *			bUpdateLcd		R	Set to TRUE to update LCD
 *			bUpdateSerial	R	Set to TRUE to update serial console
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/

PRIVATE void vUpdateDisplays(bool_t bUpdateLcd, bool_t bUpdateSerialConsole)
{

    char   acStart[] = "Start";
    char   acStop[] = "Stop ";

	char *pcStart;

    uint16 u16Per;
    uint16 u16Bad;

	/* Calculate new PER and CCA Fail values */
	if(sMasterData.u32Total == 0){
		u16Per = 0;
		u16Bad = 0;
	} else {
		u16Per = (uint16)(1000 - ((sMasterData.u32Seen * 1000) / sMasterData.u32Total));
		u16Bad = (uint16)((sMasterData.u32Errors * 1000) / sMasterData.u32Total);
	}

#ifdef PROD_TEST
	if(sMasterData.u32Total > 3000 && !bProdTestDone){
		bProdTestDone = TRUE;
		if(sMasterData.u32Seen == sMasterData.u32Total){
			vPrintf("DCAC");
		} else {
			vPrintf("DCF1");
		}
	}
#endif


	/*
	 * Figure out which mode we're in and point to
	 * relavent text for start/stop button
	 */
	if(sMasterData.u8Mode == E_JPT_SSPT_MODE_STOPPED){
		pcStart = acStart;
	} else {
		pcStart = acStop;
	}

	/* Update the LCD display content if required */
	if(bUpdateLcd){

		vInitPrintf((void*)vPutC_LCD);	/* Printf now writes to lcd shadow area */
		vLcdClear();

		switch(u8RunMode){

		case E_JPT_SSPT_MODE_LOCATE:
			vPrintf("Locating slave```");
			vDrawLcdDisplay(0);
			break;

		case E_JPT_SSPT_MODE_RUNNING_ACKS:
			/* write the static parts of the display */
			vPrintf("Channel\n"
					"Mode\n"
					"Seen\n"
					"Total\n"
					"PER %%\n"
					"CCA Fail %%\n"
					"Retries\n"
					"Chan  %s  Mode  Tries",pcStart);

			vDrawLcdDisplay(0);				/* write to lcd module */

			/* write the dynamic parts of the display */
			vPrintf("%d\nAck\n%d\n%d\n%d`%d\n%d`%d\n%d",sMasterData.u8Channel,
														 sMasterData.u32Seen,
														 sMasterData.u32Total,
														 u16Per / 10, u16Per % 10,
														 u16Bad / 10, u16Bad % 10,
														 sMasterData.u8Retries);

			vDrawLcdDisplay(60);			/* write to lcd module */
			break;

		case E_JPT_SSPT_MODE_RUNNING_NO_ACKS:
			/* write the static parts of the display */
			vPrintf("Channel\n"
					"Mode\n"
					"Seen\n"
					"Total\n"
					"PER %%\n"
					"\n"
					"\n"
					"Chan  %s  Mode  Tries",pcStart);

			vDrawLcdDisplay(0);				/* write to lcd module */

			/* write the dynamic parts of the display */
			vPrintf("%d\nNoAck\n%d\n%d\n%d`%d",sMasterData.u8Channel,
												sMasterData.u32Seen,
												sMasterData.u32Total,
												u16Per / 10, u16Per % 10);

			vDrawLcdDisplay(60);			/* write to lcd module */
			break;

		}

		if(bSlaveOk == FALSE){
			vLcdWriteText("Err", 0, 100);
		}

		vLcdRefreshAll();				/* display new data */

		vInitPrintf((void*)vPutC);		/* Printf now writes to uart console */

	}

	/* Update the serial console content if required */
#ifdef PROD_TEST
	if(bUpdateSerialConsole && bProdTestDone){
#else
	if(bUpdateSerialConsole){
#endif
		switch(u8RunMode){

		case E_JPT_SSPT_MODE_LOCATE:
			break;

		case E_JPT_SSPT_MODE_RUNNING_ACKS:
			/* write the static parts of the display */
			vPrintf("\fChannel      %d\n"
					"Mode         Ack\n"
					"Seen         %d\n"
					"Total        %d\n"
					"PER %%        %d.%d\n"
					"CCA Fail %%   %d.%d\n"
					"Retries      %d\n",sMasterData.u8Channel,
										sMasterData.u32Seen,
										sMasterData.u32Total,
										u16Per / 10, u16Per % 10,
										u16Bad / 10, u16Bad % 10,
										sMasterData.u8Retries);
			break;

		case E_JPT_SSPT_MODE_RUNNING_NO_ACKS:
			/* write the static parts of the display */
			vPrintf("\fChannel      %d\n"
					"Mode         No Ack\n"
					"Seen         %d\n"
					"Total        %d\n"
					"PER %%        %d.%d\n",sMasterData.u8Channel,
											sMasterData.u32Seen,
											sMasterData.u32Total,
											u16Per / 10, u16Per % 10);
			break;

		}

	}


}


/****************************************************************************
 *
 * NAME:       vDrawLcdDisplay
 *
 * DESCRIPTION:
 * Writes characters from the LCD scratchpad to the LCD driver
 *
 * PARAMETERS: Name     RW  Usage
 *			u32Xoffset	R	Offset in pixels from the left side of the
 *							display where content is to be written.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/

PRIVATE void vDrawLcdDisplay(uint32 u32Xoffset)
{
	int n;

	for(n = 0; n < 8; n++){
		vLcdWriteText(&au8LcdShadow[n][0], n, u32Xoffset);
		au8LcdShadow[n][0] = '\0';
	}

	u8LcdX = 0;
	u8LcdY = 0;

}


/****************************************************************************
 *
 * NAME:       vPutC_LCD
 *
 * DESCRIPTION:
 * Writes characters to the scratchpad area for constructing
 * the LCD content
 *
 * PARAMETERS: Name     RW  Usage
 *			u8Data		R	Character to write to LCD
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/

PRIVATE void vPutC_LCD(uint8 u8Data)
{

	switch(u8Data){

	case '\n':
		au8LcdShadow[u8LcdY][u8LcdX++] = '\0';
		u8LcdY++;
		break;

	case '\r':
		au8LcdShadow[u8LcdY][u8LcdX++] = '\0';
		u8LcdX = 0;
		break;

	default:
		au8LcdShadow[u8LcdY][u8LcdX++] = u8Data;
		au8LcdShadow[u8LcdY][u8LcdX] = '\0';
		break;

	}

}

/****************************************************************************
 *
 * NAME:       vPutC
 *
 * DESCRIPTION:
 * Writes characters to the UART connected to the PC
 *
 * PARAMETERS: Name     RW  Usage
 *			u8Data		R	Character to write to the UART
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/

PRIVATE void vPutC(uint8 u8Data)
{
	vUartWrite(UART_TO_PC, u8Data);
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
