/****************************************************************************/
/*
 * MODULE              JN-AN-1162 JenNet-IP Smart Home
 *
 * DESCRIPTION         OccupancyMonitor MIB Implementation
 */
/****************************************************************************/
/*
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2014. All rights reserved
 */
/****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
/* Standard includes */
#include <string.h>
/* SDK includes */
#include <jendefs.h>
/* Hardware includes */
#include <AppHardwareApi.h>
#include <PeripheralRegs.h>
/* Stack includes */
#include <Api.h>
#include <AppApi.h>
#include <JIP.h>
#include <6LP.h>
#include <AccessFunctions.h>
/* JenOS includes */
#include <dbg.h>
#include <dbg_uart.h>
#include <os.h>
#include <pdm.h>
/* Application device includes */
#include "DeviceDefs.h"
#include "Node.h"
#include "MibSensor.h"
#include "MibOccupancyMonitor.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE tsMibOccupancyMonitor *psMibOccupancyMonitor;  /* MIB data */

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME: MibOccupancyMonitor_vInit
 *
 * DESCRIPTION:
 * Initialises data
 *
 ****************************************************************************/
PUBLIC void MibOccupancyMonitor_vInit(thJIP_Mib        hMibOccupancyMonitorInit,
                         	   tsMibOccupancyMonitor *psMibOccupancyMonitorInit)
{
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_OCCUPANCY_MONITOR, "\n%sMibOccupancyMonitor_vInit() {%d}", acDebugIndent, sizeof(tsMibOccupancyMonitor));
    Node_vDebugIndent(DEBUG_MIB_OCCUPANCY_MONITOR);

    /* Valid data pointer ? */
    if (psMibOccupancyMonitorInit != (tsMibOccupancyMonitor *) NULL)
    {
        /* Take copy of pointer to data */
        psMibOccupancyMonitor = psMibOccupancyMonitorInit;
        /* Take a copy of the MIB handle */
        psMibOccupancyMonitor->hMib = hMibOccupancyMonitorInit;

        /* Load Dio mib data */
        (void) PDM_eLoadRecord(&psMibOccupancyMonitor->sDesc,
				  			   (uint16)(MIB_ID_OCCUPANCY_MONITOR & 0xFFFF),
                               (void *) &psMibOccupancyMonitor->sPerm,
                               sizeof(psMibOccupancyMonitor->sPerm),
                               FALSE);
		/* Debug */
		DBG_vPrintf(DEBUG_MIB_OCCUPANCY_MONITOR, "\n%sPDM_eLoadRecord(OccupancyMonitor) = %d", acDebugIndent, psMibOccupancyMonitor->sDesc.eState);
    }

	/* Debug */
    Node_vDebugOutdent(DEBUG_MIB_OCCUPANCY_MONITOR);
}

/****************************************************************************
 *
 * NAME: MibOccupancyMonitor_vRegister
 *
 * DESCRIPTION:
 * Registers MIB
 *
 ****************************************************************************/
PUBLIC void MibOccupancyMonitor_vRegister(void)
{
    teJIP_Status eStatus;

    /* Debug */
    DBG_vPrintf(DEBUG_MIB_OCCUPANCY_MONITOR, "\n%sMibOccupancyMonitor_vRegister()", acDebugIndent);
    Node_vDebugIndent(DEBUG_MIB_OCCUPANCY_MONITOR);

    /* Register MIB */
    eStatus = eJIP_RegisterMib(psMibOccupancyMonitor->hMib);
    DBG_vPrintf(DEBUG_MIB_OCCUPANCY_MONITOR, "\n\t%seJIP_RegisterMib(OccupancyMonitor)=%d", acDebugIndent, eStatus);

    /* Configure table structures */
	psMibOccupancyMonitor->sDeviceTable.pvData	   = (void *) psMibOccupancyMonitor->sTemp.asDeviceTable;
	psMibOccupancyMonitor->sDeviceTable.u32Size	   = sizeof(in6_addr);
	psMibOccupancyMonitor->sDeviceTable.u16Entries = MIB_OCCUPANCY_MONITOR_MAX_DEVICES;

    /* Make sure permament data is saved */
    PDM_vSaveRecord(&psMibOccupancyMonitor->sDesc);
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_OCCUPANCY_MONITOR, "\n%sPDM_vSaveRecord(OccupancyMonitor) = %d", acDebugIndent, psMibOccupancyMonitor->sDesc.eState);

	/* Debug */
    Node_vDebugOutdent(DEBUG_MIB_OCCUPANCY_MONITOR);
}

/****************************************************************************
 *
 * NAME: MibOccupancyMonitor_vTick
 *
 * DESCRIPTION:
 * Tick function called every 10ms when stack is running
 *
 ****************************************************************************/
PUBLIC void MibOccupancyMonitor_vTick(void)
{
	/* Are there any variable notifications pending ? */
	if ((psMibOccupancyMonitor->u32NotifyChanged & VAR_MASK_OCCUPANCY_MONITOR) != 0)
	{
		/* Use common function to output notifications */
		Node_vJipNotifyChanged(psMibOccupancyMonitor->hMib, &psMibOccupancyMonitor->u32NotifyChanged, VAR_MASK_OCCUPANCY_MONITOR, VAR_COUNT_OCCUPANCY_MONITOR);
	}
}

/****************************************************************************
 *
 * NAME: MibOccupancyMonitor_vSecond
 *
 * DESCRIPTION:
 * Called once per second
 *
 ****************************************************************************/
PUBLIC void MibOccupancyMonitor_vSecond(uint32 u32TimerSeconds)
{
    /* Any monitors enabled ? */
    if (psMibOccupancyMonitor->sTemp.u32Enabled != 0)
    {
		uint8		u8Device;

		/* Loop through devices */
		for (u8Device = 0; u8Device < MIB_OCCUPANCY_MONITOR_MAX_DEVICES; u8Device++)
		{
			/* Timer set for this device */
			if (psMibOccupancyMonitor->au32TimerSeconds[u8Device] > 0)
			{
				/* Timed out ? */
				if (u32TimerSeconds > psMibOccupancyMonitor->au32TimerSeconds[u8Device])
				{
					/* Zero timer for this device to avoid timing it out again */
					psMibOccupancyMonitor->au32TimerSeconds[u8Device] = 0;
					/* Clear the address so it becomes available for another device */
					memset(&psMibOccupancyMonitor->sTemp.asDeviceTable[u8Device], 0, sizeof(in6_addr));
					/* Increment table hash */
					psMibOccupancyMonitor->sDeviceTable.u16Hash++;
					/* Flag variable for trap update notification when the stack runs */
					psMibOccupancyMonitor->u32NotifyChanged |= (1 << VAR_IX_OCCUPANCY_MONITOR_DEVICE_TABLE);

					/* Device is enabled ? */
					if ((psMibOccupancyMonitor->sTemp.u32Enabled & (uint32)(1<<u8Device)) != 0)
					{
						/* Set device as disabled */
						U32_CLR_BITS(&psMibOccupancyMonitor->sTemp.u32Enabled, (uint32)(1<<u8Device));
						/* Flag variable for trap update notification when the stack runs */
						psMibOccupancyMonitor->u32NotifyChanged |= (1 << VAR_IX_OCCUPANCY_MONITOR_ENABLED);
					}
					/* Device is occupied ? */
					if ((psMibOccupancyMonitor->sTemp.u32Occupied & (uint32)(1<<u8Device)) != 0)
					{
						/* Clear device as occupied */
						U32_CLR_BITS(&psMibOccupancyMonitor->sTemp.u32Occupied, (uint32)(1<<u8Device));
						/* Flag variable for trap update notification when the stack runs */
						psMibOccupancyMonitor->u32NotifyChanged |= (1 << VAR_IX_OCCUPANCY_MONITOR_OCCUPIED);
					}
					/* Device is unoccupied ? */
					if ((psMibOccupancyMonitor->sTemp.u32Unoccupied & (uint32)(1<<u8Device)) != 0)
					{
						/* Clear device as unoccupied */
						U32_CLR_BITS(&psMibOccupancyMonitor->sTemp.u32Unoccupied, (uint32)(1<<u8Device));
						/* Flag variable for trap update notification when the stack runs */
						psMibOccupancyMonitor->u32NotifyChanged |= (1 << VAR_IX_OCCUPANCY_MONITOR_UNOCCUPIED);
					}
				}
			}
		}
	}
}

/****************************************************************************
 *
 * NAME: MibOccupancyMonitor_u8FindDevice
 *
 * DESCRIPTION:
 * Finds a device in in the table
 *
 ****************************************************************************/
PUBLIC uint8 MibOccupancyMonitor_u8FindDevice(in6_addr *psIn6Address)
{
	uint8 u8Device;
	uint8 u8Found = 0xff;

	/* Look for device */
	for (u8Device = 0; u8Device < MIB_OCCUPANCY_MONITOR_MAX_DEVICES && u8Found == 0xff; u8Device++)
	{
		/* Is this the device we're looking for ? */
		if (memcmp(psIn6Address, &psMibOccupancyMonitor->sTemp.asDeviceTable[u8Device], sizeof(in6_addr)) == 0)
		{
			/* Note its index */
			u8Found = u8Device;
		}
	}

	return u8Found;
}

/****************************************************************************
 *
 * NAME: MibOccupancyMonitor_eSetMode
 *
 * DESCRIPTION:
 * Mode set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibOccupancyMonitor_eSetMode(uint8 u8Val, void *pvCbData)
{
	teJIP_Status eReturn;

	/* Call standard function */
	eReturn = eSetUint8(u8Val, pvCbData);

    /* Make sure permament data is saved */
    PDM_vSaveRecord(&psMibOccupancyMonitor->sDesc);
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_OCCUPANCY_MONITOR, "\n%sMibOccupancyMonitor_eSetMode(%d) = %d { PDM_vSaveRecord(OccupancyMonitor) = %d }", acDebugIndent, u8Val, eReturn, psMibOccupancyMonitor->sDesc.eState);

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibOccupancyMonitor_eSetTimeout
 *
 * DESCRIPTION:
 * Timeout set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibOccupancyMonitor_eSetTimeout(uint16 u16Val, void *pvCbData)
{
	teJIP_Status eReturn;

	/* Call standard function */
	eReturn = eSetUint16(u16Val, pvCbData);

    /* Make sure permament data is saved */
    PDM_vSaveRecord(&psMibOccupancyMonitor->sDesc);
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_OCCUPANCY_MONITOR, "\n%sMibOccupancyMonitor_eSetTimeout(%d) = %d { PDM_vSaveRecord(OccupancyMonitor) = %d }", acDebugIndent, u16Val, eReturn, psMibOccupancyMonitor->sDesc.eState);

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibOccupancyMonitor_eSetOccupancy
 *
 * DESCRIPTION:
 * Generic set occupancy data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibOccupancyMonitor_eSetOccupancy(uint8 u8Val, void *pvCbData)
{
	teJIP_Status     eReturn = E_JIP_ERROR_FAILED;
	ts6LP_SockAddr   sSocketAddress;
	uint8 			u8Device;

	/* Get the source address for this write */
	iJIP_GetLastSourceAddr(&sSocketAddress);
	/* Try to find this device */
	u8Device = MibOccupancyMonitor_u8FindDevice(&sSocketAddress.sin6_addr);
	/* Didn't find this device ? */
	if (u8Device >= MIB_OCCUPANCY_MONITOR_MAX_DEVICES)
	{
		/* Try to find an unused address */
		u8Device = MibOccupancyMonitor_u8FindDevice(&psMibOccupancyMonitor->sIn6AddressNull);
		/* Found an unused address */
		if (u8Device < MIB_OCCUPANCY_MONITOR_MAX_DEVICES)
		{
			/* Take a copy of the address */
			memcpy(&psMibOccupancyMonitor->sTemp.asDeviceTable[u8Device], &sSocketAddress.sin6_addr, sizeof(in6_addr));
			/* Increment table hash */
			psMibOccupancyMonitor->sDeviceTable.u16Hash++;
			/* Flag variable for trap update notification when the stack runs */
			psMibOccupancyMonitor->u32NotifyChanged |= (1 << VAR_IX_OCCUPANCY_MONITOR_DEVICE_TABLE);
		}
	}

	/* Found a device to update ? */
	if (u8Device < MIB_OCCUPANCY_MONITOR_MAX_DEVICES)
	{
		/* Call standard function */
		eReturn = eSetUint8(u8Val, pvCbData);
		/* Set timer */
		psMibOccupancyMonitor->au32TimerSeconds[u8Device] = Node_u32TimerSeconds() + psMibOccupancyMonitor->sPerm.u16Timeout;
		/* Device not already enabled ? */
		if ((psMibOccupancyMonitor->sTemp.u32Enabled & (uint32)(1<<u8Device)) == 0)
		{
			/* Set device as enabled */
			U32_SET_BITS(&psMibOccupancyMonitor->sTemp.u32Enabled, (uint32)(1<<u8Device));
			/* Flag variable for trap update notification when the stack runs */
			psMibOccupancyMonitor->u32NotifyChanged |= (1 << VAR_IX_OCCUPANCY_MONITOR_ENABLED);
		}
		/* Remote sensor indicates occupied ? */
		if (u8Val != 0)
		{
			/* Not already flagged as occupied ? */
			if ((psMibOccupancyMonitor->sTemp.u32Occupied & (uint32)(1<<u8Device)) == 0)
			{
				/* Set device as occupied */
				U32_SET_BITS(&psMibOccupancyMonitor->sTemp.u32Occupied, (uint32)(1<<u8Device));
				/* Flag variable for trap update notification when the stack runs */
				psMibOccupancyMonitor->u32NotifyChanged |= (1 << VAR_IX_OCCUPANCY_MONITOR_OCCUPIED);
			}
			/* Currently flagged as unoccupied ? */
			if ((psMibOccupancyMonitor->sTemp.u32Unoccupied & (uint32)(1<<u8Device)) != 0)
			{
				/* Clear device as unoccupied */
				U32_CLR_BITS(&psMibOccupancyMonitor->sTemp.u32Unoccupied, (uint32)(1<<u8Device));
				/* Flag variable for trap update notification when the stack runs */
				psMibOccupancyMonitor->u32NotifyChanged |= (1 << VAR_IX_OCCUPANCY_MONITOR_UNOCCUPIED);
			}
		}
		/* Remote sensor indicates unoccupied */
		else
		{
			/* Not already flagged as unoccupied ? */
			if ((psMibOccupancyMonitor->sTemp.u32Unoccupied & (uint32)(1<<u8Device)) == 0)
			{
				/* Set device as unoccupied */
				U32_SET_BITS(&psMibOccupancyMonitor->sTemp.u32Unoccupied, (uint32)(1<<u8Device));
				/* Flag variable for trap update notification when the stack runs */
				psMibOccupancyMonitor->u32NotifyChanged |= (1 << VAR_IX_OCCUPANCY_MONITOR_UNOCCUPIED);
			}
			/* Currently flagged as occupied ? */
			if ((psMibOccupancyMonitor->sTemp.u32Occupied & (uint32)(1<<u8Device)) != 0)
			{
				/* Clear device as occupied */
				U32_CLR_BITS(&psMibOccupancyMonitor->sTemp.u32Occupied, (uint32)(1<<u8Device));
				/* Flag variable for trap update notification when the stack runs */
				psMibOccupancyMonitor->u32NotifyChanged |= (1 << VAR_IX_OCCUPANCY_MONITOR_OCCUPIED);
			}
		}
	}

	return eReturn;
}
/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
