/****************************************************************************/
/*
 * MODULE              JN-AN-1162 JenNet-IP Smart Home
 *
 * DESCRIPTION         IlluminanceScene MIB Implementation
 */
/****************************************************************************/
/*
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2014. All rights reserved
 */
/****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
/* Standard includes */
#include <string.h>
/* SDK includes */
#include <jendefs.h>
/* Hardware includes */
#include <AppHardwareApi.h>
#include <PeripheralRegs.h>
/* Stack includes */
#include <Api.h>
#include <AppApi.h>
#include <JIP.h>
#include <6LP.h>
#include <AccessFunctions.h>
/* JenOS includes */
#include <dbg.h>
#include <dbg_uart.h>
#include <os.h>
#include <pdm.h>
/* Application common includes */
#include "DeviceDefs.h"
#include "Node.h"
#include "MibSensor.h"
#include "Table.h"
/* Application device includes */
#include "MibIlluminanceControl.h"
#include "MibIlluminanceScene.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/
/* Data access */
#define PS_MIB_ILLUMINANCE_CONTROL  ((tsMibIlluminanceControl *) psMibIlluminanceScene->pvMibIlluminanceControl)

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE tsMibIlluminanceScene  *psMibIlluminanceScene;			/* Illuminance Config Mib data */

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME: MibIlluminanceScene_vInit
 *
 * DESCRIPTION:
 * Initialises data
 *
 ****************************************************************************/
PUBLIC void MibIlluminanceScene_vInit(thJIP_Mib         hMibIlluminanceSceneInit,
							   tsMibIlluminanceScene   *psMibIlluminanceSceneInit,
							   void					   *pvMibIlluminanceControlInit)
{
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_SCENE,  "\n%sMibIlluminanceScene_vInit() {%d}", acDebugIndent, sizeof(tsMibIlluminanceScene));
    Node_vDebugIndent(DEBUG_MIB_ILLUMINANCE_SCENE);

	/* Valid data pointer ? */
	if (psMibIlluminanceSceneInit != (tsMibIlluminanceScene *) NULL)
	{
		/* Take copy of pointer to data */
		psMibIlluminanceScene = psMibIlluminanceSceneInit;

		/* Take a copy of the MIB handle */
		psMibIlluminanceScene->hMib = hMibIlluminanceSceneInit;

		/* Take a copy of the illuminance control MIB pointer */
		psMibIlluminanceScene->pvMibIlluminanceControl = pvMibIlluminanceControlInit;

		/* Load IlluminanceScene mib data */
		(void) PDM_eLoadRecord(&psMibIlluminanceScene->sDesc,
							   (uint16)(MIB_ID_ILLUMINANCE_SCENE & 0xFFFF),
							   (void *) &psMibIlluminanceScene->sPerm,
						  	   sizeof(psMibIlluminanceScene->sPerm),
							   FALSE);
		/* Debug */
		DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_SCENE, "\n%sPDM_eLoadRecord(IlluminanceScene) = %d", acDebugIndent, psMibIlluminanceScene->sDesc.eState);
	}

	/* Debug */
    Node_vDebugOutdent(DEBUG_MIB_ILLUMINANCE_SCENE);
}

/****************************************************************************
 *
 * NAME: MibIlluminanceScene_vRegister
 *
 * DESCRIPTION:
 * Registers MIB
 *
 ****************************************************************************/
PUBLIC void MibIlluminanceScene_vRegister(void)
{
	teJIP_Status eStatus;

    /* Debug */
    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_SCENE, "\n%sMibIlluminanceScene_vRegister()", acDebugIndent);
    Node_vDebugIndent(DEBUG_MIB_ILLUMINANCE_SCENE);

	/* Register MIB */
	eStatus = eJIP_RegisterMib(psMibIlluminanceScene->hMib);
    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_CONFIG, "\n%seJIP_RegisterMib(IlluminanceScene) = %d", acDebugIndent, eStatus);

	/* Configure table structures */
	psMibIlluminanceScene->sSceneTable.pvData	  = (void *) psMibIlluminanceScene->sPerm.asSceneTable;
	psMibIlluminanceScene->sSceneTable.u32Size	  = sizeof(tsIlluminanceScene);
	psMibIlluminanceScene->sSceneTable.u16Entries = MIB_ILLUMINANCE_SCENE_SCENES;

    /* Make sure permament data is saved */
    PDM_vSaveRecord(&psMibIlluminanceScene->sDesc);
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_SCENE, "\n%sPDM_vSaveRecord(IlluminanceScene) = %d", acDebugIndent, psMibIlluminanceScene->sDesc.eState);

    /* Debug */
    Node_vDebugOutdent(DEBUG_MIB_ILLUMINANCE_SCENE);
}

/****************************************************************************
 *
 * NAME: MibIlluminanceScene_u8FindSceneId
 *
 * DESCRIPTION:
 * Finds a scene id in the table
 *
 ****************************************************************************/
PUBLIC uint8 MibIlluminanceScene_u8FindSceneId(uint16 u16SceneId)
{
	uint8 u8Scene;
	uint8 u8Found = 0xff;

	/* Look for scene */
	for (u8Scene = 0; u8Scene < MIB_ILLUMINANCE_SCENE_SCENES && u8Found == 0xff; u8Scene++)
	{
		/* Is this the scene we're looking for ? */
		if (psMibIlluminanceScene->sPerm.asSceneTable[u8Scene].u16Id == u16SceneId)
		{
			/* Note its index */
			u8Found = u8Scene;
		}
	}

	return u8Found;
}

/****************************************************************************
 *
 * NAME: MibIlluminanceScene_bSetScene
 *
 * DESCRIPTION:
 * AddSceneId set data callback
 *
 ****************************************************************************/
PUBLIC bool_t MibIlluminanceScene_bSetScene(uint16 u16Id, uint16 u16LuxTarget, uint16 u16LuxBand)
{
	bool_t  bReturn = FALSE;
	uint8  u8Scene;

	/* Valid scene ? */
	if (u16Id != 0)
	{
		/* Look for this scene */
		u8Scene = MibIlluminanceScene_u8FindSceneId(u16Id);
		/* Didn't find scene ? */
		if (u8Scene >= MIB_ILLUMINANCE_SCENE_SCENES)
		{
			/* Look for unused scene */
			u8Scene = MibIlluminanceScene_u8FindSceneId(0);
		}
		/* Do we have a scene index to add the scene ? */
		if (u8Scene < MIB_ILLUMINANCE_SCENE_SCENES)
		{
			/* Store the scene */
			psMibIlluminanceScene->sPerm.asSceneTable[u8Scene].u16Id        = u16Id;
			psMibIlluminanceScene->sPerm.asSceneTable[u8Scene].u16LuxTarget = u16LuxTarget;
			psMibIlluminanceScene->sPerm.asSceneTable[u8Scene].u16LuxBand   = u16LuxBand;
			/* Increment table hashes */
			psMibIlluminanceScene->sSceneTable.u16Hash++;
		    /* Make sure permament data is saved */
		    PDM_vSaveRecord(&psMibIlluminanceScene->sDesc);
		    /* Debug */
		    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_SCENE, "\n%sPDM_vSaveRecord(IlluminanceScene) = %d", acDebugIndent, psMibIlluminanceScene->sDesc.eState);
			/* Notify traps */
			vJIP_NotifyChanged(psMibIlluminanceScene->hMib, VAR_IX_ILLUMINANCE_SCENE_SCENE_TABLE);
			/* Success */
			bReturn = TRUE;
		}
	}

	return bReturn;
}

/****************************************************************************
 *
 * NAME: MibIlluminanceScene_eSetAddSceneId
 *
 * DESCRIPTION:
 * AddSceneId set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibIlluminanceScene_eSetAddSceneId(uint16 u16Val, void *pvCbData)
{
	teJIP_Status eReturn = E_JIP_ERROR_FAILED;

	/* Valid Illuminance control Mib data is available ? */
	if (PS_MIB_ILLUMINANCE_CONTROL != NULL)
	{
		/* Can we set this scene ? */
		if (MibIlluminanceScene_bSetScene(u16Val, PS_MIB_ILLUMINANCE_CONTROL->sPerm.u16LuxTarget, PS_MIB_ILLUMINANCE_CONTROL->sPerm.u16LuxBand))
		{
			/* Success */
			eReturn = eSetUint16(u16Val, pvCbData);
		}
	}

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibIlluminanceScene_eSetDelSceneId
 *
 * DESCRIPTION:
 * DelSceneId set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibIlluminanceScene_eSetDelSceneId(uint16 u16Val, void *pvCbData)
{
	teJIP_Status eReturn = E_JIP_ERROR_FAILED;
	uint8   	 u8Scene;

	/* Valid scene and Illuminance control Mib data is available ? */
	if (u16Val != 0)
	{
		/* Look for this scene */
		u8Scene = MibIlluminanceScene_u8FindSceneId(u16Val);
		/* Do we have a scene index to delete the scene ? */
		if (u8Scene < MIB_ILLUMINANCE_SCENE_SCENES)
		{
			/* Delete the scene */
			psMibIlluminanceScene->sPerm.asSceneTable[u8Scene].u16Id        = 0;
			psMibIlluminanceScene->sPerm.asSceneTable[u8Scene].u16LuxTarget = 0;
			psMibIlluminanceScene->sPerm.asSceneTable[u8Scene].u16LuxBand   = 0;
			/* Success */
			eReturn = eSetUint16(u16Val, pvCbData);
			/* Increment table hashes */
			psMibIlluminanceScene->sSceneTable.u16Hash++;
		    /* Make sure permament data is saved */
		    PDM_vSaveRecord(&psMibIlluminanceScene->sDesc);
		    /* Debug */
		    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_SCENE, "\n%sPDM_vSaveRecord(IlluminanceScene) = %d", acDebugIndent, psMibIlluminanceScene->sDesc.eState);
			/* Notify traps */
			vJIP_NotifyChanged(psMibIlluminanceScene->hMib, VAR_IX_ILLUMINANCE_SCENE_SCENE_TABLE);
		}
	}

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibilluminanceScene_eSetSetScene
 *
 * DESCRIPTION:
 * Handle remote set of a key
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibIlluminanceScene_eSetSetScene(const uint8 *pu8Val, uint8 u8Len, void *pvCbData)
{
	teJIP_Status eReturn = E_JIP_ERROR_FAILED;

	/* Has the correct amount of data being passed in ? */
	if (u8Len == sizeof(tsIlluminanceScene))
	{
		const tsIlluminanceScene *psSetScene;

		/* Cast value to get valid pointer */
		psSetScene = (tsIlluminanceScene *) pu8Val;

		/* Can we set this scene ? */
		if (MibIlluminanceScene_bSetScene(psSetScene->u16Id, psSetScene->u16LuxTarget, psSetScene->u16LuxBand))
		{
			/* Copy data */
			memcpy(pvCbData, pu8Val, u8Len);
			/* Success */
			eReturn = E_JIP_OK;
		}
	}

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibilluminanceScene_vGetSetScene
 *
 * DESCRIPTION:
 * Handle remote get of a key
 *
 ****************************************************************************/
PUBLIC void MibIlluminanceScene_vGetSetScene(thJIP_Packet hPacket, void *pvCbData)
{
	eJIP_PacketAddData(hPacket, pvCbData, sizeof(tsIlluminanceScene), 0);
}
/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
