/****************************************************************************/
/*
 * MODULE              JN-AN-1162 JenNet-IP Smart Home
 *
 * DESCRIPTION         IlluminanceConfig MIB Implementation
 */
/****************************************************************************/
/*
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2014. All rights reserved
 */
/****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
/* Standard includes */
#include <string.h>
/* SDK includes */
#include <jendefs.h>
/* Hardware includes */
#include <AppHardwareApi.h>
#include <PeripheralRegs.h>
/* Stack includes */
#include <Api.h>
#include <AppApi.h>
#include <JIP.h>
#include <6LP.h>
#include <AccessFunctions.h>
/* JenOS includes */
#include <dbg.h>
#include <dbg_uart.h>
#include <os.h>
#include <pdm.h>
/* Application device includes */
#include "DeviceDefs.h"
#include "Node.h"
#include "Address.h"
#include "MibSensor.h"
#include "MibIlluminanceConfig.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE tsMibIlluminanceConfig *psMibIlluminanceConfig;  /* MIB data */

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME: MibIlluminanceConfig_vInit
 *
 * DESCRIPTION:
 * Initialises data
 *
 ****************************************************************************/
PUBLIC void MibIlluminanceConfig_vInit(thJIP_Mib        hMibIlluminanceConfigInit,
                         	   tsMibIlluminanceConfig *psMibIlluminanceConfigInit)
{
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_CONFIG, "\n%sMibIlluminanceConfig_vInit() {%d}", acDebugIndent, sizeof(tsMibIlluminanceConfig));
    Node_vDebugIndent(DEBUG_MIB_ILLUMINANCE_CONFIG);

    /* Valid data pointer ? */
    if (psMibIlluminanceConfigInit != (tsMibIlluminanceConfig *) NULL)
    {
        /* Take copy of pointer to data */
        psMibIlluminanceConfig = psMibIlluminanceConfigInit;
        /* Take a copy of the MIB handle */
        psMibIlluminanceConfig->hMib = hMibIlluminanceConfigInit;

        /* Load Dio mib data */
        (void) PDM_eLoadRecord(&psMibIlluminanceConfig->sDesc,
							   (uint16)(MIB_ID_ILLUMINANCE_CONFIG & 0xFFFF),
                               (void *) &psMibIlluminanceConfig->sPerm,
                               sizeof(psMibIlluminanceConfig->sPerm),
                               FALSE);
		/* Debug */
		DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_CONFIG, "\n%sPDM_eLoadRecord(IlluminanceConfig) = %d", acDebugIndent, psMibIlluminanceConfig->sDesc.eState);
    }
    /* Debug */
    Node_vDebugOutdent(DEBUG_MIB_ILLUMINANCE_CONFIG);
}

/****************************************************************************
 *
 * NAME: MibIlluminanceConfig_vRegister
 *
 * DESCRIPTION:
 * Registers MIB
 *
 ****************************************************************************/
PUBLIC void MibIlluminanceConfig_vRegister(void)
{
    teJIP_Status eStatus;

    /* Debug */
    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_CONFIG, "\n%sMibIlluminanceConfig_vRegister()", acDebugIndent);
    Node_vDebugIndent(DEBUG_MIB_ILLUMINANCE_CONFIG);

	/* Address not set in flash ? */
	if (psMibIlluminanceConfig->sPerm.sStateAddress.s6_addr32[0] == 0 &&
		psMibIlluminanceConfig->sPerm.sStateAddress.s6_addr32[1] == 0 &&
		psMibIlluminanceConfig->sPerm.sStateAddress.s6_addr32[2] == 0 &&
		psMibIlluminanceConfig->sPerm.sStateAddress.s6_addr32[3] == 0)
	{
		/* Set unique group address */
		Address_vBuildGroup(&psMibIlluminanceConfig->sPerm.sStateAddress,
											  (MAC_ExtAddr_s *) pvAppApiGetMacAddrLocation(),
											  (uint16)(MIB_ID_ILLUMINANCE_CONFIG & 0xffff));
	}

    /* Register MIB */
    eStatus = eJIP_RegisterMib(psMibIlluminanceConfig->hMib);
    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_CONFIG, "\n%seJIP_RegisterMib(IlluminanceConfig) = %d", acDebugIndent, eStatus);

    /* Make sure permament data is saved */
    PDM_vSaveRecord(&psMibIlluminanceConfig->sDesc);
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_CONFIG, "\n%sPDM_vSaveRecord(IlluminanceConfig) = %d", acDebugIndent, psMibIlluminanceConfig->sDesc.eState);

    /* Debug */
    Node_vDebugOutdent(DEBUG_MIB_ILLUMINANCE_CONFIG);
}

/****************************************************************************
 *
 * NAME: MibIlluminanceConfig_eSetUint8
 *
 * DESCRIPTION:
 * Generic set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibIlluminanceConfig_eSetUint8(uint8 u8Val, void *pvCbData)
{
	teJIP_Status eReturn;

	/* Call standard function */
	eReturn = eSetUint8(u8Val, pvCbData);

    /* Make sure permament data is saved */
    PDM_vSaveRecord(&psMibIlluminanceConfig->sDesc);
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_CONFIG, "\n%sPDM_vSaveRecord(IlluminanceConfig)");

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibIlluminanceConfig_eSetUint16
 *
 * DESCRIPTION:
 * Generic set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibIlluminanceConfig_eSetUint16(uint16 u16Val, void *pvCbData)
{
	teJIP_Status eReturn;

	/* Call standard function */
	eReturn = eSetUint16(u16Val, pvCbData);

    /* Make sure permament data is saved */
    PDM_vSaveRecord(&psMibIlluminanceConfig->sDesc);
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_CONFIG, "\n%sPDM_vSaveRecord(IlluminanceConfig)");

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibIlluminanceConfig_eSetUint32
 *
 * DESCRIPTION:
 * Generic set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibIlluminanceConfig_eSetUint32(uint32 u32Val, void *pvCbData)
{
	teJIP_Status eReturn;

	/* Call standard function */
	eReturn = eSetUint32(u32Val, pvCbData);

    /* Make sure permament data is saved */
    PDM_vSaveRecord(&psMibIlluminanceConfig->sDesc);
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_CONFIG, "\n%sPDM_vSaveRecord(IlluminanceConfig)");

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibIlluminanceConfig_eSetAddress
 *
 * DESCRIPTION:
 * Handle remote set of a group address
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibIlluminanceConfig_eSetAddress(const uint8 *pu8Val, uint8 u8Len, void *pvCbData)

{
	teJIP_Status eReturn;

	/* Call helper function */
	eReturn = Address_eSet(pu8Val, u8Len, pvCbData);

	/* Success ? */
	if (eReturn == E_JIP_OK)
	{
	    /* Make sure permament data is saved */
	    PDM_vSaveRecord(&psMibIlluminanceConfig->sDesc);
	    /* Debug */
	    DBG_vPrintf(DEBUG_MIB_ILLUMINANCE_CONFIG, "\n%sPDM_vSaveRecord(IlluminanceConfig)");
	}

	return eReturn;
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
