/****************************************************************************/
/*\PROJECT 		JN-AN-1162 JenNet-IP Smart Home								*/
/*!
 *\MODULE 		MibColour
 *
 *\FILE			MibColourConfig.c
 *
 *\DESCRIPTION	ColourConfig MIB implementation
 */
/****************************************************************************/
/*
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2014. All rights reserved
 */
/****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
/* Standard includes */
#include <string.h>
/* SDK includes */
#include <jendefs.h>
/* Hardware includes */
#include <AppHardwareApi.h>
#include <PeripheralRegs.h>
/* Stack includes */
#include <Api.h>
#include <AppApi.h>
#include <JIP.h>
#include <6LP.h>
#include <AccessFunctions.h>
/* JenOS includes */
#include <dbg.h>
#include <dbg_uart.h>
#include <os.h>
#include <pdm.h>
/* Application common includes */
#include "DeviceDefs.h"
#include "Node.h"
#include "MibColour.h"
#include "Table.h"
/* Application device includes */
#include "MibColourConfig.h"
#include "ColourConversion.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        External Variables                                            ***/
/****************************************************************************/
/* MIB structures */
extern tsMibColourConfig sMibColourConfig;
extern thJIP_Mib 	     hMibColourConfig;

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/****************************************************************************/
/*!
 *\NAME 		MibColourConfig_vInit
 *
 *\DESCRIPTION	Initialises ColourConfig's MIB data.
 *
 * Loads permament data from PDM and initialises it if not available.
 *
 * Calculates the colour conversion matricies from the configuration data.
 */
/****************************************************************************/
PUBLIC void MibColourConfig_vInit(void)
{
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_COLOUR_CONFIG, "\n%sMibColourConfig_vInit() {%d}", acDebugIndent, sizeof(tsMibColourConfig));
	Node_vDebugIndent(DEBUG_MIB_COLOUR_CONFIG);

	/* Load ColourStatus mib data */
	(void) PDM_eLoadRecord(&sMibColourConfig.sDesc,
						   (uint16)(MIB_ID_COLOUR_CONFIG & 0xFFFF),
						   (void *) &sMibColourConfig.sPerm,
						   sizeof(sMibColourConfig.sPerm),
						   FALSE);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_COLOUR_CONFIG, "\n%sPDM_eLoadRecord(MibColourConfig) = %d", acDebugIndent, sMibColourConfig.sDesc.eState);

	/* Didn't load record ? */
	if (PDM_RECOVERY_STATE_RECOVERED != sMibColourConfig.sDesc.eState)
	{
		/* Initialise permament data */
		sMibColourConfig.sPerm.u32InitXYTarget   = X_Y_TO_U32(F_TO_U16(CLD_COLOURCONTROL_WHITE_X), F_TO_U16(CLD_COLOURCONTROL_WHITE_Y));
		sMibColourConfig.sPerm.u32XYPrimaryWhite = X_Y_TO_U32(F_TO_U16(CLD_COLOURCONTROL_WHITE_X), F_TO_U16(CLD_COLOURCONTROL_WHITE_Y));
		sMibColourConfig.sPerm.u32XYPrimary1     = X_Y_TO_U32(F_TO_U16(CLD_COLOURCONTROL_RED_X)  , F_TO_U16(CLD_COLOURCONTROL_RED_Y)  );
		sMibColourConfig.sPerm.u32XYPrimary2     = X_Y_TO_U32(F_TO_U16(CLD_COLOURCONTROL_GREEN_X), F_TO_U16(CLD_COLOURCONTROL_GREEN_Y));
		sMibColourConfig.sPerm.u32XYPrimary3     = X_Y_TO_U32(F_TO_U16(CLD_COLOURCONTROL_BLUE_X) , F_TO_U16(CLD_COLOURCONTROL_BLUE_Y) );
	}

	/* Generate XYZ<->RGB conversion matrices */
	eCLD_ColourControlCalculateConversionMatrices(&sMibColourConfig.sColourData,
												  U16_TO_F(U32_TO_X(sMibColourConfig.sPerm.u32XYPrimary1)),      /* Red X */
												  U16_TO_F(U32_TO_Y(sMibColourConfig.sPerm.u32XYPrimary1)),      /* Red Y */
												  U16_TO_F(U32_TO_X(sMibColourConfig.sPerm.u32XYPrimary2)),      /* Green X */
												  U16_TO_F(U32_TO_Y(sMibColourConfig.sPerm.u32XYPrimary2)),      /* Green Y */
												  U16_TO_F(U32_TO_X(sMibColourConfig.sPerm.u32XYPrimary3)),      /* Blue X */
												  U16_TO_F(U32_TO_Y(sMibColourConfig.sPerm.u32XYPrimary3)),      /* Blue Y */
												  U16_TO_F(U32_TO_X(sMibColourConfig.sPerm.u32XYPrimaryWhite)),  /* White X */
												  U16_TO_F(U32_TO_Y(sMibColourConfig.sPerm.u32XYPrimaryWhite))); /* White Y */
}

/****************************************************************************/
/*!
 *\NAME 		MibColourConfig_vRegister
 *
 *\DESCRIPTION	Registers ColourControl MIB with stack.
 *
 * Ensures permament data is saved using PDM.
 */
/****************************************************************************/
PUBLIC void MibColourConfig_vRegister(void)
{
    teJIP_Status eStatus;

	/* Debug */
	DBG_vPrintf(DEBUG_MIB_COLOUR_CONFIG, "\n%sMibColourConfig_vRegister()", acDebugIndent);
	Node_vDebugIndent(DEBUG_MIB_COLOUR_CONFIG);
	/* Register MIB */
	eStatus = eJIP_RegisterMib(hMibColourConfig);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_COLOUR_CONFIG, "\n%seJIP_RegisterMib(ColourConfig) = %d", acDebugIndent, eStatus);
	/* Make sure permament data is saved */
	PDM_vSaveRecord(&sMibColourConfig.sDesc);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_COLOUR_CONFIG, "\n%sPDM_vSaveRecord(MibColourConfig) = %d", acDebugIndent, sMibColourConfig.sDesc.eState);
	/* Debug */
	Node_vDebugOutdent(DEBUG_MIB_COLOUR_CONFIG);
}

/****************************************************************************/
/*!
 *\NAME 		MibColourConfig_eSetTransitionTime
 *
 *\DESCRIPTION	Callback for setting TranisitionTime variable.
 *
 * Saves permament data using PDM.
 */
/****************************************************************************/
PUBLIC teJIP_Status MibColourConfig_eSetTransitionTime(
	uint16 u16Val, 		/*!< New variable value. */
	void   *pvCbData) 	/*!< Pointer to variable data. */
{
	teJIP_Status eReturn = E_JIP_ERROR_BAD_VALUE;

	/* Enforce minimum transition time */
	if (u16Val < 16) u16Val = 16;
	/* Call standard function */
	eReturn = eSetUint16(u16Val, pvCbData);

	/* Make sure permament data is saved */
	PDM_vSaveRecord(&sMibColourConfig.sDesc);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_COLOUR_CONFIG, "\n%sPDM_vSaveRecord(MibColourConfig) = %d", acDebugIndent, sMibColourConfig.sDesc.eState);

	return eReturn;
}

/****************************************************************************/
/*!
 *\NAME 		MibColourConfig_eSetInitMode
 *
 *\DESCRIPTION	Callback for setting InitMode variable.
 *
 * Saves permament data using PDM.
 */
/****************************************************************************/
PUBLIC teJIP_Status MibColourConfig_eSetInitMode(
	uint8  u8Val, 		/*!< New variable value. */
	void  *pvCbData)	/*!< Pointer to variable data. */
{
	teJIP_Status eReturn;

	/* Call standard function */
	eReturn = eSetUint8(u8Val, pvCbData);

	/* Make sure permament data is saved */
	PDM_vSaveRecord(&sMibColourConfig.sDesc);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_COLOUR_CONFIG, "\n%sPDM_vSaveRecord(MibColourConfig) = %d", acDebugIndent, sMibColourConfig.sDesc.eState);

	return eReturn;
}

/****************************************************************************/
/*!
 *\NAME			MibColourConfig_eSetInitXYTarget
 *
 *\DESCRIPTION	Callback for setting InitXYTarget variable.
 *
 * Saves permament data using PDM.
 */
/****************************************************************************/
PUBLIC teJIP_Status MibColourConfig_eSetInitXYTarget(
	uint32 u32Val, 		/*!< New variable value. */
	void   *pvCbData)	/*!< Pointer to variable data. */
{
	teJIP_Status eReturn;

	/* Call standard function */
	eReturn = eSetUint32(u32Val, pvCbData);

	/* Make sure permament data is saved */
	PDM_vSaveRecord(&sMibColourConfig.sDesc);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_COLOUR_CONFIG, "\n%sPDM_vSaveRecord(MibColourConfig) = %d", acDebugIndent, sMibColourConfig.sDesc.eState);

	return eReturn;
}

/****************************************************************************/
/*!
 *\NAME			MibColourConfig_eSetPrimary
 *
 *\DESCRIPTION	Callback for setting Primary variables.
 *
 * Recalculates colour conversion matricies when called.
 *
 * Saves permament data using PDM.
 */
/****************************************************************************/
PUBLIC teJIP_Status MibColourConfig_eSetPrimary(
	uint32 u32Val, 		/*!< New variable value. */
	void   *pvCbData)	/*!< Pointer to variable data. */
{
	teJIP_Status eReturn;

	/* Call standard function */
	eReturn = eSetUint32(u32Val, pvCbData);

	/* Generate XYZ<->RGB conversion matrices */
	eCLD_ColourControlCalculateConversionMatrices(&sMibColourConfig.sColourData,
												  U16_TO_F(U32_TO_X(sMibColourConfig.sPerm.u32XYPrimary1)),      /* Red X */
												  U16_TO_F(U32_TO_Y(sMibColourConfig.sPerm.u32XYPrimary1)),      /* Red Y */
												  U16_TO_F(U32_TO_X(sMibColourConfig.sPerm.u32XYPrimary2)),      /* Green X */
												  U16_TO_F(U32_TO_Y(sMibColourConfig.sPerm.u32XYPrimary2)),      /* Green Y */
												  U16_TO_F(U32_TO_X(sMibColourConfig.sPerm.u32XYPrimary3)),      /* Blue X */
												  U16_TO_F(U32_TO_Y(sMibColourConfig.sPerm.u32XYPrimary3)),      /* Blue Y */
												  U16_TO_F(U32_TO_X(sMibColourConfig.sPerm.u32XYPrimaryWhite)),  /* White X */
												  U16_TO_F(U32_TO_Y(sMibColourConfig.sPerm.u32XYPrimaryWhite))); /* White Y */

	/* Make sure permament data is saved */
	PDM_vSaveRecord(&sMibColourConfig.sDesc);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_COLOUR_CONFIG, "\n%sPDM_vSaveRecord(MibColourConfig) = %d", acDebugIndent, sMibColourConfig.sDesc.eState);

	return eReturn;
}

/****************************************************************************/
/*!
 *\NAME			MibColourConfig_eSetCct
 *
 *\DESCRIPTION	Callback for setting CCT limit variables.
 *
 * Saves permament data using PDM.
 */
/****************************************************************************/
PUBLIC teJIP_Status MibColourConfig_eSetCct(
	uint16 u16Val, 		/*!< New variable value. */
	void   *pvCbData)	/*!< Pointer to variable data. */
{
	teJIP_Status eReturn;

	/* Call standard function */
	eReturn = eSetUint16(u16Val, pvCbData);

	/* Make sure permament data is saved */
	PDM_vSaveRecord(&sMibColourConfig.sDesc);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_COLOUR_CONFIG, "\n%sPDM_vSaveRecord(MibColourConfig) = %d", acDebugIndent, sMibColourConfig.sDesc.eState);

	return eReturn;
}
/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
