/****************************************************************************/
/*
 * MODULE              JN-AN-1162 JenNet-IP Smart Home
 *
 * DESCRIPTION         BulbStatus MIB Implementation
 */
/****************************************************************************/
/*
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2014. All rights reserved
 */
/****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
/* Standard includes */
#include <string.h>
/* SDK includes */
#include <jendefs.h>
/* Hardware includes */
#include <AppHardwareApi.h>
#include <PeripheralRegs.h>
/* Stack includes */
#include <Api.h>
#include <AppApi.h>
#include <JIP.h>
#include <6LP.h>
/* JenOS includes */
#include <dbg.h>
#include <dbg_uart.h>
#include <os.h>
#include <pdm.h>
/* Application common includes */
#include "DeviceDefs.h"
#include "Node.h"
#include "MibBulb.h"
#include "MibAdcStatus.h"
/* Application device includes */
#include "DriverBulb.h"
#include "MibBulbControl.h"
#include "MibBulbStatus.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE tsMibBulbStatus  *psMibBulbStatus;			/* Bulb Status Mib data */

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME: MibBulbStatus_vInit
 *
 * DESCRIPTION:
 * Initialises data
 *
 ****************************************************************************/
PUBLIC void MibBulbStatus_vInit(thJIP_Mib        hMibBulbStatusInit,
								tsMibBulbStatus *psMibBulbStatusInit,
								uint8  			  u8AdcSrcBusVoltsInit)
{
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_BULB_STATUS,   "\n%sMibBulbStatus_vInit() {%d}", acDebugIndent, sizeof(tsMibBulbStatus));
	Node_vDebugIndent(DEBUG_MIB_BULB_STATUS);

	/* Valid data pointer ? */
	if (psMibBulbStatusInit != (tsMibBulbStatus *) NULL)
	{
		/* Take copy of pointer to data */
		psMibBulbStatus = psMibBulbStatusInit;

		/* Take a copy of the MIB handle */
		psMibBulbStatus->hMib = hMibBulbStatusInit;

		/* Load BulbStatus mib data */
		(void) PDM_eLoadRecord(&psMibBulbStatus->sDesc,
							   (uint16)(MIB_ID_BULB_STATUS & 0xFFFF),
							   (void *) &psMibBulbStatus->sPerm,
							   sizeof(psMibBulbStatus->sPerm),
							   FALSE);
		/* Debug */
		DBG_vPrintf(DEBUG_MIB_BULB_STATUS, "\n%sPDM_eLoadRecord(MibBulbStatus) = %d", acDebugIndent, psMibBulbStatus->sDesc.eState);

		/* Note ADC source for bus voltage */
		psMibBulbStatus->u8AdcSrcBusVolts = u8AdcSrcBusVoltsInit;
	}

	/* Debug */
	Node_vDebugOutdent(DEBUG_MIB_BULB_STATUS);
}

/****************************************************************************
 *
 * NAME: MibBulbStatus_vRegister
 *
 * DESCRIPTION:
 * Registers MIB
 *
 ****************************************************************************/
PUBLIC void MibBulbStatus_vRegister(void)
{
    teJIP_Status eStatus;

	/* Debug */
	DBG_vPrintf(DEBUG_MIB_BULB_STATUS, "\n%sMibBulbStatus_vRegister()", acDebugIndent);
	Node_vDebugIndent(DEBUG_MIB_BULB_STATUS);
	/* Register MIB */
	eStatus = eJIP_RegisterMib(psMibBulbStatus->hMib);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_BULB_STATUS, "\n%seJIP_RegisterMib(BulbStatus) = %d", acDebugIndent, eStatus);
	/* Make sure permament data is saved */
	PDM_vSaveRecord(&psMibBulbStatus->sDesc);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_BULB_STATUS, "\n%sPDM_vSaveRecord(MibBulbStatus) = %d", acDebugIndent, psMibBulbStatus->sDesc.eState);
	/* Debug */
	Node_vDebugOutdent(DEBUG_MIB_BULB_STATUS);
}

/****************************************************************************
 *
 * NAME: MibBulbStatus_vTick
 *
 * DESCRIPTION:
 * Timer function called when stack is running
 *
 ****************************************************************************/
PUBLIC void MibBulbStatus_vTick(void)
{
	/* Are there any variable notifications pending ? */
	if ((psMibBulbStatus->u32NotifyChanged & VAR_MASK_BULB_STATUS) != 0)
	{
		/* Use common function to output notifications */
		Node_vJipNotifyChanged(psMibBulbStatus->hMib, &psMibBulbStatus->u32NotifyChanged, VAR_MASK_BULB_STATUS, VAR_COUNT_BULB_STATUS);
	}
}

/****************************************************************************
 *
 * NAME: MibBulbStatus_vSecond
 *
 * DESCRIPTION:
 * Timer function
 *
 ****************************************************************************/
PUBLIC void MibBulbStatus_vSecond(void)
{
	/* Bulb is on ? */
	if (DriverBulb_bOn())
	{
		/* Increment on timer */
		psMibBulbStatus->sPerm.u32OnTime++;
		/* Been on an hour ? */
		if ((psMibBulbStatus->sPerm.u32OnTime % 3600) == 0)
		{
			/* Make sure permament data is saved */
			PDM_vSaveRecord(&psMibBulbStatus->sDesc);
			/* Debug */
			DBG_vPrintf(DEBUG_MIB_BULB_STATUS, "\n%sPDM_vSaveRecord(MibBulbStatus) = %d", acDebugIndent, psMibBulbStatus->sDesc.eState);
			/* Need to notify for the on time variable */
			psMibBulbStatus->u32NotifyChanged |= (1 <<VAR_IX_BULB_STATUS_ON_TIME);
		}
	}
	/* Bulb is off */
	else
	{
		/* Increment down timer */
		psMibBulbStatus->sPerm.u32OffTime++;
		/* Been down an hour ? */
		if ((psMibBulbStatus->sPerm.u32OffTime % 3600) == 0)
		{
			/* Make sure permament data is saved */
			PDM_vSaveRecord(&psMibBulbStatus->sDesc);
			/* Debug */
			DBG_vPrintf(DEBUG_MIB_BULB_STATUS, "\n%sPDM_vSaveRecord(MibBulbStatus) = %d", acDebugIndent, psMibBulbStatus->sDesc.eState);
			/* Need to notify for the off time variable */
			psMibBulbStatus->u32NotifyChanged |= (1 <<VAR_IX_BULB_STATUS_OFF_TIME);
		}
	}
}

/****************************************************************************
 *
 * NAME: MibBulbStatus_vAnalogue
 *
 * DESCRIPTION:
 * Called when analogue readings have completed
 *
 ****************************************************************************/
PUBLIC void MibBulbStatus_vAnalogue(uint8 u8Adc)
{
	/* ADC 4 - bus voltage ? */
	if (u8Adc == psMibBulbStatus->u8AdcSrcBusVolts)
	{
		/* Pass on to bulb driver and note the returned bus voltage */
		psMibBulbStatus->sPerm.i16BusVolts = DriverBulb_i16Analogue(u8Adc, MibAdcStatus_u16Read(u8Adc));
	}
	/* ADC temperature - chip temperature ? */
	else if (u8Adc == E_AHI_ADC_SRC_TEMP)
	{
		/* Scale reading to deci-centigrade */
		psMibBulbStatus->sPerm.i16ChipTemp = (int16) MibAdcStatus_i16DeciCentigrade(u8Adc);
	}
}

/****************************************************************************
 *
 * NAME: MibBulbStatus_vOn
 *
 * DESCRIPTION:
 * Called when bulb has been turned on for monitoring purposes
 *
 ****************************************************************************/
PUBLIC void MibBulbStatus_vOn(void)
{
	/* Increment on count */
	psMibBulbStatus->sPerm.u16OnCount++;
	/* Make sure permament data is saved */
	PDM_vSaveRecord(&psMibBulbStatus->sDesc);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_BULB_STATUS, "\n%sPDM_vSaveRecord(MibBulbStatus) = %d", acDebugIndent, psMibBulbStatus->sDesc.eState);
	/* Need to notify for the on count variable */
	psMibBulbStatus->u32NotifyChanged |= (1 <<VAR_IX_BULB_STATUS_ON_COUNT);
	/* Need to notify for the off time variable */
	psMibBulbStatus->u32NotifyChanged |= (1 <<VAR_IX_BULB_STATUS_OFF_TIME);
}

/****************************************************************************
 *
 * NAME: MibBulbStatus_vOff
 *
 * DESCRIPTION:
 * Called when bulb has been turned on for monitoring purposes
 *
 ****************************************************************************/
PUBLIC void MibBulbStatus_vOff(void)
{
	/* Make sure permament data is saved */
	PDM_vSaveRecord(&psMibBulbStatus->sDesc);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_BULB_STATUS, "\n%sPDM_vSaveRecord(MibBulbStatus) = %d", acDebugIndent, psMibBulbStatus->sDesc.eState);
	/* Need to notify for the on time variable */
	psMibBulbStatus->u32NotifyChanged |= (1 <<VAR_IX_BULB_STATUS_ON_TIME);
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
