/****************************************************************************/
/*
 * MODULE              JN-AN-1162 JenNet-IP Smart Home
 *
 * DESCRIPTION         Illuminance driver implementation for TSL2550
 */
/****************************************************************************/
/*
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2014. All rights reserved
 */
/****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
#include <jendefs.h>
#include <AppHardwareApi.h>
#include "DeviceDefs.h"
#include "Node.h"
#include "DriverOccupancy.h"


/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE bool_t bOccupancy;

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME: DriverOccupancy_vInit
 *
 * DESCRIPTION:
 * Initialises the DIO occupancy sensor
 *
 * PARAMETERS: None
 *
 *
 ****************************************************************************/
PUBLIC void DriverOccupancy_vInit(void)
{
	/* Debug */
	DBG_vPrintf(DEBUG_DRIVER, "\n%sDriverOccupancy_vInit()", acDebugIndent);
	Node_vDebugIndent(DEBUG_DRIVER);

	/* Initialise DIO mask as input */
	vAHI_DioSetDirection(DEVICE_INPUT_MASK_OCCUPANCY, 0);
	vAHI_DioSetPullup(0, DEVICE_INPUT_MASK_OCCUPANCY);

	/* Debug */
	DBG_vPrintf(DEBUG_DRIVER, "\n%svAHI_DioSetDirection(0x%x, 0)", acDebugIndent, DEVICE_INPUT_MASK_OCCUPANCY);
	Node_vDebugOutdent(DEBUG_DRIVER);

	/* Valid LEDs ? */
	#if (DEVICE_OUTPUT_MASK_DRIVER_OCCUPANCY)
	{
		/* Configure as outputs */
		vAHI_DioSetDirection(0, DEVICE_OUTPUT_MASK_DRIVER_OCCUPANCY);
		vAHI_DioSetPullup(0, DEVICE_OUTPUT_MASK_DRIVER_OCCUPANCY);
		/* Turn LEDs off (by turning output on) the tick function will turn them on as appropriate */
		vAHI_DioSetOutput(DEVICE_OUTPUT_MASK_DRIVER_OCCUPANCY, 0);
	}
	#endif
}

/****************************************************************************
 *
 * NAME: DriverOccupancy_bRead
 *
 * DESCRIPTION:
 * Reads and returns occupancy sensor state
 *
 * PARAMETERS: None
 *
 * RETURNS:
 * bool_t:	TRUE if occupancy is detected
 *			FALSE if no occupancy is detected
 *
 ****************************************************************************/
PUBLIC bool_t DriverOccupancy_bRead(void)
{
	bool_t bOldOccupancy = bOccupancy;

	if ((u32AHI_DioReadInput() & DEVICE_INPUT_MASK_OCCUPANCY) == DEVICE_INPUT_MASK_OCCUPANCY)
	{
		/* Occupied */
		bOccupancy = TRUE;
		/* Valid LEDs ? */
		#if (DEVICE_OUTPUT_MASK_DRIVER_OCCUPANCY)
		{
			/* Turn LEDs off (by turning output on) the tick function will turn them on as appropriate */
			vAHI_DioSetOutput(0, DEVICE_OUTPUT_MASK_DRIVER_OCCUPANCY);
		}
		#endif
	}
	/* Motion is not detected */
	else
	{
		/* Unoccupied */
		bOccupancy = FALSE;
		/* Valid LEDs ? */
		#if (DEVICE_OUTPUT_MASK_DRIVER_OCCUPANCY)
		{
			/* Turn LEDs off (by turning output on) the tick function will turn them on as appropriate */
			vAHI_DioSetOutput(DEVICE_OUTPUT_MASK_DRIVER_OCCUPANCY, 0);
		}
		#endif
	}

	/* Debug */
	if (bOldOccupancy != bOccupancy) DBG_vPrintf(DEBUG_DRIVER, "\nOcc=%d", bOccupancy);

	return(bOccupancy);
}

/****************************************************************************
 *
 * NAME: DriverOccupancy_bOccupancy
 *
 * DESCRIPTION:
 * Returns last measured occupancy state
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Channel       R   channel to sample
 *
 * RETURNS:
 * bool_t:	TRUE if occupancy is detected
 *			FALSE if no occupancy is detected
 *
 ****************************************************************************/
PUBLIC bool_t DriverOccupancy_bOccupancy(void)
{
	return (bOccupancy);
}

/****************************************************************************
 *
 * NAME: DriverOccupancy_u8Type
 *
 * DESCRIPTION:
 * Returns the sensor type
 *
 * PARAMETERS: None
 *
 * RETURNS:
 * uint8:	0   = PIR
 *			1   = Ultrasound
 *          ?   = Reserved
 * 			255 = Unknown
 *
 ****************************************************************************/
PUBLIC uint8 DriverOccupancy_u8Type(void)
{
	/* Return PIR */
	return(0);
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
