/****************************************************************************/
/*
 * MODULE              JN-AN-1162 JenNet-IP Smart Home
 *
 * DESCRIPTION         Capacitance Touch Driver
 *
 * Replaces capacitance touch algorithms with DIO buttons and LCD display
 */
/****************************************************************************/
/*
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2014. All rights reserved
 */
/****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
#include <jendefs.h>
#include <AppHardwareApi.h>
#include <string.h>
#include <button.h>
#include "dbg.h"
#include "DriverCapTouch.h"
#include "Key.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE uint32 u32LastRead;
PRIVATE bool_t   bRunning;

/****************************************************************************/
/***        Local Constants                                               ***/
/****************************************************************************/
#ifdef  MK_DRIVER_DIO_PRG
#define DRIVER_DIO_MASK_PRG   (1<<MK_DRIVER_DIO_PRG)
#else
#define DRIVER_DIO_MASK_PRG   (0)
#endif
#ifdef  MK_DRIVER_DIO_ON
#define DRIVER_DIO_MASK_ON    (1<<MK_DRIVER_DIO_ON)
#else
#define DRIVER_DIO_MASK_ON    (0)
#endif
#ifdef  MK_DRIVER_DIO_OFF
#define DRIVER_DIO_MASK_OFF   (1<<MK_DRIVER_DIO_OFF)
#else
#define DRIVER_DIO_MASK_OFF   (0)
#endif
#ifdef  MK_DRIVER_DIO_UP
#define DRIVER_DIO_MASK_UP    (1<<MK_DRIVER_DIO_UP)
#else
#define DRIVER_DIO_MASK_UP    (0)
#endif
#ifdef  MK_DRIVER_DIO_DOWN
#define DRIVER_DIO_MASK_DOWN  (1<<MK_DRIVER_DIO_DOWN)
#else
#define DRIVER_DIO_MASK_DOWN  (0)
#endif

#define DRIVER_DIO_MASK_ALL (DRIVER_DIO_MASK_PRG|DRIVER_DIO_MASK_ON|DRIVER_DIO_MASK_OFF|DRIVER_DIO_MASK_UP|DRIVER_DIO_MASK_DOWN)

/****************************************************************************/
/***        Exported Functions                                      ***/
/****************************************************************************/
PUBLIC teTouchStatus eTouchInit(void)
{
	/* Initialise buttons and leds */
	vAHI_DioSetDirection(DRIVER_DIO_MASK_ALL, 0);
	/* Take note of the start up readings */
	u32LastRead = 0;
	DBG_vPrintf(TRUE, "\nu32LastRead=%x", u32LastRead);

	return TOUCH_STATUS_OK;
}


PUBLIC teTouchStatus eTouchSleep(void)
{
	return TOUCH_STATUS_OK;
}

PUBLIC teTouchStatus eTouchWake(void)
{
	return TOUCH_STATUS_OK;
}

PUBLIC teTouchStatus eTouchProcess(void)
{
	teTouchStatus eStatus = TOUCH_STATUS_OK;

	uint32 u32ThisRead;
	uint32 u32Changed;
	uint32 u32Pressed;
	uint32 u32Released;
	uint32 u32Held;

	/* Read buttons */
	u32ThisRead = (~u32AHI_DioReadInput() & DRIVER_DIO_MASK_ALL);;
	u32Changed  = (u32ThisRead ^ u32LastRead);
	u32Pressed  = (u32ThisRead & u32Changed);
	u32Released = (u32LastRead & u32Changed);
	u32Held    	= (u32LastRead & u32ThisRead);

	/* Running ? */
	if (bRunning)
	{
		/* No buttons were held down from the previous reading ? */
		if (u32Held == 0)
		{
			/* Buttons pressed ? */
			if (u32Pressed != 0)
			{
				/* DIO8 ? */
				#if DRIVER_DIO_MASK_PRG
				if (u32Pressed & DRIVER_DIO_MASK_PRG)
				{
					/* Issue on command */
					DBG_vPrintf(TRUE, "\nP PWR");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_PRESSED, E_KEY_PWR);
				}
				#endif
				/* SW1 ? */
				if (u32Pressed & DRIVER_DIO_MASK_ON)
				{
					/* Issue on command */
					DBG_vPrintf(TRUE, "\nP ON");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_PRESSED, E_KEY_ON);
				}
				/* SW2 ? */
				if (u32Pressed & DRIVER_DIO_MASK_OFF)
				{
					/* Issue off command */
					DBG_vPrintf(TRUE, "\nP OFF");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_PRESSED, E_KEY_OFF);
				}
				/* SW3 ? */
				if (u32Pressed & DRIVER_DIO_MASK_UP)
				{
					/* Issue up command */
					DBG_vPrintf(TRUE, "\nP UP");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_PRESSED, E_KEY_UP);
				}
				/* SW4 ? */
				if (u32Pressed & DRIVER_DIO_MASK_DOWN)
				{
					/* Issue down command */
					DBG_vPrintf(TRUE, "\nP DOWN");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_PRESSED, E_KEY_DOWN);
				}
			}
			/* Buttons released ? */
			if (u32Released != 0)
			{
				/* DIO8 ? */
				#if DRIVER_DIO_MASK_PRG
				if (u32Released & DRIVER_DIO_MASK_PRG)
				{
					/* Issue on command */
					DBG_vPrintf(TRUE, "\nR PWR");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_RELEASED, E_KEY_PWR);
				}
				#endif
				/* SW1 ? */
				if (u32Released & DRIVER_DIO_MASK_ON)
				{
					/* Issue on command */
					DBG_vPrintf(TRUE, "\nR ON");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_RELEASED, E_KEY_ON);
				}
				/* SW2 ? */
				if (u32Released & DRIVER_DIO_MASK_OFF)
				{
					/* Issue off command */
					DBG_vPrintf(TRUE, "\nR OFF");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_RELEASED, E_KEY_OFF);
				}
				/* SW3 ? */
				if (u32Released & DRIVER_DIO_MASK_UP)
				{
					/* Issue up command */
					DBG_vPrintf(TRUE, "\nR UP");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_RELEASED, E_KEY_UP);
				}
				/* SW4 ? */
				if (u32Released & DRIVER_DIO_MASK_DOWN)
				{
					/* Issue down command */
					DBG_vPrintf(TRUE, "\nR DOWN");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_RELEASED, E_KEY_DOWN);
				}
			}
		}
		/* Generic SW1 button is held ? */
		else if (u32Held == DRIVER_DIO_MASK_ON)
		{
			/* Buttons pressed ? */
			if (u32Pressed != 0)
			{
				/* SW3 ? */
				if (u32Pressed & DRIVER_DIO_MASK_UP)
				{
					/* Issue up command */
					DBG_vPrintf(TRUE, "\nP ADD");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_PRESSED, E_KEY_ADD_GROUP);
				}
				/* SW4 ? */
				if (u32Pressed & DRIVER_DIO_MASK_DOWN)
				{
					/* Issue down command */
					DBG_vPrintf(TRUE, "\nP DEL");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_PRESSED, E_KEY_DEL_GROUP);
				}
			}
			/* Buttons released ? */
			if (u32Released != 0)
			{
				/* SW3 ? */
				if (u32Released & DRIVER_DIO_MASK_UP)
				{
					/* Issue up command */
					DBG_vPrintf(TRUE, "\nR ADD");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_RELEASED, E_KEY_ADD_GROUP);
				}
				/* SW4 ? */
				if (u32Released & DRIVER_DIO_MASK_DOWN)
				{
					/* Issue down command */
					DBG_vPrintf(TRUE, "\nR DEL");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_RELEASED, E_KEY_DEL_GROUP);
				}
			}
		}
		/* Baseboard DIO8 button is held ? */
		#if DRIVER_DIO_MASK_PRG
		else if (u32Held == DRIVER_DIO_MASK_PRG)
		{
			/* Buttons pressed ? */
			if (u32Pressed != 0)
			{
				/* SW1 ? */
				if (u32Pressed & DRIVER_DIO_MASK_ON)
				{
					/* Issue on command */
					DBG_vPrintf(TRUE, "\nP BULB");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_PRESSED, E_KEY_CMSNG_BULB);
				}
				/* SW2 ? */
				if (u32Pressed & DRIVER_DIO_MASK_OFF)
				{
					/* Issue off command */
					DBG_vPrintf(TRUE, "\nP REM");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_PRESSED, E_KEY_CMSNG_REMOTE);
				}
				/* SW3 ? */
				if (u32Pressed & DRIVER_DIO_MASK_UP)
				{
					/* Issue up command */
					DBG_vPrintf(TRUE, "\nP ANY");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_PRESSED, E_KEY_CMSNG_ANY);
				}
				/* SW4 ? */
				if (u32Pressed & DRIVER_DIO_MASK_DOWN)
				{
					/* Issue down command */
					DBG_vPrintf(TRUE, "\nP DECOM");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_PRESSED, E_KEY_DCMSNG);
				}
			}
			/* Buttons released ? */
			if (u32Released != 0)
			{
				/* SW1 ? */
				if (u32Released & DRIVER_DIO_MASK_ON)
				{
					/* Issue on command */
					DBG_vPrintf(TRUE, "\nR BULB");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_RELEASED, E_KEY_CMSNG_BULB);
				}
				/* SW2 ? */
				if (u32Released & DRIVER_DIO_MASK_OFF)
				{
					/* Issue off command */
					DBG_vPrintf(TRUE, "\nR REM");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_RELEASED, E_KEY_CMSNG_REMOTE);
				}
				/* SW3 ? */
				if (u32Released & DRIVER_DIO_MASK_UP)
				{
					/* Issue up command */
					DBG_vPrintf(TRUE, "\nR BR");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_RELEASED, E_KEY_CMSNG_ANY);
				}
				/* SW4 ? */
				if (u32Released & DRIVER_DIO_MASK_DOWN)
				{
					/* Issue down command */
					DBG_vPrintf(TRUE, "\nR DECOM");
					vCbTouchEventButton(TOUCH_BUTTON_EVENT_RELEASED, E_KEY_DCMSNG);
				}
			}
		}
		#endif

		/* All buttons released ? */
		if (u32Released != 0 && u32ThisRead == 0)
		{
			/* Post all released */
			DBG_vPrintf(TRUE, "\nR ALL");
			vCbTouchEventButton(TOUCH_BUTTON_EVENT_ALL_RELEASED, E_KEY_NONE);
		}
	}
	/* Not fully running yet ? */
	else
	{
		/* All buttons released ? */
		if (u32Released != 0 && u32ThisRead == 0)
		{
			/* DIO8 ? */
			#if DRIVER_DIO_MASK_PRG
			if (u32Released == DRIVER_DIO_MASK_PRG)
			{
				DBG_vPrintf(TRUE, "\nR ALL CREATE");
				/* Issue create network command */
				vCbTouchEventButton(TOUCH_BUTTON_EVENT_ALL_RELEASED, E_KEY_CREATE);
			}
			#endif
			/* SW1 ? */
			if (u32Released == DRIVER_DIO_MASK_ON)
			{
				DBG_vPrintf(TRUE, "\nR ALL JOIN");
				/* Issue join command */
				vCbTouchEventButton(TOUCH_BUTTON_EVENT_ALL_RELEASED, E_KEY_JOIN);
			}
			/* SW2 ? */
			if (u32Released == DRIVER_DIO_MASK_OFF)
			{
				DBG_vPrintf(TRUE, "\nR ALL FACTORY");
				/* Issue factory reset command */
				vCbTouchEventButton(TOUCH_BUTTON_EVENT_ALL_RELEASED, E_KEY_FACTORY_RESET);
			}
		}
		/* No buttons are pressed ? */
		if (u32ThisRead == 0)
		{
			/* Now up and running normally */
			bRunning = TRUE;
			DBG_vPrintf(TRUE, "\nRUNNING");
		}
	}

	/* Note last reading */
	u32LastRead = u32ThisRead;

	return eStatus;
}

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
