/****************************************************************************/
/*
 * MODULE              JN-AN-1162 JenNet-IP Smart Home
 *
 * DESCRIPTION         DeviceColour DeviceScene MIB Implementation
 */
/****************************************************************************/
/*
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2014. All rights reserved
 */
/****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
/* Standard includes */
#include <string.h>
/* SDK includes */
#include <jendefs.h>
/* Hardware includes */
#include <AppHardwareApi.h>
#include <PeripheralRegs.h>
/* Stack includes */
#include <Api.h>
#include <AppApi.h>
#include <JIP.h>
#include <6LP.h>
#include <AccessFunctions.h>
/* JenOS includes */
#include <dbg.h>
#include <dbg_uart.h>
#include <os.h>
#include <pdm.h>
/* Application common includes */
#include "DeviceDefs.h"
#include "Node.h"
#include "MibCommon.h"
#include "Table.h"
/* Application device includes */
#include "MibDeviceScene.h"
#include "MibBulb.h"
#include "MibBulbScene.h"
#include "MibBulbControl.h"
#include "MibColour.h"
#include "MibColourControl.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        External Variables                                            ***/
/****************************************************************************/
/* MIB structures */
extern tsMibDeviceScene   sMibDeviceScene;
extern thJIP_Mib 	      hMibDeviceScene;

#if MK_BLD_MIB_BULB_SCENE
extern tsMibBulbScene     sMibBulbScene;
#endif

#if MK_BLD_MIB_BULB_CONTROL
extern tsMibBulbControl   sMibBulbControl;
#endif

#if MK_BLD_MIB_COLOUR_CONTROL
extern tsMibColourControl sMibColourControl;
#endif

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME: MibDeviceScene_vInit
 *
 * DESCRIPTION:
 * Initialises data
 *
 ****************************************************************************/
PUBLIC void MibDeviceScene_vInit(void)
{
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE,  "\n%sMibDeviceScene_vInit() {%d}", acDebugIndent, sizeof(tsMibDeviceScene));
    Node_vDebugIndent(DEBUG_MIB_DEVICE_SCENE);


	/* Load DeviceScene mib data */
	(void) PDM_eLoadRecord(&sMibDeviceScene.sDesc,
						   (uint16)(MIB_ID_DEVICE_SCENE & 0xFFFF),
						   (void *) &sMibDeviceScene.sPerm,
						   sizeof(sMibDeviceScene.sPerm),
						   FALSE);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, "\n%sPDM_eLoadRecord(DeviceScene) = %d", acDebugIndent, sMibDeviceScene.sDesc.eState);

	/* Building BulbScene MIB ? */
	#if MK_BLD_MIB_BULB_SCENE
	{
		uint8 u8Scene;
		/* Loop through scenes */
		for (u8Scene = 0; u8Scene < DEVICE_SCENES; u8Scene++)
		{
			/* Transfer scene from DeviceScene to BulbScene, without notifications */
			MibDeviceScene_vBulbScene(u8Scene, FALSE);
		}
	}
	#endif

	/* Debug */
    Node_vDebugOutdent(DEBUG_MIB_DEVICE_SCENE);
}

/****************************************************************************
 *
 * NAME: MibDeviceScene_vRegister
 *
 * DESCRIPTION:
 * Registers MIB
 *
 ****************************************************************************/
PUBLIC void MibDeviceScene_vRegister(void)
{
	teJIP_Status eStatus;

    /* Debug */
    DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, "\n%sMibDeviceScene_vRegister()", acDebugIndent);
    Node_vDebugIndent(DEBUG_MIB_DEVICE_SCENE);

	/* Register MIB */
	eStatus = eJIP_RegisterMib(hMibDeviceScene);
    DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, "\n%seJIP_RegisterMib(DeviceScene) = %d", acDebugIndent, eStatus);

	/* Configure table structures */
	sMibDeviceScene.sSceneTable.pvData	   = (void *) sMibDeviceScene.sPerm.asSceneTable;
	sMibDeviceScene.sSceneTable.u32Size	   = sizeof(tsDeviceScene);
	sMibDeviceScene.sSceneTable.u16Entries = DEVICE_SCENES;

    /* Make sure permament data is saved */
    PDM_vSaveRecord(&sMibDeviceScene.sDesc);
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, "\n%sPDM_vSaveRecord(DeviceScene) = %d", acDebugIndent, sMibDeviceScene.sDesc.eState);

    /* Debug */
    Node_vDebugOutdent(DEBUG_MIB_DEVICE_SCENE);
}

/****************************************************************************
 *
 * NAME: MibDeviceScene_vTick
 *
 * DESCRIPTION:
 * Timer function called when stack is running
 *
 ****************************************************************************/
PUBLIC void MibDeviceScene_vTick(void)
{
	/* Are there any variable notifications pending ? */
	if ((sMibDeviceScene.u32NotifyChanged & VAR_MASK_DEVICE_SCENE) != 0)
	{
		/* Use common function to output notifications */
		Node_vJipNotifyChanged(hMibDeviceScene, &sMibDeviceScene.u32NotifyChanged, VAR_MASK_DEVICE_SCENE, VAR_COUNT_DEVICE_SCENE);
	}
}

/****************************************************************************
 *
 * NAME: MibDeviceScene_u8FindSceneId
 *
 * DESCRIPTION:
 * Finds a scene id in the table
 *
 ****************************************************************************/
PUBLIC uint8 MibDeviceScene_u8FindSceneId(uint16 u16SceneId)
{
	uint8 u8Scene;
	uint8 u8Found = 0xff;

	/* Look for scene */
	for (u8Scene = 0; u8Scene < DEVICE_SCENES && u8Found == 0xff; u8Scene++)
	{
		/* Is this the scene we're looking for ? */
		if (sMibDeviceScene.sPerm.asSceneTable[u8Scene].u16SceneId == u16SceneId)
		{
			/* Note its index */
			u8Found = u8Scene;
		}
	}

	return u8Found;
}

/****************************************************************************
 *
 * NAME: MibDeviceScene_eMakeScene
 *
 * DESCRIPTION:
 * Make a scene from the current settings
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibDeviceScene_eMakeScene(uint16 u16SceneId)
{
	teJIP_Status eReturn = E_JIP_ERROR_FAILED;

	/* Building BulbControl or ColourControl MIBs ? */
	#if MK_BLD_MIB_BULB_CONTROL || MK_BLD_MIB_COLOUR_CONTROL
	{
		tsDeviceScene sDeviceScene;

		/* Set device type */
		sDeviceScene.u16DeviceTypeId = MK_JIP_DEVICE_TYPE;
		/* Set scene id */
		sDeviceScene.u16SceneId      = u16SceneId;
		/* Bulb Control ? */
		#if MK_BLD_MIB_BULB_CONTROL
		{
			/* Set bulb control data */
			sDeviceScene.u8BulbControlMode        = sMibBulbControl.sPerm.u8Mode;
			sDeviceScene.u8BulbControlLumTarget   = sMibBulbControl.sPerm.u8LumTarget;
		}
		#endif
		/* Colour Control ? */
		#if MK_BLD_MIB_COLOUR_CONTROL
		{
			/* Set colour control data */
			sDeviceScene.u8ColourControlMode      = sMibColourControl.sPerm.u8Mode;
			/* Temperature */
			#if MK_DRIVER_TYPE_TEMPERATURE
			{
				sDeviceScene.u16ColourControlCctTarget = sMibColourControl.sTemp.u16CctTarget;
			}
			#else
			{
				sDeviceScene.u32ColourControlXYTarget = sMibColourControl.sPerm.u32XYTarget;
			}
			#endif
		}
		#endif

		/* Attempt to add the scene */
		eReturn = MibDeviceScene_eAddScene(&sDeviceScene);
	}
	#endif

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibDeviceScene_eAddScene
 *
 * DESCRIPTION:
 * AddSceneId set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibDeviceScene_eAddScene(tsDeviceScene *psAddScene)
{
	teJIP_Status eReturn = E_JIP_ERROR_BAD_VALUE;
	uint8  u8Scene;

    /* Debug */
    DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, "\n%sMibDeviceScene_eAddScene(%d)", acDebugIndent, psAddScene->u16SceneId);
    Node_vDebugIndent(DEBUG_MIB_DEVICE_SCENE);

	/* Valid scene data ? */
	if (psAddScene->u16DeviceTypeId     == MK_JIP_DEVICE_TYPE
    &&  psAddScene->u16SceneId          != 0
    #if MK_BLD_MIB_BULB_SCENE
	&&  psAddScene->u8BulbControlMode   != VAR_VAL_BULB_CONTROL_MODE_FAILED
	#endif
	#if MK_BLD_MIB_COLOUR_SCENE
	&&  psAddScene->u8ColourControlMode != VAR_VAL_COLOUR_CONTROL_MODE_FAILED
	#endif
	)
	{
		/* Look for this scene */
		u8Scene = MibDeviceScene_u8FindSceneId(psAddScene->u16SceneId);
		/* Didn't find scene ? */
		if (u8Scene >= DEVICE_SCENES)
		{
			/* Look for unused scene */
			u8Scene = MibDeviceScene_u8FindSceneId(0);
		}
		/* Do we have a scene index to add the scene ? */
		if (u8Scene < DEVICE_SCENES)
		{
			/* Store the scene */
			memcpy((uint8 *) &sMibDeviceScene.sPerm.asSceneTable[u8Scene], (uint8 *) psAddScene, sizeof(tsDeviceScene));
			/* Increment table hashes */
			sMibDeviceScene.sSceneTable.u16Hash++;
			/* Make sure permament data is saved */
			PDM_vSaveRecord(&sMibDeviceScene.sDesc);
			/* Debug */
			DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, "\n%sPDM_vSaveRecord(DeviceScene) = %d", acDebugIndent, sMibDeviceScene.sDesc.eState);
			/* Notify traps */
			sMibDeviceScene.u32NotifyChanged |= (1 << VAR_IX_DEVICE_SCENE_SCENE_TABLE);

			/* Building BulbScene MIB ? */
			#if MK_BLD_MIB_BULB_SCENE
			{
				/* Transfer scene from DeviceScene to BulbScene, with notifications */
				MibDeviceScene_vBulbScene(u8Scene, TRUE);
			}
			#endif

			/* Return success */
			eReturn = E_JIP_OK;
		}
		/* No spare slots for scene ? */
		else
		{
			/* Return failed */
			eReturn = E_JIP_ERROR_FAILED;
		}
	}

    /* Debug */
    Node_vDebugOutdent(DEBUG_MIB_DEVICE_SCENE);
    DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, " = %d", eReturn);

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibDeviceScene_eDelScene
 *
 * DESCRIPTION:
 * DelScene handler
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibDeviceScene_eDelScene(uint16 u16SceneId)
{
	teJIP_Status eReturn = E_JIP_OK;
	uint8   	 u8Scene;

	/* Valid scene ? */
	if (u16SceneId != 0)
	{
		/* Look for this scene */
		u8Scene = MibDeviceScene_u8FindSceneId(u16SceneId);
		/* Do we have a scene index to delete the scene ? */
		if (u8Scene < DEVICE_SCENES)
		{
			/* Delete the scene */
			memset((uint8 *) &sMibDeviceScene.sPerm.asSceneTable[u8Scene], 0, sizeof(tsDeviceScene));
			/* Increment table hashes */
			sMibDeviceScene.sSceneTable.u16Hash++;
		    /* Make sure permament data is saved */
		    PDM_vSaveRecord(&sMibDeviceScene.sDesc);
		    /* Debug */
		    DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, "\n%sPDM_vSaveRecord(DeviceScene) = %d", acDebugIndent, sMibDeviceScene.sDesc.eState);
			/* Notify traps */
			sMibDeviceScene.u32NotifyChanged |= (1 << VAR_IX_DEVICE_SCENE_SCENE_TABLE);

			/* Building BulbScene MIB ? */
			#if MK_BLD_MIB_BULB_SCENE
			{
				/* Transfer scene from DeviceScene to BulbScene, with notifications */
				MibDeviceScene_vBulbScene(u8Scene, TRUE);
			}
			#endif
		}
	}

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibDeviceScene_eActivateScene
 *
 * DESCRIPTION:
 * Activates scene
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibDeviceScene_eActivateScene(uint16 u16SceneId)
{
	teJIP_Status eReturn = E_JIP_ERROR_BAD_VALUE;
	uint8  u8Scene;

    /* Debug */
    DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, "\n%sMibDeviceScene_eActivateScene(%d)", acDebugIndent, u16SceneId);
    Node_vDebugIndent(DEBUG_MIB_DEVICE_SCENE);

	/* Valid scene id and type ? */
	if (u16SceneId != 0)
	{
	    /* Debug */
    	DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, " Valid");

		/* Look for this scene */
		u8Scene = MibDeviceScene_u8FindSceneId(u16SceneId);
		/* Do we have a scene index to activate the scene and the scene is the correct type ? */
		if (u8Scene < DEVICE_SCENES)
		{
		    /* Debug */
    		DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, " Found");

    		/* Scene is correct type ? */
			if (sMibDeviceScene.sPerm.asSceneTable[u8Scene].u16DeviceTypeId == MK_JIP_DEVICE_TYPE)
			{
			    /* Debug */
    			DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, " Type");

				/* Building BulbControl MIB ? */
				#if MK_BLD_MIB_BULB_CONTROL
				{
					/* Debug */
					DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, " BulbControl");

					/* Can we set BulbControl Mode ? */
					if (E_JIP_OK == MibBulbControl_eSetMode(sMibDeviceScene.sPerm.asSceneTable[u8Scene].u8BulbControlMode, &sMibBulbControl.sPerm.u8Mode))
					{
						/* Need to notify for BulbControl Mode */
						sMibBulbControl.u32NotifyChanged |= (1 << VAR_IX_BULB_CONTROL_MODE);
					}
					/* Can we set BulbControl LumTarget ? */
					if (E_JIP_OK == MibBulbControl_eSetLumTarget(sMibDeviceScene.sPerm.asSceneTable[u8Scene].u8BulbControlLumTarget, &sMibBulbControl.sPerm.u8LumTarget))
					{
						/* Need to notify for BulbControl Mode */
						sMibBulbControl.u32NotifyChanged |= (1 << VAR_IX_BULB_CONTROL_LUM_TARGET);
					}
				}
				#endif

				/* Building ColourControl MIB ? */
				#if MK_BLD_MIB_COLOUR_CONTROL
				{
					/* Debug */
					DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, " ColourControl");
					/* Can we set ColourControl Mode ? */
					if (E_JIP_OK == MibColourControl_eSetMode(sMibDeviceScene.sPerm.asSceneTable[u8Scene].u8ColourControlMode, &sMibColourControl.sPerm.u8Mode))
					{
						/* Need to notify for BulbControl Mode */
						sMibColourControl.u32NotifyChanged |= (1 << VAR_IX_COLOUR_CONTROL_MODE);
					}

					/* Temperature */
					#if MK_DRIVER_TYPE_TEMPERATURE
					{
						/* Can we set ColourControl CCTTarget ? */
						if (E_JIP_OK == MibColourControl_eSetCctTarget(sMibDeviceScene.sPerm.asSceneTable[u8Scene].u16ColourControlCctTarget, &sMibColourControl.sTemp.u16CctTarget))
						{
							/* Need to notify for BulbControl Mode */
							sMibColourControl.u32NotifyChanged |= (1 << VAR_IX_COLOUR_CONTROL_CCT_TARGET);
						}
					}
					/* Colour ? */
					#else
					{
						/* Can we set ColourControl XYTarget ? */
						if (E_JIP_OK == MibColourControl_eSetXYTarget(sMibDeviceScene.sPerm.asSceneTable[u8Scene].u32ColourControlXYTarget, &sMibColourControl.sPerm.u32XYTarget))
						{
							/* Need to notify for BulbControl Mode */
							sMibColourControl.u32NotifyChanged |= (1 << VAR_IX_COLOUR_CONTROL_XY_TARGET);
						}
					}
					#endif
				}
				#endif

				/* Return success */
				eReturn = E_JIP_OK;
			}
		}
	}

    /* Debug */
    Node_vDebugOutdent(DEBUG_MIB_DEVICE_SCENE);
    DBG_vPrintf(DEBUG_MIB_DEVICE_SCENE, " = %d", eReturn);

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibDeviceScene_vBulbScene
 *
 * DESCRIPTION:
 * TRansfers a DeviceScene scene to a BulbScene scene
 *
 ****************************************************************************/
#if MK_BLD_MIB_BULB_SCENE
PUBLIC void MibDeviceScene_vBulbScene(uint8 u8Scene, bool_t bNotifyChange)
{
	/* Scene is valid ? */
	if (u8Scene < DEVICE_SCENES)
	{
		/* Transfer data */
		sMibBulbScene.sTemp.au16SceneId[u8Scene]       = sMibDeviceScene.sPerm.asSceneTable[u8Scene].u16SceneId;
		sMibBulbScene.sTemp.au8SceneMode[u8Scene]      = sMibDeviceScene.sPerm.asSceneTable[u8Scene].u8BulbControlMode;
		sMibBulbScene.sTemp.au8SceneLumTarget[u8Scene] = sMibDeviceScene.sPerm.asSceneTable[u8Scene].u8BulbControlLumTarget;
		/* Need to notify */
		if (bNotifyChange)
		{
			/* Increment hashes */
			sMibBulbScene.sSceneId.u16Hash++;
			sMibBulbScene.sSceneMode.u16Hash++;
			sMibBulbScene.sSceneLumTarget.u16Hash++;
			/* Flag notifications */
			sMibBulbScene.u32NotifyChanged |= (1 << VAR_IX_BULB_SCENE_SCENE_ID);
			sMibBulbScene.u32NotifyChanged |= (1 << VAR_IX_BULB_SCENE_SCENE_MODE);
			sMibBulbScene.u32NotifyChanged |= (1 << VAR_IX_BULB_SCENE_SCENE_LUM_TARGET);
		}
	}
}
#endif

/****************************************************************************
 *
 * NAME: MibDeviceScene_eSetAddSceneId
 *
 * DESCRIPTION:
 * AddSceneId set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibDeviceScene_eSetAddSceneId(uint16 u16Val, void *pvCbData)
{
	teJIP_Status eReturn = E_JIP_ERROR_FAILED;

	/* Attempt to make a scene from current settings */
	eReturn = MibDeviceScene_eMakeScene(u16Val);
	/* Did we add this scene ? */
	if (eReturn == E_JIP_OK)
	{
		/* Set this variable */
		(void) eSetUint16(u16Val, pvCbData);

		/* BulbScene MIB is being built ? */
		#if MK_BLD_MIB_BULB_SCENE
		{
			/* Echo to BulbScene MIB's AddSceneId variable */
			(void) eSetUint16(u16Val, (void *) &sMibBulbScene.sTemp.u16AddSceneId);
			/* Queue notification */
			sMibBulbScene.u32NotifyChanged |= (1 << VAR_IX_BULB_SCENE_ADD_SCENE_ID);
		}
		#endif
	}

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibDeviceScene_eSetDelSceneId
 *
 * DESCRIPTION:
 * DelSceneId set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibDeviceScene_eSetDelSceneId(uint16 u16Val, void *pvCbData)
{
	teJIP_Status eReturn = E_JIP_ERROR_FAILED;

	/* Attempt to delete scene from current settings */
	eReturn = MibDeviceScene_eDelScene(u16Val);
	/* Did we delete this scene ? */
	if (eReturn == E_JIP_OK)
	{
		/* Set this variable */
		(void) eSetUint16(u16Val, pvCbData);

		/* BulbScene MIB is being built ? */
		#if MK_BLD_MIB_BULB_SCENE
		{
			/* Echo to BulbScene MIB's DelSceneId variable */
			(void) eSetUint16(u16Val, (void *) &sMibBulbScene.sTemp.u16DelSceneId);
			/* Queue notification */
			sMibBulbScene.u32NotifyChanged |= (1 << VAR_IX_BULB_SCENE_DEL_SCENE_ID);
		}
		#endif
	}

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibilluminanceScene_eSetAddScene
 *
 * DESCRIPTION:
 * Handle remote set of a key
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibDeviceScene_eSetAddScene(const uint8 *pu8Val, uint8 u8Len, void *pvCbData)
{
	teJIP_Status eReturn = E_JIP_ERROR_BAD_BUFFER_SIZE;

	/* Has the correct amount of data being passed in ? */
	if (u8Len == sizeof(tsDeviceScene))
	{
		/* Attempt to add the scene */
		eReturn = MibDeviceScene_eAddScene((tsDeviceScene *) pu8Val);
		/* Did we add this scene ? */
		if (eReturn == E_JIP_OK)
		{
			/* Copy data */
			memcpy(pvCbData, pu8Val, u8Len);
		}
	}

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibilluminanceScene_vGetAddScene
 *
 * DESCRIPTION:
 * Handle remote get of a key
 *
 ****************************************************************************/
PUBLIC void MibDeviceScene_vGetAddScene(thJIP_Packet hPacket, void *pvCbData)
{
	eJIP_PacketAddData(hPacket, pvCbData, sizeof(tsDeviceScene), 0);
}
/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
