/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "fsl_mmc.h"
#include "sdmmc_config.h"
#include "fsl_common.h"
#include "fsl_caam.h"
#include "rpmb.h"  
/*******************************************************************************
 * Definitions
 ******************************************************************************/


/*! @brief Data block count accessed in card */
#define DATA_BLOCK_COUNT (1U)
/*! @brief Start data block number accessed in card */
#define DATA_BLOCK_START (0U)
/*! @brief Data buffer size. */
#define DATA_BUFFER_SIZE (FSL_SDMMC_DEFAULT_BLOCK_SIZE * DATA_BLOCK_COUNT)
/*! @brief Macro for using plain key, delete to use option2 */
#define PLAIN_KEY

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/*!
 * @brief printf the card information log.
 *
 * @param card Card descriptor.
 */
static void CardInformationLog(mmc_card_t *card);

/*******************************************************************************
 * Variables
 ******************************************************************************/

/*! @brief Card descriptor. */
mmc_card_t g_sd;
/*! @brief CAAM job ring interface 0 in system memory. */
AT_NONCACHEABLE_SECTION(static caam_job_ring_interface_t s_jrif0);

/* @brief decription about the read/write buffer
 * The size of the read/write buffer should be a multiple of 512, since SDHC/SDXC card uses 512-byte fixed
 * block length and this driver example is enabled with a SDHC/SDXC card.If you are using a SDSC card, you
 * can define the block length by yourself if the card supports partial access.
 * The address of the read/write buffer should align to the specific DMA data buffer address align value if
 * DMA transfer is used, otherwise the buffer address is not important.
 * At the same time buffer address/size should be aligned to the cache line size if cache is supported.
 */
/*! @brief Data written to the card */
SDK_ALIGN(uint8_t g_dataWrite[DATA_BUFFER_SIZE], BOARD_SDMMC_DATA_BUFFER_ALIGN_SIZE);
/*! @brief Data read from the card */
SDK_ALIGN(uint8_t g_dataRead[DATA_BUFFER_SIZE], BOARD_SDMMC_DATA_BUFFER_ALIGN_SIZE);
/*! @brief Data read from the card */
SDK_ALIGN(uint8_t s_KeyModifier[16], 16);

/*32 bytes key: "ultrapassword123ultrapassword123"*/
const uint8_t rpmb_key[32] = {0x75, 0x6c, 0x74, 0x72, 0x61, 0x70, 0x61, 0x73, 0x73, 0x77, 0x6f,
                                              0x72, 0x64, 0x31, 0x32, 0x33, 0x75, 0x6c, 0x74, 0x72, 0x61, 0x70,
                                              0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64, 0x31, 0x32, 0x33};
uint8_t s_blob_data[80];    // 32byte blob key + 32 bytes of data + 16bytes of MAC for this example = 80bytes in total
uint8_t s_blob_DecapData[32];
#define RNG_EXAMPLE_RANDOM_BYTES 32
/*******************************************************************************
 * Code
 ******************************************************************************/
/*******************************************************************************
 * Code
 ******************************************************************************/
static status_t AccessCard(mmc_card_t *card, bool isReadOnly)
{
    if (isReadOnly)
    {
        PRINTF("\r\nRead one data block......\r\n");
        if (kStatus_Success != MMC_ReadBlocks(card, g_dataRead, DATA_BLOCK_START, 1U))
        {
            PRINTF("Read one data block failed.\r\n");
            return kStatus_Fail;
        }

        PRINTF("Read multiple data blocks......\r\n");
        if (kStatus_Success != MMC_ReadBlocks(card, g_dataRead, DATA_BLOCK_START, DATA_BLOCK_COUNT))
        {
            PRINTF("Read multiple data blocks failed.\r\n");
            return kStatus_Fail;
        }
    }
    else
    {
        memset(g_dataWrite, 0x67U, sizeof(g_dataWrite));

        PRINTF("\r\nWrite/read one data block......\r\n");
        if (kStatus_Success != MMC_WriteBlocks(card, g_dataWrite, DATA_BLOCK_START, 1U))
        {
            PRINTF("Write one data block failed.\r\n");
            return kStatus_Fail;
        }

        memset(g_dataRead, 0U, sizeof(g_dataRead));
        if (kStatus_Success != MMC_ReadBlocks(card, g_dataRead, DATA_BLOCK_START, 1U))
        {
            PRINTF("Read one data block failed.\r\n");
            return kStatus_Fail;
        }

        PRINTF("Compare the read/write content......\r\n");
        if (memcmp(g_dataRead, g_dataWrite, FSL_SDMMC_DEFAULT_BLOCK_SIZE))
        {
            PRINTF("The read/write content isn't consistent.\r\n");
            return kStatus_Fail;
        }
        PRINTF("The read/write content is consistent.\r\n");

        PRINTF("Write/read multiple data blocks......\r\n");
        if (kStatus_Success != MMC_WriteBlocks(card, g_dataWrite, DATA_BLOCK_START, DATA_BLOCK_COUNT))
        {
            PRINTF("Write multiple data blocks failed.\r\n");
            return kStatus_Fail;
        }

        memset(g_dataRead, 0U, sizeof(g_dataRead));

        if (kStatus_Success != MMC_ReadBlocks(card, g_dataRead, DATA_BLOCK_START, DATA_BLOCK_COUNT))
        {
            PRINTF("Read multiple data blocks failed.\r\n");
            return kStatus_Fail;
        }

        PRINTF("Compare the read/write content......\r\n");
        if (memcmp(g_dataRead, g_dataWrite, FSL_SDMMC_DEFAULT_BLOCK_SIZE))
        {
            PRINTF("The read/write content isn't consistent.\r\n");
            return kStatus_Fail;
        }
        PRINTF("The read/write content is consistent.\r\n");

        PRINTF("Erase multiple data blocks......\r\n");
        if (kStatus_Success != MMC_EraseGroups(card, 0, 1))
        {
            PRINTF("Erase multiple data blocks failed.\r\n");
            return kStatus_Fail;
        }
    }

    return kStatus_Success;
}

status_t emmc_rpmb_read(mmc_card_t *card, const uint8_t *key)
{
    assert(card != NULL);
    
    uint8_t buffer[256] = {0};
    int n = 0;
    unsigned short block = 0;
    unsigned short counter = 1;
    /* RPMB read function */
    PRINTF("\r\nRPMB read function\r\n");
    PRINTF("\r\nRPMB read: block %d, counter: %d\r\n", block, counter);
    
    n = mmc_rpmb_read(card, buffer, block, counter, key);
    
    if(n != counter)
        return kStatus_Fail;
    
    PRINTF("\r\nRead data:");
    for(int i = 0; i < sizeof(buffer); i++)
    {
        PRINTF("%02X ", buffer[i]);
    }
    PRINTF("\r\n");
    
    return kStatus_Success;
}

status_t emmc_rpmb_write(mmc_card_t *card, const uint8_t *key)
{
    assert(card != NULL);
    
    uint8_t buffer[256]; 
    int n = 0;
    unsigned short block = 0;
    unsigned short counter = 1;

    /* RPMB write function */
    PRINTF("\r\nRPMB write function\r\n");
    PRINTF("\r\nRPMB write: block %d, counter: %d\r\n", block, counter);
    
    memset(buffer, 0x67U, sizeof(buffer));
    n = mmc_rpmb_write(card, buffer, block, counter, key);
    
    if(n != counter)
        return kStatus_Fail;
    
    PRINTF("\r\nWrite data:");
    for(int i = 0; i < sizeof(buffer); i++)
    {
        PRINTF("%02X ", buffer[i]);
    }
    PRINTF("\r\n");
    
    return kStatus_Success;
}


/*!
 * @brief Use Blob to encapsulate rpmb key
 */
static void BlobEnc(CAAM_Type *base, caam_handle_t *handle)
{
    status_t status;
    
    memset(s_KeyModifier, 0, sizeof(s_KeyModifier));

    /* Encrypt a original data to blob */
    status = CAAM_RedBlob_Encapsule(base, handle, s_KeyModifier,sizeof(s_KeyModifier), rpmb_key, sizeof(rpmb_key), s_blob_data);
}

/*!
 * @brief Use Blob to decapsulate rpmb key
 */
static void Blobdec(CAAM_Type *base, caam_handle_t *handle, uint8_t *s_blob_data)
{
    status_t status;
    
    memset(s_blob_DecapData, 0, sizeof(s_blob_DecapData));
    memset(s_KeyModifier, 0, sizeof(s_KeyModifier));

    /* Decrypt a blob and compare with original input data */
    status = CAAM_RedBlob_Decapsule(base, handle, s_KeyModifier, sizeof(s_KeyModifier), s_blob_data, s_blob_DecapData, sizeof(s_blob_DecapData));
}

/*!
 * @brief Main function
 */
int main(void)
{
    mmc_card_t *card = &g_sd;
    CAAM_Type *base = CAAM;
    caam_handle_t caamHandle;
    caam_config_t caamConfig;

    BOARD_ConfigMPU();
    BOARD_InitBootPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();
    BOARD_MMC_Config(card, BOARD_SDMMC_MMC_HOST_IRQ_PRIORITY);

    PRINTF("\r\nEmmc RPMB example.\r\n");
    
    /* Get default configuration. */
    CAAM_GetDefaultConfig(&caamConfig);
    caamConfig.jobRingInterface[0] = &s_jrif0;
    caamHandle.jobRing = kCAAM_JobRing0;

    /* MMC init function */
    if(MMC_Init(card) != kStatus_Success)
    {
        PRINTF("\r\nMMC init fail\r\n");
        return -1;
    }
    
    /* Init CAAM driver, including CAAM's internal RNG */
    if (CAAM_Init(base, &caamConfig) != kStatus_Success)
    {
        /* Make sure that RNG is not already instantiated (reset otherwise) */
        PRINTF("- failed to init CAAM&RNG!\r\n\r\n");
    }
    
//    PRINTF("\r\nRPMB not supported before version 4.41\r\n");
//    /*List emmc basic information */
//    CardInformationLog(card);
//
    /* Switch to the RPMB partition */
    if (kStatus_Success != switch_mmc_rpmb(card))
    {
        PRINTF("\r\nSwitch to the RPMB partition failed!\r\n");
        return -1;
    }

#ifdef PLAIN_KEY   
    /* RPMB set key function */
    /*Cautions!!!!rpmb key can only be program once*/
    /* Option1: use rpmb key directly*/
    if(kStatus_Success != mmc_rpmb_set_key(card, rpmb_key))
    {
        PRINTF("\r\nProgram rpmb key failed!\r\n");
        return -1;
    }
    
    if(kStatus_Success != emmc_rpmb_write(card, rpmb_key))
    {
        PRINTF("\r\nRPMB write fail\r\n");
    }
    
    if(kStatus_Success != emmc_rpmb_read(card, rpmb_key))
    {
        PRINTF("\r\nRPMB read fail\r\n");
    }

#else    
    /* Option2: use caam rng module generate rpmb key and store in blob*/
    /*          every time use rpmb key should decapsulate blob first*/
    if(kStatus_Success != CAAM_RNG_GetRandomData(base, &caamHandle, kCAAM_RngStateHandle0, (uint8_t *)rpmb_key, RNG_EXAMPLE_RANDOM_BYTES, kCAAM_RngDataAny, NULL))
    {
        PRINTF("\r\nUse CAAM RNG driver generate rpmb key fail!\r\n");
    }
    if(kStatus_Success != mmc_rpmb_set_key(card, rpmb_key))
    {
        PRINTF("\r\nProgram rpmb key failed!\r\n");
    }
    BlobEnc(CAAM, &caamHandle);
    Blobdec(CAAM, &caamHandle, s_blob_data);
    
    
    if(kStatus_Success != emmc_rpmb_write(card, (const uint8_t *)s_blob_DecapData))
    {
        PRINTF("\r\nRPMB write fail\r\n");
    }
    
    if(kStatus_Success != emmc_rpmb_read(card, (const uint8_t *)s_blob_DecapData))
    {
        PRINTF("\r\nRPMB read fail\r\n");
    }
    
#endif    
    MMC_CardDeinit(card);

    while (true)
    {
    }
}

static void CardInformationLog(mmc_card_t *card)
{
    assert(card);

    if(card->currentPartition == kMMC_AccessPartitionUserAera)
    {
        PRINTF("\r\nCard partition : User Aera\r\n");
    }
    else if(card->currentPartition == kMMC_AccessPartitionBoot1)
    {
        PRINTF("\r\nCard partition : Boot1\r\n");
    }
    else if(card->currentPartition == kMMC_AccessPartitionBoot2)
    {
        PRINTF("\r\nCard partition : Boot2\r\n");
    }
    else if(card->currentPartition == kMMC_AccessRPMB)
    {
        PRINTF("\r\nCard partition : RPMB\r\n");
    }
    
    PRINTF("\r\nPartition size %d * %d bytes\r\n", card->userPartitionBlocks, card->blockSize);
    PRINTF("\r\nWorking condition:\r\n");
    if (card->busWidth == kMMC_DataBusWidth1bit)
    {
        PRINTF("\r\n  MMC data bus width : 1bitV\r\n");
    }
    else if (card->busWidth == kMMC_DataBusWidth8bit)
    {
        PRINTF("\r\n  MMC data bus width : 8bitV\r\n");
    }

    PRINTF("\r\n  Bus clock frequency : %d HZ\r\n", card->busClock_Hz);
    PRINTF("\r\n  RPMB size : %d MB\r\n", card->extendedCsd.rpmbSizeMult/8);
}
