/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "board.h"
#include "fsl_usart.h"

#include "pin_mux.h"
#include <stdbool.h>
#include <stdio.h>
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define BUF_LEN 						(32)
#define FLX_USART_NUM				(8)

 enum _usart_channel
{
	CHN_0,
	CHN_1,
	CHN_2,
	CHN_3,
	CHN_4,
	CHN_5,
	CHN_6,
	CHN_7,
};

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/* USART user callback */
void USART_UserCallback(USART_Type *base, usart_handle_t *handle, status_t status, void *userData);

/*******************************************************************************
 * Variables
 ******************************************************************************/

static clock_attach_id_t s_clk_id[] = {kFRO12M_to_FLEXCOMM0, kFRO12M_to_FLEXCOMM1, kFRO12M_to_FLEXCOMM2, 
			 kFRO12M_to_FLEXCOMM3, kFRO12M_to_FLEXCOMM4, kFRO12M_to_FLEXCOMM5, kFRO12M_to_FLEXCOMM6, kFRO12M_to_FLEXCOMM7};

static USART_Type *s_demoUSART[FLX_USART_NUM] = USART_BASE_PTRS;
			 
static clock_name_t s_flexcommClks[FLX_USART_NUM] = {kCLOCK_Flexcomm0, kCLOCK_Flexcomm1, kCLOCK_Flexcomm2,
			kCLOCK_Flexcomm3,kCLOCK_Flexcomm4,kCLOCK_Flexcomm5,kCLOCK_Flexcomm6,kCLOCK_Flexcomm7};

static uint32_t s_demoUSARTClkFreq[FLX_USART_NUM];

usart_handle_t g_uartHandle[FLX_USART_NUM];
uint8_t g_channel_no[FLX_USART_NUM];
uint8_t g_txBuffer[FLX_USART_NUM][BUF_LEN] = {0};
uint8_t g_rxBuffer[FLX_USART_NUM][BUF_LEN] = {0};
volatile bool rxBufferEmpty[FLX_USART_NUM] = {true};
volatile bool txBufferFull[FLX_USART_NUM] = {false};
volatile bool txOnGoing[FLX_USART_NUM] = {false};
volatile bool rxOnGoing[FLX_USART_NUM] = {false};

uint8_t g_lenString[] = "Data Length[bytes] =         ";

uint8_t g_tipString[] =
"<Flexcomm-8 UARTs Demo>\r\n\
- UART0 used for this terminal communication\r\n\
- UART1~7 used for loopback test\r\n\
For loopback test, select all 7 UARTs by input '8' or select UART1~7 by input '1' ~ '7':\r\n";

uint8_t g_OKString[] = "UARTx loopback test passed!\r\n";
uint8_t g_ErrString[] = "UARTx loopback test failed!\r\n";
/*******************************************************************************
 * Code
 ******************************************************************************/
/* USART user callback */
void USART_UserCallback(USART_Type *base, usart_handle_t *handle, status_t status, void *userData)
{
  uint8_t n;

  n = *(uint8_t *)userData;
  if (kStatus_USART_TxIdle == status)
  {
    txBufferFull[n] = false;
    txOnGoing[n] = false;
  }
  
  if (kStatus_USART_RxIdle == status)
  {
    rxBufferEmpty[n] = false;
    rxOnGoing[n] = false;
  }
}

void USART0_sendStr(usart_transfer_t *xfer, uint8_t *str)
{

	/* Send string out via UASRT. */
	xfer[CHN_0].data = str;
	xfer[CHN_0].dataSize = strlen((const char *)str);
	txOnGoing[CHN_0] = true;

	USART_TransferSendNonBlocking(s_demoUSART[CHN_0], &g_uartHandle[CHN_0], xfer);

	/* Wait send finished */
	while (txOnGoing[CHN_0]==true)
	{
	}//do nothing, just waiting
	
	return;
}
/*!
 * @brief Main function
 */
int main(void)
{
	usart_config_t config;
	usart_transfer_t sendXfer[FLX_USART_NUM];
	usart_transfer_t receiveXfer[FLX_USART_NUM];
	uint32_t i, j;
	uint8_t ch_no;
	uint8_t ch_ok_cnt;


	/* attach 12 MHz clock to FLEXCOMM0~7 */
	for (i = CHN_0; i < FLX_USART_NUM; i++)
	{
		CLOCK_AttachClk(s_clk_id[i]);
	}

	/* reset FLEXCOMM for USART */
	RESET_PeripheralReset(kFC0_RST_SHIFT_RSTn);

	BOARD_InitPins();
	BOARD_BootClockFROHF48M();

	/*
	 * config.baudRate_Bps = 115200U;
	 * config.parityMode = kUSART_ParityDisabled;
	 * config.stopBitCount = kUSART_OneStopBit;
	 * config.loopback = false;
	 * config.enableTx = false;
	 * config.enableRx = false;
	 */
	USART_GetDefaultConfig(&config);
	config.baudRate_Bps = BOARD_DEBUG_UART_BAUDRATE;
	config.enableTx = true;
	config.enableRx = true;
	
	s_demoUSARTClkFreq[CHN_0] = CLOCK_GetFreq(s_flexcommClks[CHN_0]);
	USART_Init(s_demoUSART[CHN_0], &config, s_demoUSARTClkFreq[CHN_0]);
	g_channel_no[0] = CHN_0;
	USART_TransferCreateHandle(s_demoUSART[CHN_0], &g_uartHandle[CHN_0], USART_UserCallback, &g_channel_no[0]);
	
	config.loopback = true;
	for (i = CHN_1; i < FLX_USART_NUM; i++)
	{
		s_demoUSARTClkFreq[i] = CLOCK_GetFreq(s_flexcommClks[i]);
		USART_Init(s_demoUSART[i], &config, s_demoUSARTClkFreq[i]);
		g_channel_no[i] = i;
		USART_TransferCreateHandle(s_demoUSART[i], &g_uartHandle[i], USART_UserCallback, &g_channel_no[i]);
		
		for (j = 0; j < BUF_LEN; j++)
		{
			g_txBuffer[i][j] = j;
		}
	}

	/* Send g_tipString out via UASRT0. */  
	USART0_sendStr(&sendXfer[CHN_0], g_tipString);
	
	sprintf((char *)g_lenString, "Data Length[bytes] = %d\r\n", BUF_LEN);
	USART0_sendStr(&sendXfer[CHN_0], g_lenString);
	
	while (1)
	{
			
		/* Start to loop back test on UART1~7. */
		/* scan a char via COM with UART0*/
		receiveXfer[CHN_0].data = g_rxBuffer[CHN_0];
		receiveXfer[CHN_0].dataSize = 1;
		
		do
		{
			rxOnGoing[CHN_0] = true;
			USART_TransferReceiveNonBlocking(s_demoUSART[CHN_0], &g_uartHandle[CHN_0], &receiveXfer[CHN_0], NULL);
			while (rxOnGoing[CHN_0])
			{
			}
		} while ( ('1' > g_rxBuffer[CHN_0][0]) || (g_rxBuffer[CHN_0][0] > '8')); //1~8
		
		for (i = CHN_1; i < FLX_USART_NUM; i++)
		{
			sendXfer[i].data = g_txBuffer[i];
			sendXfer[i].dataSize = BUF_LEN;
			receiveXfer[i].data = g_rxBuffer[i];
			receiveXfer[i].dataSize = BUF_LEN;
			rxOnGoing[i] = true;
		}
		ch_no = g_rxBuffer[CHN_0][0] - 0x30;
		g_rxBuffer[CHN_0][0] = 0;
		ch_ok_cnt = 0;
		if(ch_no < 8)
		{
			USART_TransferSendNonBlocking(s_demoUSART[ch_no], &g_uartHandle[ch_no], &sendXfer[ch_no]);
			USART_TransferReceiveNonBlocking(s_demoUSART[ch_no], &g_uartHandle[ch_no], &receiveXfer[ch_no], NULL);
		}
		else
		{
			for (i = CHN_1; i < FLX_USART_NUM; i++)//send
			{
				USART_TransferSendNonBlocking(s_demoUSART[i], &g_uartHandle[i], &sendXfer[i]);
			}
			for (i = CHN_1; i < FLX_USART_NUM; i++)//receive
			{
				USART_TransferReceiveNonBlocking(s_demoUSART[i], &g_uartHandle[i], &receiveXfer[i], NULL);
			}
		}
		
		i = CHN_1;
		while(1)//check results
		{
			if(rxOnGoing[i] == false)
			{
				rxOnGoing[i] = true;
				ch_ok_cnt++;
		
				for (j = 0; j < BUF_LEN; j++)
				{
						if (g_txBuffer[i][j] != g_rxBuffer[i][j])//failed
						{
							g_ErrString[4] = i + 0x30;
							USART0_sendStr(&sendXfer[CHN_0], g_ErrString);
							while(1);//halt to see the result
						} 
				}
				/* Check if we matched on all of the bytes */
				if (j == BUF_LEN)//passed
				{
					g_OKString[4] = i + 0x30;
					USART0_sendStr(&sendXfer[CHN_0], g_OKString);
				}
				if(ch_no < 8)//single channel test
				{
					if (ch_ok_cnt)
					{
						break;
					}
				}
				else //all channels test
				{	
					if (ch_ok_cnt == FLX_USART_NUM-1)
					{
						break;
					}			
				}			
			}//if(rxOnGoing[i] == false)		
			i++;
			if (i == FLX_USART_NUM)
			{
				i = CHN_1;
			}
		}//while(1)//check results
	}
}
