/*
===============================================================================
 Name        : app_blinky.c
 Author      : $(NXP LPC MCU)
 Version     :
 Copyright   : $(NXP)
 Description : main definition
               system will re-entry secondary bootloader(SBL)
               in APP_TIMEOUT seconds
===============================================================================
*/

#include "chip.h"
#include "board.h"

#include <cr_section_macros.h>

/* Declaration in "Settings" Menu */
//#define APP_IMAGE_NO  (1)         /* image numbers, 0 - first image, 1 - second image */

#define TICKRATE_HZ1 (1)          /* 1 ticks per second */
#define TICKRATE_HZ2 (2)          /* 2 ticks per second */
#define PRESCALE_HZ2 (0xFFFF)     /* 16-bit prescale count */

#define APP_TIMEOUT  (10U)        /* Enter Secondary bootloader after 10U senconds */

volatile uint32_t dwSysTicks = 0; /* tick couter value */

/**
 * @brief Handle interrupt from 32-bit timer 0
 * @return  Nothing
 */
void TIMER32_0_IRQHandler(void)
{
  /* Check Timer Counter is Match */
  if (Chip_TIMER_MatchPending(LPC_TIMER32_0, 1))
  {
	dwSysTicks++;                             /* Increase tick counter */
    Chip_TIMER_ClearMatch(LPC_TIMER32_0, 1);  /* Clear Match status flag */
#if (APP_IMAGE_NO == 0)
    Board_LED_Toggle(0);                      /* Toggle LED 0 status -- blinky LED 0 */
#endif
  }
}

/**
 * @brief	Handle interrupt from 16-bit timer 0
 * @return	Nothing
 */
void TIMER16_0_IRQHandler(void)
{
  /* Check Timer Counter is Match */
  if (Chip_TIMER_MatchPending(LPC_TIMER16_0, 1))
  {
    Chip_TIMER_ClearMatch(LPC_TIMER16_0, 1);  /* Clear Match status flag */
#if (APP_IMAGE_NO == 1)
    Board_LED_Toggle(2);                      /* Toggle LED 2 status -- blinky LED 2 */
#endif
  }
}

/* Secondary bootloader re-entery pointer */
typedef void (*reEntrySBL)(void);
/* re-entery function declare */
volatile reEntrySBL jump2sbl;

/**
 * @brief   Main
 * @return  Nothing
 */
int main(void)
{
  volatile uint32_t timerFreq, i;

#if (APP_IMAGE_NO == 0)
  /* 1st Image VTOR address */
  SCB->RESERVED0 = 0x2000;    /* VTOR address start from app head, only for CM0+ or CM3/4 core */
#else
  /* 2nd Image VTOR address */
  SCB->RESERVED0 = 0x20000;   /* VTOR address start from app head, only for CM0+ or CM3/4 core */
#endif

  Board_Init();               /* Board system initialize */
  SystemCoreClockUpdate();    /* Update SystemCoreClock with the right system clock value */
  /* Set the LED to the state of "On" */
  Board_LED_Set(0, false);
  Board_LED_Set(2, false);

  /* Initialize 32-bit timer 0 clock */
  Chip_TIMER_Init(LPC_TIMER32_0);
  /* Initialize 16-bit timer 0 clock */
  Chip_TIMER_Init(LPC_TIMER16_0);

  /* Timer rate is system clock rate */
  timerFreq = Chip_Clock_GetSystemClockRate();

  /* Timer setup for match and interrupt at TICKRATE_HZ */
  Chip_TIMER_Reset(LPC_TIMER32_0);
  Chip_TIMER_Reset(LPC_TIMER16_0);

  /* Enable both timers to generate interrupts when time matches */
  Chip_TIMER_MatchEnableInt(LPC_TIMER32_0, 1);
  Chip_TIMER_MatchEnableInt(LPC_TIMER16_0, 1);

  /* Setup prescale value on 16-bit timer to extend range */
  Chip_TIMER_PrescaleSet(LPC_TIMER16_0, PRESCALE_HZ2);

  /* Setup 32-bit timer's duration (32-bit match time) */
  Chip_TIMER_SetMatch(LPC_TIMER32_0, 1, (timerFreq / TICKRATE_HZ1));

  /* Setup 16-bit timer's duration (16-bit match time) */
  Chip_TIMER_SetMatch(LPC_TIMER16_0, 1, (timerFreq / TICKRATE_HZ2) >> 16);

  /* Setup both timers to restart when match occurs */
  Chip_TIMER_ResetOnMatchEnable(LPC_TIMER32_0, 1);
  Chip_TIMER_ResetOnMatchEnable(LPC_TIMER16_0, 1);

  /* Start both timers */
  Chip_TIMER_Enable(LPC_TIMER32_0);
  Chip_TIMER_Enable(LPC_TIMER16_0);

  /* Clear both timers of any pending interrupts */
  NVIC_ClearPendingIRQ(TIMER_32_0_IRQn);
  NVIC_ClearPendingIRQ(TIMER_16_0_IRQn);

  /* Enable both timer interrupts */
  NVIC_EnableIRQ(TIMER_32_0_IRQn);
  NVIC_EnableIRQ(TIMER_16_0_IRQn);

  /* Wait for timers to generate interrupts (LEDs toggle in interrupt handlers) */
  while (1)
  {
    /* after 10seconds(refer APP_TIMEOUT, here is 10), system enter secondary bootload */
	  if(dwSysTicks >= APP_TIMEOUT)
	  {
		  /* Set app entry point */
		  jump2sbl =(reEntrySBL)*(uint32_t*)(0x00001F00);

		  /* Switch system clock resource to IRC */
		  /* Prevent PLL shut down and system corrupt */
		  Chip_Clock_SetMainClockSource(SYSCTL_MAINCLKSRC_IRC);
		  /* Wait for at least 580uS for osc to stabilize */
		  for (i = 0; i < 2500; i++) {}

		  /* disable interrupt */
#if   defined ( __GNUC__ )
			__ASM volatile ("cpsid i" : : : "memory");
#elif defined ( __CC_ARM )
			__ASM ("cpsid i");
#else
	XXXX  /* IDE will show Error!!! reminder for different compiler */
#endif
      	  /* jump to the secondary boot loader */
		  /* Jump_To_Application = (pFunction) JumpAddress; */
		  jump2sbl();
	  }
	  __WFI(); /* wait interrupt */
  }
  return 0 ; /* should never run to here */
}

// end file
