/****************************************************************************
 *   $Id:: LPC812_IAP_Demo                                                  $
 *   Project: NXP LPC8xx IAP example
 *
 *   Description:
 *     Demonstrate basic capability of In-Application Programming feature
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
 ****************************************************************************/

#include <LPC8xx.h>
#include "iap_driver.h"
#include "lpc8xx_gpio.h"
#include "lpc8xx_clkconfig.h"
#include "lpc8xx_uart.h"
#include <stdio.h>
#include <string.h>

void menu(void);
void CopyInterruptToSRAM(void);
void ClearStringBuffer(void);
uint8_t get_key( void );
extern volatile uint32_t UARTRxCount;
extern volatile uint32_t RxErrorCount;
extern volatile uint8_t UARTRxBuffer[BUFSIZE];

/* Flash offset where the configuration is stored */
#define CONFIG_FLASH_ADDR_OFFSET     0x2800
#define CONFIG_FLASH_PAGE            (CONFIG_FLASH_ADDR_OFFSET >> 6)
#define CONFIG_FLASH_SECTOR_FOR_PAGE (CONFIG_FLASH_PAGE >> 4)
#define CONFIG_FLASH_PAGE_SIZE       1
#define CONFIG_FLASH_SECTOR_SIZE     (CONFIG_FLASH_PAGE_SIZE >> 4)

#define SYSTICK_DELAY		(SystemCoreClock/1000)
#define MENU_SIZE 7
volatile uint8_t* menu_array[MENU_SIZE] = {
(uint8_t*)"\r\nLPC8xx IAP Test\r\n",
(uint8_t*)"'r' - Read Part ID\r\n",
(uint8_t*)"'b' - Read Boot Code Version\r\n",	
(uint8_t*)"'s' - Read Serial Number\r\n",
(uint8_t*)"'p' - Prepare sector for write\r\n",
(uint8_t*)"'f' - Erase Page\r\n",
(uint8_t*)"'0' - Write on Page\r\n",	
};
typedef enum {
	CMD_READ_ID = 'r',
	CMD_READ_BOOT = 'b',
	CMD_READ_SERIAL = 's',
	CMD_PREPARE_SECTOR = 'p',
	CMD_ERASE_PAGE = 'f',
	CMD_WRITE = '0',
}commandEnum;

const char ascii[] = "0123456789ABCDEF";
char lineBuff[80], valBuff[80];
static uint8_t demo_messg[64] = "Greetings from IAP";
uint8_t *bootFirstByte, *bootSecondByte, bootTemp, bootTemp1;
static char cmd;

int main(void) {
	
	uint32_t i, regVal,iapTemp;
	uint32_t *part_id, *serialPtr, serialTemp[4];
	__e_iap_status iap_status;
	
	/* Init the IAP driver */
	iap_init();
	/* Initialize GPIO */
	GPIOInit();


	/* Set port PIO0_7 to output */
	GPIOSetDir( PORT0, 7, 1 );	
	SysTick_Config(SystemCoreClock/1000 );
	
	/* Config CLKOUT, mostly used for debugging. */
	regVal = LPC_SWM->PINASSIGN8;
	regVal &= ~( 0xFF << 16 );
	regVal |= ( 17 << 16 );
	LPC_SWM->PINASSIGN8 = regVal;	/* P0.17 is CLKOUT, ASSIGN(23:16). */
  CLKOUT_Setup( CLKOUTCLK_SRC_MAIN_CLK );
	//	CLKOUT_Setup( CLKOUTCLK_SRC_IRC_OSC );

	//  Initialize UARTx pin connect 
  /* connect the UART0 TXD abd RXD sigals to port pins(P0.4-P0.0)*/
	regVal = LPC_SWM->PINASSIGN0 & ~( 0xFF << 0 );
	LPC_SWM->PINASSIGN0 = regVal | ( 4 << 0 );				/* P0.4 is UART0 TXD, ASSIGN0(7:0) */
	regVal = LPC_SWM->PINASSIGN0 & ~( 0xFF << 8 );
	LPC_SWM->PINASSIGN0 = regVal | ( 0 << 8 );				/* P0.0 is UART0 RXD. ASSIGN0(15:8) */
	
	regVal = LPC_SWM->PINASSIGN0 & ~( 0xFF << 16 );
	LPC_SWM->PINASSIGN0 = regVal | ( 12 << 16 );			/* P0.12 is UART0 RTS, ASSIGN0(23:16) */
	regVal = LPC_SWM->PINASSIGN0 & ~( 0xFFUL << 24 );
	LPC_SWM->PINASSIGN0 = regVal | ( 13 << 24 );			/* P0.13 is UART0 CTS. ASSIGN0(31:24) */


	UARTInit(LPC_USART0, 9600);
	menu();	

#if defined IRQ_HANDLER_IN_SRAM
	CopyInterruptToSRAM();		//remap interrupt vector to SRAM
	SCB->VTOR =0x10000000;		
	LPC_SYSCON->SYSMEMREMAP = 0x01;	//change memory map
#endif

while(1)
{ 
	ClearStringBuffer();
	cmd = get_key();
	//UARTSend(LPC_USART0, (uint8_t*)\r\n");
	switch(cmd)
	{
		case CMD_READ_ID:
			/* Part ID */
			part_id = &iapTemp;
			iap_status = (__e_iap_status)iap_read_part_id(part_id);
		
			if (iap_status == CMD_SUCCESS) 
				{ 
					sprintf(lineBuff, " Part ID is %c%c%c%c%c%c%c%c\r\n", ascii[((uint8_t)(iapTemp>>24))>>4],ascii[((uint8_t)(iapTemp>>24))& 0x0F],
					ascii[((uint8_t)(iapTemp>>16))>>4], ascii[((uint8_t)(iapTemp>>16))& 0x0F],
					ascii[((uint8_t)(iapTemp>>8))>>4], ascii[((uint8_t)(iapTemp>>8))& 0x0F],
					ascii[((uint8_t)iapTemp>>4)], ascii[((uint8_t)iapTemp& 0x0F)]);
					
					UARTSend(LPC_USART0,(uint8_t*)lineBuff);
				}
			else
			{
					/* Invalid part id */
					UARTSend(LPC_USART0, (uint8_t*)" Invalid Part ID \r\n");
			}
			break;
		case CMD_READ_BOOT:
					/* BootROM revision */
						bootFirstByte = &bootTemp;
						bootSecondByte = &bootTemp1;
						iap_status = (__e_iap_status)iap_read_bootcode_rev(bootFirstByte,bootSecondByte);
						if (iap_status !=CMD_SUCCESS) 
						{
							/* Invalid bootcode rev */
							UARTSend(LPC_USART0, (uint8_t*)" Read Boot Version Failed \r\n");
						}
						else
							{ 
								sprintf(lineBuff, " Boot code version # is %d.%d\r\n",bootTemp,bootTemp1);
								UARTSend(LPC_USART0,(uint8_t*)lineBuff);
			
							}
			break;
							
		case CMD_READ_SERIAL:
					/* Unique ID */
					serialPtr = &serialTemp[0];
					iap_status = (__e_iap_status) iap_read_unique_id(serialPtr);
					if (iap_status == CMD_SUCCESS)
				{  
					sprintf(lineBuff, "");
					
					for (i =0; i<4; i++)
					{		
							sprintf(valBuff, "%c%c%c%c%c%c%c%c ",ascii[((uint8_t)(serialTemp[i]>>24))>>4],ascii[((uint8_t)(serialTemp[i]>>24))& 0x0F],
							ascii[((uint8_t)(serialTemp[i]>>16))>>4], ascii[((uint8_t)(serialTemp[i]>>16))& 0x0F],
							ascii[((uint8_t)(serialTemp[i]>>8))>>4], ascii[((uint8_t)(serialTemp[i]>>8))& 0x0F],
							ascii[((uint8_t)serialTemp[i]>>4)], ascii[((uint8_t)serialTemp[i]& 0x0F)]);
							strcat(lineBuff, valBuff);
							
					}
					strcat(lineBuff, "\r\n");
					sprintf(valBuff, " Serial # is ");
					strcat(valBuff,lineBuff);
					UARTSend(LPC_USART0,(uint8_t*) valBuff);
				}
				else
				{
					UARTSend(LPC_USART0,(uint8_t*)" Read serial number failed\r\n");
				}
				break;

					
		case CMD_PREPARE_SECTOR:
					/* Prepare the page for erase */
				iap_status = (__e_iap_status) iap_prepare_sector(CONFIG_FLASH_SECTOR_FOR_PAGE,
					(CONFIG_FLASH_SECTOR_FOR_PAGE + CONFIG_FLASH_SECTOR_SIZE));
				if (iap_status != CMD_SUCCESS) 
				{
					UARTSend(LPC_USART0,(uint8_t*)" Preparation of Sector for Write Failed\r\n");
				}
				else
				{
					UARTSend(LPC_USART0,(uint8_t*)" Preparation of Sector for Write Successful\r\n");
				}
					break;
				
		
					
		case CMD_ERASE_PAGE:
			 /* Preparing Sector  */
				iap_status = (__e_iap_status) iap_prepare_sector(CONFIG_FLASH_SECTOR_FOR_PAGE,
					(CONFIG_FLASH_SECTOR_FOR_PAGE + CONFIG_FLASH_SECTOR_SIZE));
				if (iap_status != CMD_SUCCESS) 
				{
					UARTSend(LPC_USART0,(uint8_t*)" Erase Failed\r\n");
				}
				else
				{
					/* Erase the page */
					iap_status = (__e_iap_status) iap_erase_page(CONFIG_FLASH_PAGE,
							(CONFIG_FLASH_PAGE + CONFIG_FLASH_PAGE_SIZE - 1));
					if (iap_status != CMD_SUCCESS) 
					{
						UARTSend(LPC_USART0,(uint8_t*)" Erase Failed\r\n");
					}
					else
					{
						UARTSend(LPC_USART0,(uint8_t*) " Erase Completed\r\n");
					}
				}
					break;
		case CMD_WRITE:
				/* Prepare the page for write */
				iap_status = (__e_iap_status) iap_prepare_sector(CONFIG_FLASH_SECTOR_FOR_PAGE,(CONFIG_FLASH_SECTOR_FOR_PAGE + CONFIG_FLASH_SECTOR_SIZE)) ; 
					if (iap_status != CMD_SUCCESS) 
				{
					UARTSend(LPC_USART0,(uint8_t*)" Write Failed\r\n");
				}
				else
				{
				/* Write data to page */
				iap_status = (__e_iap_status) iap_copy_ram_to_flash(&demo_messg,(void *)CONFIG_FLASH_ADDR_OFFSET, 64);
				if (iap_status != CMD_SUCCESS) 
				{
					UARTSend(LPC_USART0,(uint8_t*)" Page  Write Failed\r\n");
				}
				else
				{
					
				UARTSend(LPC_USART0,(uint8_t*)" Page  Write Successful\r\n");
				}
			}
				break;
		
			default:
				UARTSend(LPC_USART0,(uint8_t*)" Invalid Command ! \r\n");
		}
		menu();
}
}


void CopyInterruptToSRAM(void)
{
	unsigned int * flashPtr, * ramPtr;
  unsigned int * uLimit = (unsigned int *) 0x200; 		
		
		ramPtr = (unsigned int *)0x10000000;	//load RAM starting at 0x10000000, 
		flashPtr = (unsigned int *)0x00;			//start of interrupt vector table
	  while(flashPtr < uLimit)
			{
				*ramPtr = *flashPtr;
				ramPtr++;
				flashPtr++; 
			}
}



/*********************************************************************//**
 * @brief        Clear the char buffer to output to UART
 *
 * @param[in]    void
 *
 * @return       void
 *
 **********************************************************************/
void ClearStringBuffer(void)
{
	int i;
	for (i =0; i<80; i++)
	{
		valBuff[i] = 0x00;
		lineBuff[i] = 0x00;
	}
}

/*********************************************************************//**
 * @brief        Display the menu through UART
 *
 * @param[in]    void
 *
 * @return       void
 *
 **********************************************************************/
void menu()
{
	uint32_t i;

	for (i=0; i<MENU_SIZE; i++)
	{	
		UARTSend(LPC_USART0, menu_array[i]);
	}
}






