/*********************************************************************
*                SEGGER MICROCONTROLLER GmbH & Co. KG                *
*        Solutions for real time microcontroller applications        *
**********************************************************************
*                                                                    *
*        (c) 2003-2012     SEGGER Microcontroller GmbH & Co KG       *
*                                                                    *
*        Internet: www.segger.com    Support:  support@segger.com    *
*                                                                    *
**********************************************************************

----------------------------------------------------------------------
File    : I2C_PCA9532.c
Purpose : Generic I2C_PCA9532 interface module
--------  END-OF-HEADER  ---------------------------------------------
*/

#include "lpc_i2c.h"
#include "lpc_pinsel.h"
#include "I2C_PCA9532.h"

/*********************************************************************
*
*       Defines
*
**********************************************************************
*/

//
// Register addresses
//
#define INPUT0  0
#define INPUT1  1
#define PSC0    2
#define PWM0    3
#define PSC1    4
#define PWM1    5
#define LS0     6
#define LS1     7
#define LS2     8
#define LS3     9

//PCA9532 link to I2C0 only
#define PCA9532_I2C		I2C_0

/*********************************************************************
*
*       Static data
*
**********************************************************************
*/

static uint8_t _IsInited;

/*********************************************************************
*
*       Static functions
*
**********************************************************************
*/
/*********************************************************************
*
*       _SetPinSelect
*
* Function description
*   Sets pin function select for one pin.
*
* Return value
*      0: O.K.
*   != 0: Error
*/
static uint8_t _SetPinSelect(uint8_t Addr, uint8_t Pin, uint8_t State) {
  uint8_t Data[2];
  Status r;
  I2C_M_SETUP_Type i2cData;

  if        (Pin < 4) {
    Data[0]  = LS0;
  } else if (Pin < 8) {
    Data[0]  = LS1;
    Pin     -= 4;
  } else if (Pin < 12) {
    Data[0]  = LS2;
    Pin     -= 8;
  } else if (Pin < 16) {
    Data[0]  = LS3;
    Pin     -= 12;
  } else {
    return 1;  // Error, invalid pin
  }
  Pin *= 2;

  i2cData.sl_addr7bit = Addr;
  i2cData.tx_data = &Data[0];
  i2cData.tx_length = 1;
  i2cData.rx_data = &Data[1];
  i2cData.rx_length = 1;
  i2cData.retransmissions_max = 3;

  r    = I2C_MasterTransferData(PCA9532_I2C, &i2cData, I2C_TRANSFER_POLLING);
  if (r == SUCCESS) {
    Data[1] &= ~(0x3   << Pin);
    Data[1] |=  (State << Pin);
    i2cData.tx_data = Data;
    i2cData.tx_length = 2;
    i2cData.rx_data = NULL;
    i2cData.rx_length = 0;
    i2cData.retransmissions_max = 3;
  r    = I2C_MasterTransferData(PCA9532_I2C, &i2cData, I2C_TRANSFER_POLLING);
  }
  return r;
}

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/

/*********************************************************************
*
*       I2C_PCA9532_GetPinState
*
* Function description
*   Retrieves the state for all 16 pins.
*
* Return value
*      0: O.K.
*   != 0: Error
*/
uint8_t I2C_PCA9532_GetPinState(uint32_t I2CBaseAddr, uint8_t Addr, uint16_t * pState) {
  uint8_t  Register;
  Status  r;
  I2C_M_SETUP_Type i2cData;
  Register = INPUT0;
  	i2cData.sl_addr7bit = Addr;
	i2cData.tx_data = &Register;
	i2cData.tx_length = 1;
	i2cData.rx_data = (uint8_t*)pState;
	i2cData.rx_length = 1;
	i2cData.retransmissions_max = 3;
  r    = I2C_MasterTransferData(PCA9532_I2C, &i2cData, I2C_TRANSFER_POLLING);
    if (r == SUCCESS) {
      Register++;  // Set to INPUT1
      i2cData.tx_data = &Register;
      i2cData.rx_data = (uint8_t*)pState+1;
  r    = I2C_MasterTransferData(PCA9532_I2C, &i2cData, I2C_TRANSFER_POLLING);
    }

  return r;
}

/*********************************************************************
*
*       I2C_PCA9532_SetPinSelect
*
* Function description
*   Sets pin function select for one pin.
*
* Return value
*      0: O.K.
*   != 0: Error
*/
uint8_t I2C_PCA9532_SetPinSelect(uint32_t I2CBaseAddr, uint8_t Addr, uint8_t Pin, uint8_t State) {
	Status r;

	  r = _SetPinSelect(Addr, Pin, State);
	  return r;
}

/*********************************************************************
*
*       I2C_PCA9532_SetPwm
*
* Function description
*   Sets the timing for PWM0 or PWM1 and assigns the pin to the
*   according PWM.
*
* Return value
*      0: O.K.
*   != 0: Error
*/
uint8_t I2C_PCA9532_SetPwm(uint32_t I2CBaseAddr, uint8_t Addr, uint8_t Pin, uint8_t PwmNumber, uint8_t Pwm, uint8_t Psc) {
  uint8_t Data[2];
  Status r;
  I2C_M_SETUP_Type i2cData;

  if (PwmNumber > 1) {
      return 1;  // Error, invalid PWM number
    }

    if (PwmNumber) {
      Data[0] = PSC1;
    } else {
      Data[0] = PSC0;
    }
    Data[1] = Psc;
    i2cData.sl_addr7bit = Addr;
    i2cData.tx_data = Data;
    i2cData.tx_length = 2;
    i2cData.rx_data = NULL;
    i2cData.rx_length = 0;
    i2cData.retransmissions_max = 3;
  r    = I2C_MasterTransferData(PCA9532_I2C, &i2cData, I2C_TRANSFER_POLLING);
    if (r == SUCCESS) {
      if (PwmNumber) {
        Data[0] = PWM1;
      } else {
        Data[0] = PWM0;
      }
      Data[1] = Pwm;
  r    = I2C_MasterTransferData(PCA9532_I2C, &i2cData, I2C_TRANSFER_POLLING);
      if (r == SUCCESS) {
        r = _SetPinSelect( Addr, Pin, I2C_PCA9532_PWM0 + PwmNumber);
      }
    }

  return r;
}

/*********************************************************************
*
*       I2C_PCA9532_Init()
*/
void I2C_PCA9532_Init(uint32_t ClockRate) {
  if (_IsInited == 0) {
	  PINSEL_ConfigPin (0, 27, 1);
	  PINSEL_ConfigPin (0, 28, 1);
	  I2C_Init(PCA9532_I2C, ClockRate);
	  /* Enable I2C0 operation */
	  I2C_Cmd(PCA9532_I2C, ENABLE);
    _IsInited = 1;
  }
}

/*************************** End of file ****************************/
