/**
 * @file		lpc17xx_vocoder.c
 * @purpose		Speex Vocoder
 * @version		1.1
 * @date		14. 02. 2012
 * @author		CT-PIM/NXP
 *----------------------------------------------------------------------------
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
 **********************************************************************/

/**********************************************************************/
/* Include files                                                      */
/**********************************************************************/

#include "lpc17xx_vocoder.h"
#include "lpc17xx_libcfg.h"
#include "lpc17xx_pinsel.h"
#include "lpc17xx_timer.h"
#if (BOARD_IN_USE == IAR_LPC1768_SK)
#include "lpc17xx_adc.h"
#include "lpc17xx_dac.h"
#elif (BOARD_IN_USE == CODE_RED_RDB1768)
#include "lpc17xx_i2s.h"
#include "uda1380.h"
#endif /* (BOARD_IN_USE == CODE_RED_RDB1768) */

#include "lpc17xx_gpio.h"


/**********************************************************************/
/* Private Types                                                      */
/**********************************************************************/

/* encoded (Speex) frame size */
#define SPEEX_FRAME_SIZE  	   20

#define AUDIO_SAMPLE_TOTAL_FRAMES  300

#define AUDIO_SAMPLE_RATE_8KHZ     125

#define ADC_CHANNEL__ ADC_CHANNEL_2
#define ADC_INT__     ADC_ADINTEN2

#define CODEC_STATUS_IDLE   0
#define CODEC_STATUS_REC    1
#define CODEC_STATUS_PLAY   2

/**********************************************************************/
/* Private Types Variables                                            */
/**********************************************************************/

/* Input buffer */
spx_int16_t  IN_Buffer[2][FRAME_SIZE];
int  sample_in;

/* Output buffer */
spx_int16_t  OUT_Buffer[2][FRAME_SIZE];

uint16_t     sample_index = 0;
/* Speex Bits */
SpeexBits    bits;

uint8_t codec_status = CODEC_STATUS_IDLE;

/* Decoder variables */

int  sample;
int  current_output_buffer = 0;
int  current_decode_buffer = 1;
int  decode_new_frame = 0;

int  sample_in = 0;
int  current_input_buffer = 0;
int  current_encode_buffer = 1;
int  encode_new_frame = 0;

/* Encoder variables */
void *enc_state, *dec_state;            /* Holds the state of the encoder */

/* Array to store the encoded audio */
char rec_voice[SPEEX_FRAME_SIZE*AUDIO_SAMPLE_TOTAL_FRAMES];

#if (BOARD_IN_USE == CODE_RED_RDB1768)
volatile uint8_t  I2SRXDone = 0;
#endif /* (BOARD_IN_USE == CODE_RED_RDB1768) */


int quality = 4, complexity=1, vbr=0, enh=1;/* SPEEX PARAMETERS, MUST REMAINED UNCHANGED */

/**********************************************************************/
/* Public Functions Definitions                                       */
/**********************************************************************/
#if (BOARD_IN_USE == CODE_RED_RDB1768)
void RDB1768_AudioInit(void)
{
	PINSEL_CFG_Type PinCfg;
	I2S_MODEConf_Type I2S_ClkConfig;
	I2S_CFG_Type I2S_ConfigStruct;

	/* Initialize I2S peripheral ------------------------------------*/
	/* Pin configuration:
	 * Assign: 	- P0.4  as I2SRX_CLK
	 * 			- P0.24 as I2SRX_WS
	 * 			- P0.25 as I2SRX_SDA
	 * 			- P0.7  as I2STX_CLK
	 * 			- P0.8  as I2STX_WS
	 * 			- P0.9  as I2STX_SDA
	 */
	PinCfg.Funcnum = 1;
	PinCfg.OpenDrain = 0;
	PinCfg.Pinmode = 0;
	PinCfg.Pinnum = 4;
	PinCfg.Portnum = 0;
	PINSEL_ConfigPin(&PinCfg);
	PinCfg.Pinnum = 7;
	PINSEL_ConfigPin(&PinCfg);
	PinCfg.Pinnum = 8;
	PINSEL_ConfigPin(&PinCfg);
	PinCfg.Pinnum = 9;
	PINSEL_ConfigPin(&PinCfg);

	PinCfg.Funcnum = 2;
	PinCfg.Pinnum = 24;
	PINSEL_ConfigPin(&PinCfg);
	PinCfg.Pinnum = 25;
	PINSEL_ConfigPin(&PinCfg);

	/* P4.28 as RX_MCLK and P4.29 as TX_MCLK */
	PinCfg.Funcnum = 1;
	PinCfg.OpenDrain = 0;
	PinCfg.Pinmode = 0;
	PinCfg.Pinnum = 28;
	PinCfg.Portnum = 4;
	PINSEL_ConfigPin(&PinCfg);
	PinCfg.Pinnum = 29;
	PINSEL_ConfigPin(&PinCfg);
	I2S_Init(LPC_I2S);

	/* setup:
	 * 		- wordwidth: 16 bits
	 * 		- mono mode
	 * 		- master mode for I2S_TX and slave for I2S_RX
	 * 		- ws_halfperiod is 15
	 * 		- not use mute mode
	 * 		- use reset and stop mode
	 * 		- select the fractional rate divider clock output as the source,
	 * 		- disable 4-pin mode
	 * 		- MCLK ouput is disable
	 * 		- Frequency = 16 kHz
	 * Because we use mode I2STXMODE[3:0]= 0000, I2SDAO[5]=0 and
	 * I2SRX[3:0]=0000, I2SDAI[5] = 1. So we have I2SRX_CLK = I2STX_CLK
	 * --> I2SRXBITRATE = 1 (not divide TXCLK to produce RXCLK)
	 */

	/* Audio Config*/
	I2S_ConfigStruct.wordwidth = I2S_WORDWIDTH_16;
	I2S_ConfigStruct.mono = I2S_MONO;
	I2S_ConfigStruct.stop = I2S_STOP_ENABLE;
	I2S_ConfigStruct.reset = I2S_RESET_ENABLE;
	I2S_ConfigStruct.mute = I2S_MUTE_DISABLE;

	I2S_ConfigStruct.ws_sel = I2S_MASTER_MODE;
	I2S_Config(LPC_I2S,I2S_TX_MODE,&I2S_ConfigStruct);

	I2S_ConfigStruct.ws_sel = I2S_MASTER_MODE;
	I2S_Config(LPC_I2S,I2S_RX_MODE,&I2S_ConfigStruct);

	I2S_Stop(LPC_I2S, I2S_TX_MODE);
	I2S_Stop(LPC_I2S, I2S_RX_MODE);


	/* Clock Mode Config
	 * I2S transmitter:
	 * 	- Select TX fractional rate divider clock ouput as the source
	 *  - Disable 4-wire mode
	 *  - Enable for the TX_MCLK output (optional, this setting just for
	 *  user can observe master clock on oscilloscope)
	 * I2S receiver:
	 * 	- Select TX_MCLK signal as the RX_MCLK clock source
	 *  - Disable 4-wire mode
	 *  - Disable for the RX_MCLK output
	 */
	I2S_ClkConfig.clksel = I2S_CLKSEL_FRDCLK;
	I2S_ClkConfig.fpin = I2S_4PIN_DISABLE;
	I2S_ClkConfig.mcena = I2S_MCLK_DISABLE;
	I2S_ModeConfig(LPC_I2S,&I2S_ClkConfig,I2S_TX_MODE);

	I2S_ClkConfig.clksel = I2S_CLKSEL_FRDCLK;
	I2S_ClkConfig.mcena = I2S_MCLK_ENABLE;
	I2S_ModeConfig(LPC_I2S,&I2S_ClkConfig,I2S_RX_MODE);

	I2S_FreqConfig(LPC_I2S, 16000, I2S_TX_MODE);
	I2S_FreqConfig(LPC_I2S, 16000, I2S_RX_MODE);

	I2S_IRQConfig(LPC_I2S,I2S_RX_MODE,1);

	UDA1380_CodecInit();
}
#endif /* (BOARD_IN_USE == CODE_RED_RDB1768) */

/*******************************************************************************
* @brief	Speex CODEC initialization
* @param[in]	None
* @return	None
*******************************************************************************/
void spx_Init(SPX_CODEC_CFG_Type *SpxCfg)
{
   speex_bits_init(&bits);
   /* Speex encoding initializations */ 
   enc_state = speex_encoder_init(&speex_nb_mode);
   speex_encoder_ctl(enc_state, SPEEX_SET_VBR, &SpxCfg->vbr);
   speex_encoder_ctl(enc_state, SPEEX_SET_QUALITY, &SpxCfg->quality);
   speex_encoder_ctl(enc_state, SPEEX_SET_COMPLEXITY, &SpxCfg->complexity);
   /* speex decoding intilalization */
   dec_state = speex_decoder_init(&speex_nb_mode);
   speex_decoder_ctl(dec_state, SPEEX_SET_ENH, &SpxCfg->enh);
}

/*******************************************************************************
* @brief     Audio output initialization
* @param[in] None
* @return    None
*******************************************************************************/
void AudioInput_Init(void)
{
#if (BOARD_IN_USE == CODE_RED_RDB1768)
	RDB1768_AudioInit();
#elif (BOARD_IN_USE == IAR_LPC1768_SK)
	PINSEL_CFG_Type PinCfg;

	
	PinCfg.OpenDrain = PINSEL_PINMODE_NORMAL;
	PinCfg.Pinmode   = PINSEL_PINMODE_PULLUP;
#if (ADC_CHANNEL__==ADC_CHANNEL_2)        /* select P0.25 as AD0.2 */
	PinCfg.Funcnum   = PINSEL_FUNC_1;  	  
	PinCfg.Pinnum    = PINSEL_PIN_25;
	PinCfg.Portnum   = PINSEL_PORT_0;
#elif	(ADC_CHANNEL__ == ADC_CHANNEL_5)   /* select P1.31 as AD0.5 */
	PinCfg.Funcnum   = PINSEL_FUNC_3;
	PinCfg.Pinnum    = PINSEL_PIN_31;
	PinCfg.Portnum   = PINSEL_PORT_1;
#endif
	PINSEL_ConfigPin(&PinCfg);
	/* Configuration for ADC:
	 *  select: ADC channel 2 (if using MCB1700 board)
	 *  	    ADC channel 5 (if using IAR-LPC1768 board)
	 *  ADC conversion rate = 200000KHz
	 */
	ADC_Init(LPC_ADC, 200000);
	ADC_IntConfig(LPC_ADC,ADC_INT__,DISABLE);
	ADC_ChannelCmd(LPC_ADC,ADC_CHANNEL__,ENABLE);
	/* preemption = 1, sub-priority = 1 */
	NVIC_SetPriority(ADC_IRQn, ((0x01<<3)|0x01));
#endif /* BOARD_IN_USE*/
}
/*******************************************************************************
* @brief     Audio output initialization
* @param[in] None
* @return    None
*******************************************************************************/
void AudioOutput_Init(void)
{
   TIM_TIMERCFG_Type TIM_ConfigStruct;
   TIM_MATCHCFG_Type TIM_MatchConfigStruct ;
#if (BOARD_IN_USE == IAR_LPC1768_SK)
   PINSEL_CFG_Type PinCfg;

   /*
   * Init DAC pin connect
   * AOUT on P0.26
   */
   PinCfg.Funcnum = PINSEL_FUNC_2;
   PinCfg.OpenDrain = PINSEL_PINMODE_NORMAL;
   PinCfg.Pinmode = PINSEL_PINMODE_PULLUP;
   PinCfg.Pinnum = PINSEL_PIN_26;
   PinCfg.Portnum = PINSEL_PORT_0;
   PINSEL_ConfigPin(&PinCfg);
#endif /* (BOARD_IN_USE == IAR_LPC1768_SK) */
   
   // Initialize timer 0, prescale count time of 1uS
   TIM_ConfigStruct.PrescaleOption = TIM_PRESCALE_USVAL;
   TIM_ConfigStruct.PrescaleValue  = 1;
   
   // use channel 0, MR0
   TIM_MatchConfigStruct.MatchChannel = 0;
   // Enable interrupt when MR0 matches the value in TC register
   TIM_MatchConfigStruct.IntOnMatch   = TRUE;
   //Enable reset on MR0: TIMER will reset if MR0 matches it
   TIM_MatchConfigStruct.ResetOnMatch = TRUE;
   //Stop on MR0 if MR0 matches it
   TIM_MatchConfigStruct.StopOnMatch  = FALSE;
   // unused
   TIM_MatchConfigStruct.ExtMatchOutputType = TIM_EXTMATCH_NOTHING;
   // Set Match value, count value of 125 (125 * 1uS = 125us --> 8 kHz)
   TIM_MatchConfigStruct.MatchValue   = AUDIO_SAMPLE_RATE_8KHZ;
   
   // Set configuration for Tim_config and Tim_MatchConfig
   TIM_Init(LPC_TIM0, TIM_TIMER_MODE,&TIM_ConfigStruct);
   TIM_ConfigMatch(LPC_TIM0,&TIM_MatchConfigStruct);
   
#if (BOARD_IN_USE == IAR_LPC1768_SK)
   /* init DAC structure to default
   * Maximum	current is 700 uA
   * First value to AOUT is 0
   */
   DAC_Init(LPC_DAC);
#endif /* (BOARD_IN_USE == IAR_LPC1768_SK) */
}

/******************************************************************************
 * @brief	TIMER0 interrupt handler sub-routine
 * @param[in]	None
 * @return 	None
 ******************************************************************************/
void TIMER0_IRQHandler(void)
{
   if (TIM_GetIntStatus(LPC_TIM0, TIM_MR0_INT)== SET)
   {
      if (codec_status == CODEC_STATUS_PLAY)
      {
         if (sample < FRAME_SIZE)
         {
#if (BOARD_IN_USE == IAR_LPC1768_SK)
            DAC_UpdateValue(LPC_DAC, 511-(0x3FF * (OUT_Buffer[current_output_buffer][sample++]))/0xFFFF);
#else /* (BOARD_IN_USE == IAR_LPC1768_SK) */
            I2S_Send(LPC_I2S, (OUT_Buffer[current_output_buffer][sample++]) << 16);
#endif /* (BOARD_IN_USE == IAR_LPC1768_SK) */
         }
         else
         {
            sample = 0;
            if(current_output_buffer)
            {
               current_output_buffer = 0;
               current_decode_buffer = 1;
               }
            else
            {
               current_output_buffer = 1;
               current_decode_buffer = 0;
            }
         decode_new_frame = 1;
         }
      }
#if (BOARD_IN_USE == IAR_LPC1768_SK)
      else if (codec_status == CODEC_STATUS_REC)
      {
         if (sample_in < FRAME_SIZE)
         {
            ADC_StartCmd(LPC_ADC,ADC_START_NOW);
            while (!(ADC_ChannelGetStatus(LPC_ADC,ADC_CHANNEL__,ADC_DATA_DONE)));
            IN_Buffer[current_input_buffer][sample_in++] = (ADC_ChannelGetData(LPC_ADC,ADC_CHANNEL__)<<3) - (2047<<3);
         }
         else
         {
            sample_in = 0;	 
            if(current_input_buffer)
            {
               current_input_buffer = 0;
               current_encode_buffer = 1;
            }
            else
            {
               current_input_buffer = 1;
               current_encode_buffer = 0;
            }
            encode_new_frame = 1;
         }
      }
#endif /* (BOARD_IN_USE == IAR_LPC1768_SK) */
   }
   TIM_ClearIntPending(LPC_TIM0, TIM_MR0_INT);
}

#if (BOARD_IN_USE == CODE_RED_RDB1768)
void I2S_IRQHandler(void)
{
	uint32_t RXLevel = 0;
	uint32_t tmp;

	//Check RX interrupt
	if(I2S_GetIRQStatus(LPC_I2S, I2S_RX_MODE))
	{
		RXLevel = I2S_GetLevel(LPC_I2S, I2S_RX_MODE);
		while ( RXLevel > 0 )
		{
			tmp = LPC_I2S->I2SRXFIFO;
			IN_Buffer[current_input_buffer][sample_in++] = tmp & 0xFFFF;
			IN_Buffer[current_input_buffer][sample_in++] = (tmp>>16) & 0xFFFF;
			LPC_I2S->I2STXFIFO = tmp;
			if ( sample_in == FRAME_SIZE )
			{
				sample_in = 0;
				if(current_input_buffer)
				{
				  current_input_buffer = 0;
				  current_encode_buffer = 1;
				}
				else
				{
				  current_input_buffer = 1;
				  current_encode_buffer = 0;
				}
				encode_new_frame = 1;
			}
			RXLevel--;
		}
	}
	return;
}
#endif /* (BOARD_IN_USE == CODE_RED_RDB1768) */

/******************************************************************************
 * @brief	Start Speex Encoder
 * @param[in]	None
 * @return 	None
 ******************************************************************************/
void spx_StartEncoder(void)
{
   sample_index = 0;
   encode_new_frame = 0;
   current_input_buffer = 0;
   sample_in = 0;
   current_encode_buffer = 1;
   codec_status = CODEC_STATUS_REC;

#if (BOARD_IN_USE == CODE_RED_RDB1768)
   /* preemption = 1, sub-priority = 1 */
   NVIC_EnableIRQ(I2S_IRQn);
   I2S_IRQCmd(LPC_I2S,I2S_RX_MODE,ENABLE);
   I2S_Start(LPC_I2S);
   UDA1380_EnableAudioIn(ENABLE);
#elif  (BOARD_IN_USE == IAR_LPC1768_SK)
   /* preemption = 1, sub-priority = 1 */
   NVIC_SetPriority(TIMER0_IRQn, ((0x01<<3)|0x01));
   /* Enable interrupt for timer 1 */
   NVIC_EnableIRQ(TIMER0_IRQn);
   TIM_Cmd(LPC_TIM0,ENABLE);
#endif /* (BOARD_IN_USE == CODE_RED_RDB1768) */

   while (sample_index < SPEEX_FRAME_SIZE*AUDIO_SAMPLE_TOTAL_FRAMES)
   {
      if (encode_new_frame)
      {
    	 GPIO_SetValue(1,(1<<25));
         encode_new_frame = 0;
         /*Flush all the bits in the struct so we can encode a new frame*/
         speex_bits_reset(&bits);
         /*Encode the frame*/
         speex_encode_int(enc_state, (spx_int16_t*)IN_Buffer[current_encode_buffer], &bits);
         /*Copy the bits to an array of char that can be written*/
         speex_bits_write(&bits, rec_voice + sample_index, SPEEX_FRAME_SIZE);
         sample_index += SPEEX_FRAME_SIZE;
         GPIO_ClearValue(1,(1<<25));
      }
   }

   /* Finish encoding */
   codec_status = CODEC_STATUS_IDLE;
#if (BOARD_IN_USE == CODE_RED_RDB1768)
   UDA1380_EnableAudioIn(DISABLE);
   NVIC_DisableIRQ(I2S_IRQn);
#elif  (BOARD_IN_USE == IAR_LPC1768_SK)
   TIM_Cmd(LPC_TIM0,DISABLE);
   NVIC_DisableIRQ(TIMER0_IRQn);
#endif  /* (BOARD_IN_USE == IAR_LPC1768_SK) */
}   

/******************************************************************************
 * @brief	Start Speex Decoder
 * @param[in]	None
 * @return 	None
 ******************************************************************************/
void spx_StartDecoder(void)
{
   sample_index = 0;
   codec_status = CODEC_STATUS_PLAY;

   sample = 0;
   current_output_buffer = 0;
   current_decode_buffer = 0;
   decode_new_frame = 1;
   
   /* Decoder loop */
   while (sample_index < SPEEX_FRAME_SIZE*AUDIO_SAMPLE_TOTAL_FRAMES)
   {
	 if (decode_new_frame)
     {
		GPIO_SetValue(1,(1<<25));
        decode_new_frame = 0;
        speex_bits_read_from(&bits, rec_voice + sample_index, SPEEX_FRAME_SIZE);
        speex_decode_int(dec_state, &bits, (spx_int16_t*)OUT_Buffer[current_decode_buffer]);
        sample_index += SPEEX_FRAME_SIZE;
        if (sample_index == SPEEX_FRAME_SIZE)			//after first frame is done, start decoding second frame and start playing the decoded first frame
        {
        	current_decode_buffer = 1;
        	decode_new_frame = 1;
        	sample = 0;
        	I2S_Start(LPC_I2S);
        	UDA1380_EnableAudioOut(ENABLE);
        	NVIC_EnableIRQ(TIMER0_IRQn);
        	TIM_Cmd(LPC_TIM0,ENABLE);
        }
        GPIO_ClearValue(1,(1<<25));
     }
   }

   /* Finish encoding */
   codec_status = CODEC_STATUS_IDLE;
   TIM_Cmd(LPC_TIM0,DISABLE);
   UDA1380_EnableAudioOut(DISABLE);
}

