


#include "LPC17xx.h"
#include "GLCD.h"
#include "xmodem1k.h"
#include "iap.h"
#include <stdio.h>

/*	Flash sectors to be used for image storage */
#define	IMG_START_SECTOR	0x00010000	/* Sector 16 */
#define	IMG_END_SECTOR		0x00037FFF	/* Sector 20 */

/*	BMP header information located at the beginning of the image file */
#define BMP  ((struct bmp_header*)IMG_START_SECTOR)
__packed struct bmp_header {
	uint16_t magic_number;	//Not used here
	uint32_t bmp_size;		//Used to indicated the acutal image size 
	uint32_t unused;		//Not used here
	uint32_t bmp_offset;	//Offset value to indicate when the image data starts
};

/*	Function Prototype */
static uint32_t load_image(uint8_t *data, uint16_t length);

/*	Character array workspace for GLCD print functions */
#define MAX_STRING_SIZE		50
static uint8_t string[MAX_STRING_SIZE];
static uint32_t received_data = 0;

/*	State-machine variable to control application functionality */
enum state_machine {
	READY = 0,
	MENU,
	ERASE_FLASH,
	FLASH_IMG,
	SHOW
};
enum state_machine	 cmd;

int main(void){
	/*	General purpose variables used throughout main() */
	uint32_t ints[4];

	/*	SystemClockUpdate() updates the SystemFrequency variable */
	SystemClockUpdate();

	/*	Initialize Graphics LCD on the MCB1700 */
	GLCD_Init();

	cmd = MENU;
	while(1){
		
		switch(cmd){
			case READY:
				
				/*	INT0 Button */
				if		(!(LPC_GPIO2->FIOPIN & (1<<10))){
					GLCD_ClearLn(5, 0);
					GLCD_DisplayString(5, 0, 0, "Erasing Images...");
					cmd = ERASE_FLASH;

				
				/*	Joystick "up" pushbutton */
				}else if(!(LPC_GPIO1->FIOPIN & (1<<23))){
					cmd = MENU;

				/*	Joystick "center" pushbutton */
				}else if(!(LPC_GPIO1->FIOPIN & (1<<20))){

					GLCD_ClearLn(5, 0);
					GLCD_DisplayString(5, 0, 0, "Waiting for XMODEM Xfer...");
					cmd = FLASH_IMG;
				
				/*	Joystick "down" pushbutton */
				}else if(!(LPC_GPIO1->FIOPIN & (1<<25))){
					cmd = SHOW;
				}
				break;

			case MENU:
				/*	Print menu */
				GLCD_Clear(Black);
				GLCD_SetBackColor(Black);
				GLCD_SetTextColor(White);

				GLCD_DisplayString(10, 0, 0, "Application MENU");
				GLCD_DisplayString(12, 0, 0, "Press  \"INT0\"  (P2.10) to erase image");
				GLCD_DisplayString(13, 0, 0, "Press   \"UP\"   (P1.23) to print menu");
				GLCD_DisplayString(14, 0, 0, "Press \"CENTER\" (P1.20) to transfer image");
				GLCD_DisplayString(15, 0, 0, "Press  \"DOWN\"  (P1.25) to display image");

				
				/*	Print Boot Version onto the LCD */
				if(u32IAP_ReadBootVersion (&ints[0], &ints[1]) == IAP_STA_CMD_SUCCESS){
					snprintf((char *)string, MAX_STRING_SIZE, "Boot Code version %d.%d", ints[0], ints[1]); 
					GLCD_DisplayString(24, 0, 0, string);
				}
			
				/*	Print Part ID onto the LCD */
				if(u32IAP_ReadPartID(&ints[0]) == IAP_STA_CMD_SUCCESS){
					snprintf((char *)string, MAX_STRING_SIZE, "Part ID: %d (%#x)", ints[0], ints[0]); 
					GLCD_DisplayString(26, 0, 0, string);		
				}

				/*	Print the Part's Serial Number onto the LCD */
				u32IAP_ReadSerialNumber(&ints[0], &ints[1], &ints[2], &ints[3]);
				snprintf((char *)string, MAX_STRING_SIZE, "Serial #: %08X:%08X:%08X:%08X", ints[0], ints[1], ints[2], ints[3]);
				GLCD_DisplayString(28, 0, 0, string);
				
				cmd = READY;
				break;

			case ERASE_FLASH:
				/*	Erase the images stored in flash */
				if ((u32IAP_PrepareSectors(16, 20) == IAP_STA_CMD_SUCCESS) &&
				    (u32IAP_EraseSectors  (16, 20) == IAP_STA_CMD_SUCCESS)) {

					GLCD_DisplayString(5, 17, 0, "Done");
				}else{
					GLCD_DisplayString(5, 17, 0, "FAIL");
				}				
				cmd = READY;
				break;

			case FLASH_IMG:
				/*	Clear the received data counter using in the load_mage function */
				received_data = 0;

				/*	Store a new image into flash */
				vXmodem1k_Client(&load_image);
				
				GLCD_SetBackColor(Black);
				GLCD_SetTextColor(White);
				GLCD_DisplayString(5, 26, 0, "Complete");
				cmd = READY;
				break;

			case SHOW:
				/*	Cycle through the existing images stored in flash */
				GLCD_Bmp(0,0, 320,240, (uint8_t *)(IMG_START_SECTOR + BMP->bmp_offset));
				cmd = READY;
				break;
		}
		
	}
}



static uint32_t load_image(uint8_t *data, uint16_t length){

	if(length > 0){
		
		/*	Prepare Sectors to be flashed */
		if(u32IAP_PrepareSectors(16, 20) == IAP_STA_CMD_SUCCESS){
	   		
			/*	Copy data (already) located in RAM to flash */
			if(u32IAP_CopyRAMToFlash(IMG_START_SECTOR + received_data, (uint32_t)data, length) == IAP_STA_CMD_SUCCESS){
				
				/*	Verify the flash contents with the contents in RAM */
				if(u32IAP_Compare(IMG_START_SECTOR + received_data, (uint32_t)data, length, 0) == IAP_STA_CMD_SUCCESS){
					
					/*	Update and Print Received bytes counter */
					received_data +=  length;
					snprintf((char *)string, MAX_STRING_SIZE, "Received %d of %d bytes", received_data, BMP->bmp_size);
					GLCD_SetBackColor(Black);
					GLCD_SetTextColor(White);
					GLCD_DisplayString(1, 0, 0, string);
					
					/*	Print Progress Bar */
					GLCD_SetBackColor(Blue);
					GLCD_SetTextColor(Red);
					GLCD_Bargraph(0, 20, 320, 10, (received_data*1024)/BMP->bmp_size);
					return 1;
				}
			}
		}
		/*	Error in the IAP functions */
		GLCD_DisplayString(5, 26, 0, "FAIL (RESET & ERASE IMAGE)");
		return 0;

	}else{
		return 0;
	}
}
