/*==================================================================================================
*   Project              : Safety Runtime Library - RDC Checker
*   Platform             : MPC5775E
*
*   SW Version           : 0.4.1
*
*   Copyright 2019 NXP
*   NXP Confidential. This software is owned or controlled by NXP and may only be used strictly
*   in accordance with the applicable license terms. By expressly accepting
*   such terms or by downloading, installing, activating and/or otherwise using
*   the software, you are agreeing that you have read, and that you agree to
*   comply with and are bound by, such license terms. If you do not agree to
*   be bound by the applicable license terms, then you may not retain,
*   install, activate or otherwise use the software.
==================================================================================================*/

/*==================================================================================================
*                                        INCLUDE FILES
==================================================================================================*/
#include "RDC_Checker.h"
//#include "SRF.h" /* NXP Safety Runtime Framework */

/*==================================================================================================
*                                      LOCAL VARIABLES
==================================================================================================*/ 
/**
* @brief          This variable stores the fault flags detected by low-level RDC checker.
*/ 
static unsigned long faultStatus;

/*==================================================================================================
*                                      GLOBAL VARIABLES
==================================================================================================*/ 
/**
* @brief          This structure stores the RDC data DMA-transfered from the eTPU memory.
*/ 
RDC_InputDataStr rdc_data;

/*==================================================================================================
*                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/ 
fract24 MultFract24( fract24 x, fract24 y );
fract24 ArctanFract24( fract24 s, fract24 c );
int24 SubInt24( int24 x, int24 y );

/*==================================================================================================
*                                       LOCAL FUNCTIONS
==================================================================================================*/ 
/**
* @brief          This function multiplies two 24-bit fractional values 
*                 and returns a 24-bit fractional result. 
* @param[in]      x  24-bit signed fractional number
* @param[in]      y  24-bit signed fractional number
* @retval         24-bit signed fractional result of x*y.
*/ 
fract16 MultFract16( fract16 x, fract16 y )
{
    return( ( x * y ) >> 15 );
}

/**
* @brief          This function calculates the angle given by sine and cosine
*                 components. This is the arcustangent (atan2) function. 
* @details        An aproximation method is used to calculate the arcustangent function.
*                 Maximum error is 0.0038rad (0.22deg). Refer to
*                 http://www-labs.iro.umontreal.ca/~mignotte/IFT2425/Documents/EfficientApproximationArctgFunction.pdf, 
*                 Table 2, eq. 7.
* @param[in]      c  cosine component as a 24-bit signed fractional number
* @param[in]      s  sine component as a 24-bit signed fractional number
* @retval         24-bit signed fractional angle. Range (-1, 1) corresponds to (-180, 180) deg.
*/ 
fract24 ArctanFract24( fract16 c, fract16 s )
{
    const float a0 = 1.0584;
    const float a1 = 0.273;
    const float pi = 3.14159265359;
          float cf, sf, angle, r, m;

    /* convert inputs to float and prevent from 0 value */
    cf = ( (float)c / (1<<15) ) + 0.000000001f;
    sf = ( (float)s / (1<<15) ) + 0.000000001f;
    
    /* select octant */
    if( c > 0 )
    {
        if( s > 0 )
        {
            if( c > s )
            {
                /* octant 1 */
                r = sf/cf;
                angle = 0 + r * ( a0 - a1*r ); 
            }
            else /* c < s */
            {
                /* octant 2 */
                r = cf/sf;
                angle = pi/2 - r * ( a0 - a1*r ); 
            }
        }
        else /* s < 0 */
        {
            if( c > -s )
            {
                /* octant 8 */
                r = sf/cf;
                angle = 0 + r * ( a0 + a1*r ); 
            }
            else /* c < -s */
            {
                /* octant 7 */
                r = cf/sf;
                angle = -pi/2 - r * ( a0 + a1*r ); 
            }
        }
    }
    else /* c < 0 */
    {
        if( s > 0 )
        {
            if( -c > s )
            {
                /* octant 4 */
                r = sf/cf;
                angle = pi + r * ( a0 + a1*r ); 
            }
            else /* -c < s */
            {
                /* octant 3 */
                r = cf/sf;
                angle = pi/2 - r * ( a0 + a1*r ); 
            }
        }
        else /* s < 0 */
        {
            if( -c > -s )
            {
                /* octant 5 */
                r = sf/cf;
                angle = -pi + r * ( a0 - a1*r ); 
            }
            else /* -c < -s */
            {
                /* octant 6 */
                r = cf/sf;
                angle = -pi/2 - r * ( a0 - a1*r ); 
            }
        }
    }
    
    return( (fract24)( angle/pi * ( 1<<23 ) ) );
}

/**
* @brief          This function subtracts two 24-bit integer values 
*                 and returns a 24-bit sign-extended integer result. 
* @param[in]      x  24-bit signed integer number
* @param[in]      y  24-bit signed integer number
* @retval         24-bit sign-extended integer result of x-y.
*/ 
int24 SubInt24( int24 x, int24 y )
{
    x -= y;
    x <<= 8;
    x >>= 8;
    return( x );
}


/*==================================================================================================
*                                       GLOBAL FUNCTIONS
==================================================================================================*/
/**
* @brief          This is the high-level RDC checker. This function reports low-level RDC checker
*                 results to the Safety Manager.
* @details        This function should be called periodically, while the low-level RD checker
*                 may execute asynchronously. 
*                 The low-level RD checker cummulates fault flags into the faultStatus variable.
*                 The high-level RD checker reports it to Safety Manager and clears the flags.
*/ 
void RDC_Checker( void )
{
    /* report faultStatus to the SRF Safety Manager */
  //  SRF_ReportResult( SRF_RDC_CHECKER_ID, faultStatus );
    /* clears all fault flags */
    faultStatus = 0UL;
}

/**
* @brief          This function returns and clears the faultStatus, result of all low-level checks.
* @details        This function should be called after RDC_Checker_LL(), if not running within the
*                 SRF, to get and clear the faultStatus.
*                 In SRF, the faultStatus is reported to Safety<anager and cleared by RDC_Checker().
*/ 
unsigned long RDC_Checker_FaultStatus( void )
{
    unsigned long fS;
    
    fS = faultStatus;
    /* clears all fault flags */
    faultStatus = 0UL;
    /* return faultStatus */
    return( fS );
}

/**
* @brief          This is the low-level RDC checker. This function detects faults in RDC data.
* @details        This function should be called every time new RDC data are available 
*                 in RDC_Checker_Data structure.
*/ 
void RDC_Checker_LL( void )
{
    fract16 sin08, cos08, sin16, cos16, sin24, cos24;
    static fract16 sin08_prev, cos08_prev, sin24_prev, cos24_prev;
    static fract24 angle24_prev, angle24_prev_prev, angle_ext_prev;
    static int24   timestamp08_prev, timestamp24_prev;
           fract16 vec08, vec24;
           fract24 angle08_arctan, angle24_arctan;


    /* 0) Convert 16-bit signed fract values to 32-bit signed-extended values */
    sin08 = (rdc_data.sin08 << 16) >> 16;
    sin24 = (rdc_data.sin24 << 16) >> 16;
    cos08 = (rdc_data.cos08 << 16) >> 16;
    cos24 = (rdc_data.cos24 << 16) >> 16;
    sin16 = (rdc_data.sin16 << 16) >> 16;
    cos16 = (rdc_data.cos16 << 16) >> 16;

    /* 1) Check RDC input signals */
    /* inputs stuck? */
    if(( sin08 == sin08_prev ) || ( sin24 == sin24_prev ))
    {
        faultStatus |= RDC_FAULT_SIN_STUCK;
    }
    if(( cos08 == cos08_prev ) || ( cos24 == cos24_prev ))
    {
        faultStatus |= RDC_FAULT_COS_STUCK;
    }
    sin08_prev = sin08;
    sin24_prev = sin24;
    cos08_prev = cos08;
    cos24_prev = cos24;
    /* input signals out of range? */
    if(( RDC_ABS( sin08 ) > RDC_THR_SIG_MAX_AMPL ) || ( RDC_ABS( sin24 ) > RDC_THR_SIG_MAX_AMPL ))
    {
        faultStatus |= RDC_FAULT_SIN_AMPL_OOR;
    }
    if(( RDC_ABS( cos08 ) > RDC_THR_SIG_MAX_AMPL ) || ( RDC_ABS( cos24 ) > RDC_THR_SIG_MAX_AMPL ))
    {
        faultStatus |= RDC_FAULT_COS_AMPL_OOR;
    }
    /* input signals dc-shifted? */
    if( ( sin08 + sin24 ) > RDC_THR_SIG_DC_SHFT )
    {
        faultStatus |= RDC_FAULT_SIN_DC_SHFT;
    }
    if( ( cos08 + cos24 ) > RDC_THR_SIG_DC_SHFT )
    {
        faultStatus |= RDC_FAULT_COS_DC_SHFT;
    }
    /* input signals phase-shifted? */
    if(( RDC_ABS( sin16 ) > RDC_THR_SIG_ZC_AMPL ) || ( RDC_ABS( cos16 ) > RDC_THR_SIG_ZC_AMPL ))
    {
        faultStatus |= RDC_FAULT_SIG_PHS_SHFT;
    }
    /* input signal vector on "unit" circle? */
    vec08 = MultFract16( sin08, sin08 ) + MultFract16( cos08, cos08 );
    vec24 = MultFract16( sin24, sin24 ) + MultFract16( cos24, cos24 );
    if(   ( vec08 < RDC_THR_VEC_MIN_SQR ) || ( vec08 > RDC_THR_VEC_MAX_SQR ) 
       || ( vec24 < RDC_THR_VEC_MIN_SQR ) || ( vec24 > RDC_THR_VEC_MAX_SQR ) )
    {
        faultStatus |= RDC_FAULT_VEC_OOR;
    }

    /* 2) Check RDC ATO operation */
    angle08_arctan = ArctanFract24( cos08, sin08 );
    angle24_arctan = ArctanFract24( -cos24, -sin24 );
    /* ATO result is a prediction of angle one update ahead => compare arctan angle with previous ATO result */
    if(( RDC_ABS(angle08_arctan - angle24_prev) > RDC_THR_ATO_ERR ) || ( RDC_ABS(angle24_arctan - rdc_data.angle08) > RDC_THR_ATO_ERR ))
    {
        faultStatus |= RDC_FAULT_ATO_ERR;
    }
    
    /* 3) Check RDC Extrapolation operation */
    if( SubInt24( rdc_data.angle08, angle24_prev_prev ) > 0 )
    {
        /* increasing angle */
        if(( SubInt24( angle_ext_prev, ( angle24_prev_prev - RDC_THR_EXT ) ) < 0 ) || ( SubInt24( ( rdc_data.angle08 + RDC_THR_EXT ), angle_ext_prev ) < 0 ))
        {
            faultStatus |= RDC_FAULT_EXT_ERR;
        }
    }
    else
    {
       /* decreasing angle */
       if(( SubInt24( angle_ext_prev, ( angle24_prev_prev + RDC_THR_EXT ) ) > 0 ) || ( SubInt24( ( rdc_data.angle08 - RDC_THR_EXT ), angle_ext_prev ) > 0 ))
       {
           faultStatus |= RDC_FAULT_EXT_ERR;
       }
    }
    angle_ext_prev = rdc_data.angle_ext;
    angle24_prev_prev = angle24_prev;
    angle24_prev = rdc_data.angle24;
    
    /* 4) Check RDC timestamps */
    /* a timestamp stuck? */
    if(( rdc_data.timestamp08 == timestamp08_prev ) || ( rdc_data.timestamp24 == timestamp24_prev ))
    {
        faultStatus |= RDC_FAULT_TIME_STUCK;
    }
    /* timing plausible? */
    if(    ( RDC_ABS( SubInt24( rdc_data.timestamp08, timestamp24_prev ) - (RDC_PERIOD_TCR1/2) ) > RDC_THR_TIMING )
        || ( RDC_ABS( SubInt24( rdc_data.timestamp24, rdc_data.timestamp08 ) - (RDC_PERIOD_TCR1/2) ) > RDC_THR_TIMING ) )
    {
        faultStatus |= RDC_FAULT_TIMIMNG;
    }
    timestamp08_prev = rdc_data.timestamp08;
    timestamp24_prev = rdc_data.timestamp24;

}
