
/*==================================================================================================
*   Project              : RDC Checker Test application
*   Platform             : MPC5775C
*
*   SW Version           : 0.4.3
*
*   (c) Copyright 2014 - 2016  Freescale Semiconductor, Inc.
*   Copyright 2017 - 2020 NXP
*   NXP Confidential. This software is owned or controlled by NXP and may only be used strictly
*   in accordance with the applicable license terms. By expressly accepting
*   such terms or by downloading, installing, activating and/or otherwise using
*   the software, you are agreeing that you have read, and that you agree to
*   comply with and are bound by, such license terms. If you do not agree to
*   be bound by the applicable license terms, then you may not retain,
*   install, activate or otherwise use the software.
==================================================================================================*/
/*==================================================================================================
*                                        INCLUDE FILES
==================================================================================================*/
#include "derivative.h" /* include peripheral declarations */
#include "etpu_set\etpu\resolver\etpu_resolver.h"  /* eTPU RESOLVER function API */
#include "etpu_set\etpu_gct.h"   /* eTPU configuration */
#include "edma_init.h"
#include "pit.h"
#include "sd_adc.h"
#include "siu_init.h"
#include "sys.h"	/* system clock settings */
#include "math.h"
#include "sine_wave.h"   /* 32 samples of sine wave: const float sin_wave[32] */
#include "white_noise.h" /* 1024 samples of white noise: const float white_noise[2014] */
#include "rdc_checker.h"
#include "gflib.h"


/*==================================================================================================
*                                      DEFINES AND MACROS
==================================================================================================*/
#define DEBUG_PIN_0     	    178
#define DEBUG_PIN_1     	    177
#define DEBUG_PIN_2     	    176

//#define ETPU_RES_IRQ_SOURCE	    88
#define ETPU_RES_IRQ_SOURCE	    69
#define ETPU_RES_IRQ_PRIORITY	10

#define DMA_IRQ_SOURCE	    39
#define DMA_IRQ_PRIORITY	8

#define PIT0_IRQ_SOURCE	        301
#define PIT0_IRQ_PRIORITY	    9

#define CHECKER_CNT		3600 // for debug purpose - ATO_angle array size

#define EXTRAPOL_EN     0
#define EXTRAPOL_DIS    1

#define TIMESTAMP_OK           0
#define TIMESTAMP_CORRUPTED    1
#define TIMESTAMP_STUCK        2

#define ANGLE_OK           0
#define ANGLE_CORRUPTED    1

/* Define test cases */
#define TEST_CASE_1_0 0
#define TEST_CASE_2_0 1
#define TEST_CASE_2_1 2
#define TEST_CASE_2_2 3
#define TEST_CASE_2_3 4
#define TEST_CASE_2_4 5
#define TEST_CASE_3_0 6
#define TEST_CASE_3_1 7
#define TEST_CASE_3_2 8
#define TEST_CASE_3_3 9
#define TEST_CASE_3_4 10
#define TEST_CASE_3_5 11
/*==================================================================================================
*                                      GLOBAL FUNCTION PROTOTYPES
==================================================================================================*/
extern void xcptn_xmpl(void);
extern void mcu_mode_clock_init(void);
extern void eTPU_RESOLVER_EXC_CH_Isr (void);
extern void PIT0_TIMER0_Isr              (void);
extern void esci_a_init(void);
extern void SIU_init(void);
extern void pit0_start_wait_ns(uint32_t time_ns);
extern void pit0_stop(void);
extern void pit0_irq_config(uint8_t irq);

/*==================================================================================================
*                                      GLOBAL VARIABLES
==================================================================================================*/
uint32_t etpu_a_engine_load;  /* eTPU Engine A load as a percentage */
uint32_t etpu_b_engine_load;  /* eTPU Engine B load as a percentage */

uint8_t start = 0;
uint8_t extrapolation = EXTRAPOL_EN;
uint8_t time_stamp = TIMESTAMP_OK;
uint8_t angle_corrupted = ANGLE_OK;
uint8_t ext_angle_corrupted = ANGLE_OK;

uint8_t ato_upd = 0;
uint32_t fault_cnt = 0;
uint32_t no_fault_cnt = 0;
unsigned long RDC_fault;

unsigned long first_fault = 0;	/* Transient fault captured */
unsigned long accumulated_faults = 0;  /* Next faults accumulated - excluded first transient fault */

uint8_t testcase = TEST_CASE_1_0;

unsigned long faults[CHECKER_CNT];
unsigned long ATO_angle[CHECKER_CNT];


/* Configurable signal parameters */
float delta_angle;
float sig_ampl_sin;
float sig_ampl_cos;
float dc_offset_sin;
float dc_offset_cos;
uint8_t phase_shift;
float white_noise_snr_db = 80;  // white noise SNR in dB
float wn_ampl;

float last_angle;
float last_speed;
static float next_angle;
static float next_speed;
float sin_last_angle;
float cos_last_angle;
static float sin_next_angle;
static float cos_next_angle;
float next_angle_rad;
float sin_amp;
float cos_amp;
float sin_amp_inc;
float cos_amp_inc;

/*==================================================================================================
*                                      LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
void my_interrupt_init(uint32_t source, uint8_t priority);
uint32_t get_etpu_load_a(void);
uint32_t get_etpu_load_b(void);
uint32_t random();
void hw_init(void);

void hw_init(void)
{
#if defined(TURN_ON_CPU1)
#if defined(START_FROM_FLASH)
    SIU.RSTVEC1.R = (unsigned long)0xc00400 | 0x00000001;
#else
    SIU.RSTVEC1.R = (unsigned long)0x40040000 | 0x00000001;
#endif /* defined(START_FROM_FLASH) */ 
#endif
}


/*==================================================================================================
*                                      INTERRUPT HANDLERS
==================================================================================================*/
/**
* @brief          PIT0_TIMER_Isr. Resolver feedback signals are emulated within this IRQ handler.
* @details        This handler performs calculation of sine and cosine wave samples per half period
*                 of the feedback signal depending on signal configurable parameters. Signal are
*                 written to eTPU local channel memory into signal buffers.
*                 The half-periods are distinguished and at the end of every update respective HSR
*                 is written to eTPU Resolver channel.
*/
void PIT0_TIMER0_Isr(void)
{
	uint32_t i,r,l;

    /* Debug pin ON */
    SIU.GPDO[DEBUG_PIN_1].R = 1;

    /* clear interrupt flag */
	PIT.TIMER[0].TFLG.B.TIF = 1;

	if(ato_upd) /* ATO update 2nd */
	{
		next_angle += delta_angle;
		if(next_angle > 180.0f)
		{
			next_angle -= 360.0f;
		}
		if(next_angle < -180.0f)
		{
			next_angle += 360.0f;
		}
		next_angle_rad = next_angle * (3.141592654 / 180);

		sin_next_angle = GFLIB_Sin(next_angle_rad);
		cos_next_angle = GFLIB_Cos(next_angle_rad);

		/* white noise */
		wn_ampl = (float)0x4000 / powf(10, white_noise_snr_db/20);
		r = random() % (1024-32);
		l = random() % (1024-32);

		/* write sin/cos signals to eTPU Data Ram bufferes */
		//sin_amp_inc = (sin_next_angle - sin_last_angle) * (sig_ampl_sin / 16);
		//cos_amp_inc = (cos_next_angle - cos_last_angle) * (sig_ampl_cos / 16);
		sin_amp = sin_last_angle * sig_ampl_sin;
		cos_amp = cos_last_angle * sig_ampl_cos;
		for(i=16; i<32; i++)
		{
			resolver_instance.signals_pba[i]    = (0xFFFF) & (int16_t)((sin_wave[(i + phase_shift) % 32] * sin_amp + white_noise[r+i] * (wn_ampl)) + dc_offset_sin);
			resolver_instance.signals_pba[i+32] = (0xFFFF) & (int16_t)((sin_wave[(i + phase_shift) % 32] * cos_amp + white_noise[l+i] * (wn_ampl)) + dc_offset_cos);

		//	sin_amp += sin_amp_inc;
		//	cos_amp += cos_amp_inc;
		}

        last_angle =     next_angle;
        last_speed =     next_speed;
        sin_last_angle = sin_next_angle;
        cos_last_angle = cos_next_angle;

		/* trigger ATO update 2nd */
		fs_etpu_set_hsr(ETPU_RESOLVER_ATO_CHAN, FS_ETPU_RESOLVER_HSR_UPDATE_2ND);
		ato_upd = 0;
	}
	else /* ATO update 1st */
	{
		next_angle += delta_angle;
		if(next_angle > 180.0f)
		{
			next_angle -= 360.0f;
		}
		if(next_angle < -180.0f)
		{
			next_angle += 360.0f;
		}
		next_angle_rad = next_angle * (3.141592654 / 180);

		sin_next_angle = GFLIB_Sin(next_angle_rad);
		cos_next_angle = GFLIB_Cos(next_angle_rad);

		/* white noise */
		wn_ampl = (float)0x4000 / powf(10, white_noise_snr_db/20);
		r = random() % (1024-32);
		l = random() % (1024-32);

		/* write sin/cos signals to eTPU Data Ram buffers */
	//	sin_amp_inc = (sin_next_angle - sin_last_angle) * (sig_ampl_sin / 16);
	//	cos_amp_inc = (cos_next_angle - cos_last_angle) * (sig_ampl_cos / 16);
		sin_amp = sin_last_angle * sig_ampl_sin;
		cos_amp = cos_last_angle * sig_ampl_cos;
		for(i=0; i<16; i++)
		{
			resolver_instance.signals_pba[i]    = (0xFFFF) & (int16_t)((sin_wave[(i + phase_shift) % 32] * sin_amp + white_noise[r+i] * (wn_ampl)) + dc_offset_sin);
			resolver_instance.signals_pba[i+32] = (0xFFFF) & (int16_t)((sin_wave[(i + phase_shift) % 32] * cos_amp + white_noise[l+i] * (wn_ampl)) + dc_offset_cos);

		//	sin_amp += sin_amp_inc;
		//	cos_amp += cos_amp_inc;
		}

        last_angle =     next_angle;
        last_speed =     next_speed;
        sin_last_angle = sin_next_angle;
        cos_last_angle = cos_next_angle;

		/* trigger ATO update 1st */
		fs_etpu_set_hsr(ETPU_RESOLVER_ATO_CHAN, FS_ETPU_RESOLVER_HSR_UPDATE_1ST);
		ato_upd = 1;
	}
    /* Debug pin OFF */
    SIU.GPDO[DEBUG_PIN_1].R = 0;

}

/**
* @brief          eTPU_RESOLVER_EXC_CH_Isr.
* @details        Resolver variables are updated and states are read out from eTPU channel local memory.
*                 ATO output angle is written into ATO_angle[] array - to enable display the waveform for
*                 debugging purposes. eTPU engine load is checked.
*/
void eTPU_RESOLVER_EXC_CH_Isr (void)
{
    static int32_t resolver_cnt = 0;

    /* Debug pin ON */
    SIU.GPDO[DEBUG_PIN_2].R = 1;

 /* Resolver function calls */
	fs_etpu_resolver_config(&resolver_instance, &resolver_config);
	//fs_etpu_resolver_set_thresholds(&resolver_instance, &resolver_diag_thresholds);
	fs_etpu_resolver_get_outputs_calculated(&resolver_instance, &resolver_outputs_calculated);
	//fs_etpu_resolver_get_outputs_extrapolated(&resolver_instance, &resolver_outputs_extrapolated);
	//fs_etpu_resolver_trans_outputs_el_to_mech(RES_POLE_PAIRS, &resolver_outputs_extrapolated, &resolver_outputs_mechanical);
	fs_etpu_resolver_get_states(&resolver_instance, &resolver_states);


	etpu_a_engine_load = get_etpu_load_a();
	etpu_b_engine_load = get_etpu_load_b();

	ATO_angle[resolver_cnt] = resolver_outputs_calculated.angle;
	resolver_cnt++;
	if(resolver_cnt >= CHECKER_CNT)
	{
		resolver_cnt = 0;
	}

	/* Clear the channel interrupt flag */
	fs_etpu_clear_chan_interrupt_flag(ETPU_RESOLVER_ATO_CHAN);

	/* Debug pin OFF */
    SIU.GPDO[DEBUG_PIN_2].R = 0;
}

/**
* @brief          DMA_Isr. Interrupt service routine on eDMA channel data transfer ready.
* @details        In this IRQ handler RDC_Checker_LL() is called. Faults are read to RDC_fault
*                 variable. Every fault is stored in faults[] array for debugging purposes.
*                 Resolver extrapolation is triggered within this routine. RDC checker is used
*                 as stand-alone checker. On first fautl occurence the fault is captured in
*                 first_fault variable and fault accumulation is started. Faults are accumulated
*                 for predefined time and ORed into accumulated_faults. On the time expiration
*                 the application should read the and service the faults.
*/
void DMA_Isr(void)
{
	static uint32_t checker_cnt = 0;

    /* Debug pin ON */
    SIU.GPDO[DEBUG_PIN_0].R = 1;

    if(time_stamp == TIMESTAMP_CORRUPTED)
    {
    	rdc_data.timestamp08 += 0x40000;
    }

    if(time_stamp == TIMESTAMP_STUCK)
    {
    	rdc_data.timestamp08 = 0x40000;
    }

    if(angle_corrupted == ANGLE_CORRUPTED)
    {
    	rdc_data.angle08 += 0x40000;
    }

    if(ext_angle_corrupted == ANGLE_CORRUPTED)
    {
    	rdc_data.angle_ext += 0x40000;
    }

	/* low level RDC Checker performs input data check and  */
	RDC_Checker_LL();
	/* Read the fault status from LL RDC checker */
	RDC_fault = RDC_Checker_FaultStatus();

	faults[checker_cnt] = RDC_fault;
	checker_cnt++;

	if(extrapolation == 0)
	{
		/* Trigger the extrapolation */
		fs_etpu_resolver_sample(&resolver_instance);
	}


	if(checker_cnt >= CHECKER_CNT)
	{
		checker_cnt = 0;
	}

	if( RDC_fault == 0/* no fault - normal operation */ )
	{
		no_fault_cnt++;
	}
	else
	{

		if(!first_fault) /* First fault state detected - capture the transient fault */
		{
			first_fault = RDC_fault;
		}
		else
		{
			accumulated_faults |= RDC_fault;
			fault_cnt++;

			if(fault_cnt > 100)
			{
				/* Report to Safety manager */
				RDC_Checker();
				/* Clear counters and fault variables */
				fault_cnt = 0;
				first_fault = 0;
				accumulated_faults = 0;
			}
		}
	}

    /* Clear the interrupt flag */
	DMA_A.CINT.B.CINT = 28;

	/* Debug pin OFF */
    SIU.GPDO[DEBUG_PIN_0].R = 0;

}

/*==================================================================================================
*                                       MAIN FUNCTION
==================================================================================================*/

/**
* @brief          Application initialization and test cases are performed.
* @details        All the necessary peripheral configuration is performed. eTPU Resolver function is
*                 started and after successful startup interrupt from DMA is enabled where LL RDC
*                 checker is called. In for(;;) loop the corresponding test case to configure signal
*                 parameters is entered. Test cases can be configured by assigning respective constant
*                 to  testcase variable.
*/
int main(void)
{
	uint32_t i;

	/* MCU initialization */
	mcu_mode_clock_init();
	/* Configure and Enable Interrupts */
	xcptn_xmpl();
	/* Configure custom interrupt */
	my_interrupt_init(ETPU_RES_IRQ_SOURCE,ETPU_RES_IRQ_PRIORITY);
	//my_interrupt_init(DMA_IRQ_SOURCE,DMA_IRQ_PRIORITY);
	my_interrupt_init(PIT0_IRQ_SOURCE,PIT0_IRQ_PRIORITY);
	/* SIUL2 pin configuration */
	SIU_init();
	/* Initialize eTPU */
	my_system_etpu_init();
	/* Initialize and start DMA */
	edma_init();
	edma_start();
	/* run the eTPU */
	my_system_etpu_start();

	get_etpu_load_a();
	get_etpu_load_b();

    /* Start PIT to trigger Resolver processing */
	pit0_irq_config(PIT_IRQ_ENABLE);
	pit0_start_wait_ns(99840*10);

	/* Loop forever */
	for(;;) {

		if((start == 0)&(resolver_states.state > 3))
		{
			my_interrupt_init(DMA_IRQ_SOURCE,DMA_IRQ_PRIORITY);
			start = 1;
		}


		switch(testcase)
		{
		case TEST_CASE_1_0:
			/* No fault operation */
			delta_angle = 0.1F;    /* angle change per ATO update */
			sig_ampl_sin = 0x6000; /* amplitude 0.75 in signed fract range */
			sig_ampl_cos = 0x6000; /* amplitude 0.75 in signed fract range */
			dc_offset_sin = 0;
			dc_offset_cos = 0;
			phase_shift = 0;
			extrapolation = EXTRAPOL_EN;
            time_stamp = TIMESTAMP_OK;
            angle_corrupted = ANGLE_OK;
            ext_angle_corrupted= ANGLE_OK;
			break;
		case TEST_CASE_2_0:
			/* Input signal stuck */
			delta_angle = 0.0F;     /* angle is not incrementing */
			sig_ampl_sin = 0x6000; /* amplitude 0.75 in signed fract range */
			sig_ampl_cos = 0x6000; /* amplitude 0.75 in signed fract range */
			dc_offset_sin = 0;
			dc_offset_cos = 0;
			phase_shift = 0;
			extrapolation = EXTRAPOL_EN;
            time_stamp = TIMESTAMP_OK;
            angle_corrupted = ANGLE_OK;
            ext_angle_corrupted= ANGLE_OK;
			break;
		case TEST_CASE_2_1:
			/* Signal amplitude fault */
			delta_angle = 0.1F;    /* angle change per ATO update */
			sig_ampl_sin = 0x4000; /* amplitude 0.5 in signed fract range */
			sig_ampl_cos = 0x6000; /* amplitude 0.75 in signed fract range */
			dc_offset_sin = 0;
			dc_offset_cos = 0;
			phase_shift = 0;
			extrapolation = EXTRAPOL_EN;
            time_stamp = TIMESTAMP_OK;
            angle_corrupted = ANGLE_OK;
            ext_angle_corrupted= ANGLE_OK;
			break;
		case TEST_CASE_2_2:
			/* Signal amplitude fault */
			delta_angle = 0.1F;    /* angle change per ATO update */
			sig_ampl_sin = 0x7777; /* amplitude 0.93 in signed fract range */
			sig_ampl_cos = 0x6000; /* amplitude 0.75 in signed fract range */
			dc_offset_sin = 0;
			dc_offset_cos = 0;
			phase_shift = 0;
			extrapolation = EXTRAPOL_EN;
            time_stamp = TIMESTAMP_OK;
            angle_corrupted = ANGLE_OK;
            ext_angle_corrupted= ANGLE_OK;
			break;
		case TEST_CASE_2_3:
			/* Signal DC shift fault */
			delta_angle = 0.1F;    /* angle change per ATO update */
			sig_ampl_sin = 0x6000; /* amplitude 0.75 in signed fract range */
			sig_ampl_cos = 0x6000; /* amplitude 0.75 in signed fract range */
			dc_offset_sin = 0xCCC;  /* DC shift 0.1 in signed fract range */
			dc_offset_cos = 0;
			phase_shift = 0;
			extrapolation = EXTRAPOL_EN;
            time_stamp = TIMESTAMP_OK;
            angle_corrupted = ANGLE_OK;
            ext_angle_corrupted= ANGLE_OK;
			break;
		case TEST_CASE_2_4:
			/* Signal phase shift fault */
			delta_angle = 0.1F;    /* angle change per ATO update */
			sig_ampl_sin = 0x6000; /* amplitude 0.75 in signed fract range */
			sig_ampl_cos = 0x6000; /* amplitude 0.75 in signed fract range */
			dc_offset_sin = 0;
			dc_offset_cos = 0;
			phase_shift = 2;       /* input signals phase-shifted by two samples */
			extrapolation = EXTRAPOL_EN;
            time_stamp = TIMESTAMP_OK;
            angle_corrupted = ANGLE_OK;
            ext_angle_corrupted= ANGLE_OK;
			break;
		case TEST_CASE_3_0:
			/* Angle jump fault */
			delta_angle = 120.0F;    /* angle change per ATO update */
			sig_ampl_sin = 0x6000; /* amplitude 0.75 in signed fract range */
			sig_ampl_cos = 0x6000; /* amplitude 0.75 in signed fract range */
			dc_offset_sin = 0;
			dc_offset_cos = 0;
			phase_shift = 0;
			extrapolation = EXTRAPOL_EN;
            time_stamp = TIMESTAMP_OK;
            angle_corrupted = ANGLE_OK;
            ext_angle_corrupted= ANGLE_OK;
			break;
		case TEST_CASE_3_1:
			/* Extrapolation disabled */
			delta_angle = 0.1F;    /* angle change per ATO update */
			sig_ampl_sin = 0x6000; /* amplitude 0.75 in signed fract range */
			sig_ampl_cos = 0x6000; /* amplitude 0.75 in signed fract range */
			dc_offset_sin = 0;
			dc_offset_cos = 0;
			phase_shift = 0;
			extrapolation = EXTRAPOL_DIS; /* Extrapolation is not triggered */
            time_stamp = TIMESTAMP_OK;
            angle_corrupted = ANGLE_OK;
            ext_angle_corrupted= ANGLE_OK;
			break;
		case TEST_CASE_3_2:
			/* Time-stamp corrupted */
			delta_angle = 0.1F;    /* angle change per ATO update */
			sig_ampl_sin = 0x6000; /* amplitude 0.75 in signed fract range */
			sig_ampl_cos = 0x6000; /* amplitude 0.75 in signed fract range */
			dc_offset_sin = 0;
			dc_offset_cos = 0;
			phase_shift = 0;
			extrapolation = EXTRAPOL_EN;
            time_stamp = TIMESTAMP_CORRUPTED; /* Added offset to time-stamp in rdc_data structure */
            angle_corrupted = ANGLE_OK;
            ext_angle_corrupted= ANGLE_OK;
			break;
		case TEST_CASE_3_3:
			/* ATO angle corrupted */
			delta_angle = 0.1F;    /* angle change per ATO update */
			sig_ampl_sin = 0x6000; /* amplitude 0.75 in signed fract range */
			sig_ampl_cos = 0x6000; /* amplitude 0.75 in signed fract range */
			dc_offset_sin = 0;
			dc_offset_cos = 0;
			phase_shift = 0;
			extrapolation = EXTRAPOL_EN;
            time_stamp = TIMESTAMP_OK;
            angle_corrupted = ANGLE_CORRUPTED; /* Added offset to angle in rdc_data structure */
            ext_angle_corrupted= ANGLE_OK;
			break;
		case TEST_CASE_3_4:
			/* Extrapolation angle corrupted */
			delta_angle = 0.1F;    /* angle change per ATO update */
			sig_ampl_sin = 0x6000; /* amplitude 0.75 in signed fract range */
			sig_ampl_cos = 0x6000; /* amplitude 0.75 in signed fract range */
			dc_offset_sin = 0;
			dc_offset_cos = 0;
			phase_shift = 0;
			extrapolation = EXTRAPOL_EN;
            time_stamp = TIMESTAMP_OK;
            angle_corrupted = ANGLE_OK;
            ext_angle_corrupted= ANGLE_CORRUPTED; /* Added offset to extrapolated angle in rdc_data structure */
			break;
		case TEST_CASE_3_5:
			/* Time-stamp stuck */
			delta_angle = 0.1F;    /* angle change per ATO update */
			sig_ampl_sin = 0x6000; /* amplitude 0.75 in signed fract range */
			sig_ampl_cos = 0x6000; /* amplitude 0.75 in signed fract range */
			dc_offset_sin = 0;
			dc_offset_cos = 0;
			phase_shift = 0;
			extrapolation = EXTRAPOL_EN;
            time_stamp = TIMESTAMP_STUCK; /* Time-stamp is constant value */
            angle_corrupted = ANGLE_OK;
            ext_angle_corrupted= ANGLE_OK;
			break;
		}
	}
}

/*==================================================================================================
*                                       LOCAL FUNCTIONS
==================================================================================================*/
/**
* @brief   Set the interrupt source and priority.
*
* @return  N/A
*
*/

void my_interrupt_init(uint32_t source, uint8_t priority)
{
	INTC.PSR[source].R = 0x4000 | (priority & 0xFF) ;
}

/**
* @brief   Evaluate eTPU engine A load.
*
* @warning - This function is applicable on eTPU2 only.
*          - The first call of this function does not return a correct result.
*          - This function presumes the TCR1 clock is configured for system
*            clock div 1.
*
* @return  eTPU engine A load percentage (0 - 10000).
*
*/

uint32_t get_etpu_load_a(void)
{
  static uint24_t tcr1;
         uint24_t tcr1_last;
         uint24_t idle_cnt;
         uint24_t time_cnt;

  tcr1_last = tcr1;
  tcr1 = eTPU->TB1R_A.R;
  idle_cnt = fs_etpu_get_idle_cnt_a();
  fs_etpu_clear_idle_cnt_a();

  time_cnt = 0xFFFFFF & (tcr1 - tcr1_last);
  time_cnt = time_cnt/2;
  return(10000*(time_cnt - idle_cnt)/time_cnt);
}


/**
* @brief   Evaluate eTPU engine B load.
*
* @warning - This function is applicable on eTPU2 only.
*          - The first call of this function does not return a correct result.
*          - This function presumes the TCR1 clock is configured for system
*            clock div 1.
*
* @return  eTPU engine B load percentage (0 - 10000).
*
*/

uint32_t get_etpu_load_b(void)
{
  static uint24_t tcr1;
         uint24_t tcr1_last;
         uint24_t idle_cnt;
         uint24_t time_cnt;

  tcr1_last = tcr1;
  tcr1 = eTPU->TB1R_B.R;
  idle_cnt = fs_etpu_get_idle_cnt_b();
  fs_etpu_clear_idle_cnt_b();

  time_cnt = 0xFFFFFF & (tcr1 - tcr1_last);
  time_cnt = time_cnt/2;
  return(10000*(time_cnt - idle_cnt)/time_cnt);
}



/**
* @brief   Random number generator
* *
* @return  32 bit integer random number.
*
*/
uint32_t random()
{
  static x = 123456987;
  x ^= (x << 21);
  x ^= (x >> 35);
  x ^= (x << 4);
  return x;
}
/*==================================================================================================
 *
 * REVISION HISTORY:
 *
 * FILE OWNER: Marketa Venclikova [nxa17216]
 * Revision 0.4.3  2020/01/17  nxa17216
 * updates related to changes in eTPU Resolver function
 * Revision 0.4.2  2019/10/30  nxa17216
 * Added test-cases 3.1-3.5 to test ATO and Timing checker functionality
 * Revision 0.4.1  2019/10/25  nxa17216
 * Added white noise emulations to superpose on the emulated resolver feedback signals.
 * Revision 0.4.0  2019/10/1  nxa17216
 * Initial version of the Resolver emulation with RDC checker testing application
 *
==================================================================================================*/
