/*
 * Amazon FreeRTOS Demo Bootloader V1.4.7
 * Copyright (C) 2018 Amazon.com, Inc. or its affiliates.  All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * http://aws.amazon.com/freertos
 * http://www.FreeRTOS.org
 */


/**
 * @file aws_boot_loader.h
 * @brief Boot loader header.
 */

#ifndef _AWS_BOOT_LOADER_H_
#define _AWS_BOOT_LOADER_H_

/* Standard includes.*/
#include <stdint.h>

/* Bootloader Includes.*/
#include "aws_boot_crypto.h"

/**
 * @brief Bootloader version.
 */
#define BOOTLOADER_VERSION_MAJOR    ( 0UL )
#define BOOTLOADER_VERSION_MINOR    ( 9UL )
#define BOOTLOADER_VERSION_BUILD    ( 0UL )

/**
 * @brief Magic code.
 * The Magic code should be present at start of header for every application
 * image on the device.
 * @see BOOTImageHeader_t
 */
#define BOOT_MAGIC_CODE             "@AFRTOS"

/**
 * @brief Maximum size of magic code in bytes.
 * @note Changing this size will change the size of application image header.
 * It is not expected to be null terminated.
 * @see BOOTImageHeader_t
 */
#define BOOT_MAGIC_CODE_SIZE        ( 7U )

/**
 * @brief Quad Word size.
 */
#define BOOT_QUAD_WORD_SIZE         ( 16U )

/**
 * @brief Bootloader states.
 * This enumeration defines the valid bootloader states during the execution of
 * bootloader.
 */
typedef enum
{
    /*
     * Bootloader initialization state.
     */
    eBootStateInit = 0,

    /*
     * Application image validation state.
     */
    eBootStateValidation,

    /*
     * Executing selected application image.
     */
    eBootStateExecuteImage,

    /*
     * Starting to execute default image.
     */
    eBootStateExecuteDefault,

    /*
     * Bootloader is in error state.
     */
    eBootStateError
} BOOTState_t;

/**
 * @brief Bootloader Tasks.
 * This is the main bootloader routine which executes bootloader tasks as per
 * the bootloader state.
 */
void BOOTLOADER_Tasks( void );

/**
 * @brief Application image header.
 * This structure defines the application header containing magic code and the
 * image flags. The header should be present before each application image on
 * the device. The header is not part of signed application image.
 * @see BOOT_MAGIC_CODE
 * @see BOOTImageFlag_t
 */
typedef union
{
    uint32_t ulAlign[ 2 ];
    struct
    {
        char acMagicCode[ BOOT_MAGIC_CODE_SIZE ];
        uint8_t ucImageFlags;
    };
} BOOTImageHeader_t;

/**
 * @brief Application image descriptor.
 * This structure defines the application descriptor which should be present
 * for every OTA application image. The descriptor is generated by the OTA image
 * generation script and is part of signed application binary except the header
 * field.
 * @see BOOTImageHeader_t
 */
typedef struct
{
    BOOTImageHeader_t xImageHeader; /* Application image header. */
    uint32_t ulSequenceNum;         /* OTA image Sequence number. Higher is newer. */
    void * pvStartAddress;          /* Image start address. */
    void * pvEndAddress;            /* Image end address. */
    void * pvExecAddress;           /* Execution address. */
    uint32_t ulHardwareID;          /* Unique Hardware ID. */
    uint32_t ulReserved;            /* Reserved. */
} BOOTImageDescriptor_t;

/**
 * @brief Application image trailer.
 * This structure defines the application image trailer which should be present
 * at end for each application image stored on the device. This contains the
 * crypto signature of the application and is not part of signed OTA image.
 */
typedef struct
{
    char acSignatureType[ BOOT_CRYPTO_SIGNATURE_NAME_SIZE ]; /* Signature Type.*/
    uint32_t ulSignatureSize;                                /* Signature size. */
    uint8_t aucSignature[ BOOT_CRYPTO_SIGNATURE_SIZE ];      /* Signature. */
} BOOTImageTrailer_t;

/**
 * @brief Application image flags.
 * These are the flags used by bootloader to maintain the application
 * images.
 * New (11111111b) - A new image that hasn't yet been run.
 * Commit Pending (11111110b) - Image is pending commit and is ready for self test.
 * Valid (11111100b) - The image was accepted as valid by the self test code.
 * Invalid (11111000b) - The image was NOT accepted by the self test code.
 */
typedef enum
{
    eBootImageFlagNew = 0xff,
    eBootImageFlagCommitPending = 0xfe,
    eBootImageFlagValid = 0xfc,
    eBootImageFlagInvalid = 0xf8,
} BOOTImageFlag_t;

#endif /* ifndef _AWS_BOOT_LOADER_H_ */
