/*!
* @file potentiometer_driver.c
*
* @author  
*
* @version 1.0
*
* @date Apr-01-2016
*
* @brief Driver for Potentiometer interface
*
********************************************************************************
*
* Copyright (c) 2016, Freescale Semiconductor.
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/******************************************************************************
*                               Header Files
*******************************************************************************/

// Stack header files
#include "EmbeddedTypes.h"

// Application header files
#include "potentiometer_driver.h"

// KSDK header files
#include "fsl_adc16_driver.h"
#include "fsl_pmc_hal.h"

/******************************************************************************
* Locals
*******************************************************************************/

/*! Potentiometer ADC configuration structure */
const adc16_converter_config_t potentiometerAdcConfig= {
  .lowPowerEnable = FALSE,
  .clkDividerMode = kAdc16ClkDividerOf4,
  .longSampleTimeEnable = FALSE,
  .resolution = kAdc16ResolutionBitOfSingleEndAs16,
  .clkSrc = kAdc16ClkSrcOfBusClk,
  .asyncClkEnable = FALSE,
  .highSpeedEnable = FALSE, 
  .longSampleCycleMode = kAdc16LongSampleCycleOf24,
  .hwTriggerEnable = FALSE,
  .refVoltSrc = kAdc16RefVoltSrcOfVref,
  .continuousConvEnable = FALSE,
};

/*! Channel configuration for potentiometer */
const adc16_chn_config_t potentiometerPositionChannel = {
  .chnIdx = kAdc16Chn0d,
  .convCompletedIntEnable = FALSE,
  .diffConvEnable = TRUE,
};

/******************************************************************************
* Globals
*******************************************************************************/

/******************************************************************************
* Private Function prototypes
******************************************************************************/

/******************************************************************************
* Public Functions
******************************************************************************/

potentiometer_status_t potentiometer_init (void){
  adc16_status_t result;
  adc16_calibration_param_t adcCalibrationParameters;
  pmc_bandgap_buffer_config_t pmcBandgapConfiguration = {
    .enable = TRUE,
  };
  
  /* Initialize ADC driver */
  result = ADC16_DRV_Init(POTENTIOMETER_ADC_INSTANCE, &potentiometerAdcConfig);
  
  if (result != kStatus_ADC16_Success)
    return kPotentiometerInitError;
  
  /* Calibrate ADC module */
  result = ADC16_DRV_GetAutoCalibrationParam(POTENTIOMETER_ADC_INSTANCE, &adcCalibrationParameters);
  
  if (result != kStatus_ADC16_Success)
    return kPotentiometerInitError;
  
  result =  ADC16_DRV_SetCalibrationParam(POTENTIOMETER_ADC_INSTANCE, &adcCalibrationParameters);
  
  if (result != kStatus_ADC16_Success)
    return kPotentiometerInitError;
  
  /* Enable bandgap voltage channel for reference measurement */
  PMC_HAL_BandgapBufferConfig(PMC, &pmcBandgapConfiguration);
  
  return kPotentiometerOk;
}

uint8_t potentiometer_get_position (void){
  adc16_status_t result;
  int16_t potentiometerChannelAdcReading;
  uint8_t potentiometerPosition;
  
  /* Start Potentiometer Channel Measurements */
  result = ADC16_DRV_ConfigConvChn(POTENTIOMETER_ADC_INSTANCE, 0, &potentiometerPositionChannel);
  
  if(result != kStatus_ADC16_Success)
    return kPotentiometerConversionStartError;
  
  /* Wait for potentiometer channel measurement reading */
  ADC16_DRV_WaitConvDone (POTENTIOMETER_ADC_INSTANCE, 0);
  
  /* Get potentiometer channel measurement */
  potentiometerChannelAdcReading = ADC16_DRV_GetConvValueSigned(POTENTIOMETER_ADC_INSTANCE, 0);
  
  /* Calculate potentiometer position percentage */
  potentiometerPosition = (uint8_t)((potentiometerChannelAdcReading*100)/(POTENTIOMETER_ADC_RESOLUTION*0.9));
  
  if(potentiometerPosition>100)
    potentiometerPosition = 100;
  
  return potentiometerPosition;
}

/******************************************************************************
* Private Functions
******************************************************************************/

/* End of file */