/*!
* Copyright (c) 2015, Freescale Semiconductor, Inc.
* All rights reserved.
*
* \file PhyPacketProcessor.c
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor, Inc. nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


/*! *********************************************************************************
*************************************************************************************
* Include
*************************************************************************************
********************************************************************************** */
#include "EmbeddedTypes.h"
#include "MpmInterface.h"
#include "FunctionLib.h"

#include "Phy.h"
#include "overwrites.h"
#include "KW4xXcvrDrv.h"
#include "ifr_mkw40z4_radio.h"

#include "fsl_os_abstraction.h"
#include "fsl_device_registers.h"


/*! *********************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
********************************************************************************** */
#define gPhyHwIndQueueSize_d        (128)


/*! *********************************************************************************
*************************************************************************************
* Private functions prototype
*************************************************************************************
********************************************************************************** */
#if gPhyUseNeighborTable_d
static int32_t PhyGetIndexOf( uint16_t checksum );
#endif


/*! *********************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
********************************************************************************** */
const  uint8_t gPhyIdlePwrState = gPhyDefaultIdlePwrMode_c;
const  uint8_t gPhyActivePwrState = gPhyDefaultActivePwrMode_c;

#if gPhyUseNeighborTable_d
/* Limit HW indirect queue size to ~10% */
const uint8_t gPhyIndirectQueueSize_c = gPhyHwIndQueueSize_d/10;
#else
const uint8_t gPhyIndirectQueueSize_c = gPhyHwIndQueueSize_d;
#endif


/*! *********************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
********************************************************************************** */

/*! *********************************************************************************
* \brief  Initialize the 802.15.4 Radio registers
*
********************************************************************************** */
void PhyHwInit
(
void
)
{
#if gPhyUseNeighborTable_d
    uint32_t i;
    uint32_t phyReg;
#endif

    XcvrInit( ZIGBEE );

    /* Enable 16 bit mode for TC2 - TC2 prime EN, disable all timers,
       enable AUTOACK, mask all interrupts */
    ZLL_PHY_CTRL = (gCcaCCA_MODE1_c << ZLL_PHY_CTRL_CCATYPE_SHIFT) |
                   ZLL_PHY_CTRL_TC2PRIME_EN_MASK    |
                   ZLL_PHY_CTRL_PB_ERR_MSK_MASK     |
                   ZLL_PHY_CTRL_CRC_MSK_MASK        |
                   ZLL_PHY_CTRL_PLL_UNLOCK_MSK_MASK |
                   ZLL_PHY_CTRL_FILTERFAIL_MSK_MASK |
                   ZLL_PHY_CTRL_RX_WMRK_MSK_MASK    |
                   ZLL_PHY_CTRL_CCAMSK_MASK         |
                   ZLL_PHY_CTRL_RXMSK_MASK          |
                   ZLL_PHY_CTRL_TXMSK_MASK          |
                   ZLL_PHY_CTRL_SEQMSK_MASK         |
                   ZLL_PHY_CTRL_AUTOACK_MASK        |
                   ZLL_PHY_CTRL_TRCV_MSK_MASK;

    /* Clear all PP IRQ bits to avoid unexpected interrupts immediately after init
       disable all timer interrupts */
    ZLL_IRQSTS = ZLL_IRQSTS;

    /* Enable Source Addresing Match module */
    ZLL_BWR_SAM_CTRL_SAP0_EN(ZLL, 1);
#if (gMpmIncluded_d)
    ZLL_BWR_SAM_CTRL_SAP1_EN(ZLL, 1);
    ZLL_BWR_SAM_CTRL_SAP1_START(ZLL, gPhyHwIndQueueSize_d/2);
#if gPhyUseNeighborTable_d
    ZLL_BWR_SAM_CTRL_SAA0_EN(ZLL, 1);
    ZLL_BWR_SAM_CTRL_SAA0_START(ZLL, gPhyIndirectQueueSize_c/2);
    ZLL_BWR_SAM_CTRL_SAA1_EN(ZLL, 1);
    ZLL_BWR_SAM_CTRL_SAA1_START(ZLL, gPhyHwIndQueueSize_d/2 + gPhyIndirectQueueSize_c/2);
#endif

#elif gPhyUseNeighborTable_d
    ZLL_BWR_SAM_CTRL_SAA0_EN(ZLL, 1);
    ZLL_BWR_SAM_CTRL_SAA0_START(ZLL, gPhyIndirectQueueSize_c);
#endif

    /* Clear HW indirect queue */
    ZLL_SAM_TABLE |= ZLL_SAM_TABLE_INVALIDATE_ALL_MASK;
#if gPhyUseNeighborTable_d
    for( i=0; i<gPhyHwIndQueueSize_d; i++ )
    {
        /* Invalidate current index and checksum */
        phyReg  = ZLL_SAM_TABLE & ~(ZLL_SAM_TABLE_SAM_CHECKSUM_MASK | ZLL_SAM_TABLE_SAM_INDEX_MASK);
        phyReg |= (0xFFFF << ZLL_SAM_TABLE_SAM_CHECKSUM_SHIFT) |
                  (i << ZLL_SAM_TABLE_SAM_INDEX_SHIFT)     |
                   ZLL_SAM_TABLE_SAM_INDEX_WR_MASK;
        ZLL_SAM_TABLE = phyReg;
    }
#endif

    /*  Frame Filtering
        FRM_VER[7:6] = b11. Accept FrameVersion 0 and 1 packets, reject all others */
    ZLL_RX_FRAME_FILTER = ZLL_RX_FRAME_FILTER_FRM_VER_MASK |
                          ZLL_RX_FRAME_FILTER_CMD_FT_MASK  |
                          ZLL_RX_FRAME_FILTER_DATA_FT_MASK |
                          ZLL_RX_FRAME_FILTER_BEACON_FT_MASK;

    /* Set 802.15.4 register overwrites */
//    for( i=0; i<NumberOfElements(overwrites_common); i++ )
//        *((uint32_t*)overwrites_common[i].address) = overwrites_common[i].data;
//
//    for( i=0; i<NumberOfElements(overwrites_802p15p4); i++ )
//        *((uint32_t*)overwrites_802p15p4[i].address) = overwrites_802p15p4[i].data;

    /* Set prescaller to obtain 1 symbol (16us) timebase */
    ZLL_TMR_PRESCALE = 0x05;

    /* Set CCA threshold to -75 dBm */
    ZLL_BWR_CCA_LQI_CTRL_CCA1_THRESH(ZLL, 0xB5);

    /* Set the default power level */
    PhyPlmeSetPwrLevelRequest(gPhyDefaultTxPowerLevel_d);

    /* Adjust ACK delay to fulfill the 802.15.4 turnaround requirements */
    ZLL_BWR_ACKDELAY_ACKDELAY(ZLL, -2);

    /* Enable the RxWatermark IRQ and FilterFail IRQ */
//    ZLL_PHY_CTRL &= ~ZLL_PHY_CTRL_FILTERFAIL_MSK_MASK;
    ZLL_PHY_CTRL &= ~ZLL_PHY_CTRL_RX_WMRK_MSK_MASK;
    /* Set Rx watermark level */
    ZLL_WR_RX_WTR_MARK(ZLL, 0);

    /* Set default channels */
    PhyPlmeSetCurrentChannelRequest(0x0B, 0); /* 2405 MHz */
    PhyPlmeSetCurrentChannelRequest(0x0B, 1); /* 2405 MHz */

    /* Install PHY ISR */
    PHY_InstallIsr();
}

/*! *********************************************************************************
* \brief  Aborts the current sequence and force the radio to IDLE
*
********************************************************************************** */
void PhyAbort
(
void
)
{
    ProtectFromXcvrInterrupt();

    /* Disable timer trigger (for scheduled XCVSEQ) */
    if( ZLL_PHY_CTRL & ZLL_PHY_CTRL_TMRTRIGEN_MASK )
    {
        ZLL_PHY_CTRL &= ~ZLL_PHY_CTRL_TMRTRIGEN_MASK;
        /* give the FSM enough time to start if it was triggered */
        while(XCVR_BRD_STATUS_TSM_COUNT(XCVR) == 0);
    }

    /* If XCVR is not idle, abort current SEQ */
    if( ZLL_PHY_CTRL & ZLL_PHY_CTRL_XCVSEQ_MASK )
    {
        ZLL_BWR_PHY_CTRL_XCVSEQ(ZLL, gIdle_c);
        /* wait for Sequence Idle (if not already) */
        while( ZLL_SEQ_STATE & ZLL_SEQ_STATE_SEQ_STATE_MASK );
    }

    /* Mask SEQ interrupt */
    ZLL_BWR_PHY_CTRL_SEQMSK(ZLL, 1);
    /* Stop timers */
    ZLL_BWR_PHY_CTRL_TMR2CMP_EN(ZLL, 0);
    ZLL_BWR_PHY_CTRL_TMR3CMP_EN(ZLL, 0);
    ZLL_BWR_PHY_CTRL_TC3TMOUT(ZLL, 0);
    /* clear all PP IRQ bits to avoid unexpected interrupts( do not change TMR1 and TMR4 IRQ status ) */
    ZLL_IRQSTS &= ~(ZLL_IRQSTS_TMR1IRQ_MASK | ZLL_IRQSTS_TMR4IRQ_MASK);

    PhyIsrPassRxParams(NULL);

    UnprotectFromXcvrInterrupt();
}

/*! *********************************************************************************
* \brief  Get the state of the ZLL
*
* \return  uint8_t state
*
********************************************************************************** */
uint8_t PhyPpGetState
(
void
)
{
    return ZLL_RD_PHY_CTRL_XCVSEQ(ZLL);
}

/*! *********************************************************************************
* \brief  Set the value of the MAC PanId
*
* \param[in]  pPanId
* \param[in]  pan
*
* \return  phyStatus_t
*
********************************************************************************** */
phyStatus_t PhyPpSetPanId
(
uint8_t *pPanId,
uint8_t pan
)
{
    uint16_t value;

#ifdef PHY_PARAMETERS_VALIDATION
    if(NULL == pPanId)
    {
        return gPhyInvalidParameter_c;
    }
#endif /* PHY_PARAMETERS_VALIDATION */
    
    /* Avoid unaligned memory access issues */
    FLib_MemCpy(&value, pPanId, sizeof(value));

    if( 0 == pan )
        ZLL_WR_MACSHORTADDRS0_MACPANID0( ZLL, value );
    else
        ZLL_WR_MACSHORTADDRS1_MACPANID1( ZLL, value );

    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief  Set the value of the MAC Short Address
*
* \param[in]  pShortAddr
* \param[in]  pan
*
* \return  phyStatus_t
*
********************************************************************************** */
phyStatus_t PhyPpSetShortAddr
(
uint8_t *pShortAddr,
uint8_t pan
)
{
    uint16_t value;

#ifdef PHY_PARAMETERS_VALIDATION
    if(NULL == pShortAddr)
    {
        return gPhyInvalidParameter_c;
    }
#endif /* PHY_PARAMETERS_VALIDATION */

    /* Avoid unaligned memory access issues */
    FLib_MemCpy(&value, pShortAddr, sizeof(value));
    
    if( pan == 0 )
        ZLL_WR_MACSHORTADDRS0_MACSHORTADDRS0( ZLL, value );
    else
        ZLL_WR_MACSHORTADDRS1_MACSHORTADDRS1( ZLL, value );

    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief  Set the value of the MAC extended address
*
* \param[in]  pLongAddr
* \param[in]  pan
*
* \return  phyStatus_t
*
********************************************************************************** */
phyStatus_t PhyPpSetLongAddr
(
uint8_t *pLongAddr,
uint8_t pan
)
{
    uint32_t addrLo;
    uint32_t addrHi;

#ifdef PHY_PARAMETERS_VALIDATION
    if(NULL == pLongAddr)
    {
        return gPhyInvalidParameter_c;
    }
#endif /* PHY_PARAMETERS_VALIDATION */

    /* Avoid unaligned memory access issues */
    FLib_MemCpy(&addrLo, pLongAddr, sizeof(addrLo));
    pLongAddr += sizeof(addrLo);
    FLib_MemCpy(&addrHi, pLongAddr, sizeof(addrHi));

    if( 0 == pan )
    {
        ZLL_WR_MACLONGADDRS0_LSB(ZLL, addrLo );
        ZLL_WR_MACLONGADDRS0_MSB(ZLL, addrHi );
    }
    else
    {
        ZLL_WR_MACLONGADDRS1_LSB(ZLL, addrLo );
        ZLL_WR_MACLONGADDRS1_MSB(ZLL, addrHi );
    }

    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief  Set the MAC PanCoordinator role
*
* \param[in]  macRole
* \param[in]  pan
*
* \return  phyStatus_t
*
********************************************************************************** */
phyStatus_t PhyPpSetMacRole
(
  bool_t macRole,
  uint8_t pan
)
{
    uint8_t panCoord;

    if(gMacRole_PanCoord_c == macRole)
        panCoord = 1;
    else
        panCoord = 0;

    if( 0 == pan )
    {
        ZLL_BWR_PHY_CTRL_PANCORDNTR0(ZLL, panCoord);
    }
    else
    {
        ZLL_BWR_DUAL_PAN_CTRL_PANCORDNTR1(ZLL, panCoord);
    }

    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief  Set the PHY in Promiscuous mode
*
* \param[in]  mode
*
********************************************************************************** */
void PhyPpSetPromiscuous
(
bool_t mode
)
{
    if( mode )
    {
        if( (ZLL_PHY_CTRL & ZLL_PHY_CTRL_PROMISCUOUS_MASK) ||
            (ZLL_RX_FRAME_FILTER & ZLL_RX_FRAME_FILTER_ACTIVE_PROMISCUOUS_MASK) )
            return;

        ZLL_BWR_PHY_CTRL_PROMISCUOUS(ZLL, 1);
    /* FRM_VER[7:6] = b00. Any FrameVersion accepted (0,1,2 & 3) */
    /* All frame types accepted*/
        ZLL_RX_FRAME_FILTER &= ~ZLL_RX_FRAME_FILTER_FRM_VER_MASK;
        ZLL_RX_FRAME_FILTER |= (ZLL_RX_FRAME_FILTER_ACK_FT_MASK |
                                ZLL_RX_FRAME_FILTER_NS_FT_MASK);
    }
    else
    {
        ZLL_BWR_PHY_CTRL_PROMISCUOUS(ZLL, 0);
        /* FRM_VER[7:6] = b11. Accept FrameVersion 0 and 1 packets, reject all others */
        /* Beacon, Data and MAC command frame types accepted */
        ZLL_RX_FRAME_FILTER &= ~(ZLL_RX_FRAME_FILTER_FRM_VER_MASK |
                                 ZLL_RX_FRAME_FILTER_ACK_FT_MASK  |
                                 ZLL_RX_FRAME_FILTER_NS_FT_MASK   |
                                 ZLL_RX_FRAME_FILTER_ACTIVE_PROMISCUOUS_MASK);
        ZLL_RX_FRAME_FILTER |= 0x03 << ZLL_RX_FRAME_FILTER_FRM_VER_SHIFT;
    }
}

/*! *********************************************************************************
* \brief  Set the PHY in ActivePromiscuous mode
*
* \param[in]  state
*
********************************************************************************** */
void PhySetActivePromiscuous(bool_t state)
{
    if( state )
    {
        if( !(ZLL_PHY_CTRL & ZLL_PHY_CTRL_PROMISCUOUS_MASK) )
            return;

        /* Disable Promiscuous mode */
        ZLL_BWR_PHY_CTRL_PROMISCUOUS(ZLL, 0);
        ZLL_WR_RX_FRAME_FILTER_ACTIVE_PROMISCUOUS(ZLL, 1);
    }
    else
    {
        if( !(ZLL_RX_FRAME_FILTER & ZLL_RX_FRAME_FILTER_ACTIVE_PROMISCUOUS_MASK) )
            return;

        ZLL_WR_RX_FRAME_FILTER_ACTIVE_PROMISCUOUS(ZLL, 0);
        /* Enable Promiscuous mode */
        ZLL_BWR_PHY_CTRL_PROMISCUOUS(ZLL, 1);
    }
}

/*! *********************************************************************************
* \brief  Get the state of the ActivePromiscuous mode
*
* \return  bool_t state
*
********************************************************************************** */
bool_t PhyGetActivePromiscuous
(
void
)
{
    return ZLL_RD_RX_FRAME_FILTER_ACTIVE_PROMISCUOUS(ZLL);
}

/*! *********************************************************************************
* \brief  Set the state of the SAM HW module
*
* \param[in]  state
*
********************************************************************************** */
void PhyPpSetSAMState
(
  bool_t state
)
{
    ZLL_BWR_SAM_CTRL_SAP0_EN(ZLL, state);
#if gMpmIncluded_d
    ZLL_BWR_SAM_CTRL_SAP1_EN(ZLL, state);
#endif
}

/*! *********************************************************************************
* \brief  Add a new element to the PHY indirect queue
*
* \param[in]  index
* \param[in]  checkSum
* \param[in]  instanceId
*
* \return  phyStatus_t
*
********************************************************************************** */
phyStatus_t PhyPp_IndirectQueueInsert
(
uint8_t  index,
uint16_t checkSum,
instanceId_t instanceId
)
{
    uint32_t temp;

    (void)instanceId;
    if( index >= gPhyHwIndQueueSize_d )
        return gPhyInvalidParameter_c;

    temp = ZLL_SAM_TABLE;
    temp &= ~(ZLL_SAM_TABLE_SAM_INDEX_MASK | ZLL_SAM_TABLE_SAM_CHECKSUM_MASK);

    temp |= (index << ZLL_SAM_TABLE_SAM_INDEX_SHIFT) |
            (checkSum << ZLL_SAM_TABLE_SAM_CHECKSUM_SHIFT) |
            ZLL_SAM_TABLE_SAM_INDEX_WR_MASK |
            ZLL_SAM_TABLE_SAM_INDEX_EN_MASK;
    ZLL_SAM_TABLE = temp;

    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief  Remove an eleent from the PHY indirect queue
*
* \param[in]  index
* \param[in]  instanceId
*
* \return  phyStatus_t
*
********************************************************************************** */
phyStatus_t PhyPp_RemoveFromIndirect
(
uint8_t index,
instanceId_t instanceId
)
{
    uint32_t temp;
    if( index >= gPhyHwIndQueueSize_d )
        return gPhyInvalidParameter_c;

    temp = ZLL_SAM_TABLE;
    temp &= ~(ZLL_SAM_TABLE_SAM_INDEX_MASK);
    temp |= (index << ZLL_SAM_TABLE_SAM_INDEX_SHIFT) | ZLL_SAM_TABLE_SAM_INDEX_INV_MASK;
    ZLL_SAM_TABLE = temp;

    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief  Return TRUE if the received packet is a PollRequest
*
* \return  bool_t
*
********************************************************************************** */
bool_t PhyPpIsPollIndication
(
void
)
{
    return ZLL_RD_IRQSTS_PI(ZLL);
}

/*! *********************************************************************************
* \brief  Return the state of the FP bit of the received ACK
*
* \return  bool_t
*
********************************************************************************** */
bool_t PhyPpIsRxAckDataPending
(
void
)
{
    return ZLL_RD_IRQSTS_RX_FRM_PEND(ZLL);
}

/*! *********************************************************************************
* \brief  Return TRUE if there is data pending for the Poling Device
*
* \return  bool_t
*
********************************************************************************** */
bool_t PhyPpIsTxAckDataPending
(
void
)
{
    if( ZLL_SAM_CTRL & (ZLL_SAM_CTRL_SAP0_EN_MASK | ZLL_SAM_CTRL_SAP1_EN_MASK) )
    {
        return ZLL_RD_IRQSTS_SRCADDR(ZLL);
    }
    else
    {
        return ZLL_RD_SAM_TABLE_ACK_FRM_PND(ZLL);
    }
}

/*! *********************************************************************************
* \brief  Set the state of the FP bit of an outgoing ACK frame
*
* \param[in]  FP  the state of the FramePending bit
*
********************************************************************************** */
void PhyPpSetFpManually
(
  bool_t FP
)
{
    /* Disable the Source Address Matching feature and set FP manually */
    ZLL_SAM_TABLE |= ZLL_SAM_TABLE_ACK_FRM_PND_CTRL_MASK;

    if( FP )
    {
        ZLL_SAM_TABLE |= ZLL_SAM_TABLE_ACK_FRM_PND_MASK;
    }
    else
    {
        ZLL_SAM_TABLE &= ~ZLL_SAM_TABLE_ACK_FRM_PND_MASK;
    }
}

/*! *********************************************************************************
* \brief  Set the value of the CCA threshold
*
* \param[in]  ccaThreshold
*
* \return  phyStatus_t
*
********************************************************************************** */
phyStatus_t PhyPpSetCcaThreshold
(
uint8_t ccaThreshold
)
{
    ZLL_BWR_CCA_LQI_CTRL_CCA1_THRESH( ZLL, ccaThreshold );
    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief  This function will set the value for the FAD threshold
*
* \param[in]  FADThreshold   the FAD threshold
*
* \return  phyStatus_t
*
********************************************************************************** */
uint8_t PhyPlmeSetFADThresholdRequest(uint8_t FADThreshold)
{
    XCVR_WR_FAD_THR(XCVR, FADThreshold );
    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief  This function will enable/disable the FAD
*
* \param[in]  state   the state of the FAD
*
* \return  phyStatus_t
*
********************************************************************************** */
uint8_t PhyPlmeSetFADStateRequest(bool_t state)
{
    ZLL_BWR_FAD_CTRL_FAD_EN(ZLL, state);

    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief  This function will set the LQI mode
*
* \return  uint8_t
*
********************************************************************************** */
uint8_t PhyPlmeSetLQIModeRequest(uint8_t lqiMode)
{
    ZLL_BWR_CCA_LQI_CTRL_CCA3_AND_NOT_OR(ZLL, (lqiMode>0));

    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief  This function will return the RSSI level
*
* \return  uint8_t
*
********************************************************************************** */
uint8_t PhyPlmeGetRSSILevelRequest(void)
{
    return ZLL_RD_LQI_AND_RSSI_RSSI(ZLL);
}

/*! *********************************************************************************
* \brief  This function will enable/disable the ANTX
*
* \param[in]  state   the state of the ANTX
*
* \return  phyStatus_t
*
********************************************************************************** */
uint8_t PhyPlmeSetANTXStateRequest(bool_t state)
{
    ZLL_BWR_FAD_CTRL_ANTX_EN(ZLL, state);

    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief Invert the logic of the ANT pads
*
* \param[in] invAntA - invert the ANT_A pad
* \param[in] invAntB - invert the ANT_A pad
* \param[in] invTx   - invert the ANT_TX pad
* \param[in] invRx   - invert the ANT_RX pad
*
* \return gPhySuccess
*
********************************************************************************** */
uint8_t PhyPlmeSetANTPadInvertedRequest(bool_t invAntA, bool_t invAntB, bool_t invTx, bool_t invRx)
{
    uint32_t settings = 0;

    if( invAntA ) 
        settings |= (1 << 0 );
    if( invAntB ) 
        settings |= (1 << 1 );
    if( invTx )   
        settings |= (1 << 2 );
    if( invRx )   
        settings |= (1 << 3 );
    
    ZLL_BWR_FAD_CTRL_ANTX_POL(ZLL, settings);

    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief Enable the ANT pads
*
* \param[in] antAB_on - 
* \param[in] rxtxSwitch_on - 
*
* \return gPhySuccess
*
********************************************************************************** */
uint8_t PhyPlmeSetANTPadStateRequest(bool_t antAB_on, bool_t rxtxSwitch_on)
{
    uint32_t settings = 0;

    if( antAB_on ) 
        settings |= (1 << 1 );

    if( rxtxSwitch_on ) 
        settings |= (1 << 0 );
    
    ZLL_BWR_FAD_CTRL_ANTX_EN(ZLL, settings);

    return gPhySuccess_c;
}

/*! *********************************************************************************
* \brief  This function will retrn the state of the ANTX
*
* \return  uint8_t
*
********************************************************************************** */
uint8_t PhyPlmeGetANTXStateRequest(void)
{
    return ZLL_BRD_FAD_CTRL_ANTX_EN(ZLL);
}

/*! *********************************************************************************
* \brief  Set the state of the Dual Pan Auto mode
*
* \param[in]  mode TRUE/FALSE
*
********************************************************************************** */
void PhyPpSetDualPanAuto
(
bool_t mode
)
{
    ZLL_BWR_DUAL_PAN_CTRL_DUAL_PAN_AUTO(ZLL, mode);
}

/*! *********************************************************************************
* \brief  Get the state of the Dual Pan Auto mode
*
* \return  bool_t state
*
********************************************************************************** */
bool_t PhyPpGetDualPanAuto
(
void
)
{
    return ZLL_BRD_DUAL_PAN_CTRL_DUAL_PAN_AUTO(ZLL);
}

/*! *********************************************************************************
* \brief  Set the dwell for the Dual Pan Auto mode
*
* \param[in]  dwell
*
********************************************************************************** */
void PhyPpSetDualPanDwell
(
uint8_t dwell
)
{
    ZLL_BWR_DUAL_PAN_CTRL_DUAL_PAN_DWELL( ZLL, dwell );
}

/*! *********************************************************************************
* \brief  Get the dwell for the Dual Pan Auto mode
*
* \return  uint8_t PAN dwell
*
********************************************************************************** */
uint8_t PhyPpGetDualPanDwell
(
void
)
{
    return ZLL_BRD_DUAL_PAN_CTRL_DUAL_PAN_DWELL(ZLL);
}

/*! *********************************************************************************
* \brief  Get the remeining time before a PAN switch occures
*
* \return  uint8_t remaining time
*
********************************************************************************** */
uint8_t PhyPpGetDualPanRemain
(
void
)
{
  return ZLL_BRD_DUAL_PAN_CTRL_DUAL_PAN_REMAIN(ZLL);
}

/*! *********************************************************************************
* \brief  Set the current active Nwk
*
* \param[in]  nwk index of the nwk
*
********************************************************************************** */
void PhyPpSetDualPanActiveNwk
(
uint8_t nwk
)
{
    ZLL_BWR_DUAL_PAN_CTRL_ACTIVE_NETWORK(ZLL, (nwk > 0));
}

/*! *********************************************************************************
* \brief  Return the index of the Acive PAN
*
* \return  uint8_t index
*
********************************************************************************** */
uint8_t PhyPpGetDualPanActiveNwk
(
void
)
{
  return ZLL_BRD_DUAL_PAN_CTRL_ACTIVE_NETWORK(ZLL);
}

/*! *********************************************************************************
* \brief  Returns the PAN bitmask for the last Rx packet.
*         A packet can be received on multiple PANs
*
* \return  uint8_t bitmask
*
********************************************************************************** */
uint8_t PhyPpGetPanOfRxPacket(void)
{
  uint8_t PanBitMask = 0;

  if( ZLL_DUAL_PAN_CTRL & ZLL_DUAL_PAN_CTRL_DUAL_PAN_AUTO_MASK )
  {
      if( ZLL_DUAL_PAN_CTRL & ZLL_DUAL_PAN_CTRL_RECD_ON_PAN0_MASK )
          PanBitMask |= (1<<0);

      if( ZLL_DUAL_PAN_CTRL & ZLL_DUAL_PAN_CTRL_RECD_ON_PAN1_MASK )
          PanBitMask |= (1<<1);
  }
  else
  {
      if(ZLL_DUAL_PAN_CTRL & ZLL_DUAL_PAN_CTRL_ACTIVE_NETWORK_MASK )
          PanBitMask |= (1<<1);
      else
          PanBitMask |= (1<<0);
	  
  }

  return PanBitMask;
}

/*! *********************************************************************************
* \brief  Get the indirect queue level at which the HW queue will be split between PANs
*
* \return  uint8_t level
*
********************************************************************************** */
uint8_t PhyPpGetDualPanSamLvl(void)
{
    return ZLL_BRD_SAM_CTRL_SAP1_START(ZLL);
}

/*! *********************************************************************************
* \brief  Set the indirect queue level at which the HW queue will be split between PANs
*
* \param[in]  level
*
********************************************************************************** */
void PhyPpSetDualPanSamLvl( uint8_t level )
{
    ZLL_BWR_SAM_CTRL_SAP1_START(ZLL,level);
    ZLL_BWR_SAM_CTRL_SAP1_EN(ZLL, (level > 0) );
}

/*! *********************************************************************************
* \brief  This function compute the hash code for an 802.15.4 device
*
* \param[in]  pAddr     Pointer to an 802.15.4 address
* \param[in]  addrMode  The 802.15.4 addressing mode
* \param[in]  PanId     The 802.15.2 PAN Id
*
* \return  hash code
*
********************************************************************************** */
uint16_t PhyGetChecksum(uint8_t *pAddr, uint8_t addrMode, uint16_t PanId)
{
    uint16_t checksum;
    
    /* Short address */
    checksum  = PanId;
    checksum += *pAddr++;
    checksum += (*pAddr++) << 8;
    
    if( addrMode == 3 )    
    {
        /* Extended address */
        checksum += *pAddr++;
        checksum += (*pAddr++) << 8;
        checksum += *pAddr++;
        checksum += (*pAddr++) << 8;
        checksum += *pAddr++;
        checksum += (*pAddr++) << 8;
    }

    return checksum;
}

/*! *********************************************************************************
* \brief  This function adds an 802.15.4 device to the neighbor table.
*         If a polling device is not in the neighbor table, the ACK will have FP=1
*
* \param[in]  pAddr     Pointer to an 802.15.4 address
* \param[in]  addrMode  The 802.15.4 addressing mode
* \param[in]  PanId     The 802.15.2 PAN Id
*
********************************************************************************** */
uint8_t PhyAddToNeighborTable(uint8_t *pAddr, uint8_t addrMode, uint16_t PanId)
{
#if gPhyUseNeighborTable_d
    int32_t index;
    uint32_t phyReg;
    uint16_t checksum = PhyGetChecksum(pAddr, addrMode, PanId);
    uint32_t min = ZLL_RD_SAM_CTRL_SAA0_START(ZLL);
    uint32_t max = gPhyHwIndQueueSize_d;

    if( PhyGetIndexOf(checksum) != -1 )
    {
        /* Device is allready in the table */
        return 0;
    }

    /* Find first free index */
    phyReg = ZLL_SAM_TABLE;
    phyReg &= ~(ZLL_SAM_TABLE_SAM_INDEX_WR_MASK  |
                ZLL_SAM_TABLE_SAM_INDEX_INV_MASK |
                ZLL_SAM_TABLE_SAM_INDEX_EN_MASK  |
                ZLL_SAM_TABLE_FIND_FREE_IDX_MASK |
                ZLL_SAM_TABLE_INVALIDATE_ALL_MASK );
    ZLL_SAM_TABLE |= phyReg | ZLL_SAM_TABLE_FIND_FREE_IDX_MASK;
    
    while( ZLL_SAM_TABLE & ZLL_SAM_TABLE_SAM_BUSY_MASK );
    
    index = ZLL_BRD_SAM_FREE_IDX_SAA0_1ST_FREE_IDX(ZLL);

    if( (index >= min) && (index < max) )
    {
        PhyPp_IndirectQueueInsert((uint8_t)index, checksum, 0);
        return 0;
    }
#endif
    return 1;
}

/*! *********************************************************************************
* \brief  This function removes an 802.15.4 device to the neighbor table.
*         If a polling device is not in the neighbor table, the ACK will have FP=1
*
* \param[in]  pAddr     Pointer to an 802.15.4 address
* \param[in]  addrMode  The 802.15.4 addressing mode
* \param[in]  PanId     The 802.15.2 PAN Id
*
********************************************************************************** */
uint8_t PhyRemoveFromNeighborTable(uint8_t *pAddr, uint8_t addrMode, uint16_t PanId)
{
#if gPhyUseNeighborTable_d
    uint32_t phyReg;
    uint16_t checksum;
    int32_t  index;

    checksum = PhyGetChecksum(pAddr, addrMode, PanId);
    index    = PhyGetIndexOf(checksum);
    
    if( index != -1 )
    {
        /* Invalidate current index and checksum */
        phyReg  = ZLL_SAM_TABLE & ~(ZLL_SAM_TABLE_SAM_CHECKSUM_MASK | ZLL_SAM_TABLE_SAM_INDEX_MASK);
        phyReg |= (0xFFFF << ZLL_SAM_TABLE_SAM_CHECKSUM_SHIFT) |
                  (index << ZLL_SAM_TABLE_SAM_INDEX_SHIFT)     |
                  ZLL_SAM_TABLE_SAM_INDEX_INV_MASK | ZLL_SAM_TABLE_SAM_INDEX_WR_MASK;
        ZLL_SAM_TABLE = phyReg;
        return 0;
    }
#endif
    return 1;
}

/*! *********************************************************************************
* \brief  This function checks if an 802.15.4 device is in the neighbor table.
*         If a polling device is not in the neighbor table, the ACK will have FP=1
*
* \param[in]  checksum     hash code generated by PhyGetChecksum()
*
* \return  TRUE if the device is present in the neighbor table, FALSE if not.
*
********************************************************************************** */
bool_t PhyCheckNeighborTable(uint16_t checksum)
{
#if gPhyUseNeighborTable_d
    if( PhyGetIndexOf(checksum) != -1 )
    {
        return TRUE;
    }
#endif
    return FALSE;
}

/*! *********************************************************************************
* \brief  This function returns the table index of the specified checksum.
*
* \param[in]  checksum     hash code generated by PhyGetChecksum()
*
* \return  The table index where the checksum was found or
*          -1 if no entry was found with the specified chacksum
*          
*
********************************************************************************** */
#if gPhyUseNeighborTable_d
static int32_t PhyGetIndexOf( uint16_t checksum )
{
    uint32_t i, phyReg;
    uint32_t start = ZLL_RD_SAM_CTRL_SAA0_START(ZLL);
    uint32_t stop = gPhyHwIndQueueSize_d;
    
    for(i=start; i<stop; i++)
    {
        /* Set the index value */
        phyReg = ZLL_SAM_TABLE;
        phyReg &= ~(ZLL_SAM_TABLE_SAM_INDEX_MASK     |
                    ZLL_SAM_TABLE_SAM_INDEX_WR_MASK  |
                    ZLL_SAM_TABLE_SAM_INDEX_INV_MASK |
                    ZLL_SAM_TABLE_SAM_INDEX_EN_MASK  |
                    ZLL_SAM_TABLE_FIND_FREE_IDX_MASK |
                    ZLL_SAM_TABLE_INVALIDATE_ALL_MASK );
        phyReg |= (i << ZLL_SAM_TABLE_SAM_INDEX_SHIFT) | ZLL_SAM_TABLE_SAM_INDEX_INV_MASK;
        ZLL_SAM_TABLE = phyReg;
        /* Read checksum located at the specified index */
        phyReg = ZLL_SAM_TABLE;
        phyReg = (phyReg & ZLL_SAM_TABLE_SAM_CHECKSUM_MASK) >> ZLL_SAM_TABLE_SAM_CHECKSUM_SHIFT;

        if( phyReg == checksum )
        {
            return i;
        }
    }
    
    return -1;
}
#endif

phyStatus_t PhyPlmeSetPwrState( uint8_t state )
{
    return gPhySuccess_c;
}