/*! *********************************************************************************
 * \addtogroup Accelerometer Service
 * @{
 ********************************************************************************** */
/*!
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * \file accelerometer_service.c
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/
#include "FunctionLib.h"
#include "ble_general.h"
#include "gatt_db_app_interface.h"
#include "gatt_server_interface.h"
#include "gap_interface.h"
#include "accelerometer_interface.h"
/************************************************************************************
*************************************************************************************
* Private constants & macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/
/*! Accelerometer Service - Subscribed Client*/
static deviceId_t mAcs_SubscribedClientId;

//Characteristic declarations
const bleUuid_t accelerometerScaleCharacteristicUuid128 = {
  .uuid128 = {0xE0, 0x1C, 0x4B, 0x5E, 0x1E, 0xEB, 0xA1, 0x5C, 0xEE, 0xF4, 0x5E, 0xBA, 0x05, 0x57, 0xFF, 0x02}
};
const bleUuid_t accelerometerReadingsCharacteristicUuid128 = {
  .uuid128 = {0xE0, 0x1C, 0x4B, 0x5E, 0x1E, 0xEB, 0xA1, 0x5C, 0xEE, 0xF4, 0x5E, 0xBA, 0x06, 0x57, 0xFF, 0x02}
};

/************************************************************************************
*************************************************************************************
* Private functions prototypes
*************************************************************************************
************************************************************************************/
static void Acs_SendAccelerometerMeasurementNotification (uint16_t handle);

/************************************************************************************
*************************************************************************************
* Public functions prototypes
*************************************************************************************
************************************************************************************/
extern void App_AccelSignalAppThread(bool_t isSensitivityUpdateEvent, uint8_t newSensitivityValue);

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/
bleResult_t Acs_Start (acsConfig_t *pServiceConfig)
{    
    mAcs_SubscribedClientId = gInvalidDeviceId_c;
    
    return Acs_RecordAccelerometerMeasurement (pServiceConfig->serviceHandle, 
                                               pServiceConfig->xAxisValue,
                                               pServiceConfig->yAxisValue,
                                               pServiceConfig->zAxisValue
                                               );
}

bleResult_t Acs_Stop (acsConfig_t *pServiceConfig)
{
    return Acs_Unsubscribe();
}

bleResult_t Acs_Subscribe(deviceId_t deviceId)
{
    mAcs_SubscribedClientId = deviceId;

    return gBleSuccess_c;
}

bleResult_t Acs_Unsubscribe()
{
    mAcs_SubscribedClientId = gInvalidDeviceId_c;
    return gBleSuccess_c;
}

bleResult_t Acs_RecordAccelerometerMeasurement (uint16_t serviceHandle, int16_t newXAxisValue, 
                                                int16_t newYAxisValue, int16_t newZAxisValue)
{
    uint16_t  handle;
    bleResult_t result;
    
    acs_readings_packer_t accelerometerReadings = {
      .xAxisReading = newXAxisValue,
      .yAxisReading = newYAxisValue,
      .zAxisReading = newZAxisValue
    };

    /* Get handle of Accelerometer Readings characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType128_c, (bleUuid_t*)&accelerometerReadingsCharacteristicUuid128, &handle);

    if (result != gBleSuccess_c)
        return result;

    /* Update characteristic value */
    result = GattDb_WriteAttribute(handle, sizeof(accelerometerReadings), (uint8_t*)&accelerometerReadings);

    if (result != gBleSuccess_c)
        return result;

    Acs_SendAccelerometerMeasurementNotification(handle);

    return gBleSuccess_c;
}

bleResult_t Acs_RecordAccelerometerScale (uint16_t serviceHandle, uint8_t scale)
{
    uint16_t  handle;
    bleResult_t result;
    
    /* Verify that the new scale is compatible */
    if (scale >= kScaleNotSupported)
      return gCustomScaleConfigurationNotSupported_c;
    
    /* Update accelerometer scale */
    App_AccelSignalAppThread(TRUE, scale);

    /* Get handle of Accelerometer Scale characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType128_c, (bleUuid_t*)&accelerometerScaleCharacteristicUuid128, &handle);

    if (result != gBleSuccess_c)
        return result;

    /* Update characteristic value */ 
    result = GattDb_WriteAttribute(handle, sizeof(uint8_t), (uint8_t*)&scale);

    if (result != gBleSuccess_c)
        return result;

    return gBleSuccess_c;
}


/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/
static void Acs_SendAccelerometerMeasurementNotification
(
  uint16_t handle
)
{
    uint16_t  hCccd;
    bool_t isNotificationActive;

    /* Get handle of CCCD */
    if (GattDb_FindCccdHandleForCharValueHandle(handle, &hCccd) != gBleSuccess_c)
        return;

    if (gBleSuccess_c == Gap_CheckNotificationStatus
        (mAcs_SubscribedClientId, hCccd, &isNotificationActive) &&
        TRUE == isNotificationActive)
    {
        GattServer_SendNotification(mAcs_SubscribedClientId, handle);
    }
}
/*! *********************************************************************************
 * @}
 ********************************************************************************** */
