/*******************************************************************************
* FILE NAME: etpu_CEA709.c           COPYRIGHT (c) Freescale Semiconductor 2006
*                                              All Rights Reserved
* DESCRIPTION:
* This file contains the CEA709 eTPU function API.
*===============================================================================
* REV      AUTHOR      DATE        DESCRIPTION OF CHANGE
* ---   -----------  ----------    ---------------------
* 1.0    M. Princ     08/Jun/07    Initial version of file.
*******************************************************************************/
#include "etpu_util.h"          /* Utility routines for working with the eTPU */
#include "etpu_CEA709.h"       /* eTPU CEA709 API */

extern uint32_t fs_etpu_data_ram_start;
extern uint32_t fs_etpu_data_ram_end;
extern uint32_t fs_etpu_data_ram_ext;
extern uint32_t etpu_a_tcr1_freq;
extern uint32_t etpu_a_tcr2_freq;
extern uint32_t etpu_b_tcr1_freq;
extern uint32_t etpu_b_tcr2_freq;

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_init_1unit
*PURPOSE      : To initialize one CEA709 MAC unit.
*INPUTS NOTES : This function has 17 parameters:
*  rx_channel          - RX channel number (CP0)
*  tx_channel          - TX channel number (CP1)
*  preamble_length     - length of preamble for direct mode (in number of bits)
*  packet_cycle        - packet cycle duration (in number of bits)
*  beta2_control       - beta2 slots width (in number of bits)
*  xmit_interpacket    - interpacket padding after transmitting 
*                        (in number of bits)
*  receive_interpacket - interpacket padding after receiving (in number of bits)
*  channel_priorities  - number of priority slots on the channel
*  node_priority       - priority slot used by the node when sending priority 
*                        messages on the channel (1  255). It should not be 
*                        greater than the number of priority slots on the 
*                        channel. If the node has no priority slot allocated,
*                        this is 0.
*  bit_sync_threshold  - number of sync bits
*  dma_control_channel - DMA control channel number necessary for external DMA 
*                        request generation.
*  dma_trigger_mode    - This parameter defines the polarity on DMA CONTROL
*                        channel. This parameter should be assigned a value of:
*                        FS_ETPU_CEA709_DMA_TRIGGER_RISING_EDGE or
*                        FS_ETPU_CEA709_DMA_TRIGGER_FALLING_EDGE or,
*                        FS_ETPU_CEA709_DMA_TRIGGER_INTERNAL.
*  tx_en_channel       - transmit enable channel number (CP2,if required);
*                        If not applicable then set to 
*                        FS_ETPU_CEA709_CHAN_NOT_USED.
*  cd_channel          - collision detection channel number (CP4,if required)
*                        If not applicable then set to 
*                        FS_ETPU_CEA709_CHAN_NOT_USED.
*  cd_preamble         - this parameter determines whether the cd signal is
*                        checked at the end of the packet preamble or not; 
*                        this parameter should be assigned a value of:
*                        FS_ETPU_CEA709_CDPREAMBLE_NO or
*                        FS_ETPU_CEA709_CDPREAMBLE_YES
*  cd_tail             - this parameter determines whether the cd signal is
*                        checked at the end of the packet or not; this 
*                        parameter should be assigned a value of:
*                        FS_ETPU_CEA709_CDTAIL_NO or
*                        FS_ETPU_CEA709_CDTAIL_YES
*  cd_to_end_packet    - how close to the end of packet the cd signal
*                        is checked (in number of bits)
*
* RETURNS NOTES: Error codes which can be returned are: FS_ETPU_ERROR_VALUE,
*                FS_ETPU_ERROR_MALLOC
******************************************************************************/
int32_t fs_etpu_CEA709_init_1unit( uint8_t   rx_channel,
                                   uint8_t   tx_channel,
                                   uint8_t   preamble_length,
                                   uint24_t  packet_cycle,
                                   uint24_t  beta2_control,
                                   uint24_t  xmit_interpacket,
                                   uint24_t  receive_interpacket,
                                   uint8_t   channel_priorities,
                                   uint8_t   node_priority,
                                   uint8_t   bit_sync_threshold,
                                   uint8_t   dma_control_channel,
                                   uint8_t   dma_trigger_mode,
                                   uint8_t   tx_en_channel,
                                   uint8_t   cd_channel,
                                   uint8_t   cd_preamble,
                                   uint8_t   cd_tail,
                                   uint24_t  cd_to_end_packet)
{
   uint32_t * pba;
   uint32_t * tx_pri_buffer_start_pointer;
   uint32_t * tx_nonpri_buffer_start_pointer;
   uint32_t * rx_buffer_start_pointer;

   uint32_t etpu_tcr1_freq;

   /****************************************
    * Parameters bounds check.
    ***************************************/
   #ifdef FS_ETPU_MC_PARAM_CHECK
   if(((rx_channel>31)&&(rx_channel<64))||(rx_channel>95)||
      ((tx_channel>31)&&(tx_channel<64))||(tx_channel>95)||
      ((dma_control_channel>31)&&(dma_control_channel<64))||(dma_control_channel>95)||
      ((tx_en_channel>31)&&(tx_en_channel<64))||(tx_en_channel>95)||
      ((cd_channel>31)&&(cd_channel<64))||(cd_channel>95)||
      (channel_priorities>255)||
      (node_priority>255)||
      (dma_trigger_mode>FS_ETPU_CEA709_DMA_TRIGGER_INTERNAL))
   {
      return(FS_ETPU_ERROR_VALUE);
   }
   #endif

   /****************************************
    * PRAM allocation.
    ***************************************/
   if ((pba=fs_etpu_malloc(FS_ETPU_CEA709_RX_NUM_PARMS))== 0)
   {
      return(FS_ETPU_ERROR_MALLOC);
   }

   /* CEA709 Buffers have to start at addresses with 0 at the end, i.e. aligned by 16 bytes */
   fs_free_param += 2;
   fs_free_param = (uint32_t*)((uint32_t)fs_free_param>>4);
   fs_free_param = (uint32_t*)((uint32_t)fs_free_param<<4);
   tx_pri_buffer_start_pointer = fs_free_param;
   fs_free_param += (((48+7)>>3)<<1);

   if (fs_free_param > (uint32_t *)fs_etpu_data_ram_end)
       return(FS_ETPU_ERROR_MALLOC);

   tx_nonpri_buffer_start_pointer = tx_pri_buffer_start_pointer + 4;
   rx_buffer_start_pointer = tx_pri_buffer_start_pointer + 8;

   /****************************************
    * Write channel configuration registers
    * and FM (function mode) bits.
    ***************************************/
   /* RX, TX CHANNEL */
   eTPU->CHAN[rx_channel].CR.R = (FS_ETPU_CEA709_RX_TABLE_SELECT << 24) +
                                 (FS_ETPU_CEA709_RX_FUNCTION_NUMBER << 16) +
                                 (((uint32_t)pba - fs_etpu_data_ram_start) >> 3);
   eTPU->CHAN[rx_channel].SCR.R = 0;
   eTPU->CHAN[tx_channel].CR.R = (FS_ETPU_CEA709_TX_TABLE_SELECT << 24) +
                                 (FS_ETPU_CEA709_TX_FUNCTION_NUMBER << 16) +
                                 (((uint32_t)pba - fs_etpu_data_ram_start) >> 3);
   eTPU->CHAN[tx_channel].SCR.R = 0;

   /* DMA CONTROL CHANNEL */
   eTPU->CHAN[dma_control_channel].CR.R = (FS_ETPU_CEA709_DMA_CONTROL_TABLE_SELECT << 24) +
                                          (FS_ETPU_CEA709_DMA_CONTROL_FUNCTION_NUMBER << 16) +
                                          (((uint32_t)pba - fs_etpu_data_ram_start) >> 3);
   eTPU->CHAN[dma_control_channel].SCR.R = FS_ETPU_CEA709_DMA_CTR_FM0_INCREMENT_COUNTER_YES;
   if(dma_trigger_mode!=FS_ETPU_CEA709_DMA_TRIGGER_INTERNAL)
   {
      eTPU->CHAN[rx_channel].SCR.R |= FS_ETPU_CEA709_FM0_DMA_CTR_ON;
      eTPU->CHAN[tx_channel].SCR.R |= FS_ETPU_CEA709_FM0_DMA_CTR_ON;
   }

   /* TXEN CHANNEL */
   if(tx_en_channel!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      eTPU->CHAN[rx_channel].SCR.R |= FS_ETPU_CEA709_FM1_TXEN_ON;
      eTPU->CHAN[tx_channel].SCR.R |= FS_ETPU_CEA709_FM1_TXEN_ON;
      eTPU->CHAN[tx_en_channel].CR.R = (FS_ETPU_CEA709_TXEN_TABLE_SELECT << 24) +
                                       (FS_ETPU_CEA709_TXEN_FUNCTION_NUMBER << 16) +
                                       (((uint32_t)pba - fs_etpu_data_ram_start) >> 3);
      eTPU->CHAN[tx_en_channel].SCR.R = 0;
   }

   /* CD CHANNEL */
   if(cd_channel!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      eTPU->CHAN[cd_channel].CR.R = (FS_ETPU_CEA709_CD_TABLE_SELECT << 24) +
                                    (FS_ETPU_CEA709_CD_FUNCTION_NUMBER << 16) +
                                    (((uint32_t)pba - fs_etpu_data_ram_start) >> 3);
      eTPU->CHAN[cd_channel].SCR.R = 0;
   }


   /****************************************
    * Write parameters.
    ***************************************/
   if (rx_channel<64)
   {
      etpu_tcr1_freq = etpu_a_tcr1_freq;
   }
   else
   {
      etpu_tcr1_freq = etpu_b_tcr1_freq;
   }

   *(pba + ((FS_ETPU_CEA709_TRANSMIT_PERIOD_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125;
   *(pba + ((FS_ETPU_CEA709_STATUS_OFFSET - 1)>>2)) = 0;
   *(pba + ((FS_ETPU_CEA709_BETA1_TRANSMIT_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*xmit_interpacket;
   *(pba + ((FS_ETPU_CEA709_BETA1_RECEIVE_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*receive_interpacket;
   *(pba + ((FS_ETPU_CEA709_BETA2_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*beta2_control;
   *(pba + ((FS_ETPU_CEA709_CD_TO_END_PACKET_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*cd_to_end_packet;
   *(pba + ((FS_ETPU_CEA709_PACKET_CYCLE_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*packet_cycle;
   *(pba + ((FS_ETPU_CEA709_P_TX_PRI_BUF_START_OFFSET - 1)>>2)) = (uint24_t)((uint32_t)tx_pri_buffer_start_pointer - fs_etpu_data_ram_start);
   *(pba + ((FS_ETPU_CEA709_P_TX_NONPRI_BUF_START_OFFSET - 1)>>2)) = (uint24_t)((uint32_t)tx_nonpri_buffer_start_pointer - fs_etpu_data_ram_start);
   *(pba + ((FS_ETPU_CEA709_P_RX_BUF_START_OFFSET - 1)>>2)) = (uint24_t)((uint32_t)rx_buffer_start_pointer - fs_etpu_data_ram_start);
   *(pba + ((FS_ETPU_CEA709_LAST_RX_PACKET_LENGTH_OFFSET - 1)>>2)) = 0;
   *(pba + ((FS_ETPU_CEA709_LAST_RX_PACKET_TIME_STAMP_HIGH_OFFSET - 1)>>2)) = 0;
   *(pba + ((FS_ETPU_CEA709_LAST_RX_PACKET_TIME_STAMP_LOW_OFFSET - 1)>>2)) = 0;
   *((uint8_t*)pba + FS_ETPU_CEA709_STATE_OFFSET) = 0;
   *((uint8_t*)pba + FS_ETPU_CEA709_CHANNEL_ACCESS_STATE_OFFSET) = 0;
   *((uint8_t*)pba + FS_ETPU_CEA709_BUFFERS_OFFSET) = 0;
   *((uint8_t*)pba + FS_ETPU_CEA709_PREAMBLE_LENGTH_OFFSET) = preamble_length;
   *((uint8_t*)pba + FS_ETPU_CEA709_NODE_PRIORITY_OFFSET) = node_priority;
   *((uint8_t*)pba + FS_ETPU_CEA709_CHANNEL_PRIORITIES_OFFSET) = channel_priorities;
   *((uint8_t*)pba + FS_ETPU_CEA709_BACKLOG_OFFSET) = 1;
   *((uint8_t*)pba + FS_ETPU_CEA709_DMA_CONTROL_CHAN_OFFSET) = dma_control_channel;
   *((uint8_t*)pba + FS_ETPU_CEA709_RX_CHAN_OFFSET) = rx_channel;
   *((uint8_t*)pba + FS_ETPU_CEA709_TX_ENABLE_CHAN_OFFSET) = tx_en_channel;
   *((uint8_t*)pba + FS_ETPU_CEA709_BIT_SYNC_THRESHOLD_OFFSET) = bit_sync_threshold;
   *((uint8_t*)pba + FS_ETPU_CEA709_LAST_RX_PACKET_DATAGRAM_INFO_OFFSET) = 0;
   if(cd_channel!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      *((uint8_t*)pba + FS_ETPU_CEA709_CD_OPTIONS_OFFSET) = cd_tail + cd_preamble;
   }
   else
   {
      *((uint8_t*)pba + FS_ETPU_CEA709_CD_OPTIONS_OFFSET) = 0;
   }

   /****************************************
    * Write HSR.
    ***************************************/
   eTPU->CHAN[rx_channel].HSRR.R = FS_ETPU_CEA709_RX_INIT;
   eTPU->CHAN[tx_channel].HSRR.R = FS_ETPU_CEA709_TX_INIT;
   
   if(dma_trigger_mode == FS_ETPU_CEA709_DMA_TRIGGER_FALLING_EDGE)
   {
      eTPU->CHAN[dma_control_channel].HSRR.R = FS_ETPU_CEA709_DMA_CONTROL_INIT_FALLING_TRIGGER;
   }
   else
   {
      /* This HSR can be used to initialize the DMA_CONTROL_CHANNEL either 
           a) for external DMA request signal generation - rising edge triggers
           b) for 48bit timestamp counter operation only, without DMA request generation 
              (rx/tx channels generate the applicable DMA requests internally) */
      eTPU->CHAN[dma_control_channel].HSRR.R = FS_ETPU_CEA709_DMA_CONTROL_INIT_RISING_TRIGGER;
   }
   
   if(tx_en_channel!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      eTPU->CHAN[tx_en_channel].HSRR.R = FS_ETPU_CEA709_TXEN_INIT;
   }
   
   if(cd_channel!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      eTPU->CHAN[cd_channel].HSRR.R = FS_ETPU_CEA709_CD_INIT;
   }

   return(0);
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_init_2units
*PURPOSE      : To initialize one CEA709 MAC unit.
*INPUTS NOTES : This function has 34 parameters:
*  rx_channel_1        - RX channel number (CP0) of the unit 1
*  tx_channel_1        - TX channel number (CP1) of the unit 1
*  preamble_length_1   - length of preamble for direct mode (in number of bits)
*                        of the unit 1
*  packet_cycle_1      - packet cycle duration (in number of bits) of the unit 1
*  beta2_control_1     - beta2 slots width (in number of bits) of the unit 1
*  xmit_interpacket_1  - interpacket padding after transmitting 
*                        (in number of bits) of the unit 1
*  receive_interpacket_1 - interpacket padding after receiving (in number of bits)
*                          of the unit 1
*  channel_priorities_1  - number of priority slots on the channel of the unit 1
*  node_priority_1     - unit 1 priority slot used by the node when sending  
*                        priority messages on the channel (1  255). It should
*                        not be greater than the number of priority slots on the 
*                        channel. If the node has no priority slot allocated,
*                        this is 0.
*  bit_sync_threshold_1  - number of sync bits of the unit 1
*  dma_control_channel_1 - DMA control channel number necessary for external DMA 
*                        request generation of the unit 1
*  dma_trigger_mode_1  - This parameter defines the polarity on DMA CONTROL
*                        channel of the unit 1. This parameter should be 
*                        assigned a value of:
*                        FS_ETPU_CEA709_DMA_TRIGGER_RISING_EDGE or
*                        FS_ETPU_CEA709_DMA_TRIGGER_FALLING_EDGE or,
*                        FS_ETPU_CEA709_DMA_TRIGGER_INTERNAL.
*  tx_en_channel_1     - transmit enable channel number (CP2,if required)
*                        of the unit 1;
*                        If not applicable then set to 
*                        FS_ETPU_CEA709_CHAN_NOT_USED.
*  cd_channel_1        - collision detection channel number (CP4,if required)
*                        of the unit 1;
*                        If not applicable then set to 
*                        FS_ETPU_CEA709_CHAN_NOT_USED.
*  cd_preamble_1       - this parameter determines whether the cd signal is
*                        checked at the end of the packet preamble or not; 
*                        this parameter should be assigned a value of:
*                        FS_ETPU_CEA709_CDPREAMBLE_NO or
*                        FS_ETPU_CEA709_CDPREAMBLE_YES
*  cd_tail_1           - this parameter determines whether the cd signal is
*                        checked at the end of the packet or not; this 
*                        parameter should be assigned a value of:
*                        FS_ETPU_CEA709_CDTAIL_NO or
*                        FS_ETPU_CEA709_CDTAIL_YES
*  cd_to_end_packet_1  - how close to the end of packet the cd signal
*                        is checked (in number of bits)
*  rx_channel_2        - RX channel number (CP0) of the unit 2
*  tx_channel_2        - TX channel number (CP1) of the unit 2
*  preamble_length_2   - length of preamble for direct mode (in number of bits)
*                        of the unit 2
*  packet_cycle_2      - packet cycle duration (in number of bits) of the unit 2
*  beta2_control_2     - beta2 slots width (in number of bits) of the unit 2
*  xmit_interpacket_2  - interpacket padding after transmitting 
*                        (in number of bits) of the unit 2
*  receive_interpacket_2 - interpacket padding after receiving (in number of bits)
*                          of the unit 2
*  channel_priorities_2  - number of priority slots on the channel of the unit 2
*  node_priority_2     - unit 2 priority slot used by the node when sending  
*                        priority messages on the channel (1  255). It should
*                        not be greater than the number of priority slots on the 
*                        channel. If the node has no priority slot allocated,
*                        this is 0.
*  bit_sync_threshold_2  - number of sync bits of the unit 2
*  dma_control_channel_2 - DMA control channel number necessary for external DMA 
*                        request generation of the unit 2
*  dma_trigger_mode_2  - This parameter defines the polarity on DMA CONTROL
*                        channel of the unit 2. This parameter should be 
*                        assigned a value of:
*                        FS_ETPU_CEA709_DMA_TRIGGER_RISING_EDGE or
*                        FS_ETPU_CEA709_DMA_TRIGGER_FALLING_EDGE or,
*                        FS_ETPU_CEA709_DMA_TRIGGER_INTERNAL.
*  tx_en_channel_2     - transmit enable channel number (CP2,if required)
*                        of the unit 2;
*                        If not applicable then set to 
*                        FS_ETPU_CEA709_CHAN_NOT_USED.
*  cd_channel_2        - collision detection channel number (CP4,if required)
*                        of the unit 2;
*                        If not applicable then set to 
*                        FS_ETPU_CEA709_CHAN_NOT_USED.
*  cd_preamble_2       - this parameter determines whether the cd signal is
*                        checked at the end of the packet preamble or not; 
*                        this parameter should be assigned a value of:
*                        FS_ETPU_CEA709_CDPREAMBLE_NO or
*                        FS_ETPU_CEA709_CDPREAMBLE_YES
*  cd_tail_2           - this parameter determines whether the cd signal is
*                        checked at the end of the packet or not; this 
*                        parameter should be assigned a value of:
*                        FS_ETPU_CEA709_CDTAIL_NO or
*                        FS_ETPU_CEA709_CDTAIL_YES
*  cd_to_end_packet_2  - how close to the end of packet the cd signal
*                        is checked (in number of bits)
*
* RETURNS NOTES: Error codes which can be returned are: FS_ETPU_ERROR_VALUE,
*                FS_ETPU_ERROR_MALLOC
******************************************************************************/
int32_t fs_etpu_CEA709_init_2units( uint8_t   rx_channel_1,
                                    uint8_t   tx_channel_1,
                                    uint8_t   preamble_length_1,
                                    uint24_t  packet_cycle_1,
                                    uint24_t  beta2_control_1,
                                    uint24_t  xmit_interpacket_1,
                                    uint24_t  receive_interpacket_1,
                                    uint8_t   channel_priorities_1,
                                    uint8_t   node_priority_1,
                                    uint8_t   bit_sync_threshold_1,
                                    uint8_t   dma_control_channel_1,
                                    uint8_t   dma_trigger_mode_1,
                                    uint8_t   tx_en_channel_1,
                                    uint8_t   cd_channel_1,
                                    uint8_t   cd_preamble_1,
                                    uint8_t   cd_tail_1,
                                    uint24_t  cd_to_end_packet_1,
                                    uint8_t   rx_channel_2,
                                    uint8_t   tx_channel_2,
                                    uint8_t   preamble_length_2,
                                    uint24_t  packet_cycle_2,
                                    uint24_t  beta2_control_2,
                                    uint24_t  xmit_interpacket_2,
                                    uint24_t  receive_interpacket_2,
                                    uint8_t   channel_priorities_2,
                                    uint8_t   node_priority_2,
                                    uint8_t   bit_sync_threshold_2,
                                    uint8_t   dma_control_channel_2,
                                    uint8_t   dma_trigger_mode_2,
                                    uint8_t   tx_en_channel_2,
                                    uint8_t   cd_channel_2,
                                    uint8_t   cd_preamble_2,
                                    uint8_t   cd_tail_2,
                                    uint24_t  cd_to_end_packet_2)
{
   uint32_t * pba_1;
   uint32_t * pba_2;
   uint32_t * tx_pri_buffer_start_pointer_1;
   uint32_t * tx_nonpri_buffer_start_pointer_1;
   uint32_t * rx_buffer_start_pointer_1;
   uint32_t * tx_pri_buffer_start_pointer_2;
   uint32_t * tx_nonpri_buffer_start_pointer_2;
   uint32_t * rx_buffer_start_pointer_2;

   uint32_t etpu_tcr1_freq;

   /****************************************
    * Parameters bounds check.
    ***************************************/
   #ifdef FS_ETPU_MC_PARAM_CHECK
   if(((rx_channel_1>31)&&(rx_channel_1<64))||(rx_channel_1>95)||
      ((tx_channel_1>31)&&(tx_channel_1<64))||(tx_channel_1>95)||
      ((dma_control_channel_1>31)&&(dma_control_channel_1<64))||(dma_control_channel_1>95)||
      ((tx_en_channel_1>31)&&(tx_en_channel_1<64))||(tx_en_channel_1>95)||
      ((cd_channel_1>31)&&(cd_channel_1<64))||(cd_channel_1>95)||
      (channel_priorities_1>255)||
      (node_priority_1>255)||
      (dma_trigger_mode_1>FS_ETPU_CEA709_DMA_TRIGGER_INTERNAL)||
      ((rx_channel_2>31)&&(rx_channel_2<64))||(rx_channel_2>95)||
      ((tx_channel_2>31)&&(tx_channel_2<64))||(tx_channel_2>95)||
      ((dma_control_channel_2>31)&&(dma_control_channel_2<64))||(dma_control_channel_2>95)||
      ((tx_en_channel_2>31)&&(tx_en_channel_2<64))||(tx_en_channel_2>95)||
      ((cd_channel_2>31)&&(cd_channel_2<64))||(cd_channel_2>95)||
      (channel_priorities_2>255)||
      (node_priority_2>255)||
      (dma_trigger_mode_2>FS_ETPU_CEA709_DMA_TRIGGER_INTERNAL))
   {
      return(FS_ETPU_ERROR_VALUE);
   }
   #endif

   /****************************************
    * PRAM allocation.
    ***************************************/
   if ((pba_1=fs_etpu_malloc(FS_ETPU_CEA709_RX_NUM_PARMS))== 0)
   {
      return(FS_ETPU_ERROR_MALLOC);
   }

   /* CEA709 Buffers have to start at addresses with 0 at the end, i.e. aligned by 16 bytes */
   fs_free_param += 2;
   fs_free_param = (uint32_t*)((uint32_t)fs_free_param>>4);
   fs_free_param = (uint32_t*)((uint32_t)fs_free_param<<4);
   tx_pri_buffer_start_pointer_1 = fs_free_param;
   fs_free_param += (((48+7)>>3)<<1);

   if (fs_free_param > (uint32_t *)fs_etpu_data_ram_end)
       return(FS_ETPU_ERROR_MALLOC);

   tx_nonpri_buffer_start_pointer_1 = tx_pri_buffer_start_pointer_1 + 4;
   rx_buffer_start_pointer_1 = tx_pri_buffer_start_pointer_1 + 8;

   if ((pba_2=fs_etpu_malloc(FS_ETPU_CEA709_RX_NUM_PARMS))== 0)
   {
      return(FS_ETPU_ERROR_MALLOC);
   }

   /* CEA709 Buffers have to start at addresses with 0 at the end, i.e. aligned by 16 bytes */
   fs_free_param += 2;
   fs_free_param = (uint32_t*)((uint32_t)fs_free_param>>4);
   fs_free_param = (uint32_t*)((uint32_t)fs_free_param<<4);
   tx_pri_buffer_start_pointer_2 = fs_free_param;
   fs_free_param += (((48+7)>>3)<<1);

   if (fs_free_param > (uint32_t *)fs_etpu_data_ram_end)
       return(FS_ETPU_ERROR_MALLOC);

   tx_nonpri_buffer_start_pointer_2 = tx_pri_buffer_start_pointer_2 + 4;
   rx_buffer_start_pointer_2 = tx_pri_buffer_start_pointer_2 + 8;

   /****************************************
    * Write channel configuration registers
    * and FM (function mode) bits.
    ***************************************/
   /* RX1, TX1 CHANNEL */
   eTPU->CHAN[rx_channel_1].CR.R = (FS_ETPU_CEA709_RX_TABLE_SELECT << 24) +
                                   (FS_ETPU_CEA709_RX_FUNCTION_NUMBER << 16) +
                                   (((uint32_t)pba_1 - fs_etpu_data_ram_start) >> 3);
   eTPU->CHAN[rx_channel_1].SCR.R = 0;
   eTPU->CHAN[tx_channel_1].CR.R = (FS_ETPU_CEA709_TX_TABLE_SELECT << 24) +
                                   (FS_ETPU_CEA709_TX_FUNCTION_NUMBER << 16) +
                                   (((uint32_t)pba_1 - fs_etpu_data_ram_start) >> 3);
   eTPU->CHAN[tx_channel_1].SCR.R = 0;

   /* DMA1 CONTROL CHANNEL */
   eTPU->CHAN[dma_control_channel_1].CR.R = (FS_ETPU_CEA709_DMA_CONTROL_TABLE_SELECT << 24) +
                                            (FS_ETPU_CEA709_DMA_CONTROL_FUNCTION_NUMBER << 16) +
                                            (((uint32_t)pba_1 - fs_etpu_data_ram_start) >> 3);
   eTPU->CHAN[dma_control_channel_1].SCR.R = FS_ETPU_CEA709_DMA_CTR_FM0_INCREMENT_COUNTER_YES;
   eTPU->CHAN[rx_channel_1].SCR.R |= FS_ETPU_CEA709_FM0_DMA_CTR_ON;
   eTPU->CHAN[tx_channel_1].SCR.R |= FS_ETPU_CEA709_FM0_DMA_CTR_ON;

   /* TXEN1 CHANNEL */
   if(tx_en_channel_1!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      eTPU->CHAN[rx_channel_1].SCR.R |= FS_ETPU_CEA709_FM1_TXEN_ON;
      eTPU->CHAN[tx_channel_1].SCR.R |= FS_ETPU_CEA709_FM1_TXEN_ON;
      eTPU->CHAN[tx_en_channel_1].CR.R = (FS_ETPU_CEA709_TXEN_TABLE_SELECT << 24) +
                                         (FS_ETPU_CEA709_TXEN_FUNCTION_NUMBER << 16) +
                                         (((uint32_t)pba_1 - fs_etpu_data_ram_start) >> 3);
      eTPU->CHAN[tx_en_channel_1].SCR.R = 0;
   }

   /* CD1 CHANNEL */
   if(cd_channel_1!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      eTPU->CHAN[cd_channel_1].CR.R = (FS_ETPU_CEA709_CD_TABLE_SELECT << 24) +
                                      (FS_ETPU_CEA709_CD_FUNCTION_NUMBER << 16) +
                                      (((uint32_t)pba_1 - fs_etpu_data_ram_start) >> 3);
      eTPU->CHAN[cd_channel_1].SCR.R = 0;
   }

   /* RX2, TX2 CHANNEL */
   eTPU->CHAN[rx_channel_2].CR.R = (FS_ETPU_CEA709_RX_TABLE_SELECT << 24) +
                                   (FS_ETPU_CEA709_RX_FUNCTION_NUMBER << 16) +
                                   (((uint32_t)pba_2 - fs_etpu_data_ram_start) >> 3);
   eTPU->CHAN[rx_channel_2].SCR.R = 0;
   eTPU->CHAN[tx_channel_2].CR.R = (FS_ETPU_CEA709_TX_TABLE_SELECT << 24) +
                                   (FS_ETPU_CEA709_TX_FUNCTION_NUMBER << 16) +
                                   (((uint32_t)pba_2 - fs_etpu_data_ram_start) >> 3);
   eTPU->CHAN[tx_channel_2].SCR.R = 0;

   /* DMA2 CONTROL CHANNEL */
   eTPU->CHAN[dma_control_channel_2].CR.R = (FS_ETPU_CEA709_DMA_CONTROL_TABLE_SELECT << 24) +
                                            (FS_ETPU_CEA709_DMA_CONTROL_FUNCTION_NUMBER << 16) +
                                            (((uint32_t)pba_2 - fs_etpu_data_ram_start) >> 3);
   eTPU->CHAN[dma_control_channel_2].SCR.R = FS_ETPU_CEA709_DMA_CTR_FM0_INCREMENT_COUNTER_NO;
   eTPU->CHAN[rx_channel_2].SCR.R |= FS_ETPU_CEA709_FM0_DMA_CTR_ON;
   eTPU->CHAN[tx_channel_2].SCR.R |= FS_ETPU_CEA709_FM0_DMA_CTR_ON;

   /* TXEN2 CHANNEL */
   if(tx_en_channel_2!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      eTPU->CHAN[rx_channel_2].SCR.R |= FS_ETPU_CEA709_FM1_TXEN_ON;
      eTPU->CHAN[tx_channel_2].SCR.R |= FS_ETPU_CEA709_FM1_TXEN_ON;
      eTPU->CHAN[tx_en_channel_2].CR.R = (FS_ETPU_CEA709_TXEN_TABLE_SELECT << 24) +
                                         (FS_ETPU_CEA709_TXEN_FUNCTION_NUMBER << 16) +
                                         (((uint32_t)pba_2 - fs_etpu_data_ram_start) >> 3);
      eTPU->CHAN[tx_en_channel_2].SCR.R = 0;
   }

   /* CD2 CHANNEL */
   if(cd_channel_2!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      eTPU->CHAN[cd_channel_2].CR.R = (FS_ETPU_CEA709_CD_TABLE_SELECT << 24) +
                                      (FS_ETPU_CEA709_CD_FUNCTION_NUMBER << 16) +
                                      (((uint32_t)pba_2 - fs_etpu_data_ram_start) >> 3);
      eTPU->CHAN[cd_channel_2].SCR.R = 0;
   }

   /****************************************
    * Write parameters.
    ***************************************/
   if (rx_channel_1<64)
   {
      etpu_tcr1_freq = etpu_a_tcr1_freq;
   }
   else
   {
      etpu_tcr1_freq = etpu_b_tcr1_freq;
   }

   *(pba_1 + ((FS_ETPU_CEA709_TRANSMIT_PERIOD_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125;
   *(pba_1 + ((FS_ETPU_CEA709_STATUS_OFFSET - 1)>>2)) = 0;
   *(pba_1 + ((FS_ETPU_CEA709_BETA1_TRANSMIT_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*xmit_interpacket_1;
   *(pba_1 + ((FS_ETPU_CEA709_BETA1_RECEIVE_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*receive_interpacket_1;
   *(pba_1 + ((FS_ETPU_CEA709_BETA2_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*beta2_control_1;
   *(pba_1 + ((FS_ETPU_CEA709_CD_TO_END_PACKET_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*cd_to_end_packet_1;
   *(pba_1 + ((FS_ETPU_CEA709_PACKET_CYCLE_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*packet_cycle_1;
   *(pba_1 + ((FS_ETPU_CEA709_P_TX_PRI_BUF_START_OFFSET - 1)>>2)) = (uint24_t)((uint32_t)tx_pri_buffer_start_pointer_1 - fs_etpu_data_ram_start);
   *(pba_1 + ((FS_ETPU_CEA709_P_TX_NONPRI_BUF_START_OFFSET - 1)>>2)) = (uint24_t)((uint32_t)tx_nonpri_buffer_start_pointer_1 - fs_etpu_data_ram_start);
   *(pba_1 + ((FS_ETPU_CEA709_P_RX_BUF_START_OFFSET - 1)>>2)) = (uint24_t)((uint32_t)rx_buffer_start_pointer_1 - fs_etpu_data_ram_start);
   *(pba_1 + ((FS_ETPU_CEA709_LAST_RX_PACKET_LENGTH_OFFSET - 1)>>2)) = 0;
   *(pba_1 + ((FS_ETPU_CEA709_LAST_RX_PACKET_TIME_STAMP_HIGH_OFFSET - 1)>>2)) = 0;
   *(pba_1 + ((FS_ETPU_CEA709_LAST_RX_PACKET_TIME_STAMP_LOW_OFFSET - 1)>>2)) = 0;
   *((uint8_t*)pba_1 + FS_ETPU_CEA709_STATE_OFFSET) = 0;
   *((uint8_t*)pba_1 + FS_ETPU_CEA709_CHANNEL_ACCESS_STATE_OFFSET) = 0;
   *((uint8_t*)pba_1 + FS_ETPU_CEA709_BUFFERS_OFFSET) = 0;
   *((uint8_t*)pba_1 + FS_ETPU_CEA709_PREAMBLE_LENGTH_OFFSET) = preamble_length_1;
   *((uint8_t*)pba_1 + FS_ETPU_CEA709_NODE_PRIORITY_OFFSET) = node_priority_1;
   *((uint8_t*)pba_1 + FS_ETPU_CEA709_CHANNEL_PRIORITIES_OFFSET) = channel_priorities_1;
   *((uint8_t*)pba_1 + FS_ETPU_CEA709_BACKLOG_OFFSET) = 1;
   *((uint8_t*)pba_1 + FS_ETPU_CEA709_DMA_CONTROL_CHAN_OFFSET) = dma_control_channel_1;
   *((uint8_t*)pba_1 + FS_ETPU_CEA709_RX_CHAN_OFFSET) = rx_channel_1;
   *((uint8_t*)pba_1 + FS_ETPU_CEA709_TX_ENABLE_CHAN_OFFSET) = tx_en_channel_1;
   *((uint8_t*)pba_1 + FS_ETPU_CEA709_BIT_SYNC_THRESHOLD_OFFSET) = bit_sync_threshold_1;
   *((uint8_t*)pba_1 + FS_ETPU_CEA709_LAST_RX_PACKET_DATAGRAM_INFO_OFFSET) = 0;
   if(cd_channel_1!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      *((uint8_t*)pba_1 + FS_ETPU_CEA709_CD_OPTIONS_OFFSET) = cd_tail_1 + cd_preamble_1;
   }
   else
   {
      *((uint8_t*)pba_1 + FS_ETPU_CEA709_CD_OPTIONS_OFFSET) = 0;
   }

   *(pba_2 + ((FS_ETPU_CEA709_TRANSMIT_PERIOD_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125;
   *(pba_2 + ((FS_ETPU_CEA709_STATUS_OFFSET - 1)>>2)) = 0;
   *(pba_2 + ((FS_ETPU_CEA709_BETA1_TRANSMIT_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*xmit_interpacket_2;
   *(pba_2 + ((FS_ETPU_CEA709_BETA1_RECEIVE_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*receive_interpacket_2;
   *(pba_2 + ((FS_ETPU_CEA709_BETA2_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*beta2_control_2;
   *(pba_2 + ((FS_ETPU_CEA709_CD_TO_END_PACKET_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*cd_to_end_packet_2;
   *(pba_2 + ((FS_ETPU_CEA709_PACKET_CYCLE_OFFSET - 1)>>2)) = etpu_tcr1_freq/78125*packet_cycle_2;
   *(pba_2 + ((FS_ETPU_CEA709_P_TX_PRI_BUF_START_OFFSET - 1)>>2)) = (uint24_t)((uint32_t)tx_pri_buffer_start_pointer_2 - fs_etpu_data_ram_start);
   *(pba_2 + ((FS_ETPU_CEA709_P_TX_NONPRI_BUF_START_OFFSET - 1)>>2)) = (uint24_t)((uint32_t)tx_nonpri_buffer_start_pointer_2 - fs_etpu_data_ram_start);
   *(pba_2 + ((FS_ETPU_CEA709_P_RX_BUF_START_OFFSET - 1)>>2)) = (uint24_t)((uint32_t)rx_buffer_start_pointer_2 - fs_etpu_data_ram_start);
   *(pba_2 + ((FS_ETPU_CEA709_LAST_RX_PACKET_LENGTH_OFFSET - 1)>>2)) = 0;
   *(pba_2 + ((FS_ETPU_CEA709_LAST_RX_PACKET_TIME_STAMP_HIGH_OFFSET - 1)>>2)) = 0;
   *(pba_2 + ((FS_ETPU_CEA709_LAST_RX_PACKET_TIME_STAMP_LOW_OFFSET - 1)>>2)) = 0;
   *((uint8_t*)pba_2 + FS_ETPU_CEA709_STATE_OFFSET) = 0;
   *((uint8_t*)pba_2 + FS_ETPU_CEA709_CHANNEL_ACCESS_STATE_OFFSET) = 0;
   *((uint8_t*)pba_2 + FS_ETPU_CEA709_BUFFERS_OFFSET) = 0;
   *((uint8_t*)pba_2 + FS_ETPU_CEA709_PREAMBLE_LENGTH_OFFSET) = preamble_length_2;
   *((uint8_t*)pba_2 + FS_ETPU_CEA709_NODE_PRIORITY_OFFSET) = node_priority_2;
   *((uint8_t*)pba_2 + FS_ETPU_CEA709_CHANNEL_PRIORITIES_OFFSET) = channel_priorities_2;
   *((uint8_t*)pba_2 + FS_ETPU_CEA709_BACKLOG_OFFSET) = 1;
   *((uint8_t*)pba_2 + FS_ETPU_CEA709_DMA_CONTROL_CHAN_OFFSET) = dma_control_channel_2;
   *((uint8_t*)pba_2 + FS_ETPU_CEA709_RX_CHAN_OFFSET) = rx_channel_2;
   *((uint8_t*)pba_2 + FS_ETPU_CEA709_TX_ENABLE_CHAN_OFFSET) = tx_en_channel_2;
   *((uint8_t*)pba_2 + FS_ETPU_CEA709_BIT_SYNC_THRESHOLD_OFFSET) = bit_sync_threshold_2;
   *((uint8_t*)pba_2 + FS_ETPU_CEA709_LAST_RX_PACKET_DATAGRAM_INFO_OFFSET) = 0;
   if(cd_channel_2!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      *((uint8_t*)pba_2 + FS_ETPU_CEA709_CD_OPTIONS_OFFSET) = cd_tail_2 + cd_preamble_2;
   }
   else
   {
      *((uint8_t*)pba_2 + FS_ETPU_CEA709_CD_OPTIONS_OFFSET) = 0;
   }

   /****************************************
    * Write HSR.
    ***************************************/
   eTPU->CHAN[rx_channel_1].HSRR.R = FS_ETPU_CEA709_RX_INIT;
   eTPU->CHAN[tx_channel_1].HSRR.R = FS_ETPU_CEA709_TX_INIT;
   
   if(dma_trigger_mode_1 == FS_ETPU_CEA709_DMA_TRIGGER_FALLING_EDGE)
   {
      eTPU->CHAN[dma_control_channel_1].HSRR.R = FS_ETPU_CEA709_DMA_CONTROL_INIT_FALLING_TRIGGER;
   }
   else
   {
      eTPU->CHAN[dma_control_channel_1].HSRR.R = FS_ETPU_CEA709_DMA_CONTROL_INIT_RISING_TRIGGER;
   }
   
   if(tx_en_channel_1!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      eTPU->CHAN[tx_en_channel_1].HSRR.R = FS_ETPU_CEA709_TXEN_INIT;
   }
   
   if(cd_channel_1!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      eTPU->CHAN[cd_channel_1].HSRR.R = FS_ETPU_CEA709_CD_INIT;
   }

   eTPU->CHAN[rx_channel_2].HSRR.R = FS_ETPU_CEA709_RX_INIT;
   eTPU->CHAN[tx_channel_2].HSRR.R = FS_ETPU_CEA709_TX_INIT;
   
   if(dma_trigger_mode_2 == FS_ETPU_CEA709_DMA_TRIGGER_FALLING_EDGE)
   {
      eTPU->CHAN[dma_control_channel_2].HSRR.R = FS_ETPU_CEA709_DMA_CONTROL_INIT_FALLING_TRIGGER;
   }
   else
   {
      eTPU->CHAN[dma_control_channel_2].HSRR.R = FS_ETPU_CEA709_DMA_CONTROL_INIT_RISING_TRIGGER;
   }
   
   if(tx_en_channel_2!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      eTPU->CHAN[tx_en_channel_2].HSRR.R = FS_ETPU_CEA709_TXEN_INIT;
   }
   
   if(cd_channel_2!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      eTPU->CHAN[cd_channel_2].HSRR.R = FS_ETPU_CEA709_CD_INIT;
   }

   return(0);
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_get_tx_pri_buffer_start
*PURPOSE      : To get pointer to the beginning of the transmit priority buffer.
*INPUTS NOTES : This function has 1 parameter:
*   rx_channel       - This is the CEA709 RX channel number.
*
*RETURNS NOTES: This function returns the pointer to the beginning of the 
*               transmit priority buffer as unsigned int32.
******************************************************************************/
uint32_t fs_etpu_CEA709_get_tx_pri_buffer_start(uint8_t rx_channel)
{
   uint32_t tx_pri_buffer_start_addr;

   tx_pri_buffer_start_addr = fs_etpu_data_ram_start + 
     fs_etpu_get_chan_local_24(rx_channel, FS_ETPU_CEA709_P_TX_PRI_BUF_START_OFFSET);

   return(tx_pri_buffer_start_addr);
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_get_tx_nonpri_buffer_start
*PURPOSE      : To get pointer to the beginning of the transmit non-priority buffer.
*INPUTS NOTES : This function has 1 parameter:
*   rx_channel       - This is the CEA709 RX channel number.
*
*RETURNS NOTES: This function returns the pointer to the beginning of the 
*               transmit non-priority buffer as unsigned int32.
******************************************************************************/
uint32_t fs_etpu_CEA709_get_tx_nonpri_buffer_start(uint8_t rx_channel)
{
   uint32_t tx_nonpri_buffer_start_addr;

   tx_nonpri_buffer_start_addr = fs_etpu_data_ram_start + 
     fs_etpu_get_chan_local_24(rx_channel, FS_ETPU_CEA709_P_TX_NONPRI_BUF_START_OFFSET);

   return(tx_nonpri_buffer_start_addr);
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_get_rx_buffer_start
*PURPOSE      : To get pointer to the beginning of the receive buffer.
*INPUTS NOTES : This function has 1 parameter:
*   rx_channel       - This is the CEA709 RX channel number.
*
*RETURNS NOTES: This function returns the pointer to the beginning of the 
*               receive buffer as unsigned int32.
******************************************************************************/
uint32_t fs_etpu_CEA709_get_rx_buffer_start(uint8_t rx_channel)
{
   uint32_t rx_buffer_start_addr;

   rx_buffer_start_addr = fs_etpu_data_ram_start + 
     fs_etpu_get_chan_local_24(rx_channel, FS_ETPU_CEA709_P_RX_BUF_START_OFFSET);

   return(rx_buffer_start_addr);
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_enable
*PURPOSE      : To enable eTPU channels of the given transciever.
*INPUTS NOTES : This function has 5 parameter:
*  rx_channel - RX channel number (CP0)
*  tx_channel - TX channel number (CP1)
*  dma_control_channel - DMA control channel number necessary for external DMA request generation
*  tx_en_channel - transmit enable channel number (CP2,if required)
*  cd_channel - collision detection channel number (CP4,if required)
*
*RETURNS NOTES: 
******************************************************************************/
uint32_t fs_etpu_CEA709_enable(uint8_t   rx_channel,
                               uint8_t   tx_channel,
                               uint8_t   dma_control_channel,
                               uint8_t   tx_en_channel,
                               uint8_t   cd_channel)
{
   fs_etpu_enable(rx_channel, FS_ETPU_PRIORITY_HIGH);
   fs_etpu_enable(tx_channel, FS_ETPU_PRIORITY_HIGH);
   fs_etpu_enable(dma_control_channel, FS_ETPU_PRIORITY_LOW);

   if(tx_en_channel!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      fs_etpu_enable(tx_en_channel, FS_ETPU_PRIORITY_MIDDLE);
   }
   if(cd_channel!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      fs_etpu_enable(cd_channel, FS_ETPU_PRIORITY_MIDDLE);
   }
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_disable
*PURPOSE      : To disable eTPU channels of the given transciever.
*INPUTS NOTES : This function has 5 parameter:
*  rx_channel - RX channel number (CP0)
*  tx_channel - TX channel number (CP1)
*  dma_control_channel - DMA control channel number necessary for external DMA request generation
*  tx_en_channel - transmit enable channel number (CP2,if required)
*  cd_channel - collision detection channel number (CP4,if required)
*
*RETURNS NOTES: 
******************************************************************************/
uint32_t fs_etpu_CEA709_disable(uint8_t   rx_channel,
                                uint8_t   tx_channel,
                                uint8_t   dma_control_channel,
                                uint8_t   tx_en_channel,
                                uint8_t   cd_channel)
{
   fs_etpu_disable(rx_channel);
   fs_etpu_disable(tx_channel);
   fs_etpu_disable(dma_control_channel);

   if(tx_en_channel!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      fs_etpu_disable(tx_en_channel);
   }
   if(cd_channel!=FS_ETPU_CEA709_CHAN_NOT_USED)
   {
      fs_etpu_disable(cd_channel);
   }
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_transmit_packet
*PURPOSE      : To request for transmitting the next packet. Each time the CPU 
*               has a new data to transmit, it writes the data to the 
*               applicable tx buffer, set applicable status flags and call this 
*               function.
*INPUTS NOTES : This function has 1 parameter:
*   rx_channel       - This is the CEA709 RX channel number.
*
*RETURNS NOTES: This function returns 0 if the request for transmitting was 
*               accepted. In case tx channel has any pending HSR the request 
*               is not accepted and this function should be called again
*               later. In this case the pending HSR number is returned.
*******************************************************************************/
int32_t  fs_etpu_CEA709_transmit_packet(uint8_t rx_channel)
{
   uint32_t  hsrr;

   hsrr = eTPU->CHAN[rx_channel+1].HSRR.R;
   if (hsrr != 0) return ((int32_t)hsrr);
   fs_etpu_set_hsr(rx_channel+1, FS_ETPU_CEA709_TX_RUN);

   return(0);
    
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_get_status_flags
*PURPOSE      : To get actual MAC/Link layer status flags for callback function.
*INPUTS NOTES : This function has 1 parameter:
*   rx_channel       - This is the CEA709 RX channel number.
*
*RETURNS NOTES: This function returns the actual MAC/Link layer status flags
*               as unsigned int24.
*******************************************************************************/
uint24_t  fs_etpu_CEA709_get_status_flags(uint8_t rx_channel)
{
	return(fs_etpu_get_chan_local_24(rx_channel, FS_ETPU_CEA709_STATUS_OFFSET));
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_set_status_flag
*PURPOSE      : To set appropriate status flag.
*INPUTS NOTES : This function has 2 parameters:
*   rx_channel       - This is the CEA709 RX channel number.
*   mask             - determines which status flag should be set
*   FS_ETPU_CEA709_STATUS_RX_FRAME_END               0x000001 // End of frame = line code violation
*   FS_ETPU_CEA709_STATUS_RX_ERROR                   0x000002 // Error occurred when receiving                                      
*   FS_ETPU_CEA709_STATUS_TX_PRI_BUFFER_FULL         0x000004 // TX priority buffer contains new data to be transmitted
*   FS_ETPU_CEA709_STATUS_TX_NONPRI_BUFFER_FULL      0x000008 // TX non-priority buffer contains new data to be transmitted
*   FS_ETPU_CEA709_STATUS_TX_TRANSMISSION_BEGINNING  0x000010 // Transmission begins              
*   FS_ETPU_CEA709_STATUS_TX_TRANSMISSION_SUCCESSFUL 0x000020 // Data transmitted sucessfully
*   FS_ETPU_CEA709_STATUS_TX_ERROR                   0x000040 // Error occurred when transmitting
*   FS_ETPU_CEA709_STATUS_BACKLOG_OVERFLOW           0x000080 // Backlog overflow
*   FS_ETPU_CEA709_STATUS_COLLISION_DETECTED         0x000100 // Collision detected either during the preamble or at the end of packet
*
*RETURNS NOTES: 
*******************************************************************************/
uint32_t fs_etpu_CEA709_set_status_flag(uint8_t rx_channel, uint32_t mask)
{
   uint24_t status;
   
   status = fs_etpu_get_chan_local_24(rx_channel, FS_ETPU_CEA709_STATUS_OFFSET);
   status |= mask;
   fs_etpu_set_chan_local_24(rx_channel, FS_ETPU_CEA709_STATUS_OFFSET,
                             (uint24_t)status);
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_clear_status_flag
*PURPOSE      : To clear appropriate status flag.
*INPUTS NOTES : This function has 2 parameters:
*   rx_channel       - This is the CEA709 RX channel number.
*   mask             - determines which status flag should be cleared
*   FS_ETPU_CEA709_STATUS_RX_FRAME_END               0x000001 // End of frame = line code violation
*   FS_ETPU_CEA709_STATUS_RX_ERROR                   0x000002 // Error occurred when receiving                                      
*   FS_ETPU_CEA709_STATUS_TX_PRI_BUFFER_FULL         0x000004 // TX priority buffer contains new data to be transmitted
*   FS_ETPU_CEA709_STATUS_TX_NONPRI_BUFFER_FULL      0x000008 // TX non-priority buffer contains new data to be transmitted
*   FS_ETPU_CEA709_STATUS_TX_TRANSMISSION_BEGINNING  0x000010 // Transmission begins              
*   FS_ETPU_CEA709_STATUS_TX_TRANSMISSION_SUCCESSFUL 0x000020 // Data transmitted sucessfully
*   FS_ETPU_CEA709_STATUS_TX_ERROR                   0x000040 // Error occurred when transmitting
*   FS_ETPU_CEA709_STATUS_BACKLOG_OVERFLOW           0x000080 // Backlog overflow
*   FS_ETPU_CEA709_STATUS_COLLISION_DETECTED         0x000100 // Collision detected either during the preamble or at the end of packet
*
*RETURNS NOTES: 
*******************************************************************************/
uint32_t fs_etpu_CEA709_clear_status_flag(uint8_t rx_channel, uint32_t mask)
{
   uint24_t status;
   
   status = fs_etpu_get_chan_local_24(rx_channel, FS_ETPU_CEA709_STATUS_OFFSET);
   status &= ~mask;
   fs_etpu_set_chan_local_24(rx_channel, FS_ETPU_CEA709_STATUS_OFFSET,
                             (uint24_t)status);
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_get_last_rx_packet_length
*PURPOSE      : To get the length of the last received packet.
*INPUTS NOTES : This function has 1 parameter:
*   rx_channel       - This is the CEA709 RX channel number.
*
*RETURNS NOTES: This function returns the length of the last received packet
*               as unsigned int24.
*******************************************************************************/
uint24_t  fs_etpu_CEA709_get_last_rx_packet_length(uint8_t rx_channel)
{
   return(fs_etpu_get_chan_local_24(rx_channel, FS_ETPU_CEA709_LAST_RX_PACKET_LENGTH_OFFSET));
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_get_last_rx_packet_time_stamp_high24
*PURPOSE      : To get high 24 bits of the last received packet timestamp.
*INPUTS NOTES : This function has 1 parameter:
*   rx_channel       - This is the CEA709 RX channel number.
*
*RETURNS NOTES: This function returns high 24 bits of the last received packet
*               timestamp as unsigned int24.
*******************************************************************************/
uint24_t  fs_etpu_CEA709_get_last_rx_packet_time_stamp_high24(uint8_t rx_channel)
{
   return(fs_etpu_get_chan_local_24(rx_channel, FS_ETPU_CEA709_LAST_RX_PACKET_TIME_STAMP_HIGH_OFFSET));
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_get_last_rx_packet_time_stamp_low24
*PURPOSE      : To get low 24 bits of the last received packet timestamp.
*INPUTS NOTES : This function has 1 parameter:
*   rx_channel       - This is the CEA709 RX channel number.
*
*RETURNS NOTES: This function returns low 24 bits of the last received packet
*               timestamp as unsigned int24.
*******************************************************************************/
uint24_t  fs_etpu_CEA709_get_last_rx_packet_time_stamp_low24(uint8_t rx_channel)
{
   return(fs_etpu_get_chan_local_24(rx_channel, FS_ETPU_CEA709_LAST_RX_PACKET_TIME_STAMP_LOW_OFFSET));
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_get_last_rx_packet_datagram_info
*PURPOSE      : To get the additional datagram info of the last received packet.
*INPUTS NOTES : This function has 1 parameter:
*   rx_channel       - This is the CEA709 RX channel number.
*
*RETURNS NOTES: This function returns:
*               0x00 - if the last received datagram is valid
*               0x01 - if the last received datagram is invalid - incorrect CRC
*******************************************************************************/
uint8_t  fs_etpu_CEA709_get_last_rx_packet_datagram_info(uint8_t rx_channel)
{
   return(fs_etpu_get_chan_local_8(rx_channel, FS_ETPU_CEA709_LAST_RX_PACKET_DATAGRAM_INFO_OFFSET));
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_get_timestamp_counter_value_high24
*PURPOSE      : To get high 24 bits of the timestamp counter value.
*INPUTS NOTES : This function has 1 parameter:
*   rx_channel       - This is the CEA709 RX channel number.
*
*RETURNS NOTES: High 24 bits of the timestamp counter value is returned as uint24_t.
*******************************************************************************/
uint24_t fs_etpu_CEA709_get_timestamp_counter_value_high24(uint8_t rx_channel)
{
   return(fs_etpu_get_global_24(FS_ETPU_CEA709_TIME_STAMP_COUNTER_HIGH24_OFFSET));
}

/*******************************************************************************
*FUNCTION     : fs_etpu_CEA709_get_timestamp_counter_value_low24
*PURPOSE      : To get low 24 bits of the timestamp counter value.
*INPUTS NOTES : This function has 1 parameter:
*   rx_channel       - This is the CEA709 RX channel number.
*
*RETURNS NOTES: Low 24 bits of the timestamp counter value is returned as uint24_t.
*******************************************************************************/
uint24_t fs_etpu_CEA709_get_timestamp_counter_value_low24(uint8_t rx_channel)
{
   return(eTPU->TB1R_A.R);
}

/*********************************************************************
 *
 * Copyright:
 *  Freescale Semiconductor, INC. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Freescale
 *  Semiconductor, Inc. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, Freescale
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  Freescale Semiconductor assumes no responsibility for the
 *  maintenance and support of this software
 ********************************************************************/
