/*
 * Timer.c
 *
 *  Created on: Jan 18, 2016
 *      Author: B55840
 */

#include <hidef.h>
#include "derivative.h"
#include "Timer.h"

#ifndef TIMER_C_
#define TIMER_C_


/* Service Interrupt Routines Callbacks */
void (*TIM0_CH0_callback)(unsigned int);
void (*TIM0_CH1_callback)(unsigned int);
void (*TIM1_CH0_callback)(unsigned int);
void (*TIM1_CH1_callback)(unsigned int);

/********************************* General timer functions **************************************************************************************************/

/**	Initializes the desired timer and sets its prescaler
 * 
 * 	This function initializes the desired timer, set the callback pointers to 0 and the prescaler of the timer.
 *	The prescaler must be selected from the options specified in the header file.
 * 	
 *	@param TIM_select						Desired timer to be configured. TIM0/TIM1/TIM2
 *	@param bus_divider_selection			Selection of the prescaler for the timer frequency
 *	 
 * */
 
void TIM_init(unsigned char TIM_select, TIM_frequency bus_divider_selection){
	TIM0_CH0_callback = 0;
	TIM0_CH1_callback = 0;
	TIM1_CH0_callback = 0;
	TIM1_CH1_callback = 0;
	
	if(TIM_select==0){
		TIM0TSCR2_PR = bus_divider_selection;
	}
	else if(TIM_select==1){
		TIM1TSCR2_PR = bus_divider_selection;
	}
}



/**	Configures the seleted timer's channel as desired
 * 
 * 	This function initializes the desired timer, set the callback pointers to 0 and the prescaler of the timer.
 *	The prescaler must be selected from the options specified in the header file.
 * 	
 *	@param TIM_select						Desired timer to be configured. TIM0/TIM1/TIM2
 *	@param bus_divider_selection			Selection of the prescaler for the timer frequency
 *	 
 * */

void TIM_channel_config(unsigned char TIM_select, unsigned char channel_select, TIM_CONFIG channel_configuration){
	if(TIM_select==0){
		if(channel_configuration == TIM_IC_INT_EN){
			TIM0TIOS |= (0x00 << channel_select); // 0x00 for IC
			TIM0TIE |= (0x01 << channel_select); // Interrupt on channel enabled
		}
		else if(channel_configuration == TIM_IC_INT_DIS){
			TIM0TIOS |= (0x00 << channel_select); // 0x00 for IC
			TIM0TIE |= (0x00 << channel_select); // Interrupt on channel disabled
		}
		else if(channel_configuration == TIM_OC_INT_EN){
			TIM0TIOS |= (0x01 << channel_select); // 0x01 for OC
			TIM0TIE |= (0x01 << channel_select); // Interrupt on channel enabled
		}
		else if(channel_configuration == TIM_OC_INT_DIS){
			TIM0TIOS |= (0x01 << channel_select); // 0x01 for OC
			TIM0TIE |= (0x00 << channel_select); // Interrupt on channel disabled
		}
	}
	else if(TIM_select==1){
		if(channel_configuration == TIM_IC_INT_EN){
			TIM1TIOS |= (0x00 << channel_select); // 0x00 for IC
			TIM1TIE |= (0x01 << channel_select); // Interrupt on channel enabled
		}
		else if(channel_configuration == TIM_IC_INT_DIS){
			TIM1TIOS |= (0x00 << channel_select); // 0x00 for IC
			TIM1TIE |= (0x00 << channel_select); // Interrupt on channel disabled
		}
		else if(channel_configuration == TIM_OC_INT_EN){
			TIM1TIOS |= (0x01 << channel_select); // 0x01 for OC
			TIM1TIE |= (0x01 << channel_select); // Interrupt on channel enabled
		}
		else if(channel_configuration == TIM_OC_INT_DIS){
			TIM1TIOS |= (0x01 << channel_select); // 0x01 for OC
			TIM1TIE |= (0x00 << channel_select); // Interrupt on channel disabled
		}
	}
}

void TIM_run(unsigned char TIM_select){
	if(TIM_select==0){
		TIM0TSCR1_TEN = 1;
	}
	else if(TIM_select==1){
		TIM1TSCR1_TEN = 1;
	}
}

unsigned int TIM_get_time(unsigned int TIM_select){
	if(TIM_select==0){
		return TIM0TCNT;
	}
	else if(TIM_select==1){
		return TIM1TCNT;
	}
}
unsigned char TIM_OF_status(unsigned char TIM_select){
	if(TIM_select==0){
		return TIM0TFLG2_TOF;
	}
	else if(TIM_select==1){
		return TIM1TFLG2_TOF;
	}
}

/* Output compare functions */
void TIM_OC_set_compare(unsigned char TIM_select, unsigned char channel_select, unsigned int compare_time){
	if(TIM_select==0){
		if(channel_select==0){
			TIM0TC0 = compare_time;
		}
		else if(channel_select==1){
			TIM0TC1 = compare_time;
		}
	}
	else if(TIM_select==1){
		if(channel_select==0){
			TIM1TC0 = compare_time;
		}
		else if(channel_select==1){
			TIM1TC1 = compare_time;
		}
	}
}
unsigned char TIM_OC_flag_status(unsigned char TIM_select, unsigned char channel_select){
	if(TIM_select==0){
		if(TIM0TFLG1>>channel_select){
			TIM0TFLG1 = TIM0TFLG1&(0x01<<channel_select); // clears flag
			return 1;
		}
		return 0;
	}
	else if(TIM_select==1){
		if(TIM1TFLG1>>channel_select){
			TIM1TFLG1 = TIM1TFLG1&(0x01<<channel_select); // clears flag
			return 1;
		}
		return 0;
	}
}

/* Input capture functions */
void TIM_IC_capture_config(unsigned char TIM_select, unsigned char channel_select,TIM_IC_MODE input_capture_mode){
	if(TIM_select==0){
		if(input_capture_mode == TIM_IC_CAP_DIS){
			TIM0TCTL4 |= (0x00<<(2*channel_select)); 
		}
		else if(input_capture_mode == TIM_IC_CAP_RIS){
			TIM0TCTL4 |= (0x01<<(2*channel_select)); 
		}
		else if(input_capture_mode == TIM_IC_CAP_FALL){
			TIM0TCTL4 |= (0x10<<(2*channel_select)); 
		}
		else if(input_capture_mode == TIM_IC_CAP_ANY){
			TIM0TCTL4 |= (0x11<<(2*channel_select)); 
		}
	}
	else if(TIM_select==1){
		if(input_capture_mode == TIM_IC_CAP_DIS){
			TIM1TCTL4 |= (0x00<<(2*channel_select)); 
		}
		else if(input_capture_mode == TIM_IC_CAP_RIS){
			TIM1TCTL4 |= (0x01<<(2*channel_select)); 
		}
		else if(input_capture_mode == TIM_IC_CAP_FALL){
			TIM1TCTL4 |= (0x10<<(2*channel_select)); 
		}
		else if(input_capture_mode == TIM_IC_CAP_ANY){
			TIM1TCTL4 |= (0x11<<(2*channel_select)); 
		}
	}
}
unsigned char TIM_IC_flag_status(unsigned char TIM_select, unsigned char channel_select){
	if(TIM_select==0){
		if(TIM0TFLG1>>channel_select){
			TIM0TFLG1 = TIM0TFLG1&(0x01<<channel_select); // clears flag
			return 1;
		}
		return 0;
	}
	else if(TIM_select==1){
		if(TIM1TFLG1>>channel_select){
			TIM1TFLG1 = TIM1TFLG1&(0x01<<channel_select); // clears flag
			return 1;
		}
		return 0;
	}
}

/*callback function*/

void TIM0_set_CH0_callback(void (*ptr)(unsigned int)){
	TIM0_CH0_callback = ptr;
}
void TIM0_set_CH1_callback(void (*ptr)(unsigned int)){
	TIM0_CH1_callback = ptr;
}
void TIM1_set_CH0_callback(void (*ptr)(unsigned int)){
	TIM1_CH0_callback = ptr;
}
void TIM1_set_CH1_callback(void (*ptr)(unsigned int)){
	TIM1_CH1_callback = ptr;
}


// Channels interruptions

void interrupt VectorNumber_Vtim0ch0 TIM0_CH0_ISR(){
	if(TIM0_CH0_callback!=0){
		TIM0_CH0_callback(TIM_get_time(TIM0));
	}
	TIM0TFLG1_C0F = 1;
}
void interrupt VectorNumber_Vtim0ch1 TIM0_CH1_ISR(){
	if(TIM0_CH1_callback!=0){
		TIM0_CH1_callback(TIM_get_time(TIM0));
	}
	TIM0TFLG1_C1F = 1;
}
void interrupt VectorNumber_Vtim1ch0 TIM1_CH0_ISR(){
	if(TIM1_CH0_callback!=0){
		TIM1_CH0_callback(TIM_get_time(TIM1));
	}
	TIM1TFLG1_C0F = 1;
}
void interrupt VectorNumber_Vtim1ch1 TIM1_CH1_ISR(){
	if(TIM1_CH1_callback!=0){
		TIM1_CH1_callback(TIM_get_time(TIM1));
	}
	TIM1TFLG1_C1F = 1;
}

#endif /* TIMER_C_ */
