/*
 * LIN.c
 *
 *  Created on: May 10, 2016
 *      Author: B55840
 */

#include "LIN.h"
#include "Timer.h"
#include "derivative.h"

/* change according to baud rate */
#define bit_time 104


/***********************************************************************************************
*
* @brief    LIN_init  - Initialize the LIN phy communication
* @param    none
* @return   none
*
************************************************************************************************/
void LIN_init(){
	/* Timer initialization */
	TIM_init(TIM0,bus_clk_div_32);
	TIM_channel_config(TIM0,0,TIM_OC_INT_DIS);
	TIM_run(TIM0);
	
	MODRR0_S0L0RR=0; 
	LP0CR_LPPUE = 1;    /* Pull up to strong signal  */
	LP0SLRM = 0x01;		/* LIN Slew Rate selection */
	/* With the LIN baudrate selected as 9600bps, 
    the Slew Rate is set as 01 which is optimized for 10,400 Bit/s */
	LP0CR_LPE = 1;      /* Enable LIN Phy  */

}

/***********************************************************************************************
*
* @brief    LIN_send_data - Send a LIN frame
* @param    ID of the frame, number of bytes to be transmitted, data to be transmited
* @return   none
*
************************************************************************************************/
void LIN_send_data(unsigned char ID, unsigned char data_lenght, unsigned char data[]){
	unsigned char parity0,parity1,temp,i;
	unsigned int checksum;
	SCI0_send_break(); /* send break field */
	SCI0_send_char(0x55); /* send sync field */
	
	/* calculate parity */
	parity0 = ((ID)^(ID>>1)^(ID>>2)^(ID>>4))&0x01;
	parity1 = ((~((ID>>1)^(ID>>3)^(ID>>4)^(ID>>5)))&0x01);
	
	temp = ID|(parity0<<6)|(parity1<<7);
	SCI0_send_char(temp); /* send ID with parity */
	for(i=0;i<data_lenght;i++){
		SCI0_send_char(data[i]);
	}
	
	/* Checksum calculation */
	checksum = temp;
	while(data_lenght>0){
		data_lenght--;
		checksum+=data[data_lenght];
		if(checksum>0xFF){
			checksum-=0xFF;
		}
	}
	checksum = ~(unsigned char)checksum; 
	SCI0_send_char(checksum);
	
}

/***********************************************************************************************
*
* @brief    LIN_read_char - Read a response from an slave
* @param    ID of the header
* @return   Slave response to that ID
*
************************************************************************************************/
unsigned char LIN_read_char(unsigned char ID){
	unsigned char parity0,parity1,temp,i;
	unsigned char message_received_flag = 0;
	unsigned char slave_message[2];;
	while(message_received_flag==0){
		SCI0_send_break();
		SCI0_send_char(0x55);
		parity0 = ((ID)^(ID>>1)^(ID>>2)^(ID>>4))&0x01;
		parity1 = ((~((ID>>1)^(ID>>3)^(ID>>4)^(ID>>5)))&0x01);
		temp = ID|(parity0<<6)|(parity1<<7);
		SCI0_send_char(temp);
		while(SCI0SR1_TDRE==0){}
		SCI0CR2_RE = 1;				/* Enable Rx */
		SCI0SR1;
		SCI0DRL;
		message_received_flag = 1;
		TIM0TFLG1_C0F = 1;
		TIM0TC0 = TIM0TCNT+bit_time*20;
		/* receive data and checksum */
		for(i = 0; i<2;i++){
			while(SCI0SR1_RDRF==0){
				if(TIM0TFLG1){
					message_received_flag = 0;
					i=1;
					break;
				}
			}
			slave_message[i] = SCI0DRL;
		}


		SCI0CR2_RE = 0;				/* disable receiver */
		if(slave_message[0]==0x041){
			message_received_flag = 1;
		}
		else{
			SCI0_send_char(~(temp)); /* send checksum */
			delay_us(50000); 	/* delay of 50 ms for next transmision of reception frame */ 
		}

	}
	return slave_message[0];
}

/***********************************************************************************************
*
* @brief    delay_us - Generate a delay of the desire microseconds
* @param    amount of micro seconds
* @return   none
*
************************************************************************************************/
void delay_us(unsigned int micro_seconds){
	TIM_OC_flag_status(TIM0,0); /* Clear flag if set */
	TIM_OC_set_compare(TIM0,0,micro_seconds+TIM_get_time(TIM0));
	while(TIM_OC_flag_status(TIM0,0)==0){}
}

