/*******************************************************************************
 *
 * Freescale Semiconductor Inc.
 * (c) Copyright 2015 Freescale Semiconductor, Inc.
 * ALL RIGHTS RESERVED.
 *
 ****************************************************************************//*!
*
* @file    reacm2_util.h
*
* @author  Marketa Venclikova [b52107]
* 
* @version 1.0
* 
* @date    11-Nov-2015
*
* @brief   This file contains useful functions for using the RECM2
* 
****************************************************************************//*!
*
* @mainpage
*
* The REACM2 Utilities @ref reacm2_util.c/.h includes low-level functions for
* using the REACM2 module. The upper software levels should prevent
* from direct access to REACM2 registers by using these utilities. Note that these 
* routines can only be used to program REACM2 module, they might not be compatible 
* with previous versions of REACM.
*
* Routines can be classified into several groups according to application usage:
*-# REACM2 Module and channel initialization
* - @ref fs_reacm2_module_init
* - @ref fs_reacm2_channel_init
*-# REACM2 Run-time module control
*   - @ref fs_reacm2_module_enable, @ref fs_reacm2_module_disable
* - @ref fs_reacm2_hold_timer_enable, @ref fs_reacm2_hold_timer_disable
* - @ref fs_reacm2_shared_timer_enable, fs_reacm2_shared_timer_disable
* - @ref fs_reacm2_clear_adc_ovr_flag
*-# REACM2 Run-time channel control
* - @ref fs_reacm2_channel_enable, @ref fs_reacm2_channel_disable
* - @ref fs_reacm2_channel_sw_start, @ref fs_reacm2_channel_sw_stop
*-# single REACM2 module settings
* - @ref fs_reacm2_set_hold_timer_prs
* - @ref fs_reacm2_set_shared_timer_prs
* - @ref fs_reacm2_period_generator_config
* - @ref fs_reacm2_period_shift_delay_set
* - @ref fs_reacm2_adc_max_limit_set
* - @ref fs_reacm2_set_modulation_range
* - @ref fs_reacm2_threshold_router_set
*-# REACM2 Modulation Control Words setting
* - @ref fs_reacm2_modword_set
* - @ref fs_reacm2_modword_bank_status
* - @ref fs_reacm2_modword_countpointer_thrs
* - @ref fs_reacm2_modword_countpointer_shtime
* - @ref fs_reacm2_modword_countpointer_holdtime
*-# signle REACM2 threshold and timers value settings
* - @ref fs_reacm2_threshold_set_reg
* - @ref fs_reacm2_threshold_bank_status
* - @ref fs_reacm2_sh_time_set_reg
* - @ref fs_reacm2_sh_time_bank_status
* - @ref fs_reacm2_hold_time_set_reg
* - @ref fs_reacm2_hold_time_bank_status
*******************************************************************************/
/*******************************************************************************
 * Includes
 *******************************************************************************/
#include "reacm2_util.h"    /* prototypes and useful defines */

/*******************************************************************************
 * Global variables
 *******************************************************************************/
uint8_t mod_words_top = 0; // top of the register bank
uint8_t threshold_top = 0; // top of the register bank
uint8_t sh_time_top = 0;   // top of the register bank
uint8_t hold_time_top = 0; // top of the register bank
 
 /*******************************************************************************
* FUNCTION: fs_reacm2_module_init
****************************************************************************//*!
* @brief    This function initializes the REACM2 module.
*
* @note   Following actions are performed:
*     -# Initialize global module registers:
*       - Module Control Register MCR
*       - Timer Configuration Register TCR
*       - Threshold Router Register THRR
*       - Period Generator Configuration Register PCR
*       - Period Shift Delay Configuration Register PSCR
*       - ADC Result Maximum Limit Check ADCMAX
*       - Modulation Range Pulse Width RANGE_PWD
*       - Modulation Minimum Pulse Width MIN_PWD
* @param  p_reacm2_init - pointer to  a configuration structure to
*     initialize REACM2. This structure has to be created prior this
*     function call.  
* @warning  This function does not perform pin configuration, only REACM2
*     module initialization.  
*******************************************************************************/
void fs_reacm2_module_init(REACM2_INIT_T * p_reacm2_init)
{
  // clear the overrun flag here
  REACM2.MCR.B.MDIS = TRUE;
  REACM2.MCR.R = p_reacm2_init->mcr;
  REACM2.TCR.R = p_reacm2_init->tcr;
  REACM2.THRR.R = p_reacm2_init->thrr;
  REACM2.PCR.R = p_reacm2_init->pcr;
  REACM2.PSCR.R = p_reacm2_init->pscr;
  REACM2.ADCMAX.R = p_reacm2_init->adcmax;
  REACM2.RANGE_PWD.R = p_reacm2_init->range_pwd;
  REACM2.MIN_PWD.R = p_reacm2_init->min_pwd;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_channel_init
****************************************************************************//*!
* @brief  This function configures the REACM2 channel.
*
* @note   This function should be called after REACM2 module initialization and 
*     Modulation words settings. Following actions are performed:
*     -# Channel flags are cleared in Channel Status Register
*     -# Channel n Router Register is set (selection of the
*     time window signal input for channel n and ADC result
*     router is set - to be compared with incoming TAG)
*     -# Channel n Configuration Register is set
* @param  ch_num - index of the channel to be set
* @param  p_channel_init - this is a pointer to channel configuration
*     structure where input,ADC result TAG, control mode and
*     interrupts are specified
* @param  modulation_adress - address of the first modulation word to be
*     executed by channel 
*******************************************************************************/
void fs_reacm2_channel_init( 
  uint8_t ch_num,
  REACM2_CHANNEL_INIT_T *p_channel_init,
  uint8_t modulation_adress
)
{
  REACM2.CH[ch_num].CHSR.R = FS_REACM2_CHSR_CLEAR_ALL;  // clear all the flags
  REACM2.CH[ch_num].CHRR.R = p_channel_init->chrr;
  REACM2.CH[ch_num].CHCR.R = (p_channel_init->chcr | FS_REACM2_MOD_ADDR(modulation_adress)) & 0x3FFFFFFF;
} 

/*******************************************************************************
* FUNCTION: fs_reacm2_modword_set
****************************************************************************//*!
* @brief  This function performs settings of modulation words specified by
*     the structure MODULATION_WORD_T phases.                       
* @note   Following actions are performed:
*     -# checking the Modulation Control word Bank, Threshold
*     bank, Shared Timer Bank and Hold Timer Bank availability
*     -# storage of threshold and timer values into respective
*     banks
*     -# creation and storage of the modulation words into bank 
* @param  phases - pointer to the array of structures that contain 
*     parameters of the modulation words to be stored in Modulation
*     Control Word bank
* @param  num_of_phases - number of modulation words to be stored
* @return   if there is enough empty registers for storage the function
*     performs storage and returns index of the modulation word
*     register where the first modulation word of the phases was 
*     stored, otherwise it returns ERROR code
*******************************************************************************/
int8_t fs_reacm2_modword_set(MODULATION_WORD_T * phases, uint8_t num_of_phases)
{
  uint8_t i;

  if (fs_reacm2_modword_bank_status(num_of_phases)&& fs_reacm2_threshold_bank_status(fs_reacm2_modword_countpointer_thrs(phases, num_of_phases)) && fs_reacm2_sh_time_bank_status(fs_reacm2_modword_countpointer_shtime(phases, num_of_phases)) && fs_reacm2_hold_time_bank_status(fs_reacm2_modword_countpointer_holdtime(phases, num_of_phases))) // if there is an empty space in modword bank to write all the phases..
  {
    for (i = 0; i < num_of_phases; i++) 
    {

      REACM2.MWBK[mod_words_top].R = phases[i].loopback |
      phases[i].init_output_val |
      phases[i].mod_mode |
      phases[i].seq_mode |
      phases[i].hod_c |
      phases[i].hod_b |
      phases[i].hod_a |
      phases[i].lod_c |
      phases[i].lod_b |
      phases[i].lod_a;
      if(phases[i].thrs_val_1 != NULL)
      {
        if(phases[i].thrs_val_1->idx == NAN)
        {
          phases[i].thrs_val_1->idx = fs_reacm2_threshold_set_reg(phases[i].thrs_val_1->thrs, threshold_top);
          threshold_top++;
        }
        REACM2.MWBK[mod_words_top].B.THRESPT= phases[i].thrs_val_1->idx;
      }

      if(phases[i].thrs_val_2 != NULL)
      {
        if(phases[i].thrs_val_2->idx == NAN)
        {
          phases[i].thrs_val_2->idx = fs_reacm2_threshold_set_reg(phases[i].thrs_val_2->thrs, threshold_top);  
          threshold_top++;
        }
      }

      if(phases[i].sh_time_val != NULL)
      {
        if(phases[i].sh_time_val->idx == NAN)
        {
          phases[i].sh_time_val->idx = fs_reacm2_sh_time_set_reg(phases[i].sh_time_val->sh_time, sh_time_top);
          sh_time_top++;
        }
        REACM2.MWBK[mod_words_top].B.STPT= phases[i].sh_time_val->idx;
      }

      if(phases[i].hold_time_val != NULL)
      {
        if(phases[i].hold_time_val->idx == NAN)
        {
          phases[i].hold_time_val->idx = fs_reacm2_hold_time_set_reg(phases[i].hold_time_val->hold_time, hold_time_top);
          hold_time_top++; 
        }
        REACM2.MWBK[mod_words_top].B.HDOFFTPT = phases[i].hold_time_val->idx;
      }

      mod_words_top++;
    }
    return mod_words_top - num_of_phases;
  }
  else
  {
    return ERROR;
  }
}

/*******************************************************************************
* FUNCTION: fs_reacm2_threshold_set_reg
****************************************************************************//*!
* @brief  This function performs storage of the value into the Threshold Bank 
*     on a position determined by idx
* @param  value - threshold value to be stored in Threshold Bank
* @param  idx - index of the Threshold Bank register where the threshold value 
*     is to be stored
* @return index of the register, where the value was stored
*******************************************************************************/
uint8_t fs_reacm2_threshold_set_reg ( uint16_t value, uint8_t idx)
{
  REACM2.THBK[idx].B.THRESHOLD_VALUE = value;
  return idx;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_sh_time_set_reg
****************************************************************************//*!
* @brief  This function performs storage of the value into the Shared Timer Bank 
*     on a position determined by idx
* @param  value - time value to be stored in Shared Timer Bank
* @param  idx - index of the Shared Timer Bank register where the time value
*     is to be stored
* @return index of the register, where the value was stored
*******************************************************************************/
uint8_t fs_reacm2_sh_time_set_reg ( uint16_t value, uint8_t idx)
{
  REACM2.STBK[idx].B.SHARED_TIMER = value;
  return idx;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_hold_time_set_reg
****************************************************************************//*!
* @brief  This function performs storage of the value into the Hold Timer Bank 
*     on a position determined by idx
* @param  value - time value to be stored in Hold Timer Bank
* @param  idx - index of the Hold Timer Bank register where the time value is 
*     to be stored
* @return index of the register, where the value was stored
*******************************************************************************/
uint8_t fs_reacm2_hold_time_set_reg ( uint16_t value, uint8_t idx)
{
  REACM2.HOTBK[idx].B.HOLD_TIM = value;
  return idx;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_modword_countpointer_thrs
****************************************************************************//*!
* @brief  This function counts the number of the threshold values in array 
*     "phases" that are to be stored into the Threshold Bank
* @param  phases - pointer to the array of structures that contain 
*     parameters of the modulation words to be stored in Modulation
*     Control Word bank
* @param  length - number of the consecutive modulation words to be stored
* @return   number that corresponds to the count of threshold values that are to
*     be stored into the Threshold Bank
*******************************************************************************/
int8_t fs_reacm2_modword_countpointer_thrs (MODULATION_WORD_T * phases, uint8_t length)
{
  uint8_t temp_top = 0;
  uint16_t temp[32];
  int i;

  for (i = 0;i < length;i++)
  {
    if (phases[i].thrs_val_1  != NULL)
    {
      temp[temp_top++] = phases[i].thrs_val_1->thrs;
    }
    if (phases[i].thrs_val_2 != NULL)
    {
      temp[temp_top++] = phases[i].thrs_val_2->thrs;
    }
  }
  return temp_top;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_modword_countpointer_shtime
****************************************************************************//*!
* @brief  This function counts the number of the shared time values in array 
*     "phases" that are to be stored into the Shared Timer Bank
* @param  phases - pointer to the array of structures that contain 
*     parameters of the modulation words to be stored in Modulation
*     Control Word bank
* @param  length - number of the consecutive modulation words to be stored
* @return   number that corresponds to the count of shared time values that are to
*     be stored into the Shared Timer Bank
*******************************************************************************/
int8_t fs_reacm2_modword_countpointer_shtime(MODULATION_WORD_T * phases, uint8_t length)
{
  uint8_t temp_top = 0;
  uint16_t temp[5];
  int i;

  for (i = 0;i < length;i++)
  {
    if (phases[i].sh_time_val != NULL)
    {
      temp[temp_top++] = phases[i].sh_time_val->sh_time;
    }
  }
  return temp_top;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_modword_countpointer_holdtime
****************************************************************************//*!
* @brief  This function counts the number of the hold time values in array 
*     "phases" that are to be stored into the Hold Timer Bank
* @param  phases - pointer to the array of structures that contain 
*     parameters of the modulation words to be stored in Modulation
*     Control Word bank
* @param  length - number of the consecutive modulation words to be stored
* @return   number that corresponds to the count of hold time values that are to
*     be stored into the Hold Timer Bank
*******************************************************************************/
int8_t fs_reacm2_modword_countpointer_holdtime(MODULATION_WORD_T * phases, uint8_t length)
{
  uint8_t temp_top = 0;
  uint16_t temp[5];
  int i;

  for (i = 0;i < length;i++)
  {
    if (phases[i].hold_time_val != NULL)
    {
      temp[temp_top++] = phases[i].hold_time_val->hold_time;
    }
  }
  return temp_top;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_modword_bank_status
****************************************************************************//*!
* @brief  This function is to find out whether new modulation words can be 
*     stored into the Modulation Control Word Bank
* @param  length - number of the consecutive modulation words to be stored
* @return   status of the Modulation Control Word Bank : either FS_REACM2_BANK_FULL (all 
*     the modulation words cannot be stored) or FS_REACM2_BANK_OK (if there is an 
*     empty space to store all consecutive modulation words)
*******************************************************************************/
uint8_t fs_reacm2_modword_bank_status(uint8_t length)
{
  if ((mod_words_top + length) > FS_REACM2_MOD_WORD_BANK_SIZE)
  {
    return FS_REACM2_BANK_FULL;
  }
  else
  {
    return FS_REACM2_BANK_OK;
  }
}

/*******************************************************************************
* FUNCTION: fs_reacm2_threshold_bank_status
****************************************************************************//*!
* @brief  This function is to find out whether new threshold values can be 
*     stored into the Threshold Bank
* @param  count - number of threshold values to be stored in Threshold Bank
* @return   status of the Threshold Bank : either FS_REACM2_BANK_FULL (all the thresholds
*     cannot be stored) or FS_REACM2_BANK_OK (if there is an empty space to store
*      all thresholds)
*******************************************************************************/
uint8_t fs_reacm2_threshold_bank_status(uint8_t count)
{
  if ((threshold_top + count) > FS_REACM2_THRESHOLD_BANK_SIZE)
  {
    return FS_REACM2_BANK_FULL;
  }
  else
  {
    return FS_REACM2_BANK_OK;
  }
}

/*******************************************************************************
* FUNCTION: fs_reacm2_sh_time_bank_status
****************************************************************************//*!
* @brief  This function is to find out whether new time values can be stored 
*     into the Shared Timer Bank
* @param  count - number of time values to be stored in Shared Timer Bank
* @return   status of the Shared Timer Bank : either FS_REACM2_BANK_FULL (all the time
*     values cannot be stored) or FS_REACM2_BANK_OK (if there is an empty space 
*     to store all time values)
*******************************************************************************/
uint8_t fs_reacm2_sh_time_bank_status(uint8_t count)
{
  if ((sh_time_top + count) > FS_REACM2_SH_TIMER_BANK_SIZE)
  {
    return FS_REACM2_BANK_FULL;
  }
  else
  {
    return FS_REACM2_BANK_OK;
  }
}

/*******************************************************************************
* FUNCTION: fs_reacm2_hold_time_bank_status
****************************************************************************//*!
* @brief  This function is to find out whether new time values can be stored 
*     into the Hold Timer Bank
* @param  count - number of time values to be stored in Hold Timer Bank
* @return   status of the Hold Timer Bank : either FS_REACM2_BANK_FULL (all the time
*     values cannot be stored) or FS_REACM2_BANK_OK (if there is an empty space 
*     to store all time values)
*******************************************************************************/
uint8_t fs_reacm2_hold_time_bank_status(uint8_t count)
{
  if ((hold_time_top + count) > FS_REACM2_HOLD_TIMER_BANK_SIZE)
  {
    return FS_REACM2_BANK_FULL;
  }
  else
  {
    return FS_REACM2_BANK_OK;
  }
}

 /*******************************************************************************
* FUNCTION: fs_reacm2_module_enable
****************************************************************************//*!
* @brief  This function enables REACM2 module
*******************************************************************************/
void fs_reacm2_module_enable(void)
{
  REACM2.MCR.B.MDIS = FALSE;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_module_disable
****************************************************************************//*!
* @brief  This function disables REACM2 module
*******************************************************************************/
void fs_reacm2_module_disable(void)
{
  REACM2.MCR.B.MDIS = TRUE;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_hold_timer_enable
****************************************************************************//*!
* @brief  This function enables hold timer
*******************************************************************************/
void fs_reacm2_hold_timer_enable(void)
{
  REACM2.MCR.B.HPREN = TRUE;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_hold_timer_disable
****************************************************************************//*!
* @brief  This function disables hold timer
*******************************************************************************/
void fs_reacm2_hold_timer_disable(void)
{
  REACM2.MCR.B.HPREN = FALSE;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_shared_timer_enable
****************************************************************************//*!
* @brief  This function enables shared timer
*******************************************************************************/
void fs_reacm2_shared_timer_enable(void)
{
  REACM2.MCR.B.TPREN = TRUE;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_shared_timer_disable
****************************************************************************//*!
* @brief  This function disables shared timer
*******************************************************************************/
void fs_reacm2_shared_timer_disable(void)
{
  REACM2.MCR.B.TPREN = FALSE;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_set_hold_timer_prs
****************************************************************************//*!
* @brief  This function is to set the prescaler of the timer input clock
* @note   As soon as timer prescaler is enabled timer starts new count sequence
* @param  prs_value - resultant prescaler value to scale timer clock, the  
*     actual value written to the register is lowered by one 
*******************************************************************************/
void fs_reacm2_set_hold_timer_prs(uint16_t prs_value)
{
  REACM2.TCR.B.HPRE = prs_value - 1;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_set_shared_timer_prs
****************************************************************************//*!
* @brief  This function is to set the prescaler of the timer input clock
* @note   As soon as timer prescaler is enabled timer starts new count sequence
* @param  prs_value - resultant prescaler value to scale timer clock, the  
*     actual value written to the register is lowered by one 
*******************************************************************************/
void fs_reacm2_set_shared_timer_prs(uint8_t prs_value)
{
  REACM2.TCR.B.TPRE = prs_value - 1;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_clear_adc_ovr_flag
****************************************************************************//*!
* @brief  This function clears ADC overrun flag
* @note   The OVR flag is used to indicate that an overrun condition was 
*     detected at the ADC Interface
*******************************************************************************/
void fs_reacm2_clear_adc_ovr_flag(void)
{
  REACM2.MCR.B.OVRC = 1;
  while( REACM2.GEFR.B.OVR == 1); //wait for flag to be cleared
}

/*******************************************************************************
* FUNCTION: fs_reacm2_period_generator_config
****************************************************************************//*!
* @brief  This function configures period generator in PCR register
* @note   Configuration of the period generator is necessary for 
*     threshold/period modulation
* @param  per_presc - period prescaler rate value, to be used by the Period
*     counter and Delay counter. If per_presc value is 0, the rate is 1, 
*       and so forth. Maximum PERPRESC value is 255, thus the maximum scale 
*     rate is 256
* @param  period - the value for the internal period generator counter. When 
*     it is set to 0, the counter is stopped. If it is not 0, the period 
*     time in clock cycles is given by multiplying (PER+1) with the 
*     prescaler value(PERPRESC+1).    
*******************************************************************************/  
void fs_reacm2_period_generator_config(uint8_t per_presc, uint16_t period)
{
  REACM2.PCR.B.PERPRESC = per_presc - 1; 
  REACM2.PCR.B.PER = period;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_period_shift_delay_set
****************************************************************************//*!
* @brief  This function configures shift delay in PSCR register
* @note   Configuration of the shift delay is used to set delay time between
*     period pulses of two consecutive channels. The delay multiplied by 
*     the number of channels must be less than or equal to the period.
* @param  delay_value - Delay value between two consecutive channel period pulses. 
*     The correct time depends on the prescaler counter. Also,if DLY=0 then 
*     the period pulses of all reaction channels occur at the same time. 
*     If DLY is not all 0, then the shift delay time in clock cycles is 
*     given by multiplying DLY with the prescaler (PERPRESC+1).
*******************************************************************************/  
void fs_reacm2_period_shift_delay_set(uint16_t delay_value)
{
  REACM2.PSCR.B.DLY = delay_value;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_adc_max_limit_set
****************************************************************************//*!
* @brief  This function sets the maximum limit allowed for the ADC result
* @note   The Reaction Module ADC Maximum Limit Check register holds the
*     maximum expected value of the ADC result. If a value greater than
*     or equal to ADC_MAX_LIMIT is received the MAXL bit in the 
*     corresponding Channel Status Register Error is asserted.
* @param  max_limit - the ADC_MAX_LIMIT value indicates the maximum expected 
*     value for the ADC result.
*******************************************************************************/
void fs_reacm2_adc_max_limit_set(uint16_t max_limit)
{
  REACM2.ADCMAX.B.ADC_MAX_LIMIT = max_limit;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_set_modulation_range
****************************************************************************//*!
* @brief  This function configures the pulse width range of the modulation
* @note   The REACM2 Range PulseWidth Detection Check registers provide the 
*     value used to check if the PWM pulse width generated during the 
*     modulation process is larger than a maximum pulse width specified 
*     by (MIN_PWD + RANGE_PWD).
* @param  range - defines the maximum pulse width allowed by the channel PWM 
*     generation.
* @param  minimum - defines the minimum pulse width allowed by the channel 
*     PWM generation
*******************************************************************************/
void fs_reacm2_set_modulation_range(uint16_t range, uint16_t minimum)
{
  REACM2.RANGE_PWD.B.RANGE_PWD = range;
  REACM2.MIN_PWD.B.MIN_PWD = minimum;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_threshold_router_set
****************************************************************************//*!
* @brief  This function configures routing of the ADC result to the Threshold 
*     bank
* @note   Since a TAG is assigned for each ADC result coming from the on-chip
*     ADC module, this TAG is specified in this register and used for the 
*     routing process. The ADC result is written to the Threshold Value 
*     Bank as soon as it is received by the Reaction Module.
* @param  wren - specifies, whether the write into threshold bank on the
*     address 0 or 1 is enabled or disabled 
* @param  adc_tag - specifies the TAG of the ADC result to be written into 
*     Threshold Bank
*******************************************************************************/
void fs_reacm2_threshold_router_set (uint32_t wren, uint32_t adc_tag)
{
  REACM2.THRR.R = wren | adc_tag ;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_channel_enable
****************************************************************************//*!
* @brief  This function enables the channel specified with ch_num in the 
*     pre-set control mode (timer control / sw control)
* @note   The mode is pre-set by the fs_reacm2_channel_set function call
* @param  ch_num - number of channel to be enabled
* @param  p_channel_init - this is a pointer to channel configuration
*     structure where input,ADC result TAG, control mode and
*     interrupts are specified
*******************************************************************************/
void fs_reacm2_channel_enable(uint8_t ch_num, REACM2_CHANNEL_INIT_T *p_channel_init )
{
  REACM2.CH[ch_num].CHCR.B.CHEN |= p_channel_init->chcr >> 30;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_channel_disable
****************************************************************************//*!
* @brief  This function disables the channel specified with ch_num 
* @param  ch_num - number of channel to be disabled
*******************************************************************************/
void fs_reacm2_channel_disable(uint8_t ch_num)
{
  REACM2.CH[ch_num].CHCR.B.CHEN = FS_REACM2_DISABLE;    
}

/*******************************************************************************
* FUNCTION: fs_reacm2_channel_sw_start
****************************************************************************//*!
* @brief  This function call initiates the channel modulation when channel is
*     enabled in software mode control only
* @param  ch_num - number of channel where modulation is to be started
*******************************************************************************/
void fs_reacm2_channel_sw_start(uint8_t ch_num)
{
  REACM2.CH[ch_num].CHCR.B.SWMC = FS_REACM2_ENABLE;
}

/*******************************************************************************
* FUNCTION: fs_reacm2_channel_sw_stop
****************************************************************************//*!
* @brief  This function call stops the channel modulation when channel is
*     enabled in software mode control only
* @param  ch_num - number of channel where modulation is to be stopped
*******************************************************************************/
void fs_reacm2_channel_sw_stop(uint8_t ch_num)
{
  REACM2.CH[ch_num].CHCR.B.SWMC = FS_REACM2_DISABLE;
}

/*******************************************************************************
 *
 * Copyright:
 *  Freescale Semiconductor, INC. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Freescale
 *  Semiconductor, Inc. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, Freescale
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  Freescale Semiconductor assumes no responsibility for the
 *  maintenance and support of this software
 ******************************************************************************/
/*******************************************************************************
 *
 * REVISION HISTORY:
 *
 * FILE OWNER: Marketa Venclikova [b52107]
 *
 * Revision 1.0  2015/11/11  b52107
 * Initial version.
 *
 ******************************************************************************/
