/*************************************************************************//**
 * @file SIPI.c
 * @copyright Freescale 2014 All Rights Reserved                             
 * @version 1.0                                                              
 * @brief This file contains application examples for API function calls 
 * @date 22-Nov-14
 * @author MSG-GDL                                                          */
/*===========================================================================*
 * UPDATE HISTORY                                                            *
 * REV      AUTHOR      DATE       	DESCRIPTION OF CHANGE                    *
 * ---   -----------  ---------    	---------------------                    * 
 * 1.0     MSG-GDL    22-Nov-12     Original source from GDL                 *
 *                                                                           *
 *===========================================================================*
 * COPYRIGHT:                                                                *
 *  Freescale Semiconductor, INC. All Rights Reserved. You are hereby        *
 *  granted a copyright license to use, modify, and distribute the           *
 *  SOFTWARE so long as this entire notice is retained without alteration    *
 *  in any modified and/or redistributed versions, and that such modified    *
 *  versions are clearly identified as such. No licenses are granted by      *
 *  implication, estoppel or otherwise under any patentsor trademarks        *
 *  of Freescale Semiconductor, Inc. This software is provided on an         *
 *  "AS IS" basis and without warranty.                                      *
 *                                                                           *
 *  To the maximum extent permitted by applicable law, Freescale             *
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,       *
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A         *
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH REGARD     *
 *  TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF) AND ANY        *
 *  ACCOMPANYING WRITTEN MATERIALS.                                          *
 *                                                                           *
 *  To the maximum extent permitted by applicable law, IN NO EVENT           *
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER       *
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,     *
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER            *
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.     *
 *                                                                           *
 *  Freescale Semiconductor assumes no responsibility for the                *
 *  maintenance and support of this software                                 *
 *                                                                           *
 ****************************************************************************/

/***************Files to be included******************************************/
#include "Zipwire.h"
#include "LFAST.h"
#include "SIPI_API.h"
#include "SIPI_HSSL_Header_v4.h"

/*****************************************************************************/
extern void LFAST_Configure(unsigned char master);

#pragma ghs section data=".sipi_data"
/** readAddress, writeAddress and temp_data are placed in a specific area on memory to ensure reads and writes are fixed to those addresses.*/
volatile unsigned long readAddress = 0xAABBCCDD;
volatile unsigned long writeAddress = 0xFFEEDDCC;
uint32_t temp_data[8] = {0};
uint32_t temp_data_ID[8] = {0};
uint32_t temp_data_multiple_read_no_dma[8] = {0};
uint32_t temp_data_multiple_write_no_dma[8] = {0};
uint32_t temp_data_stream[8] = {0};
uint32_t temp_data_multiple_write_dma[8] = {0};
#pragma ghs section data=default



DATA_TEMPLATE_t test_data_structure, test_data_structure_array[8], test_data_structure_array_multiple_read_no_dma[8],test_data_structure_array_stream[8],
                test_data_structure_array_multiple_write_no_dma[8];
CHANNEL_t test_channel_structure;


unsigned char sipi_app_note_read(void);
unsigned char sipi_app_note_write(void);
unsigned char sipi_app_note_multiple_read_no_dma(void);
unsigned char sipi_app_note_multiple_write_no_dma(void);
unsigned char sipi_app_note_multiple_write_dma(void);
unsigned char sipi_app_note_stream_write(void);
unsigned char sipi_app_note_ID(void);
unsigned char sipi_app_note_event(void);


/** \fn sipi_app_note_read(void)

\brief Exercises the SIPI_read() and SIPI_read_channel_data functions to send a read request and verifies that the read values are the expected values.
\return 
	- 0 = Successful
	- 1 = Invalid Width
	- 2 = Channel Busy
	- 3 = Invalid Channel
	- 4 = Timeout Error
	- 10 = Wrong read value
*/
unsigned char sipi_app_note_read(void)
{
  uint8_t test_result;
  test_result = 0;
  
  // LFAST_Configure(CONFIGURED_AS_MASTER);
  
  #if(1==CONFIGURED_AS_MASTER)
    //Assign a known pattern to the address that will be read
    readAddress = 0x00112233;
    SIPI_init_INITIATOR(1024);
    test_data_structure.Size = 32;
    test_data_structure.Address = (uint32_t)&readAddress;
    test_channel_structure.Number = 0;
    test_result =  SIPI_read(test_data_structure, test_channel_structure, 0);

    if(test_result!=0)
    {
      dummyPrintf("READ OPERATION FAILED");
      return(test_result);
    }
    readAddress = SIPI_read_channel_data(test_channel_structure);
    if(readAddress!= 0x00224466)
    {
      dummyPrintf("READ FAILED");
      return(10);
    }
  #else
    //Assign a known pattern that will be read by the initiator
    readAddress = 0x00224466;
    SIPI_init_TARGET(0, 0, 0);
  #endif
  return(test_result);
}
/** \fn sipi_app_note_write(void)

\brief Exercises the SIPI_write(), SIPI_read() and SIPI_read_channel_data functions to send a write request and then verify that the address was actually written.
\return
	- 0 = Successfully Set Up
	- 1 = Invalid Data Size
	- 2 = Channel Busy
	- 3 = Invalid Channel
	- 4 = Timeout Error / Wrong Acknowledge
	- 10 = Wrong Read Value
*/
unsigned char sipi_app_note_write(void)
{
  uint8_t test_result;
  test_result = 0;
  // LFAST_Configure(CONFIGURED_AS_MASTER);
  
  #if(1==CONFIGURED_AS_MASTER)
    //Assign a known pattern to write in the target address
    writeAddress = 0x00112233;
    readAddress = 0x00224466;
    SIPI_init_INITIATOR(1024);
    test_data_structure.Size = 32;
    test_data_structure.Address = (uint32_t)&writeAddress;
    test_data_structure.Data = (uint32_t)writeAddress;
    test_channel_structure.Number = 0;

    test_result = SIPI_write(test_data_structure, test_channel_structure, 0);
    if(test_result!=0)
    {
      dummyPrintf("WRITE TRANSFER FAILED");
      return(test_result);
    }
    
    //Verify data was correctly written, verify in the target that data was written.
    test_data_structure.Size = 32;
    test_data_structure.Address = (uint32_t)&writeAddress;
    test_channel_structure.Number = 0;

    test_result = SIPI_read(test_data_structure, test_channel_structure, 0);
    
    if(test_result!=0)
    {
      dummyPrintf("READ TRANSFER FAILED");
      return(test_result);
    }
    
    //Read what was read from the target device
    readAddress = SIPI_read_channel_data(test_channel_structure);
    if(readAddress!= 0x00112233)
    {
      dummyPrintf("READ FAILED");
      return(10);
    }
    
  #else
    writeAddress = 0x00224466;
    readAddress = 0x00224466;
    SIPI_init_TARGET(0, 0, 0);
  #endif
  return(test_result);
}


/** \fn sipi_app_note_ID(void)

\brief Exercises the SIPI_ID() and SIPI_read_channel_data functions to send the ID command and read the CAR to verify that the ID was received.
\return
	- 0 = Successfully Set Up
	- 2 = Channel Busy
	- 3 = Invalid Channel
	- 4 = Timeout Error
	- 10 = ID Not received properly
*/
unsigned char sipi_app_note_ID(void)
{

  uint8_t test_result;
  test_result = 0;
  
  // LFAST_Configure(CONFIGURED_AS_MASTER);

  #if(1==CONFIGURED_AS_MASTER)
    readAddress = 0x00224466;
    SIPI_init_INITIATOR(1024);
    temp_data_ID[0] = 0x00224466;
    
    test_channel_structure.Number = 0;
    test_result = SIPI_ID(temp_data_ID, test_channel_structure);
    
    /* Clear flags */
    SIPI_0.CCR0.R = 0;
    SIPI_0.CSR0.B.RAR = 1;
    
    if(0!=test_result)
    {
      dummyPrintf("ID TRANSFER FAIL");
      return(test_result);
    }
    readAddress = SIPI_read_channel_data(test_channel_structure);
    if(0x00224466 == readAddress)
    {
      dummyPrintf("ID NOT RECEIVED");
      return(10);
    }
    dummyPrintf("ID RECEIVED");
  #else
    readAddress = 0x00112233;
    SIPI_init_TARGET(0, 0, 0);
  #endif
  return(test_result);
}

/** \fn sipi_app_note_event(void)

\brief Exercises SIPI_Trigger() to verify that a trigger event can be initiated using the SIPI interface.
\return 
	- 0 = Successfully Set Up
	- 1 = Incorrect Channel 
	- 2 = Channel Busy
*/
unsigned char sipi_app_note_event(void)
{
  uint8_t test_result;
  
  // LFAST_Configure(CONFIGURED_AS_MASTER);
  
  #if(1==CONFIGURED_AS_MASTER)
    uint8_t trigger_index;
    test_result = 0;
    SIPI_init_INITIATOR(1024);
    for(trigger_index=0;trigger_index<4;trigger_index++)
    {
      test_channel_structure.Number = trigger_index;

      test_result = SIPI_Trigger(test_channel_structure);
      if(0!=test_result)
      {
        dummyPrintf("TEST FAILED");
        return(test_result);
      }
      //Acknowledge Properly received
      dummyPrintf("ACKNOWLEDGE RECEIVED");
    }
    
    /* Clear flags after success */
    SIPI_0.CCR0.R = 0;
    SIPI_0.CCR1.R = 0;
    SIPI_0.CCR2.R = 0;
    SIPI_0.CCR3.R = 0;
    SIPI_0.CSR0.R = 0xF8;
    SIPI_0.CSR1.R = 0xF8;
    SIPI_0.CSR2.R = 0xF8;
    SIPI_0.CSR3.R = 0xF8;
    
  #else
    test_result = 0;
    SIPI_init_TARGET(0, 0, 0);
    
    //Wait until Initiator sends the command
    while(SIPI_0.SR.B.TE!=0x0F){}
    SIPI_0.SR.R = 0xFFFF;
    
    dummyPrintf("TARGET EVENT SENT");
  #endif
  return(test_result);
}


/** \fn sipi_app_note_multiple_read_no_dma(void)

\brief Exercises SIPI_multiple_read() without using DMA support. Each array will be read using CPU and the read contents will be verified.
\return 
	- 0 = Successfully Set Up
	- 1 = Incorrect Channel 
	- 2 = Channel Busy
	- 4 = Timeout Error
	- 10 = Data not read properly
*/
unsigned char sipi_app_note_multiple_read_no_dma(void)
{
  uint8_t test_result;
  test_result = 0;
  // LFAST_Configure(CONFIGURED_AS_MASTER);
  
  #if(1==CONFIGURED_AS_MASTER)
    SIPI_init_INITIATOR(1024);
    test_data_structure_array_multiple_read_no_dma[0].Data = 0x11111111;
    test_data_structure_array_multiple_read_no_dma[1].Data = 0x22222222;
    test_data_structure_array_multiple_read_no_dma[2].Data = 0x33333333;
    test_data_structure_array_multiple_read_no_dma[3].Data = 0x44444444;

    test_data_structure_array_multiple_read_no_dma[0].Address = (uint32_t)&test_data_structure_array_multiple_read_no_dma[0].Data;
    test_data_structure_array_multiple_read_no_dma[1].Address = (uint32_t)&test_data_structure_array_multiple_read_no_dma[1].Data;
    test_data_structure_array_multiple_read_no_dma[2].Address = (uint32_t)&test_data_structure_array_multiple_read_no_dma[2].Data;
    test_data_structure_array_multiple_read_no_dma[3].Address = (uint32_t)&test_data_structure_array_multiple_read_no_dma[3].Data;

    test_data_structure_array_multiple_read_no_dma[0].Size = 32;
    test_data_structure_array_multiple_read_no_dma[1].Size = 32;
    test_data_structure_array_multiple_read_no_dma[2].Size = 32;
    test_data_structure_array_multiple_read_no_dma[3].Size = 32;

    test_channel_structure.Number = 0;

    test_result = SIPI_multiple_read(test_data_structure_array_multiple_read_no_dma, 4, test_channel_structure, 0, temp_data_multiple_read_no_dma);
    if(0!=test_result)
    {
      dummyPrintf("TEST FAILED");
      return(test_result);
    }
    
    if((temp_data_multiple_read_no_dma[0] != 0x55555555)||
       (temp_data_multiple_read_no_dma[1] != 0x66666666)||
       (temp_data_multiple_read_no_dma[2] != 0x77777777)||
       (temp_data_multiple_read_no_dma[3] != 0x88888888))
    {
      dummyPrintf("MULTIPLE READ FAIL");
      return(10);
    }
    dummyPrintf("ALL READS OK");
  
  #else
    SIPI_init_TARGET(0, 0, 0);
    test_data_structure_array_multiple_read_no_dma[0].Data = 0x55555555;
    test_data_structure_array_multiple_read_no_dma[1].Data = 0x66666666;
    test_data_structure_array_multiple_read_no_dma[2].Data = 0x77777777;
    test_data_structure_array_multiple_read_no_dma[3].Data = 0x88888888;
  #endif
  return(test_result);
}

/** \fn sipi_app_note_multiple_write_no_dma(void)

\brief Exercises SIPI_multiple_write() and SIPI_multiple_read() to send multiple write requests. Once completed, will read the written content to verify that the data was written correctly.
\return 
	- 0 = Successfully Set Up
	- 1 = Incorrect Channel 
	- 2 = Channel Busy
	- 4 = Timeout Error
	- 10 = Data not read properly
*/

unsigned char sipi_app_note_multiple_write_no_dma(void)
{
  uint8_t test_result;
  
  // LFAST_Configure(CONFIGURED_AS_MASTER);
  
  #if(1==CONFIGURED_AS_MASTER)
    uint8_t data_length, fill_index;
    test_result = 0;
    SIPI_init_INITIATOR(1024);

    test_data_structure_array_multiple_write_no_dma[0].Data = 0x11111111;
    test_data_structure_array_multiple_write_no_dma[1].Data = 0x22222222;
    test_data_structure_array_multiple_write_no_dma[2].Data = 0x33333333;
    test_data_structure_array_multiple_write_no_dma[3].Data = 0x44444444;

    test_data_structure_array_multiple_write_no_dma[0].Address = (uint32_t)&test_data_structure_array_multiple_write_no_dma[0].Data;
    test_data_structure_array_multiple_write_no_dma[1].Address = (uint32_t)&test_data_structure_array_multiple_write_no_dma[1].Data;
    test_data_structure_array_multiple_write_no_dma[2].Address = (uint32_t)&test_data_structure_array_multiple_write_no_dma[2].Data;
    test_data_structure_array_multiple_write_no_dma[3].Address = (uint32_t)&test_data_structure_array_multiple_write_no_dma[3].Data;

    //Prepare this data in a temporary array, so it is DMA friendly
    data_length = 8;

    for(fill_index=0; fill_index<data_length; fill_index+=2)
    {
      temp_data_multiple_write_no_dma[fill_index] = test_data_structure_array_multiple_write_no_dma[fill_index/2].Data;
      temp_data_multiple_write_no_dma[fill_index+1] = test_data_structure_array_multiple_write_no_dma[fill_index/2].Address;
    }
    
    test_data_structure_array_multiple_write_no_dma[0].Size = 32;
    test_data_structure_array_multiple_write_no_dma[1].Size = 32;
    test_data_structure_array_multiple_write_no_dma[2].Size = 32;
    test_data_structure_array_multiple_write_no_dma[3].Size = 32;

    test_channel_structure.Number = 0;

    test_result = SIPI_multiple_write(test_data_structure_array_multiple_write_no_dma, 4, test_channel_structure, 0, 0, temp_data_multiple_write_no_dma);
    if(0!=test_result)
    {
      dummyPrintf("MULTIPLE WRITE FAILED");
      return(test_result);
    }
    //Read received data
    //Assign a multiple but arbitrary data to the temp_structure that will be read
    temp_data_multiple_write_no_dma[0] = 0x55555555;
    temp_data_multiple_write_no_dma[1] = 0x66666666;
    temp_data_multiple_write_no_dma[2] = 0x77777777;
    temp_data_multiple_write_no_dma[3] = 0x88888888;

    test_result = SIPI_multiple_read(test_data_structure_array_multiple_write_no_dma, 4, test_channel_structure, 0, temp_data_multiple_write_no_dma);
    if(0!=test_result)
    {
      dummyPrintf("MULTIPLE READ FAILED");
      return(test_result);
    }
    
    if((temp_data_multiple_write_no_dma[0] != 0x11111111)||
       (temp_data_multiple_write_no_dma[1] != 0x22222222)||
       (temp_data_multiple_write_no_dma[2] != 0x33333333)||
       (temp_data_multiple_write_no_dma[3] != 0x44444444))
    {
      dummyPrintf("MULTIPLE READ FAIL");
      return(10);
    }
    dummyPrintf("ALL READS OK");
  
  #else
    test_result = 0;
    test_data_structure_array_multiple_write_no_dma[0].Data = 0x55555555;
    test_data_structure_array_multiple_write_no_dma[1].Data = 0x66666666;
    test_data_structure_array_multiple_write_no_dma[2].Data = 0x77777777;
    test_data_structure_array_multiple_write_no_dma[3].Data = 0x88888888;
    SIPI_init_TARGET(0, 0, 0);
  #endif
  return(test_result);
}

/** \fn sipi_app_note_multiple_write_dma(void)

\brief Exercises SIPI_multiple_write() using DMA support and SIPI_multiple_read() to send multiple write requests. Once completed, will read the written content to verify that the data was written correctly.
\return 
	- 0 = Successfully Set Up
	- 1 = Incorrect Channel 
	- 2 = Channel Busy
	- 4 = Timeout Error
	- 10 = Data not read properly
*/

unsigned char sipi_app_note_multiple_write_dma(void)
{
  uint8_t test_result;
  
    
  // LFAST_Configure(CONFIGURED_AS_MASTER);
  
  #if(1==CONFIGURED_AS_MASTER)
    uint8_t data_length, fill_index;
    test_result = 0;
    SIPI_init_INITIATOR(1024);

    test_data_structure_array[0].Data = 0x11111111;
    test_data_structure_array[1].Data = 0x22222222;
    test_data_structure_array[2].Data = 0x33333333;
    test_data_structure_array[3].Data = 0x44444444;

    test_data_structure_array[0].Address = (uint32_t)&test_data_structure_array[0].Data;
    test_data_structure_array[1].Address = (uint32_t)&test_data_structure_array[1].Data;
    test_data_structure_array[2].Address = (uint32_t)&test_data_structure_array[2].Data;
    test_data_structure_array[3].Address = (uint32_t)&test_data_structure_array[3].Data;

    //Prepare this data in a temporary array, so it is DMA friendly
    data_length = 8;

    for(fill_index=0; fill_index<data_length; fill_index+=2)
    {
      temp_data[fill_index] = test_data_structure_array[fill_index/2].Data;
      temp_data[fill_index+1] = test_data_structure_array[fill_index/2].Address;
    }
    
    test_data_structure_array[0].Size = 32;
    test_data_structure_array[1].Size = 32;
    test_data_structure_array[2].Size = 32;
    test_data_structure_array[3].Size = 32;

    test_channel_structure.Number = 0;

    test_result = SIPI_multiple_write(test_data_structure_array, 4, test_channel_structure, 0, 1, temp_data);
    if(0!=test_result)
    {
      dummyPrintf("MULTIPLE WRITE FAILED");
      return(test_result);
    }
    
    //Assign a multiple but arbitrary data to the temp_structure that will be read
    temp_data[0] = 0x55555555;
    temp_data[1] = 0x66666666;
    temp_data[2] = 0x77777777;
    temp_data[3] = 0x88888888;

    test_result = SIPI_multiple_read(test_data_structure_array, 4, test_channel_structure, 0, temp_data);
    if(0!=test_result)
    {
      dummyPrintf("MULTIPLE READ FAILED");
      return(test_result);
    }
    
    if((temp_data[0] != 0x11111111)||
       (temp_data[1] != 0x22222222)||
       (temp_data[2] != 0x33333333)||
       (temp_data[3] != 0x44444444))
    {
      dummyPrintf("MULTIPLE READ FAIL");
      return(10);
    }
    dummyPrintf("ALL READS OK");
  
  #else
    test_result = 0;
    test_data_structure_array[0].Data = 0x55555555;
    test_data_structure_array[1].Data = 0x66666666;
    test_data_structure_array[2].Data = 0x77777777;
    test_data_structure_array[3].Data = 0x88888888;
    SIPI_init_TARGET(0, 0, 0);
  
  #endif
  return(test_result);
}

/** \fn sipi_app_note_stream_write(void)

\brief Exercises SIPI_stream() and SIPI_multiple_read() to stream write a series of words and then read them back to verify the data.
\return 
	- 0 = Successfully Set Up
	- 1 = Incorrect Channel 
	- 2 = Channel Busy
	- 4 = Timeout Error
	- 10 = Data not read properly
*/

unsigned char sipi_app_note_stream_write(void)
{
  // LFAST_Configure(CONFIGURED_AS_MASTER);

  #if(1==CONFIGURED_AS_MASTER)
  
    temp_data_stream[0] = 0x11111111;
    temp_data_stream[1] = 0x22222222;
    temp_data_stream[2] = 0x33333333;
    temp_data_stream[3] = 0x44444444;
    temp_data_stream[4] = 0x55555555;
    temp_data_stream[5] = 0x66666666;
    temp_data_stream[6] = 0x77777777;
    temp_data_stream[7] = 0x88888888;
    //Stream data
    SIPI_stream_transfer(temp_data_stream, CONFIGURED_AS_MASTER, 8);

    //Verify that stream transfer worked.
    temp_data_stream[0] = 0xAAAAAAAA;
    temp_data_stream[1] = 0xAAAAAAAA;
    temp_data_stream[2] = 0xAAAAAAAA;
    temp_data_stream[3] = 0xAAAAAAAA;
    temp_data_stream[4] = 0xAAAAAAAA;
    temp_data_stream[5] = 0xAAAAAAAA;
    temp_data_stream[6] = 0xAAAAAAAA;
    temp_data_stream[7] = 0xAAAAAAAA;

    test_data_structure_array_stream[0].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[1].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[2].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[3].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[4].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[5].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[6].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[7].Data = 0xAAAAAAAA;

    test_data_structure_array_stream[0].Size = 32;
    test_data_structure_array_stream[1].Size = 32;
    test_data_structure_array_stream[2].Size = 32;
    test_data_structure_array_stream[3].Size = 32;
    test_data_structure_array_stream[4].Size = 32;
    test_data_structure_array_stream[5].Size = 32;
    test_data_structure_array_stream[6].Size = 32;
    test_data_structure_array_stream[7].Size = 32;

    test_data_structure_array_stream[0].Address = (uint32_t)&temp_data_stream[0];
    test_data_structure_array_stream[1].Address = (uint32_t)&temp_data_stream[1];
    test_data_structure_array_stream[2].Address = (uint32_t)&temp_data_stream[2];
    test_data_structure_array_stream[3].Address = (uint32_t)&temp_data_stream[3];
    test_data_structure_array_stream[4].Address = (uint32_t)&temp_data_stream[4];
    test_data_structure_array_stream[5].Address = (uint32_t)&temp_data_stream[5];
    test_data_structure_array_stream[6].Address = (uint32_t)&temp_data_stream[6];
    test_data_structure_array_stream[7].Address = (uint32_t)&temp_data_stream[7];

    test_channel_structure.Number = 2;
    
    SIPI_multiple_read(test_data_structure_array_stream, 8, test_channel_structure, 0, temp_data_stream);

    if((temp_data_stream[0]!=0x11111111) ||
       (temp_data_stream[1]!=0x22222222) ||
       (temp_data_stream[2]!=0x33333333) ||
       (temp_data_stream[3]!=0x44444444) ||
       (temp_data_stream[4]!=0x55555555) ||
       (temp_data_stream[5]!=0x66666666) ||
       (temp_data_stream[6]!=0x77777777) ||
       (temp_data_stream[7]!=0x88888888))
    {
      dummyPrintf("STREAM WRITE FAIL");
      return(2);
    }

    dummyPrintf("STREAM WRITE PASS");
    return(0);

  #else
    test_data_structure_array_stream[0].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[1].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[2].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[3].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[4].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[5].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[6].Data = 0xAAAAAAAA;
    test_data_structure_array_stream[7].Data = 0xAAAAAAAA;

    temp_data_stream[0] = 0xAAAAAAAA;
    temp_data_stream[1] = 0xAAAAAAAA;
    temp_data_stream[2] = 0xAAAAAAAA;
    temp_data_stream[3] = 0xAAAAAAAA;
    temp_data_stream[4] = 0xAAAAAAAA;
    temp_data_stream[5] = 0xAAAAAAAA;
    temp_data_stream[6] = 0xAAAAAAAA;
    temp_data_stream[7] = 0xAAAAAAAA;

    SIPI_stream_transfer(temp_data_stream, CONFIGURED_AS_MASTER, 8);
  return(0);
  #endif
}

void dummyPrintf (const char * format)
{
    const char *data;
    data = &format[0];
    data++;
}
